//	Program name..	Zinc Interface Library
//	Filename......	UI_DSP.HPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef UI_DSP_HPP
#define UI_DSP_HPP
#include <ui_gen.hpp>

/* ----- Definition of the UI_PALETTE structure ----- */
#define attrib(foreground, background)	(((background) << 4) + (foreground))

struct UI_PALETTE
{
	/* Text mode */
	UCHAR fillCharacter;		// Fill character.
	UCHAR colorAttribute;		// Color attribute.
	UCHAR monoAttribute;		// Mono attribute.

	/* Graphics mode */
	UCHAR fillPattern;			// Fill pattern.
	UCHAR color;				// Foreground and background EGA/VGA colors.
	UCHAR bwColor;				// Black & White colors (2 color).
	UCHAR grayScaleColor;		// Monochrome colors (3+ color).
};

extern UI_PALETTE *_backgroundPalette;

/* ----- Definition of the DISPLAY_REGION class ----- */
class UI_REGION_ELEMENT : public UI_ELEMENT
{
public:
	int screenID;
	UI_REGION region;

	UI_REGION_ELEMENT(int a_screenID, const UI_REGION *a_region) :
		screenID(a_screenID), region(*a_region) { }
	UI_REGION_ELEMENT(int a_screenID, int left, int top, int right,
		int bottom) : screenID(a_screenID) { region.left = left,
		region.top = top, region.right = right, region.bottom = bottom; }

	int Encompassed(const UI_REGION &sRegion)
		{ return(region.left >= sRegion.left && region.top >= sRegion.top &&
		  region.right <= sRegion.right && region.bottom <= sRegion.bottom); }
	UI_REGION_ELEMENT *Next(void)
		{ return((UI_REGION_ELEMENT *)next); }
	int Overlap(const UI_POSITION &position)
		{ return(position.column >= region.left &&
		  position.column <= region.right && position.line >= region.top &&
		  position.line <= region.bottom); }
	int Overlap(const UI_REGION &a_region);
	int Overlap(const UI_REGION &s_region, UI_REGION &t_region);
	UI_REGION_ELEMENT *Previous(void)
		{ return((UI_REGION_ELEMENT *)previous); }
};

/* ----- Definition of the UI_REGION_LIST class ----- */
class UI_REGION_LIST : public UI_LIST
{
public:
	UI_REGION_ELEMENT *First(void)
		{ return((UI_REGION_ELEMENT *)first); }
	UI_REGION_ELEMENT *Last(void)
		{ return((UI_REGION_ELEMENT *)last); }
	void Split(int screenID, const UI_REGION &region);
};

/* ----- Definition of screen identification ----- */
const int ID_SCREEN					= 0;	// Special value. see ui_map.hpp

const int BITMAP_WIDTH 				= 0;
const int BITMAP_HEIGHT 			= 1;

/* ----- Definition of the DISPLAY class ----- */
const UCHAR MONO_BLACK	= 0x00;
const UCHAR MONO_DIM	= 0x08;
const UCHAR MONO_NORMAL	= 0x07;
const UCHAR MONO_HIGH	= 0x0F;

const UCHAR BW_BLACK	= 0x00;
const UCHAR BW_WHITE	= 0x01;

const UCHAR GS_BLACK	= 0x00;
const UCHAR GS_GRAY		= 0x01;
const UCHAR GS_BLINKING	= 0x02;
const UCHAR GS_WHITE	= 0x03;

enum TEXT_DISPLAY_MODE
{
	TDM_25x80,
	TDM_25x40,
	TDM_43x80,
	TDM_AUTO
};

enum DISPLAY_CODE
{
	DC_NO_DISPLAY 		= 0x00,
	DC_MONOCHROME 		= 0x01,
	DC_CGA		  		= 0x02,
	DC_EGA_COLOR  		= 0x04,
	DC_EGA_MONO   		= 0x05,
	DC_PGA_COLOR   		= 0x06,
	DC_PS2_ANALOG_MONO	= 0x07,
	DC_PS2_ANALOG_COLOR = 0x08,
	DC_M30_ANALOG_MONO  = 0x0B,
	DC_M30_ANALOG_COLOR = 0x0C,
	DC_UNKNOWN			= -1
};

extern class UI_EVENT_MANAGER;
class UI_DISPLAY
{
public:
	static UCHAR usingActiveDisplay;
	static UCHAR usingAlternateDisplay;
	UCHAR installed;
	UCHAR isActiveDisplay;
	const UCHAR isText;
    const UCHAR cellHeight;
	const UCHAR cellWidth;
	int	lines;
	int	columns;
	UI_EVENT_MANAGER *eventManager;

	UI_DISPLAY(UCHAR a_isText, UCHAR a_cellWidth, UCHAR a_cellHeight);
	virtual ~UI_DISPLAY(void);

	UCHAR ActiveDisplayCode(void) { return activeDisplayCode; }
	UCHAR AlternateDisplayCode(void) { return alternateDisplayCode; }
	virtual void Bitmap(int screenID, const UI_REGION &region,
		const USHORT *bitmap, const UI_PALETTE *palette,
		int fillBackground = TRUE) = 0;
	UCHAR ForceMono(void)
		{ return (forceMono); }
	void Fill(int screenID, int left, int top, int right, int bottom,
		const UI_PALETTE *palette);
	virtual void Fill(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette) = 0;
	virtual void FillXOR(const UI_REGION &region) = 0;
	virtual void Line(int screenID, int left, int top, int right,
		int bottom, const UI_PALETTE *palette, int width = 1) = 0;
	virtual void MakeActive(void) = 0;
	virtual void Rectangle(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette, int width = 1) = 0;
	virtual void RectangleXOR(const UI_REGION &region) = 0;
	virtual void RectangleXORDiff(const UI_REGION &oldRegion,
		const UI_REGION &newRegion) = 0;
	virtual void RegionConvert(UI_REGION &region) = 0;
	void RegionDefine(int screenID, UI_REGION const &region);
	virtual void Text(int screenID, int left, int top, const char *text,
		const UI_PALETTE *palette, int length = -1,
		int fill_graphics = TRUE) = 0;
	virtual int TextHeight(const char *string) = 0;
	virtual int TextWidth(const char *string) = 0;
	virtual void TextXOR(int left, int top, int length) = 0;

protected:
	UI_REGION_LIST regionList;
	UCHAR forceMono;
	UCHAR displayCode;
	UCHAR videoMode;
	UCHAR equipmentFlags;

private:
	static UCHAR activeDisplayCode;
	static UCHAR alternateDisplayCode;
	static UCHAR oldEquipmentFlags;
	static UCHAR oldVideoMode;
};

/* ----- Definition of the GRAPHICS_DISPLAY class ----- */

class UI_DOS_BGI_DISPLAY : public UI_DISPLAY
{
public:
	UI_DOS_BGI_DISPLAY(int driver = 0, int mode = 0);
	virtual ~UI_DOS_BGI_DISPLAY(void);

	void Bitmap(int screenID, const UI_REGION &region, const USHORT *bitmap,
		const UI_PALETTE *palette, int fillBackground = TRUE);
	int Driver(void) { return driver; }
	void Fill(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette);
	void FillXOR(const UI_REGION &region);
	void Line(int screenID, int left, int top, int right, int bottom,
		const UI_PALETTE *palette, int width = 1);
	void MakeActive(void);
	int  MapColor(const UI_PALETTE *palette, int isForeground);
	int  Mode(void) { return mode; }
	void Rectangle(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette, int width = 1);
	void RectangleXOR(const UI_REGION &region);
	void RectangleXORDiff(const UI_REGION &oldRegion,
		const UI_REGION &newRegion);
	void RegionConvert(UI_REGION &region);
	void Text(int screenID, int left, int top, const char *text,
		const UI_PALETTE *palette, int length = -1,
		int fillBackground = TRUE);
	int TextHeight(const char *string);
	int TextWidth(const char *string);
	void TextXOR(int left, int top, int length);

protected:
	int driver;
	int mode;
	int maxColor;
};

/* ----- Definition of the TEXT_DISPLAY class ----- */
class UI_DOS_TEXT_DISPLAY : public UI_DISPLAY
{
public:
	UCHAR isMono;
	USHORT onCursorValue;
	USHORT offCursorValue;
	USHORT originalCursorValue;
	UCHAR  originalCursorEmulationBit;

	UI_DOS_TEXT_DISPLAY(TEXT_DISPLAY_MODE displayMode = TDM_AUTO);
	virtual ~UI_DOS_TEXT_DISPLAY(void);

	void Bitmap(int screenID, const UI_REGION &region, const USHORT *bitmap,
		const UI_PALETTE *palette, int fillBackground = TRUE);
	void Fill(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette);
	void FillXOR(const UI_REGION &region);
	void Line(int screenID, int left, int top, int right, int bottom,
		const UI_PALETTE *palette, int width = 1);
	void Rectangle(int screenID, const UI_REGION &region,
		const UI_PALETTE *palette, int width = 1);
	void RectangleXOR(const UI_REGION &region);
	void RectangleXORDiff(const UI_REGION &oldRegion,
		const UI_REGION &newRegion);
	void RegionConvert(UI_REGION &region);
	void SetBlink(int enableBlink);
	void Text(int screenID, int left, int top, const char *text,
		const UI_PALETTE *palette, int length = -1,
		int fillBackground = TRUE);
	int TextHeight(const char *string);
	int TextWidth(const char *string);
	void TextXOR(int left, int top, int length);
	void MakeActive(void);

protected:
	UCHAR retraceWait;
	UCHAR desqview;
	UCHAR topview;
	USHORT segment;
	USHORT offset;
	UCHAR blinkState;

	void DrawText(const UI_PALETTE *palette, int column, int line,
		const char *text, int length);
	void FillLine(int column, int line, int noOfCharacters,
		const UI_PALETTE *palette);
	void XorAttribute(int column, int line, int swapNibbles = FALSE);
};

#include <ui_evt.hpp>
#endif /* UI_DSP_HPP */
