//	Program name..	Zinc Interface Library
//	Filename......	UI_EVT.HPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef UI_EVT_HPP
#define UI_EVT_HPP
#include <ui_dsp.hpp>

/* ----- Definition of UI_KEY information -----
 *	The event type is E_KEY.
 *	The event rawCode is the key's scan code.
 *	The event line and column are unused.
 */
/* KEY.shiftState */
const UCHAR S_RIGHT_SHIFT		= 0x0001;
const UCHAR S_LEFT_SHIFT		= 0x0002;
const UCHAR S_CTRL				= 0x0004;
const UCHAR S_ALT				= 0x0008;
const UCHAR S_SCROLL_LOCK		= 0x0010;
const UCHAR S_NUM_LOCK			= 0x0020;
const UCHAR S_CAPS_LOCK			= 0x0040;
const UCHAR S_INSERT			= 0x0080;

/* KEY.value is the low 8 bits of the scan code. */

struct UI_KEY
{
	UCHAR shiftState;			// The keyboard's shift state.
	UCHAR value;				// The key's ascii value.
};

/* ----- Definition of mouse information -----
 *	The event type is E_MOUSE.
 *	The event rawCode is the mouse shift state and button status.
 *	The event line is the mouse line.
 *	The event column is the mouse column.
 */
/* The mouse shift state is defined by KEY.shiftState */

/* The mouse button state */
const USHORT M_LEFT				= 0x0100;
const USHORT M_RIGHT			= 0x0200;
const USHORT M_MIDDLE			= 0x0400;
const USHORT M_LEFT_CHANGE		= 0x1000;
const USHORT M_RIGHT_CHANGE		= 0x2000;
const USHORT M_MIDDLE_CHANGE	= 0x4000;

const USHORT M_TOP_CHANGE		= 0x0800;	// only used in sizing operation
const USHORT M_BOTTOM_CHANGE	= 0x8000;	// only used in sizing operation

/* ----- Definition of the UI_EVENT structure ----- */
/* UI_EVENT.type */
/* Unmapped event types: 0 through 99 */
const int E_KEY 				= 10;		// Raw DEVICE events
const int E_MOUSE 				= 30;
const int E_DIGITIZER			= 31;
const int E_CURSOR				= 50;
const int E_DEVICE				= 99;

/* System event types: -1 through -999 */
const int S_ERROR				= -1;		// No union information
const int S_UNKNOWN				= -2;
const int S_NO_OBJECT			= -3;
const int S_MINIMIZE			= -4;
const int S_MAXIMIZE			= -5;
const int S_ALT_KEY				= -6;
const int S_CONTINUE			= -7;
const int S_ERROR_RESPONSE		= -8;

const int S_CREATE				= -10;
const int S_DELETE				= -11;
const int S_CURRENT				= -12;
const int S_NON_CURRENT			= -13;
const int S_MOVE				= -14;
const int S_REDISPLAY			= -17;
const int S_DELETE_LEVEL		= -18;
const int S_CANCEL				= -19;
const int S_CLEAR				= -20;

const int S_SIZE				= -200;		// UI_REGION union information
const int S_CHANGE				= -201;
const int S_DISPLAY_ACTIVE		= -202;
const int S_DISPLAY_INACTIVE	= -203;

const int S_SCROLL_VERTICAL		= -300;		// data union information
const int S_SCROLL_HORIZONTAL	= -301;
const int S_SCROLL_VERTICAL_SET	= -302;
const int S_SCROLL_HORIZONTAL_SET	= -303;

const int S_DEFAULT_MATCH		= -1000;	// Special palette matching
const int S_SELECTED			= -1001;
const int S_VIEW_ONLY			= -1002;

const int S_CHECK_HOT_KEY		= -2000;

struct UI_EVENT
{
	int type;						// The type of event.
	USHORT rawCode;					// The raw value for the event.
	union
	{
		UI_KEY key;					// Information specific to keyboard events.
		UI_REGION region;
		UI_POSITION position;
		UI_SCROLL_INFORMATION scroll;
		void *data;
	};
};

/* ----- Definition of the UI_DEVICE class ----- */
/* UI_EVENT.rawCode */
const USHORT D_OFF				= 0x0000;
const USHORT D_ON				= 0x0001;
const USHORT D_HIDE				= 0x0002;
const USHORT D_SHOW				= 0x0003;
const USHORT D_POSITION			= 0x0004;
const USHORT D_INITIALIZE		= 0x0005;
const USHORT D_RESTORE			= 0x0006;
const USHORT D_STATE			= 0x0007;

/* Private device codes: 0x0100 through 0xFFFF */
const USHORT DM_VIEW			= 0x0100;	// Mouse
const USHORT DM_EDIT			= 0x0101;
const USHORT DM_WAIT			= 0x0102;
const USHORT DM_MOVE			= 0x0103;
const USHORT DM_HORIZONTAL		= 0x0104;
const USHORT DM_VERTICAL		= 0x0105;
const USHORT DM_DIAGONAL_ULLR	= 0x0106;
const USHORT DM_DIAGONAL_LLUR	= 0x0107;
const USHORT DM_RANGE			= 0x0108;

const USHORT DC_INSERT			= 0x0110;	// Cursor
const USHORT DC_OVERTYPE		= 0x0111;
const USHORT DC_BLINK			= 0x0112;
const USHORT DC_INVALID			= 0x0113;
const USHORT DC_INSERT_OFF		= 0x0114;	// Cursor
const USHORT DC_OVERTYPE_OFF	= 0x0115;

// These are the states for the "altPressed" member variable below.
enum
{
	ALT_NOT_PRESSED,
	ALT_PRESSED_AND_NO_EVENTS_YET,
	ALT_PRESSED_AND_EVENTS_RECEIVED
};

class UI_DEVICE : public UI_ELEMENT
{
protected:
	friend class UI_EVENT_MANAGER;

	int type;
	USHORT state;
	USHORT enabled;
	UI_DISPLAY *display;
	UI_EVENT_MANAGER *eventManager;

	UI_DEVICE(int a_type, USHORT initialState);
	static int CompareFunction(void *device1, void *device2);
	virtual void Poll(void) = 0;

public:
	static int altPressed;

	virtual int Event(const UI_EVENT &event) = 0;
	UI_DEVICE *Next(void)
		{ return((UI_DEVICE *)next); }
	UI_DEVICE *Previous(void)
		{ return((UI_DEVICE *)previous); }
};

/* ----- Definition of the UI_CURSOR class ----- */
class UI_CURSOR : public UI_DEVICE
{
public:
	UI_CURSOR(USHORT initialState = D_OFF);

	virtual int Event(const UI_EVENT &event);

protected:
	USHORT cursorMask;
	int blink;
	int line;
	int column;
	USHORT lastTime;

	void Display(USHORT displayCursorMask, USHORT displayState);
	virtual void Poll(void);
};

/* ----- Definition of the UI_BIOS_KEYBOARD class ----- */
class UI_BIOS_KEYBOARD : public UI_DEVICE
{
public:
	UI_BIOS_KEYBOARD(USHORT initialState = D_ON);
	~UI_BIOS_KEYBOARD(void);

	virtual int Event(const UI_EVENT &event);

protected:
	UCHAR enhancedBIOS;
	UCHAR breakState;
	static UCHAR breakHandlerSet;

	virtual void Poll(void);
};

/* ----- Definition of the UI_MS_MOUSE class ----- */
class UI_MS_MOUSE : public UI_DEVICE
{
public:
	UI_MS_MOUSE(USHORT initialState = DM_VIEW);
	~UI_MS_MOUSE(void);

	virtual int Event(const UI_EVENT &event);
	void SetSensitivity(short horizontalSensitivity, short verticalSensitivity, short doubleSpeedThreshold);

protected:
	int installed;
	USHORT *graphicsCursorMask;
	USHORT textScreenMask;
	USHORT textCursorMask;
	int line;
	int column;

	void Display(void);
	virtual void Poll(void);
};

/* ----- Definition of the UI_EVENT_MANAGER class ----- */
/* QUEUE.flags */
const USHORT Q_NORMAL			= 0x0000;	// block, begin, destroy and poll
const USHORT Q_BLOCK			= 0x0000;	// use with caution
const USHORT Q_NO_BLOCK			= 0x0001;
const USHORT Q_BEGIN			= 0x0000;	// use with caution
const USHORT Q_END				= 0x0002;
const USHORT Q_DESTROY			= 0x0000;	// use with caution
const USHORT Q_NO_DESTROY		= 0x0004;
const USHORT Q_POLL				= 0x0000;	// use with caution
const USHORT Q_NO_POLL			= 0x0008;

class UI_QUEUE_ELEMENT : public UI_ELEMENT
{
public:
	UI_EVENT event;

	UI_QUEUE_ELEMENT(void) {}

	UI_QUEUE_ELEMENT *Next(void)
		{ return((UI_QUEUE_ELEMENT *)next); }
	UI_QUEUE_ELEMENT *Previous(void)
		{ return((UI_QUEUE_ELEMENT *)previous); }
};

class UI_QUEUE_BLOCK : public UI_LIST_BLOCK
{
public:
	UI_QUEUE_BLOCK(int a_noOfElements);
	~UI_QUEUE_BLOCK(void);

	UI_QUEUE_ELEMENT *First(void)
		{ return((UI_QUEUE_ELEMENT *)first); }
	UI_QUEUE_ELEMENT *Last(void)
		{ return((UI_QUEUE_ELEMENT *)last); }
};

class UI_EVENT_MANAGER
{
protected:
	static int CompareFunction(void *device1, void *device2);

	UI_DISPLAY *display;
	UI_QUEUE_BLOCK queueBlock;
	UI_LIST deviceList;
	int level;

public:
	UI_EVENT_MANAGER(int a_noOfElements, UI_DISPLAY *a_display);
	void Add(UI_DEVICE *device);
	void Subtract(UI_DEVICE *device);
	void DevicesHide(const UI_REGION &region);
	void DevicesShow(const UI_REGION &region);
	void DevicePosition(int deviceType, int column, int line);
	int DeviceState(int deviceType, USHORT deviceState);
	int Event(const UI_EVENT &event);
	UI_DEVICE *First(void)
		{ return((UI_DEVICE *)deviceList.first); }
	int Get(UI_EVENT &event, USHORT flags = Q_NORMAL);
	UI_DEVICE *Last(void)
		{ return((UI_DEVICE *)deviceList.last); }
	void Put(const UI_EVENT &event, USHORT flags = Q_END);

	UI_EVENT_MANAGER &operator+(UI_DEVICE *device)
		{ Add(device); return(*this); }
	UI_EVENT_MANAGER &operator-(UI_DEVICE *device)
		{ Subtract(device); return(*this); }
};

#endif /* UI_EVT_HPP */
