//	Program name..	Zinc Interface Library
//	Filename......	UI_WIN.HPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef UI_WIN_HPP
#define UI_WIN_HPP
#include <ui_map.hpp>

/* ----- Definition of the UI_WINDOW_OBJECT class ----- */
const int NO_HELP_CONTEXT = 0;

enum INFORMATION_REQUEST
{
	GET_DEFAULT_HEIGHT,
	GET_DEFAULT_WIDTH,
	GET_MINIMUM_HEIGHT,
	GET_MINIMUM_WIDTH,
	GET_MAXIMUM_HEIGHT,
	GET_MAXIMUM_WIDTH,
};

/* UI_WINDOW_OBJECT.woFlags */
const USHORT WOF_NO_FLAGS			= 0x0000;
const USHORT WOF_JUSTIFY_CENTER		= 0x0001;	// center justify object's data
const USHORT WOF_JUSTIFY_RIGHT		= 0x0002;	// right justify object's data
const USHORT WOF_BORDER				= 0x0004;	// draw a border around the object
const USHORT WOF_NO_ALLOCATE_DATA	= 0x0008;	// do not allocate the data buffer
const USHORT WOF_VIEW_ONLY			= 0x0010;	// object can be viewed, but not edited
const USHORT WOF_NO_UNANSWERED		= 0x0020;	// Prevents "Leave Unanswered" option
const USHORT WOF_NO_INVALID			= 0x0040;	// Prevents "Leave Invalid" option
const USHORT WOF_UNANSWERED			= 0x0080;	// Sets initial status bit to "unanswered"
const USHORT WOF_INVALID			= 0x0100;	// Sets initial status bit to "invalid"
const USHORT WOF_NON_FIELD_REGION	= 0x0200;	// object does not share screen space 
const USHORT WOF_NON_SELECTABLE		= 0x0400;	// object cannot be selected
const USHORT WOF_AUTO_CLEAR			= 0x0800;	// auto clear option for edit fields

/* UI_WINDOW_OBJECT.woAdvancedFlags */
const USHORT WOAF_NO_FLAGS			= 0x0000;
const USHORT WOAF_OUTSIDE_REGION	= 0x0001;	// object occupies space outside the true region
const USHORT WOAF_NON_CURRENT		= 0x0002;	// object cannot be made current
const USHORT WOAF_TEMPORARY			= 0x0004;	// object is only temporarily on the display
const USHORT WOAF_HOT_REGION		= 0x0008;
const USHORT WOAF_NO_DESTROY		= 0x0010;	// don't destroy the object
const USHORT WOAF_NORMAL_HOT_KEYS	= 0x0020;	// look for hot keys on normal input.
const USHORT WOAF_NO_SIZE			= 0x0040;	// object cannot be sized
const USHORT WOAF_NO_MOVE			= 0x0080;	// object cannot be moved
const USHORT WOAF_MODAL				= 0x0100;	// object is modal (gets all events)
const USHORT WOAF_LOCKED			= 0x0200;	// object cannot be removed from the window manager

/* UI_WINDOW_OBJECT.woStatus */
const USHORT WOS_NO_STATUS			= 0x0000;
const USHORT WOS_GRAPHICS			= 0x0001;	// object region is in graphics coordinates
const USHORT WOS_CURRENT			= 0x0002;	// object is current
const USHORT WOS_CHANGED			= 0x0004;	// object data has changed
const USHORT WOS_SELECTED			= 0x0008;	// object has been selected
const USHORT WOS_UNANSWERED			= 0x0010;	// object data is answered
const USHORT WOS_INVALID			= 0x0020;	// object has invalid data
const USHORT WOS_NO_AUTO_CLEAR		= 0x0040;	// edit keys have been used

/* UI_WINDOW_OBJECT.woAdvancedStaus */
const USHORT WOAS_NO_STATUS			= 0x0000;
const USHORT WOAS_INVALID_REGION	= 0x0001;	// object space is invalid
const USHORT WOAS_MAXIMIZED			= 0x0002;	// object is maximized
const USHORT WOAS_MINIMIZED			= 0x0004;	// object is minimized
const USHORT WOAS_TOO_SMALL			= 0x0008;	// sized too small to be selected
const USHORT WOAS_NEED_VALIDATE		= 0x0010;	// data needs validation
const USHORT WOAS_REDISPLAY			= 0x0020;	// data needs updating

class UI_WINDOW_OBJECT : public UI_ELEMENT
{
	friend class UI_WINDOW_MANAGER;

public:
	static UI_EVENT_MAP *eventMapTable;
	static int defaultDepth;

	USHORT woFlags;
	USHORT woAdvancedFlags;
	USHORT woStatus;
	USHORT woAdvancedStatus;
	UI_REGION true;
	UI_REGION relative;
	UI_WINDOW_OBJECT *parent;
	UI_DISPLAY *display;
	UI_EVENT_MANAGER *eventManager;
	UI_WINDOW_MANAGER *windowManager;
	UI_PALETTE_MAP *paletteMapTable;
	UI_PALETTE *lastPalette;
	int (*Validate)(void *object, int ccode);
	UCHAR hotKey;

	UI_WINDOW_OBJECT(int left, int top, int width, int height,
		USHORT a_woFlags, USHORT a_woAdvancedFlags);
	virtual ~UI_WINDOW_OBJECT(void);

	virtual int Event(const UI_EVENT &event);
	virtual void Information(INFORMATION_REQUEST request, void *data);
	void InformationSet(int a_screenID, UI_DISPLAY *a_display,
		UI_EVENT_MANAGER *a_eventManager, UI_WINDOW_MANAGER *a_windowManager,
		UI_PALETTE_MAP *a_paletteMapTable, UI_WINDOW_OBJECT *a_parent);
	UI_WINDOW_OBJECT *Next(void)
		{ return((UI_WINDOW_OBJECT *)next); }
	int Overlap(const UI_REGION &region);
	int Overlap(const UI_POSITION &position)
		{ return(position.column >= true.left &&
		  position.column <= true.right && position.line >= true.top &&
		  position.line <= true.bottom); }
	UI_WINDOW_OBJECT *Previous(void)
		{ return((UI_WINDOW_OBJECT *)previous); }
	void RegionMax(int leftTop);
	int Touching(const UI_POSITION &position)
		{ return(position.column == true.left || 
		  position.column == true.right || position.line == true.top ||
		  position.line == true.bottom); }
	void WindowID(int position, int id) { windowID[position] = id; }

protected:
	int screenID;
	int windowID[4];

	void Border(int ccode, UI_REGION &region, const UI_PALETTE *palette);
	int LogicalEvent(const UI_EVENT &event, int currentID)
		{ return (MapEvent(eventMapTable, event, currentID, windowID[0],
		  windowID[1], windowID[2], windowID[3])); }
	UI_PALETTE *LogicalPalette(int ccode);
	int NeedsUpdate(const UI_EVENT &event, int ccode);
	int NeedsValidation(void);
	void Redisplay(int fromRoot);
	void Shadow(UI_REGION &region, int depth);
	void Text(char *string, int depth, int ccode, const UI_PALETTE *palette);
};

/* ----- Definition of the UI_WINDOW_MANAGER class ----- */
class UI_WINDOW_MANAGER
{
public:
	UI_WINDOW_MANAGER(UI_DISPLAY *a_display, UI_EVENT_MANAGER *a_eventManager);
	~UI_WINDOW_MANAGER(void);
	void Add(UI_WINDOW_OBJECT *object);
	void Subtract(UI_WINDOW_OBJECT *object);
	int Event(const UI_EVENT &event);
	UI_WINDOW_OBJECT *First(void)
		{ return((UI_WINDOW_OBJECT *)windowList.first); }
	UI_WINDOW_OBJECT *Last(void)
		{ return((UI_WINDOW_OBJECT *)windowList.last); }

	UI_WINDOW_MANAGER &operator+(void *object)
		{ Add((UI_WINDOW_OBJECT *)object); return(*this); }
	UI_WINDOW_MANAGER &operator-(void *object)
		{ Subtract((UI_WINDOW_OBJECT *)object); return(*this); };

protected:
	static UI_EVENT_MAP *eventMapTable;
	static int screenID;
	UI_DISPLAY *display;
	UI_EVENT_MANAGER *eventManager;
	UI_LIST windowList;
	UI_WINDOW_OBJECT *errorObject;

	void Erase(UI_WINDOW_OBJECT *object, const UI_REGION *newRegion);
	void Modify(UI_WINDOW_OBJECT *object, const UI_EVENT &event);
	void ToFront(UI_WINDOW_OBJECT *object, int refreshOld, int newWindow);
};

/* ----- Definition of the UI_BORDER class ----- */
class UIW_BORDER : public UI_WINDOW_OBJECT
{
public:
	UIW_BORDER(void);
	virtual ~UIW_BORDER(void) {}

	virtual int Event(const UI_EVENT &event);

private:
	int depth;
	static int size;
};

/* ----- Definition of the UIW_BUTTON class ----- */
/* UIW_BUTTON.btFlags */
const USHORT BTF_NO_FLAGS			= 0x0000;
const USHORT BTF_NO_TOGGLE			= 0x0001;	// don't toggle (select) the button
const USHORT BTF_DOWN_CLICK			= 0x0002;	// complete button action on down click
const USHORT BTF_CHECK_MARK			= 0x0004;	// display a check mark when selected.
const USHORT BTF_REPEAT				= 0x0008;	// repeat the down-click action
const USHORT BTF_DOUBLE_CLICK		= 0x0010;	// perform the action on a double-mouse click.

/* UIW_BUTTON.btStatus */
const USHORT BTS_NO_STATUS			= 0x0000;
const USHORT BTS_DEPRESSED			= 0x0001;	// button is depressed

class UIW_BUTTON : public UI_WINDOW_OBJECT
{
public:
	int depth;
	USHORT btFlags;

	UIW_BUTTON(int left, int top, int width, char *a_string, USHORT a_btFlags,
		USHORT a_woFlags, void (*userFunction)(void *object, UI_EVENT &event) = 0);
	virtual ~UIW_BUTTON(void);

	const char *DataGet(void) { return (string); }
	void DataSet(char *string);
	virtual int Event(const UI_EVENT &event);

protected:
	USHORT btStatus;
	char *string;
	void (*userFunction)(void *object, UI_EVENT &event);
};

/* ----- Definition of the UIW_MAXIMIZE_BUTTON class ----- */
class UIW_MAXIMIZE_BUTTON : public UIW_BUTTON
{
public:
	UIW_MAXIMIZE_BUTTON(void);
	virtual ~UIW_MAXIMIZE_BUTTON(void) {}

	virtual int Event(const UI_EVENT &event);

protected:
	static void MaximizeUserFunction(void *button, UI_EVENT &event);
};

/* ----- Definition of the UIW_MINIMIZE_BUTTON class ----- */
class UIW_MINIMIZE_BUTTON : public UIW_BUTTON
{
public:
	UIW_MINIMIZE_BUTTON(void);
	virtual ~UIW_MINIMIZE_BUTTON(void) {}

	virtual int Event(const UI_EVENT &event);

protected:
	static void MinimizeUserFunction(void *button, UI_EVENT &event);
};

/* ----- Definition of the UI_ICON class ----- */
const USHORT ICF_NO_FLAGS			= 0x0000;
const USHORT ICF_TRANSPARENT		= 0x0001;

class UIW_ICON : public UI_WINDOW_OBJECT
{
public:
	static USHORT repeatRate;
	USHORT icFlags;

	UIW_ICON(int left, int top, USHORT **bitmapArray,
		UI_PALETTE *paletteArray, USHORT a_icFlags, USHORT a_woFlags,
		void (*userFunction)(void *object, UI_EVENT &event) = 0);
	virtual ~UIW_ICON(void) {}

	void DataGet(USHORT ***a_bitmapArray, UI_PALETTE **a_paletteArray)
		{ *a_bitmapArray = bitmapArray, *a_paletteArray = paletteArray; }
	void DataSet(USHORT **bitmapArray, UI_PALETTE *paletteArray);
	virtual int Event(const UI_EVENT &event);

private:
	void (*userFunction)(void *object, UI_EVENT &event);
	USHORT **bitmapArray;
	UI_PALETTE *paletteArray;
	USHORT lastTime;
};

/* ----- Definition of the UI_PROMPT class ----- */
class UIW_PROMPT : public UI_WINDOW_OBJECT
{
public:
	UIW_PROMPT(int left, int top, const char *a_prompt, USHORT a_woFlags);
	~UIW_PROMPT(void) { delete prompt; }

	virtual int Event(const UI_EVENT &event);

private:
	char *prompt;
};

/* ----- Definition of the UIW_TITLE class ----- */
class UIW_TITLE : public UI_WINDOW_OBJECT
{
public:
	UIW_TITLE(char *a_title, USHORT a_woFlags = WOF_JUSTIFY_CENTER);
	~UIW_TITLE(void);

	const char *DataGet(void) { return(title); }
	void DataSet(char *a_title);
	virtual int Event(const UI_EVENT &event);

private:
	char *title;
};

/* ----- Definition of the UIW_WINDOW class ----- */
class UIW_WINDOW : public UI_WINDOW_OBJECT
{
public:
	UIW_WINDOW(int left, int top, int width, int height, USHORT a_woFlags,
		USHORT a_woAdvancedFlags, int helpContext = NO_HELP_CONTEXT);
	virtual ~UIW_WINDOW(void) {}

	static UIW_WINDOW *GENERIC(int left, int top, int width, int height,
		USHORT woFlags, USHORT woAdvancedFlags, int helpContext, char *title);

	void Add(UI_WINDOW_OBJECT *object);
	void Subtract(UI_WINDOW_OBJECT *object);
	virtual int Event(const UI_EVENT &event);
	UI_WINDOW_OBJECT *First(void)
		{ return((UI_WINDOW_OBJECT *)windowList.first); }
	UI_WINDOW_OBJECT *Last(void)
		{ return((UI_WINDOW_OBJECT *)windowList.last); }

	UIW_WINDOW &operator + (void *object)
		{ Add((UI_WINDOW_OBJECT *)object); return(*this); }
	UIW_WINDOW &operator - (void *object)
		{ Subtract((UI_WINDOW_OBJECT *)object); return(*this); }

protected:
	USHORT wnStatus;
	UI_REGION_LIST regionList;
	UI_LIST windowList;
	int helpContext;
	UI_WINDOW_OBJECT *errorObject;

	static int FindStatus(void *object, void *matchData);
	virtual void Information(INFORMATION_REQUEST request, void *data);
	void RegionsCompute(void);
	UI_WINDOW_OBJECT *ToFront(UI_WINDOW_OBJECT *object, int checkCurrent);

private:
	USHORT max_woAdvancedFlags;
	USHORT max_woAdvancedStatus;
	UI_REGION max_region;

	USHORT min_woAdvancedFlags;
	USHORT min_woAdvancedStatus;
	UI_REGION min_region;

	void Maximize(void);
	void Minimize(void);
};

/* ----- Definition of the UIW_POP_UP_WINDOW class ----- */
class UIW_POP_UP_WINDOW : public UIW_BUTTON
{
public:
	UIW_POP_UP_WINDOW(int left, int top, int width, char *a_string,
		UIW_WINDOW *window, USHORT a_btFlags, USHORT a_woFlags,
		void (*userFunction)(void *object, UIW_WINDOW *window, int ccode) = 0);
	virtual ~UIW_POP_UP_WINDOW(void);

	virtual int Event(const UI_EVENT &event);
	UI_WINDOW_OBJECT *First(void)
		{ return((UI_WINDOW_OBJECT *)window->First()); }
	UI_WINDOW_OBJECT *Last(void)
		{ return((UI_WINDOW_OBJECT *)window->Last()); }

	UIW_POP_UP_WINDOW &operator + (void *object)
		{ window->Add((UI_WINDOW_OBJECT *)object); return(*this); }
	UIW_POP_UP_WINDOW &operator - (void *object)
		{ window->Subtract((UI_WINDOW_OBJECT *)object); return(*this); }

protected:
	UIW_WINDOW *window;
	UI_REGION windowRegion;
	USHORT windowAdvancedFlags;

	void UserFunction(UI_EVENT &event);

private:
	void (*windowUserFunction)(void *object, UIW_WINDOW *window, int ccode);
	static void PopUpWindowUserFunction(void *button, UI_EVENT &event);
};

/* ----- Definition of the UIW_POP_UP_ITEM class ----- */
/* UIW_POP_UP_ITEM.mniFlags */
const USHORT MNIF_NO_FLAGS			= 0x0000;
const USHORT MNIF_SEPARATOR			= 0x0001;	// Menu item seperator
const USHORT MNIF_MAXIMIZE			= 0x0002;	// Item is tied to maximize button.
const USHORT MNIF_MINIMIZE			= 0x0004;	// Item is tied to minimize button.
const USHORT MNIF_MOVE				= 0x0008;	// Item is tied to moving.
const USHORT MNIF_SIZE				= 0x0010;	// Item is tied to sizing.
const USHORT MNIF_DUAL_MONITOR		= 0x0020;	// Item is tied to dual monitor support.
const USHORT MNIF_RESTORE			= 0x0040;	// Maximize/minimize restoration.
const USHORT MNIF_CLOSE				= 0x0080;	// Close the parent window.

class UIW_POP_UP_ITEM : public UIW_BUTTON
{
	friend class UIW_POP_UP_ITEM;

public:
	USHORT mniFlags;

	UIW_POP_UP_ITEM(void);
	UIW_POP_UP_ITEM(char *string, USHORT mniFlags, USHORT btFlags,
		USHORT woFlags, void (*userFunction)(void *object, UI_EVENT &event) = 0);
	UIW_POP_UP_ITEM(int left, int top, int width, char *string,
		USHORT mniFlags, USHORT btFlags, USHORT woFlags,
		void (*userFunction)(void *object, UI_EVENT &event) = 0);
	virtual ~UIW_POP_UP_ITEM(void) {}

	virtual int Event(const UI_EVENT &event);

	UIW_POP_UP_ITEM *Next(void)
		{ return((UIW_POP_UP_ITEM *)next); }
	UIW_POP_UP_ITEM *Previous(void)
		{ return((UIW_POP_UP_ITEM *)previous); }
};

/* ----- Definition of the UIW_POP_UP_MENU class ----- */
/* UIW_POP_UP_MENU.mnFlags */
const USHORT MNF_NO_FLAGS			= 0x0000;
const USHORT MNF_SELECT_ONE			= 0x0001;
const USHORT MNF_AUTO_SORT			= 0x0002;
const USHORT MNF_SELECT_LEFT		= 0x0004;
const USHORT MNF_SELECT_RIGHT		= 0x0008;

class UIW_POP_UP_MENU : public UIW_WINDOW
{
public:
	USHORT mnFlags;

	UIW_POP_UP_MENU(int left, int top, USHORT mnFlags, USHORT woFlags,
		USHORT woAdvancedFlags);
	virtual ~UIW_POP_UP_MENU(void) {}

	virtual int Event(const UI_EVENT &event);
	UIW_POP_UP_ITEM *First(void)
		{ return((UIW_POP_UP_ITEM *)windowList.first); }
	UIW_POP_UP_ITEM *Last(void)
		{ return((UIW_POP_UP_ITEM *)windowList.last); }

private:
	static int CompareFunction(void *element1, void *element2);
};

/* ----- Definition of the UIW_PULL_DOWN_ITEM class ----- */
class UIW_PULL_DOWN_ITEM : public UIW_BUTTON
{
	friend class UIW_PULL_DOWN_MENU;

public:
	UIW_PULL_DOWN_ITEM(char *string, USHORT mnFlags,
		void (*userFunction)(void *object, UI_EVENT &event) = 0);
	virtual ~UIW_PULL_DOWN_ITEM(void) {}

	void Add(UIW_POP_UP_ITEM *object) { menu.Add(object); }
	void Subtract(UIW_POP_UP_ITEM *object) { menu.Subtract(object); }
	virtual int Event(const UI_EVENT &event);
	UIW_PULL_DOWN_ITEM *Next(void)
		{ return((UIW_PULL_DOWN_ITEM *)next); }
	UIW_PULL_DOWN_ITEM *Previous(void)
		{ return((UIW_PULL_DOWN_ITEM *)previous); }

	UIW_PULL_DOWN_ITEM &operator + (void *object)
		{ menu.Add((UIW_POP_UP_ITEM *)object); return(*this); }
	UIW_PULL_DOWN_ITEM &operator - (void *object)
		{ menu.Subtract((UIW_POP_UP_ITEM *)object); return(*this); }

private:
	UIW_POP_UP_MENU menu;

	static void PullDownUserFunction(void *button, UI_EVENT &event);
	void UserFunction(UI_EVENT &event);
};

/* ----- Definition of the UI_PULL_DOWN_MENU class ----- */
class UIW_PULL_DOWN_MENU : public UIW_WINDOW
{
public:
	UIW_PULL_DOWN_MENU(int indentation, USHORT woFlags,
		USHORT woAdvancedFlags);
	virtual ~UIW_PULL_DOWN_MENU(void) {}

	virtual int Event(const UI_EVENT &event);
	UIW_PULL_DOWN_ITEM *First(void)
		{ return((UIW_PULL_DOWN_ITEM *)windowList.first); }
	UIW_PULL_DOWN_ITEM *Last(void)
		{ return((UIW_PULL_DOWN_ITEM *)windowList.last); }

protected:
	int indentation;
};

/* ----- Definition of the UI_SCROLL_BAR class ----- */
/* UI_SCROLL_BAR.sbFlags */
const USHORT SBF_NO_FLAGS			= 0x0000;
const USHORT SBF_CORNER				= 0x0001;
const USHORT SBF_VERTICAL			= 0x0002;
const USHORT SBF_HORIZONTAL			= 0x0004;

class UIW_SCROLL_BAR : public UIW_WINDOW
{
public:
	USHORT sbFlags;

	UIW_SCROLL_BAR(int left, int top, int width, int height, USHORT sbFlags, USHORT woFlags);
	virtual ~UIW_SCROLL_BAR(void) {}

	virtual int Event(const UI_EVENT &event);

protected:
	int current;
	int maximum;
	int showing;

private:
	UIW_BUTTON *button[5];
	static void ScrollBarUserFunction(void *button, UI_EVENT &event);
};

/* ----- Definition of the UIW_MATRIX class ----- */
/* UIW_MATRIX.mxFlags */
const USHORT MXF_NO_FLAGS			= 0x0000;
const USHORT MXF_COLUMNS_FILL		= 0x0001;
const USHORT MXF_ROWS_FILL			= 0x0002;
const USHORT MXF_AUTO_FILL			= 0x0004;
const USHORT MXF_SELECT_ONE			= 0x0008;

class UIW_MATRIX : public UIW_WINDOW
{
public:
	USHORT mxFlags;

	UIW_MATRIX(int left, int top, int width, int height, int maxRowsColumns,
		int cellWidth, int cellHeight,
		int (*a_compareFunction)(void *element1, void *element2), 
		USHORT mxFlags, USHORT woFlags, USHORT a_woAdvancedFlags);
	virtual ~UIW_MATRIX(void) {}

	virtual int Event(const UI_EVENT &event);

protected:
	int maxRowsColumns;
	int cellWidth;
	int cellHeight;
};

/* ----- Definition of the UIW_SYSTEM_BUTTON class ----- */
class UIW_SYSTEM_BUTTON : public UIW_BUTTON
{
public:
	UIW_SYSTEM_BUTTON(void);
	virtual ~UIW_SYSTEM_BUTTON(void) {}

	virtual int Event(const UI_EVENT &event);
	void UserFunction(UI_EVENT &event);

	static UIW_SYSTEM_BUTTON *GENERIC(void);

	UIW_SYSTEM_BUTTON &operator + (void *object)
		{ menu.Add((UI_WINDOW_OBJECT *)object); btFlags |= BTF_DOWN_CLICK; return(*this); }
	UIW_SYSTEM_BUTTON &operator - (void *object)
		{ menu.Subtract((UI_WINDOW_OBJECT *)object);
		  if (!menu.First()) btFlags &= ~BTF_DOWN_CLICK; return(*this); }

protected:
	UIW_POP_UP_MENU menu;

private:
	static void SystemUserFunction(void *button, UI_EVENT &event);
};

/* ------ Definition of the UI_UNDO_RECORD ------- */
enum UNDO_OPERATION
{
	MOVE_POSITION,
	DELETE_TEXT,
	ADD_TEXT,
	REPLACE_CHAR
};

class UI_UNDO_RECORD : public UI_ELEMENT
{
public:
	USHORT size;			// Size of the record in bytes.

	virtual ~UI_UNDO_RECORD(void) {}
};

class UI_UNDO_LIST
{
public:
	UI_UNDO_RECORD *head;
	UI_UNDO_RECORD *tail;
	short count;
	long totalSize;

	UI_UNDO_LIST(void) { head = tail = 0; count = 0; }
	virtual ~UI_UNDO_LIST(void);

	void Add(UI_UNDO_RECORD *element, USHORT a_size);
	void Subtract(UI_UNDO_RECORD *element);
	void MoveFront(UI_UNDO_RECORD *element);
	void Unlink(UI_UNDO_RECORD *element);
};

typedef UI_UNDO_RECORD *UI_UNDO_HANDLE;

class UI_EDIT_INFO
{
public:
	static void UndoStrategy(int maxObjects, long maxBytes, int maxUndos,
		long maxBytesPerObject, int maxUndosPerObject);

protected:
	UI_UNDO_HANDLE undoHandle;
	USHORT markColumn;					// Column where mouse mark initiated.
	static char *pasteBuffer;			// There is only one global paste buffer.
	static short pasteLength;
	static short numEditObjects;

	UI_EDIT_INFO(void);
	virtual ~UI_EDIT_INFO(void);
};

extern UI_UNDO_HANDLE UndoCreate(void);
extern void UndoDestroy(UI_UNDO_HANDLE handle);
extern void UndoAdd(UI_UNDO_HANDLE handle, UI_UNDO_RECORD *undo, USHORT a_size);
extern UI_UNDO_RECORD *UndoCurrent(UI_UNDO_HANDLE handle);
extern void UndoRedoPush(UI_UNDO_HANDLE handle);
extern void UndoRedoPop(UI_UNDO_HANDLE handle);
extern UI_UNDO_RECORD *UndoRedoCurrent(UI_UNDO_HANDLE handle);

/* ----- Definition of the UIW_STRING class ----- */
const USHORT STF_NO_FLAGS			= 0x0000;

struct STRING_STATE
{
	char *screenTop;
	char *cursor;
	char *markedBlock;
};

class STRING_UNDO : public UI_UNDO_RECORD
{
public:
	char *textAdd;
	short textLength;
	char *buff;
	UNDO_OPERATION operation;
	STRING_STATE info;

	STRING_UNDO(void);
	virtual ~STRING_UNDO(void);
};

class UIW_STRING : public UI_WINDOW_OBJECT, public UI_EDIT_INFO
{
public:
	USHORT stFlags;

	UIW_STRING(int left, int top, int width, char *text, short maxLength,
		USHORT stFlags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	virtual ~UIW_STRING(void);

	const char *DataGet(void) { return text; }
	void DataSet(char *newText, short maxLength = -1);
	virtual int Event(const UI_EVENT &event);

protected:
	char *text;
	UCHAR fieldWidth;

	char *screenTop;
	char *textNull;
	char *textTail;
	char *markedBlock;
	char *markStart;
	char *markTail;
	UCHAR insertMode;
	char *cursor;
	UCHAR originalWidth;
	char *screen;
	UCHAR screenInvalid;
	short screenSize;
	char *screenMarkStart;
	char *screenMarkTail;

	STRING_UNDO *AllocateUndo(void);
	void BackspaceKey(void);
	int CalcLeading(int width);
	void CheckLeftScroll(void);
	void CheckRightScroll(void);
	STRING_UNDO *CopyBlock(void);
	void CutBlock(void);
	void DeleteBlock(STRING_UNDO *undo, char *block, USHORT length);
	void DeleteChar(STRING_UNDO *undo);
	void DeleteEol(void);
	void DeleteKey(void);
	void DeleteWord(void);
	void InsertBlock(STRING_UNDO *undo, char *insert_point, char *block, USHORT length);
	void LeftArrow(void);
	void MoveOperation(void);
	int NonWhiteSpace(int value);
	void PasteBlock(void);
	void Redisplay(int ccode);
	void RegularKey(USHORT key);
	void ReplaceChar(STRING_UNDO *undo, char *buff, USHORT key);
	void SetMark(void);
	int StateInfoChanged(STRING_STATE *state);
	void StateInfoRestore(STRING_STATE *state);
	void StateInfoSave(STRING_STATE *state);
	void UndoOperation(STRING_UNDO *undo);
	void UpdateCursor(UI_REGION &region, int width);
	int VerticalCenterOffset(UI_REGION *region);
	int WhiteSpace(int value);
	void WordTabLeft(void);
};

/* ----- Definition of the UIW_TEXT class ----- */
const USHORT TXF_NO_FLAGS = 0x0000;

struct TEXT_STATE
{
	char *screenTop;
	char *cursor;
	char *markedBlock;
	short cursorDesiredCol;
};

class TEXT_UNDO : public UI_UNDO_RECORD
{
public:
	char *textAdd;
	short textLength;
	char *buff;
	UNDO_OPERATION operation;
	TEXT_STATE info;

	TEXT_UNDO(void);
	virtual ~TEXT_UNDO(void);
};

class UIW_TEXT : public UIW_STRING
{
public:
	USHORT txFlags;

	UIW_TEXT(int left, int top, int width, int height, char *text,
		short maxLength, USHORT txFlags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	virtual ~UIW_TEXT(void);

	const char *DataGet(void) { return text; }
	void DataSet(char *newText, short maxLength = -1);
	virtual int Event(const UI_EVENT &event);

protected:
	short cellHeight;
	short cellWidth;
	short width;
	short height;
	short cursorRow;
	short cursorCol;
	short cursorDesiredCol;
	short pixelBorder;

	char *AdvanceColumns(char *ptr,	int num_cols, int *result_col);
	TEXT_UNDO *AllocateUndo(void);
	void BackspaceKey(void);
	void ComputeMousePosition(int mouse_col, int mouse_row);
	void ComputeRegion(int ccode, UI_REGION &region, UI_PALETTE *palette);
	void ComputeScroll(void);
	TEXT_UNDO *CopyBlock(void);
	void CutBlock(void);
	void DeleteBlock(TEXT_UNDO *undo,	char *block, USHORT length);
	void DeleteChar(TEXT_UNDO *undo);
	void DeleteEol(void);
	void DeleteKey(void);
	void DeleteWord(void);
	void EnterKey(void);
	void Home(void);
	void InsertBlock(TEXT_UNDO *undo,	char *insert_point,	char *block, USHORT length);
	int  LastLine(void);
	void LeftArrow(void);
	void MoveDown(USHORT numLines, int ccode);
	void MoveOperation(void);
	void MoveUp(USHORT numLines, int ccode);
	int NonWhiteSpace(int value);
	void PasteBlock(void);
	char *PreviousLine(char *line, USHORT num_lines, int col, char *base);
	void Redisplay(UI_REGION &region, int ccode);
	void RegularKey(USHORT key);
	void ReplaceChar(TEXT_UNDO *undo,	char *buff,	USHORT key);
	int StateInfoChanged(TEXT_STATE *state);
	void StateInfoRestore(TEXT_STATE *state);
	void StateInfoSave(TEXT_STATE *state);
	void UndoOperation(TEXT_UNDO *undo);
	void UpdateCursor(UI_REGION &region);
	int WhiteSpace(int value);
	void WordTabLeft(void);
};

/* ----- Definition of the UI_FORMATTED_STRING class ----- */
class FS_STATE
{
public:
	UCHAR isMarked;
	UCHAR cursor;
	char  *text;

	FS_STATE() { isMarked = FALSE; cursor = 0; text = 0; }
	FS_STATE(FS_STATE& src);
	~FS_STATE(void) { if (text) delete text; }

	void operator = (FS_STATE& src);
	int operator == (FS_STATE& rightOperand);
	int operator != (FS_STATE& rightOperand);
};

class FS_UNDO : public UI_UNDO_RECORD
{
public:
	FS_STATE info;

	FS_UNDO(FS_STATE &src) : info(src) {}
};

class UIW_FORMATTED_STRING : public UI_WINDOW_OBJECT, public UI_EDIT_INFO
{
public:
	UIW_FORMATTED_STRING(int left, int top, int width, char *text,
		char *editMask, char *literalMask, USHORT a_woFlags,
		int (*validate)(void *object, int ccode) = 0 );
	virtual ~UIW_FORMATTED_STRING(void);

	const char *DataGet(void) { return state.text; }
	void DataSet(char *newText);
	virtual int Event(const UI_EVENT &event);

protected:
	UCHAR fieldWidth;
	UCHAR maskLen;
	char *editMask;
	char *literalMask;
	FS_STATE state;

	void BackspaceKey(void);
	int CalcLeading(int width, char *dText);
	void CopyBlock(void);
	void CutBlock(void);
	void DeleteKey(void);
	int Expand(char *expandedText);
	int ExpandedColumn(void);
	void FixText(void);
	void PasteBlock(void);
	void Redisplay(UI_REGION &region, short width, char *dText);
	void RegularKey(USHORT key);
	void SetCursor(int column, int width);
	int TextLength(void);
	void UndoOperation(FS_UNDO *undo);
	void UpdateCursor(UI_REGION &region, int width, char *dText, int cursorColumn);
	int ValidateMask(int editValue, int textValue);
};

/* ----- Definition of the UIW_DATE class ----- */
class UIW_DATE : public UIW_STRING
{
public:
	UIW_DATE(int left, int top, int width, UI_DATE *date, 
		const char *range, USHORT dtFlags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	virtual ~UIW_DATE(void);

	const UI_DATE *DataGet(void);
	void DataSet(UI_DATE *newDate);
	virtual int Event(const UI_EVENT &event);

protected:
	USHORT dtFlags;
	UI_DATE *datePtr;
	UI_DATE date;	 			// Working copy of date.
	char dateWidth;
	char *range;

	void DateError(DTI_RESULT errorCode);
	DTI_RESULT DateRangeCheck(void);
	void DateRangeError(char *formattedString);
};

/* ----- Definition of the UIW_TIME class ----- */

class UIW_TIME : public UIW_STRING
{
public:
	UIW_TIME(int left, int top, int width, UI_TIME *time, char *range,
		USHORT tmFlags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	virtual ~UIW_TIME(void);

	const UI_TIME *DataGet(void);
	void DataSet(UI_TIME *time);
	virtual int Event(const UI_EVENT &event);

protected:
	USHORT tmFlags;
	UI_TIME *timePtr;
	UI_TIME time;						// Working copy of time
	char *range;

	void TimeError(TMI_RESULT errorCode);
	TMI_RESULT TimeRangeCheck(void);
	void TimeRangeError(char *formattedString);
};

/* ----- Definition of the UIW_NUMBER classes ----- */
#define	NUM_MAX_TEXT_LEN		25

#define	NMF_DECIMAL(num)	(((num & 0xF) + 1))
const USHORT NMF_NO_FLAGS			= 0x0000;
const USHORT NMF_DECIMAL_FLOAT		= 0x000F;
const USHORT NMF_CURRENCY			= 0x0010;	// Currency symbol
const USHORT NMF_CREDIT				= 0x0020;	// Credit symbols on negative numbers.
const USHORT NMF_COMMAS				= 0x0040;	// Commas
const USHORT NMF_PERCENT			= 0x0080;	// Percent
const USHORT NMF_SCIENTIFIC			= 0x0100;	// Scientific notation
const USHORT NMF_UNSIGNED			= 0x0200;

enum NUM_TYPE
{
	NUM_CHAR,
	NUM_SHORT,
	NUM_LONG,
	NUM_FLOAT,
	NUM_DOUBLE
};

class NUM_STATE
{
public:
	UCHAR isNegative;
	UCHAR isMarked;
	UCHAR cursor;				// Index to cursor position.
	char text[NUM_MAX_TEXT_LEN];

	int operator == (NUM_STATE& rightOperand);
	int operator != (NUM_STATE& rightOperand);
};

class NUM_UNDO : public UI_UNDO_RECORD
{
public:
	NUM_STATE info;
};

class UIW_NUMBER : public UI_WINDOW_OBJECT, public UI_EDIT_INFO
{
public:
	USHORT nmFlags;

	UIW_NUMBER(int left, int top, int width, signed char *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, unsigned char *a_value, 
		char *a_range, USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, short *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, unsigned short *a_value,
		char *a_range, USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, int *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, unsigned int *a_value, 
		char *a_range, USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, long *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, unsigned long *a_value, 
		char *a_range, USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, float *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	UIW_NUMBER(int left, int top, int width, double *a_value, char *a_range, 
		USHORT flags, USHORT woFlags,
		int (*validate)(void *object, int ccode) = 0);
	virtual ~UIW_NUMBER(void);

	void DataSet(void *value);
	const void *DataGet(void) { return value; }
	virtual int Event(const UI_EVENT &event);

protected:
	NUM_STATE state;
	UCHAR maxTextLen;
	UCHAR insertMode;
	UCHAR decimal;				// 0 = none, 0xFF = floating
	UCHAR type;
	void *value;
	char *range;
	void (UIW_NUMBER::*BinaryToAscii)(void);
	void (UIW_NUMBER::*AsciiToBinary)(void);
	int  (UIW_NUMBER::*ValidNumber)(void);
	UCHAR switchedToReplaceMode;

	void AsciiToReal(void);
	void AsciiToWhole(void);
	NUM_UNDO *AllocateUndo(void);
	void BackspaceKey(void);
	int CalcLeading(int width, char *dText);
	void Constructor(void *value, int width, USHORT flags, char *a_range, 
		void (UIW_NUMBER::*BinaryToAscii)(void),
		void (UIW_NUMBER::*AsciiToBinary)(void),
		int  (UIW_NUMBER::*ValidNumber)(void),
		int (*validate)(void *object, int ccode) = 0);
	void CopyBlock(void);
	void CutBlock(void);
	void DeleteChar(void);
	void DeleteKey(void);
	void Expand(char *expText, int *cursorColumn);
	void FixText(void);
	void FormatNumber(void);
	void FormatNumericBuffer(char *buffer, UCHAR maximumLength);
	void PasteBlock(void);
	int RangeError(char *a_range);
	void RealToAscii(void);
	void Redisplay(UI_REGION &region, int width, char *dText);
	void RegularChar(USHORT key);
	void RegularKey(USHORT key);
	void SetCursor(int column, int width);
	void ShiftDecimal(int shiftCount);
	void UndoOperation(NUM_UNDO *undo);
	void UpdateCursor(UI_REGION &region, int width, char *dText, int cursorColumn);
	int ValidWhole(void);
	int ValidWholeInRange(char *a_range);
	int ValidWholeBetween(char *minValue, char *maxValue);
	int ValidRealBetween(char *minValue, char *maxValue);
	int ValidRealInRange(char *a_range);
	int ValidRealNumber(void);
	int ValidWholeNumber(void);
	void WholeToAscii(void);
};

/* ----- Definition of the ERROR system classes ----- */
class UI_ERROR_SYSTEM
{
public:
	virtual ~UI_ERROR_SYSTEM(void) {}

	virtual void Beep(void);
	virtual void ReportError(UI_WINDOW_MANAGER *windowManager, 
		USHORT objectFlags, char *format, ...);
};

class UI_ERROR_WINDOW_SYSTEM : public UI_ERROR_SYSTEM
{
public:
	UI_ERROR_WINDOW_SYSTEM(void);
	virtual ~UI_ERROR_WINDOW_SYSTEM(void);

	virtual void ReportError(UI_WINDOW_MANAGER *windowManager,
		USHORT objectFlags, char *format, ...);

protected:
	UIW_WINDOW *errorWindow;
	UIW_TEXT *textField;
	UIW_PULL_DOWN_ITEM *continueEditField;
	UIW_PULL_DOWN_ITEM *leaveInvalidField;
	UIW_PULL_DOWN_ITEM *leaveUnansweredField;
	char *message;
};

extern UI_ERROR_SYSTEM *_errorSystem;

/* ----- Definition of the HELP system classes ----- */
const short HELP_MAGIC_WORD   	= 0x05AF;
const short HELP_VERSION		= 1;
const short MAX_HELP_CONTEXTS 	= 10000;

struct HELP_HEADER
{
	short magicWord;
	short helpVersion;
	short numContexts;
	long indexPtr;						// Pointer to the index table.
};

struct TEXT_INDEX
{
	long offset;
	short length;
	short titleIndex;
};

class UI_HELP_SYSTEM
{
public:
	virtual ~UI_HELP_SYSTEM(void) {}

	virtual void DisplayHelp(UI_WINDOW_MANAGER *windowManager,
		int helpContext);
};

class UI_HELP_WINDOW_SYSTEM : public UI_HELP_SYSTEM
{
public:
	UCHAR installed;

	UI_HELP_WINDOW_SYSTEM(char *helpFileName,
		UI_WINDOW_MANAGER *windowManager,
		int defaultHelpContext = NO_HELP_CONTEXT);
	~UI_HELP_WINDOW_SYSTEM(void);

	void DisplayHelp(UI_WINDOW_MANAGER *windowManager, int helpContext);

protected:
	UI_WINDOW_MANAGER *windowManager;
	UIW_WINDOW *helpWindow;
	UIW_TITLE *titleField;
	UIW_TEXT *textField;
	char *title;
	char *message;
	int helpFile;
	int defaultHelpContext;

	HELP_HEADER header;
	TEXT_INDEX *index;
	char *strings;

	void Close(void);
	int Read(void *buff, unsigned length);
	int Seek(long filePos);
};

extern UI_HELP_SYSTEM *_helpSystem;

#endif /* UI_WIN_HPP */
