//	Program name..	Zinc Interface Library
//	Filename......	GDISPLAY.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <stdlib.h>
#include <string.h>
#include <graphics.h>
#include "ui_dsp.hpp"

static UI_PALETTE xorPalette = { '\260', attrib(BLUE, BLACK),
	attrib(MONO_DIM, MONO_BLACK), SOLID_FILL, attrib(LIGHTGRAY, LIGHTGRAY),
	attrib(BW_WHITE, BW_WHITE), attrib(GS_GRAY, GS_GRAY)};
UI_PALETTE *_xorPalette = &xorPalette;

extern void VideoInt(void);

// ----- Constructor & Destructor -------------------------------------------

UI_DOS_BGI_DISPLAY::UI_DOS_BGI_DISPLAY(int requestedDriver, int requestedMode):
	UI_DISPLAY(FALSE, 8, 14)
{
	if (requestedDriver == DETECT)
	{
		// Initialize the display.
		driver = DETECT;
		detectgraph(&driver, &mode);
		if (driver < 0)
			return;
	}
	else
	{
		driver = requestedDriver;
		mode = requestedMode;
	}

	switch(driver)
	{
	case CGA:
		if (displayCode < DC_CGA)
			return;
		break;

	case MCGA:
		if (displayCode < DC_EGA_COLOR)
			return;
		break;

	case EGA:
	case EGAMONO:
	case VGA:
		if (displayCode < DC_EGA_COLOR ||
			// Following is kludge to get around EGA Wonder 3.05 BIOS bug.
			(isActiveDisplay && AlternateDisplayCode()== DC_PS2_ANALOG_COLOR &&
			ActiveDisplayCode()== DC_PS2_ANALOG_MONO))
			return;
		break;

	case HERCMONO:
		if (displayCode != DC_MONOCHROME)
			return;
		break;

	case EGA64:					// 64K EGA not supported.
	case IBM8514:				// IBM 8514 not supported.
	case ATT400:				// AT&T 400 not supported.
	case PC3270:				// PC 3270 not supported.
	default:
		return;
	}

	// Try to initialize the graphics driver.
	int tDriver = -3;
	int tMode;
	_path->SetFileName(0);
	char *pathToDriver = 0;
	do
	{
		tDriver = driver;
		tMode = mode;
		initgraph(&tDriver, &tMode, pathToDriver);
		pathToDriver = _path->NextPathName();
	} while (tDriver == -3 && pathToDriver);
	driver = tDriver;
	mode = tMode;
	if (tDriver < 0)
		return;

	delay(60);	  		// For benefit of suspected VEGA VGA BIOS bug.
	_AH = 0x0F;
	VideoInt();
	videoMode = _AL;
	lines = getmaxy()+ 1;
	columns = getmaxx()+ 1;
	maxColor = getmaxcolor();

	// Fill the screen according to the specified palette.
	settextstyle(DEFAULT_FONT, HORIZ_DIR, 1);
	setfillstyle(_backgroundPalette->fillPattern, MapColor(_backgroundPalette, FALSE));
	setviewport(0, 0, columns - 1, lines - 1, TRUE);
	bar(0, 0, columns - 1, lines - 1);

	// Define the screen display region.
	regionList.Add(0, new UI_REGION_ELEMENT(ID_SCREEN, 0, 0, columns - 1, lines - 1));
	installed = TRUE;
}

UI_DOS_BGI_DISPLAY::~UI_DOS_BGI_DISPLAY(void)
{
	if (installed)
	{
		closegraph();		// Restore the display.
		delay(60);	  		// For benefit of suspected VEGA VGA BIOS bug.
	}
}

// ----- Member functions ---------------------------------------------------

void UI_DOS_BGI_DISPLAY::Bitmap(int screenID, const UI_REGION &a_region,
	const USHORT *bitmap, const UI_PALETTE *palette, int fillBackground)
{
	USHORT width = *bitmap++;
	USHORT height = *bitmap++;
	int wordsPerRow = (width - 1)/ 16 + 1;
	UI_REGION tRegion;
	UI_REGION region = a_region;

	// Compute the region coordinates.
	if (region.right - region.left >= width)
		region.right = region.left + width - 1;
	if (region.bottom - region.top >= height)
		region.bottom = region.top + height - 1;

	// Draw the bitmap on the display.
	int hidden = FALSE;
	for (UI_REGION_ELEMENT *d_region = regionList.First(); d_region; d_region = d_region->Next())
		if (d_region->screenID == screenID &&
			d_region->Overlap(region, tRegion))
		{
			if (!hidden && eventManager)
			{
				hidden = TRUE;
				eventManager->DevicesHide(region);
				setviewport(0, 0, columns - 1, lines - 1, TRUE);
			}
			for (USHORT row = tRegion.top; row <= tRegion.bottom; row++)
			{
				int bit_offset = (row - region.top)* wordsPerRow;
				int col = tRegion.left;
				int colDelta = col - region.left;
				unsigned int bit = (1 << (15 - (colDelta & 0xF)));
				while (col <= tRegion.right)
				{
					if (*(bitmap + bit_offset + colDelta / 16)& bit)
						putpixel(col, row, MapColor(palette, TRUE));
					else if (fillBackground)
						putpixel(col, row, MapColor(palette, FALSE));
					col++;
					colDelta++;
					bit = bit >> 1;
					if (bit == 0)
						bit = 0x8000;
				}
			}
		}

	// Show the screen devices.
	if (hidden && eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::Fill(int screenID, const UI_REGION &region,
	const UI_PALETTE *palette)
{
	UI_REGION tRegion;

	// Fill a zone on the display.
	int hidden = FALSE;
	for (UI_REGION_ELEMENT *d_region = regionList.First(); d_region; d_region = d_region->Next())
		if (d_region->screenID == screenID &&
			d_region->Overlap(region, tRegion))
		{
			if (!hidden && eventManager)
			{
				hidden = TRUE;
				eventManager->DevicesHide(region);
				setfillstyle(palette->fillPattern, MapColor(palette, FALSE));
				setviewport(0, 0, columns - 1, lines - 1, TRUE);
			}
			bar(tRegion.left, tRegion.top, tRegion.right, tRegion.bottom);
		}

	// Show the screen devices.
	if (hidden && eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::FillXOR(const UI_REGION &region)
{
	// Hide the screen devices.
	if (eventManager)
		eventManager->DevicesHide(region);

	// Draw an XOR rectangle on the display.
	setwritemode(XOR_PUT);
	setcolor(MapColor(_xorPalette, TRUE));
	setviewport(0, 0, columns - 1, lines - 1, TRUE);
	for (int i = region.top; i <= region.bottom; i++)
		line(region.left, i, region.right, i);
	setwritemode(COPY_PUT);

	// Show the screen devices.
	if (eventManager)
		eventManager->DevicesShow(region);
}

#pragma argsused
void UI_DOS_BGI_DISPLAY::Line(int screenID, int left, int top, int right,
	int bottom, const UI_PALETTE *palette, int width)
{
	UI_REGION tRegion;

	// Set up the line region.
	UI_REGION region;
	region.left = left;
	region.top = top;
	region.right = right;
	region.bottom = bottom;

	// Draw a line on the display.
	int hidden = FALSE;
	for (UI_REGION_ELEMENT *d_region = regionList.First(); d_region; d_region = d_region->Next())
		if (d_region->screenID == screenID &&
			d_region->Overlap(region, tRegion))
		{
			if (!hidden && eventManager)
			{
				hidden = TRUE;
				eventManager->DevicesHide(region);
				setcolor(MapColor(palette, FALSE));
				setviewport(0, 0, columns - 1, lines - 1, TRUE);
			}
			line(tRegion.left, tRegion.top, tRegion.right, tRegion.bottom);
		}

	// Show the screen devices.
	if (hidden && eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::MakeActive(void)
{
	if (equipmentFlags != *((UCHAR far *)0x410L))
	{
		*((UCHAR far *)0x410L)= equipmentFlags;
		_AX = (USHORT)videoMode | 0x80;
		VideoInt();
		delay(60);	  		// For benefit of suspected VEGA VGA BIOS bug.
	}
}

int UI_DOS_BGI_DISPLAY::MapColor(const UI_PALETTE *palette, int foreground)
{
	UCHAR colors;
	if (maxColor == 15)
		colors = palette->color;
	else if (maxColor == 1)
		colors = palette->bwColor;
	else
		colors = palette->grayScaleColor;
	return foreground ? colors & 0xF : (colors >> 4)& 0xF;
}

#pragma argsused
void UI_DOS_BGI_DISPLAY::Rectangle(int screenID, const UI_REGION &region,
	const UI_PALETTE *palette, int width)
{
	UI_REGION tRegion;

	// Draw a box on the display.
	int hidden = FALSE;
	for (UI_REGION_ELEMENT *d_region = regionList.First(); d_region; d_region = d_region->Next())
		if (d_region->screenID == screenID &&
			d_region->Overlap(region, tRegion))
		{
			if (!hidden && eventManager)
			{
				hidden = TRUE;
				eventManager->DevicesHide(region);
				setcolor(MapColor(palette, FALSE));
			}
			setviewport(tRegion.left, tRegion.top, tRegion.right, tRegion.bottom, TRUE);
			rectangle(region.left - tRegion.left, region.top - tRegion.top,
				region.right - tRegion.left, region.bottom - tRegion.top);
		}

	// Show the screen devices.
	if (hidden && eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::RectangleXOR(const UI_REGION &region)
{
	// Hide the screen devices.
	if (eventManager)
		eventManager->DevicesHide(region);

	// Draw an XOR rectangle on the display.
	setwritemode(XOR_PUT);
	setcolor(MapColor(_xorPalette, TRUE));
	setviewport(0, 0, columns - 1, lines - 1, TRUE);
	rectangle(region.left, region.top, region.right, region.bottom);
	setwritemode(COPY_PUT);

	// Show the screen devices.
	if (eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::RectangleXORDiff(const UI_REGION &oldRegion,
	const UI_REGION &newRegion)
{
	// See if the regions match.
	if (oldRegion.left == newRegion.left && oldRegion.top == newRegion.top &&
		oldRegion.right == newRegion.right && oldRegion.bottom == newRegion.bottom)
		return;

	// Hide the screen devices.
	UI_REGION region;
	region.left = min(oldRegion.left, newRegion.left);
	region.top = min(oldRegion.top, newRegion.top);
	region.right = max(oldRegion.right, newRegion.right);
	region.bottom = max(oldRegion.bottom, newRegion.bottom);
	if (eventManager)
		eventManager->DevicesHide(region);

	// Draw the XOR rectangles on the display.
	setwritemode(XOR_PUT);
	setcolor(MapColor(_xorPalette, TRUE));
	setviewport(0, 0, columns - 1, lines - 1, TRUE);
	rectangle(oldRegion.left, oldRegion.top, oldRegion.right, oldRegion.bottom);
	rectangle(newRegion.left, newRegion.top, newRegion.right, newRegion.bottom);
	setwritemode(COPY_PUT);

	// Show the screen devices.
	if (eventManager)
		eventManager->DevicesShow(region);
}

void UI_DOS_BGI_DISPLAY::RegionConvert(UI_REGION &region)
{
	// These variables handle Text<-->Graphics coordinate problems.
	int width = (region.right - region.left)* 8;
	int height = (region.bottom - region.top)* 14;

	region.left *= 8;
	region.top *= 14;
	region.right = region.left + width + 7;
	region.bottom = region.top + height + 12;
}

void UI_DOS_BGI_DISPLAY::Text(int screenID, int left, int top,
	const char *text, const UI_PALETTE *palette, int length,
	int fillBackground)
{
	UI_REGION tRegion;
	UI_REGION region;
	int offset;

	// Convert the cell coordinates.
	int height = textheight((char *)text); // Typecast Borland bug
	int cell_width = textwidth("W");
	int width = (length == -1 ? textwidth((char *)text): length * cell_width); // Typecast Borland bug
	region.left = left;
	region.top = top;
	region.right = region.left + width - 1;
	region.bottom = region.top + height - 1;

	// Draw the text on the display.
	int hidden = FALSE;
	for (UI_REGION_ELEMENT *d_region = regionList.First(); d_region; d_region = d_region->Next())
	{
		if (d_region->screenID == screenID && d_region->Overlap(region, tRegion))
		{
			int text_left;
			int text_top;

			if (tRegion.left + cell_width <= tRegion.right + 1)
			{		// Patch for Borland setviewport bug.
				if (!hidden && eventManager)
				{
					hidden = TRUE;
					eventManager->DevicesHide(region);
					if (fillBackground)
						setfillstyle(palette->fillPattern, MapColor(palette, FALSE));
					setcolor(MapColor(palette, TRUE));
				}
				if (tRegion.top > region.top || tRegion.bottom < region.bottom)
				{
					// This is the case where the text is going to get 
					// clipped because it overlaps a region boundary.  Try to
					// find any regions of the same ID that are adjacent
					// above or below and coalesce them temporarily.
					for (UI_REGION_ELEMENT *a_reg = regionList.First(); a_reg; a_reg = a_reg->Next())
						if (a_reg->screenID == screenID &&
							a_reg != d_region &&
							a_reg->region.left < tRegion.right &&
							a_reg->region.right > tRegion.left)
						{
							if ((tRegion.top > region.top &&
			   					tRegion.top == a_reg->region.bottom + 1)||
									(tRegion.bottom < region.bottom &&
				 					tRegion.bottom == a_reg->region.top - 1))
							{
								tRegion.left = max(tRegion.left, a_reg->region.left);
								tRegion.top = min(tRegion.top, a_reg->region.top);
								tRegion.right = min(tRegion.right, a_reg->region.right);
								tRegion.bottom = max(tRegion.bottom, a_reg->region.bottom);
							}
						}
				}
				setviewport(tRegion.left, tRegion.top, tRegion.right, tRegion.bottom, TRUE);
				if (region.left < tRegion.left)
				{
					offset = (tRegion.left - region.left - 1)/ cell_width + 1;
					text_left = offset * cell_width - tRegion.left + region.left;
				}
				else
				{
					offset = 0;
					text_left = region.left - tRegion.left;
				}
				text_top = region.top - tRegion.top;
				if (fillBackground)
					bar(text_left, text_top, text_left + width - 1, text_top + height - 1);
				outtextxy(text_left, text_top, (char *)&text[offset]); // Typecast Borland bug
			}
		}
	}

	// Show the screen devices.
	if (hidden && eventManager)
		eventManager->DevicesShow(region);
}

int UI_DOS_BGI_DISPLAY::TextHeight(const char *string)
{
	return textheight((char *)string); // Typecast Borland bug
}

int UI_DOS_BGI_DISPLAY::TextWidth(const char *string)
{
	return textwidth((char *)string); // Typecast Borland bug
}

void UI_DOS_BGI_DISPLAY::TextXOR(int left, int top, int length)
{
	// Set up the text region.
	UI_REGION region;
	region.left = left;
	region.top = top;
	region.right = region.left + length * textwidth("W")- 1;
	region.bottom = region.top + textheight("W")- 1;

	// Hide the screen devices.
	if (eventManager)
		eventManager->DevicesHide(region);

	// Draw an XOR rectangle on the display.
	setwritemode(XOR_PUT);
	setcolor(MapColor(_xorPalette, TRUE));
	setviewport(0, 0, columns - 1, lines - 1, TRUE);
	for (int i = region.top; i <= region.bottom; i++)
		line(region.left, i, region.right, i);
	setwritemode(COPY_PUT);

	// Show the screen devices.
	if (eventManager)
		eventManager->DevicesShow(region);
}

