//	Program name..	Zinc Interface Library
//	Filename......	KEYBOARD.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#pragma inline

#include <dos.h>
#include "ui_evt.hpp"

UCHAR UI_BIOS_KEYBOARD::breakHandlerSet = FALSE;

int enhancedBiosFlag;

static int CheckEnhancedBios(void)
{
	asm {
		mov		ah, 5
		mov		cx, 0FFFFH
		int		16H
		or		al, al
		jnz		NotEnhanced

		mov		cx, 16
	}

readLoop:
	asm {
		push	cx
		mov		ah, 10H
		int		16H
		pop		cx
		cmp		ax, 0FFFFH
		je		EnhancedKeyboard

		loop	readLoop
	}

NotEnhanced:
	return (0x00);

EnhancedKeyboard:
	return (0x10);
}

static int KeystrokeWaiting(void)
{
	_AH = 1 + enhancedBiosFlag;
	geninterrupt(0x16);
	asm		jz	no_key

	return (TRUE);
no_key:
	return (FALSE);
}

UI_BIOS_KEYBOARD::UI_BIOS_KEYBOARD(
	USHORT initialState) :
	UI_DEVICE(E_KEY, initialState)
{
	/* Initialize the keyboard device */
	enhancedBiosFlag = enhancedBIOS = CheckEnhancedBios();
	_AX = 0x3300;
	geninterrupt(0x21);			// Get state of Control-Break checking
	breakState = _DL;			// and save it
	if (!breakHandlerSet)
	{
		breakHandlerSet = TRUE;
		asm	{
			push	ds
			lea		dx, IntRet
			push	cs
			pop		ds
			mov		ax, 2523H
			int		21H
			pop		ds
			jmp		short SetDone

IntRet:		iret				// Note: this produces a warning, but works OK.
		}

SetDone:;
	}
	_DL = 0;
	_AX = 0x3301;
	geninterrupt(0x21);			// Now disable Control-Break handling
}

UI_BIOS_KEYBOARD::~UI_BIOS_KEYBOARD(void)
{
	_DL = breakState;			// Restore original break checking state.
	_AX = 0x3301;
	geninterrupt(0x21);
}

void UI_BIOS_KEYBOARD::Poll(void)
{
	/* See if a keystroke is already waiting */
	if (!KeystrokeWaiting())
	{
		// Check the ALT key at the BIOS level.
		UCHAR shiftState = *((UCHAR far *) 0x417L) & 0x0F;
		if ( FlagSet(shiftState, S_ALT) && altPressed == ALT_NOT_PRESSED )
			altPressed = ALT_PRESSED_AND_NO_EVENTS_YET;
		else if ( !FlagSet(shiftState, S_ALT) )
		{
			if (altPressed == ALT_PRESSED_AND_NO_EVENTS_YET)
			{
				UI_EVENT event;
				event.rawCode = 0;
				event.key.value = 0;
				event.key.shiftState = 0;
				event.type = S_ALT_KEY;
				if (state != D_OFF)
					eventManager->Put(event, Q_END);
			}
			altPressed = ALT_NOT_PRESSED;
		}
		return;
	}

	/* Set the key information and call the event manager */
	if (altPressed == ALT_PRESSED_AND_NO_EVENTS_YET)
		altPressed = ALT_PRESSED_AND_EVENTS_RECEIVED;
	UI_EVENT event;
	_AH = enhancedBIOS;
	geninterrupt(0x16);
	event.rawCode = _AX;
	event.key.value = _AL;
	event.key.shiftState = *((UCHAR far *) 0x417L) & 0x0F;
	event.type = E_KEY;
	if (enabled && eventManager)
		eventManager->Put(event, Q_END);
}

int UI_BIOS_KEYBOARD::Event(
	const UI_EVENT &event)
{
	/* Switch on the rawCode */
	switch (event.rawCode)
	{
	case D_OFF:
	case D_ON:
		state = event.rawCode;
		enabled = (event.rawCode == D_OFF) ? FALSE : TRUE;
		break;
	}

	/* Return the keyboard state */
	return (state);
}

