//	Program name..	Zinc Interface Library
//	Filename......	NUMBERE.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

#define	NMF_DECIMAL_FLAGS	0x000F

extern int ValidNumChar(char val, int decimalPoint);
extern void PlaceCommas(char *buff);
extern void PlaceDecimal(char *buff, int decimal);
extern void FixNegative(char *buff, int credit);

// Constructor & Destructor -------------------------------------------------

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, char *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_CHAR;
	Constructor(a_value, width, flags, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, unsigned char *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_CHAR;
	Constructor(a_value, width, flags | NMF_UNSIGNED, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, int *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_SHORT;
	Constructor(a_value, width, flags, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, unsigned int *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_SHORT;
	Constructor(a_value, width, flags | NMF_UNSIGNED, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, short *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_SHORT;
	Constructor(a_value, width, flags, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, unsigned short *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_SHORT;
	Constructor(a_value, width, flags | NMF_UNSIGNED, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, long *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_LONG;
	Constructor(a_value, width, flags, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::UIW_NUMBER(int left, int top, int width, unsigned long *a_value,
	char *a_range, USHORT flags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	type = NUM_LONG;
	Constructor(a_value, width, flags | NMF_UNSIGNED, a_range,
		UIW_NUMBER::WholeToAscii, UIW_NUMBER::AsciiToWhole,
		UIW_NUMBER::ValidWholeNumber, validate);
}

UIW_NUMBER::~UIW_NUMBER(void)
{
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete value;
	if (range)
		delete range;
}

// Member functions ---------------------------------------------------------

void UIW_NUMBER::DataSet(void *newValue)
{
	if (newValue)
	{
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			value = newValue;
		else
			memcpy(value, newValue, sizeof(double));
	}
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

int UIW_NUMBER::Event(const UI_EVENT &event)
{
	int viewOnly = (FlagSet(woFlags, WOF_VIEW_ONLY) ||
		FlagSet(woAdvancedStatus, WOAS_TOO_SMALL)) ? TRUE : FALSE;
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_STRING);
	if (ccode == S_CREATE)
	{
		UndoDestroy(undoHandle);
		undoHandle = 0;
		FormatNumber();
	}
	short cellWidth = display->cellWidth;
	UI_REGION region;
	UI_WINDOW_OBJECT::Border(0, region, 0);
	short width = (region.right + 1 - region.left) / cellWidth;
	USHORT key;
	NUM_STATE oldStateInfo = state;
	UCHAR forceDisplay = FALSE;
	UI_PALETTE *palette;
	int isCurrent = woStatus & WOS_CURRENT;
	UCHAR reDisplay = TRUE;
	UCHAR displayCheckRegion = FALSE;
	int needValidate = NeedsValidation();

	if (!isCurrent)
		state.isMarked = FALSE;
	switch (ccode)
	{
	case E_KEY:
		key = event.rawCode & 0xFF;
		if ((key == '\b' || key >= ' ') && !viewOnly)
		{
			if (key == '\b')
				BackspaceKey();
			else
				RegularKey(key);
		}
		else
		{
			reDisplay = FALSE;
			ccode = S_UNKNOWN;
		}
		break;

	case L_MOVE_LEFT:
		if (state.cursor > 0)
		{
			state.cursor--;
			woStatus |= WOS_NO_AUTO_CLEAR;
		}
		else
			ccode = S_UNKNOWN;
		break;

	case L_MOVE_RIGHT:
		if (state.text[state.cursor])
		{
			state.cursor++;
			woStatus |= WOS_NO_AUTO_CLEAR;
		}
		else
			ccode = S_UNKNOWN;
		break;

	case L_MOVE_BOL:
		state.cursor = 0;
		woStatus |= WOS_NO_AUTO_CLEAR;
		break;

	case L_MOVE_EOL:
		state.cursor = ui_strlen(state.text);
		woStatus |= WOS_NO_AUTO_CLEAR;
		break;

	case L_DELETE:
		if (!viewOnly)
		{
			if (state.isMarked)
				CutBlock();
			else
				DeleteKey();
		}
		break;

	case L_INSERT_TOGGLE:
		insertMode = !insertMode;
		woStatus |= WOS_NO_AUTO_CLEAR;
		forceDisplay = TRUE;
		break;

	case L_MARK:
		state.isMarked = !state.isMarked;
		break;

	case L_MOVE:
		eventManager->DeviceState(event.type, DM_EDIT);
		break;

	case L_CUT:
		if (!viewOnly)
		{
			if (state.isMarked)
				CutBlock();
			else
				reDisplay = FALSE;
			break;
		}
		// Continue to L_COPY_MARK if view only field.

	case L_COPY_MARK:
		if (state.isMarked)
			CopyBlock();
		else
			reDisplay = FALSE;
		break;

	case L_CUT_PASTE:
		if (state.isMarked)
		{
			if (viewOnly)
				CopyBlock();
			else
				CutBlock();
			break;
		}
		// Continue to L_PASTE.

	case L_PASTE:
		if (!viewOnly)
		{
			if (event.position.line != -1)
				SetCursor((event.position.column - region.left) / cellWidth, width);
			PasteBlock();
		}
		break;

	case L_UNDO:
		{
			NUM_UNDO *undo = (NUM_UNDO *)UndoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPush(undoHandle);
			}
		}
		break;

	case L_REDO:
		{
			NUM_UNDO *undo = (NUM_UNDO *)UndoRedoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPop(undoHandle);
			}
		}
		break;

	case L_CONTINUE_MARK:
		if (markColumn != (event.position.column - region.left) / cellWidth)
			state.isMarked = TRUE;
		break;

	case L_BEGIN_MARK:
		state.isMarked = FALSE;
		markColumn = (event.position.column - region.left) / cellWidth;
		SetCursor(markColumn, width);
		forceDisplay = TRUE;			// Force cursor to get updated.
		break;

	case S_CURRENT:
		switchedToReplaceMode = FALSE;
		if (needValidate)
			(*Validate)(this, ccode);

		// Clear the WOS_UNANSWERED and WOS_NO_AUTO_CLEAR bits
		woStatus &= ~(WOS_UNANSWERED | WOS_NO_AUTO_CLEAR);
		isCurrent = TRUE;
		if (!undoHandle)
			undoHandle = UndoCreate();
		palette = UI_WINDOW_OBJECT::LogicalPalette(ccode);
		displayCheckRegion = TRUE;
		break;

	case S_DISPLAY_ACTIVE:
		palette = UI_WINDOW_OBJECT::LogicalPalette(ccode);
		displayCheckRegion = TRUE;
		break;

	case S_NON_CURRENT:
		if (!FlagSet(woStatus, WOS_UNANSWERED))
		{
			if ((this->*ValidNumber)())
				(this->*AsciiToBinary)();
			else
				ccode = S_ERROR;
			if (ccode == S_ERROR ||
			 	(needValidate && (*Validate)(this, ccode) != 0))
			{
				ccode = S_ERROR;
				woStatus |= WOS_INVALID;
				reDisplay = FALSE;
				break;
			}
		}
		FormatNumber();							// Re-format from the binary.
		woStatus &= ~WOS_INVALID;
		// Continue to S_DISPLAY_INACTIVE.

	case S_DISPLAY_INACTIVE:
		state.isMarked = FALSE;
		palette = UI_WINDOW_OBJECT::LogicalPalette(ccode);
		displayCheckRegion = TRUE;
		break;

	case S_ERROR_RESPONSE:
		forceDisplay = TRUE;
		woStatus |= event.rawCode;
		if (event.rawCode == WOS_UNANSWERED)
			FormatNumber();
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		reDisplay = FALSE;
		break;
	}

	if (displayCheckRegion)
	{
		forceDisplay = TRUE;
		if (!UI_WINDOW_OBJECT::NeedsUpdate(event, ccode) &&
			oldStateInfo == state)
			reDisplay = FALSE;
		else
			UI_WINDOW_OBJECT::Border(ccode, region, palette);
	}

	char dText[256];
	int cursorColumn;
	if (reDisplay)
	{
		FixText();
		if (ccode != L_UNDO && ccode != L_REDO)
		{
			NUM_STATE newStateInfo = state;
			if (oldStateInfo != state)
			{
				state = oldStateInfo;
				AllocateUndo();						// Save original for undo.
				state = newStateInfo;
			}
		}
		Expand(dText, &cursorColumn);
		int maxWidth = min(maxTextLen - 1, width - 1);
		if (strlen(dText) > maxWidth &&
			(oldStateInfo != state || ccode == E_KEY))
			_errorSystem->Beep();
		if (forceDisplay || oldStateInfo != state)
		{
			Redisplay(region, width, dText);
			UpdateCursor(region, width, dText, cursorColumn);
		}
	}
	else if (ccode == S_CURRENT)
	{
		Expand(dText, &cursorColumn);
		UpdateCursor(region, width, dText, cursorColumn);
	}

	// Return the control code.
	return (ccode);
}

void UIW_NUMBER::FormatNumber(void)
{
	(this->*BinaryToAscii)();
	if (state.text[0] == '-')
	{
		strcpy(state.text, state.text + 1);
		state.isNegative = TRUE;
	}
	else
		state.isNegative = FALSE;
	state.cursor = 0;				// Default to cursor at beginning.
	FixText();
}

NUM_UNDO *UIW_NUMBER::AllocateUndo(void)
{
	NUM_UNDO *undo = new NUM_UNDO;
	undo->info = state;
	UndoAdd(undoHandle, undo, sizeof(NUM_UNDO));
	return undo;
}

void UIW_NUMBER::BackspaceKey(void)
{
	if (state.cursor > 0)
	{
		state.cursor--;
		DeleteChar();
	}
}

int UIW_NUMBER::CalcLeading(int width, char *dText)
{
	int maxWidth = min(maxTextLen - 1, width - 1);
	int textLen = strlen(dText);
	int leading = 0;
	if (woFlags & WOF_JUSTIFY_CENTER)
		leading = min((maxWidth - textLen) / 2, width);
	else if (woFlags & WOF_JUSTIFY_RIGHT)
		leading = min(maxWidth - textLen, width);
	if (leading < 0)
		leading = 0;
	return leading;
}

void UIW_NUMBER::Constructor(void *a_value, int width, USHORT flags,
	char *a_range, void (UIW_NUMBER::*a_BinaryToAscii)(void),
	void (UIW_NUMBER::*a_AsciiToBinary)(void),
	int (UIW_NUMBER::*a_ValidNumber)(void),
	int (*a_Validate)(void *object, int ccode))
{
	windowID[0] = ID_STRING;

	if (type < NUM_FLOAT)
		flags &= ~NMF_SCIENTIFIC;	// Scientific not allowed for whole nums.
	if ( FlagSet(flags, NMF_SCIENTIFIC) )
	{
		flags |= NMF_DECIMAL_FLOAT;	// Fixed-point not allowed w/scientific.
		flags &= ~NMF_COMMAS;		// Nor are commas.
	}
	BinaryToAscii = a_BinaryToAscii;
	AsciiToBinary = a_AsciiToBinary;
	ValidNumber   = a_ValidNumber;
	Validate = a_Validate;
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		value = a_value;
	else
	{
		value = new double;		// NOTE: This doesn't suck floating point code.
		if (a_value)
			memcpy(value, a_value, sizeof(double));
		else
			woStatus |= WOS_UNANSWERED;
	}
	GetCountryInfo();			// Insure we have country info.
	insertMode = TRUE;
	decimal = flags & NMF_DECIMAL_FLAGS;
	if ((type >= NUM_FLOAT && decimal == 0) || decimal == NMF_DECIMAL_FLAGS)
		decimal = 0xFF;
	else if (decimal)
		decimal--;
	// Disallow floating decimal point with whole number types.
	if (type < NUM_FLOAT && decimal == 0xFF)
		decimal = 0;
	nmFlags = flags;
	range = (a_range) ? ui_strdup(a_range) : 0;
	state.isMarked = FALSE;
	if (FlagSet(woFlags, WOF_BORDER))
		if (display->isText)
			width -= 2;
		else
			width -= 1;
	maxTextLen = min(width + 1, NUM_MAX_TEXT_LEN);
}

void UIW_NUMBER::CopyBlock(void)
{
	char dText[256];
	int  cursorColumn;

	Expand(dText, &cursorColumn);
	pasteLength = (USHORT) strlen(dText);
	if (pasteBuffer)
		delete pasteBuffer;
	pasteBuffer = new char[pasteLength];
	if (pasteBuffer)
		memcpy(pasteBuffer, dText, pasteLength);
	state.isMarked = FALSE;
}

void UIW_NUMBER::CutBlock(void)
{
	woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
	CopyBlock();
	state.text[0] = '\0';
	state.cursor = 0;
}

void UIW_NUMBER::DeleteChar(void)
{
	woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
	if (state.text[state.cursor] == 'E')		// Delete exponent.
		state.text[state.cursor] = '\0';
	else 
		memmove(state.text + state.cursor, state.text + state.cursor + 1,
			strlen(state.text + state.cursor) );
}

void UIW_NUMBER::DeleteKey(void)
{
	if (state.text[state.cursor])
		DeleteChar();
}

void UIW_NUMBER::Expand(char *expText, int *cursorColumn)
{
	char *currencyFmt;
	char newText[256];

	newText[0] = '\0';
	if (FlagSet(nmFlags, NMF_CURRENCY))
	{
		currencyFmt = (_countryInfo.co_currstyle < 2 ? "%s%s" : "%s %s");
		if ( _countryInfo.co_currstyle == 0 ||
			_countryInfo.co_currstyle == 2 )
			sprintf(newText, currencyFmt, _countryInfo.co_curr, state.text);
		else
			sprintf(newText, currencyFmt, state.text, _countryInfo.co_curr);
	}
	else
		strcpy(newText, state.text);
	if (FlagSet(nmFlags, NMF_PERCENT))
		strcat(newText, "%");
	PlaceDecimal(newText, decimal);
	if (FlagSet(nmFlags, NMF_COMMAS))
		PlaceCommas(newText);
	if (state.isNegative)
		FixNegative(newText, nmFlags & NMF_CREDIT);
	strcpy(expText, newText);
	int match = 0;
	for (int index = 0; index <= state.cursor && newText[match]; index++)
	{
		while (newText[match] != state.text[index] && newText[match] &&
			newText[match] != '%' && newText[match] != ')')
			match++;
		if (index < state.cursor && newText[match] == state.text[index])
			match++;
	}
	*cursorColumn = match;
}

void UIW_NUMBER::FixText(void)
{
	if (state.text[0] == '\0')
	{
		strcpy(state.text, "0");
		state.cursor = 1;
	}
	if (decimal && decimal != 0xFF)
	{
		while (strlen(state.text) <= decimal)
		{
			memmove(state.text + 1, state.text, strlen(state.text) + 1);
			state.cursor++;
			state.text[0] = '0';
		}
	}
	// Strip leading zeroes, but only if we have enough decimal places.
	while (state.text[0] == '0' && isdigit(state.text[1]) )
	{
		if (decimal && decimal != 0xFF && strlen(state.text) <= decimal + 1)
			break;
		memmove(state.text, state.text + 1, strlen(state.text) );
		if (state.cursor)
			state.cursor--;
	}
}

void UIW_NUMBER::PasteBlock(void)
{
	char *src = pasteBuffer;
	int src_len = pasteLength;

	if (pasteBuffer && !state.isMarked)
	{
		woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
		state.cursor = 0;
		while (state.cursor < maxTextLen - 1 && src_len)
		{
			if (ValidNumChar(*src, decimal))
				state.text[state.cursor++] = *src;
			src++;
			src_len--;
		}
		state.text[state.cursor] = '\0';
		state.cursor = 0;
	}
}

void UIW_NUMBER::Redisplay(UI_REGION &region, int width, char *dText)
{
	UI_PALETTE *palette = lastPalette;
	// Pre-fill for graphics or unanswered field.
	if (!display->isText || (woStatus & WOS_UNANSWERED))
		display->Fill(screenID,	region.left, region.top, region.right,
			region.bottom, palette);
	int textLen = strlen(dText);
	int leading = CalcLeading(width, dText);
	textLen = min(textLen, width - leading);
	int trailing = width - textLen - leading;
	int textHeight = display->TextHeight("W");
	int bottom = region.top + textHeight - 1;
	if (bottom <= region.bottom && (woStatus & WOS_UNANSWERED) == 0)
	{
		if (!display->isText)			// Center text vertically.
		{
			int vertRoom = region.bottom + 1 - region.top - textHeight;
			region.top += (vertRoom + 1) / 2;
			region.bottom += (vertRoom + 1) / 2;
		}
		if (leading && display->isText)
			display->Fill(screenID, region.left, region.top,
				region.left + leading - 1, region.bottom, palette);
		display->Text(screenID, region.left + leading * display->cellWidth,
			region.top, dText, palette, textLen, FALSE);
		if (trailing && display->isText)
			display->Fill(screenID, region.left + leading + textLen,
				region.top, region.left + leading + textLen + trailing - 1,
				region.bottom, palette);
		if (state.isMarked)
			display->TextXOR(region.left, region.top, width);
	}
}

void UIW_NUMBER::RegularChar(USHORT key)
{
	int autoClear = FALSE;
	if (FlagSet(woFlags, WOF_AUTO_CLEAR) &&
		!FlagSet(woStatus, WOS_NO_AUTO_CLEAR))
	{
		autoClear = TRUE;
		int index = state.cursor;
		while (index)
		{
			index--;
			if (state.text[index] != '0')
				autoClear = FALSE;
		}
	}
	if (autoClear)
	{
		state.text[0] = key;
		state.text[1] = '\0';
		state.cursor = 1;
	}
	else
	{
		if (!insertMode && state.text[state.cursor])
			DeleteChar();		// Replace mode - delete and then insert.
		if (ui_strlen(state.text) < maxTextLen - 1)
		{
			memmove(state.text + state.cursor + 1, state.text + state.cursor,
				strlen(state.text + state.cursor) + 1);
			state.text[state.cursor++] = key;
		}
		if (state.text[state.cursor] == '\0' && switchedToReplaceMode)
		{
			insertMode = TRUE;
			switchedToReplaceMode = FALSE;
		}
	}
}

void UIW_NUMBER::RegularKey(USHORT key)
{
	char valid_list[256];

	sprintf(valid_list, "0123456789%c",
		decimal ? _countryInfo.co_desep[0] : '\0');
	if (!FlagSet(nmFlags, NMF_UNSIGNED))
		strcat(valid_list, "-+");
	if (FlagSet(nmFlags, NMF_SCIENTIFIC))
		strcat(valid_list, "Ee");
	if (strchr(valid_list, key))
	{
		if (key == _countryInfo.co_desep[0])
		{
			if (decimal == 0xFF)	// Floating decimal - replace the point.
			{
				char *exp = strchr(state.text, 'E');
				if (!exp || exp >= state.text + state.cursor)
				{	// Don't allow decimal inside the exponent.
			 		char *dec = strchr(state.text, _countryInfo.co_desep[0]);
					if (dec && (insertMode || state.text[state.cursor] != *dec))
					{
						memmove(dec, dec + 1, strlen(dec));
						if (state.text + state.cursor > dec)
							state.cursor--;
					}
					RegularChar(key);
				}
			}
			else					// Fixed decimal.
			{
				int nullPos = ui_strlen(state.text);
				int position = (int)(nullPos - state.cursor);
				if (position < decimal)
				{
					int extra = decimal - position;
					memset(state.text + nullPos, '0', extra);
					state.text[nullPos + extra] = '\0';
				}
				else if (position > decimal)
				{
					int clip = position - decimal;
					state.text[nullPos - clip] = '\0';
				}
				if (insertMode)
				{
					insertMode = FALSE;
					switchedToReplaceMode = TRUE;
				}
			}
		}
		else if (key == '-' || key == '+')
		{
			char *exp;
			if (FlagSet(nmFlags, NMF_SCIENTIFIC) &&
				(exp = strchr(state.text, 'E')) &&
				state.text + state.cursor >= exp)
			{
				exp++;		// Point just after the 'E'.
				if (*exp == '-')
				{
					memmove(exp, exp + 1, strlen(exp));
					if (state.text + state.cursor >= exp + 1)
						state.cursor--;
				}
				else if (key == '-')
				{
					memmove(exp + 1, exp, strlen(exp) + 1);
					*exp = '-';
					if (state.text + state.cursor >= exp)
						state.cursor++;
				}
			}
			else
				state.isNegative = key == '+' ? FALSE : !state.isNegative;
		}
		else if (toupper(key) == 'E')
		{
			char *exp = strchr(state.text, 'E');
			if (exp)
			{
				state.cursor = (UCHAR)(exp + 1 - state.text);
				if (state.text[state.cursor] == '-')
					state.cursor++;
			}
			else
			{
				state.cursor = ui_strlen(state.text);
				RegularChar('E');
			}
		}
		else // digit
			RegularChar(key);
		woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
	}
}

void UIW_NUMBER::SetCursor(int column, int width)
{
	char expandedText[256];
	int cursorColumn;
	int maxWidth = min(maxTextLen - 1, width - 1);

	Expand(expandedText, &cursorColumn);
	int textLen = strlen(expandedText);
	int leading = 0;
	if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		leading = min((maxWidth - textLen) / 2, width);
	else if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		leading = min(maxWidth - textLen, width);
	if (leading < 0)
		leading = 0;
	char *screen_cursor = expandedText + column - leading;
	char *expNull = strchr(expandedText, '\0');
	if (screen_cursor > expNull)
		screen_cursor = expNull;
	int match = 0;
	for (char *cp = expandedText; cp < screen_cursor && state.text[match]; cp++)
		if (strchr(state.text, *cp))
			match++;
	state.cursor = match;
}

void UIW_NUMBER::UndoOperation(NUM_UNDO *undo)
{
	NUM_STATE stateInfo = state;

	state = undo->info;
	undo->info = stateInfo;
}

void UIW_NUMBER::UpdateCursor(UI_REGION &region, int width, char *dText,
	int cursorColumn)
{
	int leading = CalcLeading(width, dText);
	if (FlagSet(woStatus, WOS_CURRENT) && (leading + cursorColumn < width))
	{
		int trueColumn =
			region.left + (leading + cursorColumn) * display->cellWidth;
		if (!FlagSet(woFlags, WOF_VIEW_ONLY) &&
			!FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) && trueColumn >= 0 && 
			trueColumn < display->columns && region.top >= 0 && 
			region.top < display->lines)
		{
			eventManager->DevicePosition(E_CURSOR, trueColumn, region.top);
			eventManager->DeviceState(E_CURSOR, ((insertMode) ? DC_INSERT : DC_OVERTYPE));
		}
	}
}

void UIW_NUMBER::WholeToAscii(void)
{
	switch (type)
	{
	case NUM_CHAR:
		if (FlagSet(woStatus, WOS_UNANSWERED))
			*((char *)value) = 0;
		sprintf(state.text, "%d", nmFlags & NMF_UNSIGNED ?
			*((unsigned char *)value) : *((char *)value));
		break;

	case NUM_SHORT:
		if (FlagSet(woStatus, WOS_UNANSWERED))
			*((short *)value) = 0;
		sprintf(state.text, nmFlags & NMF_UNSIGNED ? "%u" : "%d",
			nmFlags & NMF_UNSIGNED ? *((unsigned short *)value) :
			*((short *)value));
		break;

	case NUM_LONG:
		if (FlagSet(woStatus, WOS_UNANSWERED))
			*((long *)value) = 0;
		sprintf(state.text, nmFlags & NMF_UNSIGNED ? "%lu" : "%ld",
			nmFlags & NMF_UNSIGNED ? *((unsigned long *)value) :
			*((long *)value));
		break;
	}
}

void UIW_NUMBER::AsciiToWhole(void)
{
	switch (type)
	{
	case NUM_CHAR:
		if (FlagSet(nmFlags, NMF_UNSIGNED))
			*((unsigned char *)value) = atoi(state.text);
		else
		{
			*((char *)value) = atoi(state.text);
			if (state.isNegative)
				*((char *)value) = - *((char *)value);
		}
		break;

	case NUM_SHORT:
		if (FlagSet(nmFlags, NMF_UNSIGNED))
			*((unsigned short *)value) = (unsigned short) atol(state.text);
		else
		{
			*((short *)value) = atoi(state.text);
			if (state.isNegative)
				*((short *)value) = - *((short *)value);
		}
		break;

	case NUM_LONG:
		if (FlagSet(nmFlags, NMF_UNSIGNED))
			*((unsigned long *)value) = strtoul(state.text, 0, 10);
		else
		{
			*((long *)value) = atol(state.text);
			if (state.isNegative)
				*((long *)value) = - *((long *)value);
		}
		break;
	}
}
