//	Program name..	Zinc Interface Library
//	Filename......	TEXT.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

extern char *AdvanceLine(char *ptr, USHORT numLines, int col, USHORT *actual,
	int width);

TEXT_UNDO::TEXT_UNDO(void)
{
	textAdd = 0;
	operation = MOVE_POSITION;
}

TEXT_UNDO::~TEXT_UNDO(void)
{
	if (textAdd)
		delete textAdd;
}
 
// Constructor & Destructor -------------------------------------------------

UIW_TEXT::UIW_TEXT(int left, int top, int a_width, int a_height,
	char *a_text, short maxLength, USHORT a_txFlags, USHORT woFlags,
	int (*validate)(void *object, int ccode)) :
	UIW_STRING(left, top, a_width, a_text, maxLength, STF_NO_FLAGS, woFlags,
		validate)
{
	windowID[0] = (woFlags & WOF_VIEW_ONLY) ? ID_TEXT_VIEW : ID_TEXT;
	windowID[1] = ID_STRING;
	txFlags = a_txFlags;

	relative.bottom = relative.top + a_height - 1;
	true.bottom = true.top + a_height - 1;
	cellHeight = -1;
}

UIW_TEXT::~UIW_TEXT(void)
{
}

// Member functions ---------------------------------------------------------

int _scrollCount;

int UIW_TEXT::Event(const UI_EVENT &event)
{
	int viewOnly = FlagSet(woFlags, WOF_VIEW_ONLY) ||
		FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) ? TRUE : FALSE;
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event,
		viewOnly ? ID_TEXT_VIEW : ID_TEXT);
	UI_REGION region;
	char *ptr;
	int row;
	char *nextLine;
	int numLines;
	USHORT key;
	USHORT count;
	UCHAR reDisplay = TRUE;
	UCHAR displayCheckRegion = FALSE;
	_scrollCount = 0;

	if (ccode == S_CREATE || ccode == S_SIZE)
	{
		UndoDestroy(undoHandle);
		undoHandle = 0;
		cellHeight = (display->isText) ?
			display->cellHeight : display->cellHeight - 2;
		cellWidth = display->cellWidth;
		if (ccode == S_CREATE)
		{
			markedBlock = 0;
			screenTop = text;
			cursor = text;
			cursorDesiredCol = -1;
		}
		else
			cursor = screenTop;

		// Determine the size of the text field.  We have to do this 
		// twice to make sure both fields are properly initialized.
		UI_WINDOW_OBJECT::Event(event);
		ComputeRegion(0, region, 0);
	}
	else
		ComputeRegion(0, region, 0);

	// Switch on the event type.
	switch (ccode)
	{
	case E_KEY:
		key = event.rawCode & 0xFF;
		if ((key == '\b' || key == '\r' || key >= ' ') && !viewOnly)
		{
			if (key == '\b')
				BackspaceKey();
			else if (key == '\r')
			{
				SetMark();
				if (markStart)
					CopyBlock();
				else
					EnterKey();
			}
			else
				RegularKey(key);
		}
		else
		{
			reDisplay = FALSE;
			ccode = S_UNKNOWN;
		}
		break;

	case L_MOVE_LEFT:
		if (cursor == text)
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			LeftArrow();
		}
		break;

	case L_MOVE_RIGHT:
		if (*cursor)
		{
			MoveOperation();
			cursor++;
			if (*cursor == '\n' && *(cursor - 1) == '\r')
				cursor++;
			cursorDesiredCol = -1;
		}
		else
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		break;

	case S_SCROLL_VERTICAL:
		MoveOperation();
		if (event.scroll.delta < 0)
		{
			screenTop = AdvanceLine(screenTop, -event.scroll.delta, 0, 0, width);
			cursor = AdvanceLine(cursor, -event.scroll.delta, 0, 0, width);
		}
		else
		{
			screenTop = PreviousLine(screenTop, event.scroll.delta, 0, text);
			cursor = PreviousLine(cursor, event.scroll.delta, 0, text);
		}
		if (!screenTop)
			screenTop = text;
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			cursor = screenTop;
		reDisplay = TRUE;
		break;

	case L_MOVE_UP:
		if (screenTop == text && cursorRow == 0)
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			MoveUp(viewOnly && cursorRow ? cursorRow + 1 : 1, ccode);
		}
		break;

	case L_MOVE_DOWN:
		if (LastLine())
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			MoveDown(!viewOnly || (cursorRow == height - 1) ? 1 :
				height - cursorRow, ccode);
		}
		break;

	case L_MOVE_PAGE_UP:
		if (screenTop == text && cursorRow == 0)
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			numLines = cursorRow == 0 ?
				height - 1 : viewOnly ? cursorRow + height - 1 : cursorRow;
			MoveUp(numLines, ccode);
		}
		break;

	case L_MOVE_PAGE_DOWN:
		if (LastLine())
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			numLines = cursorRow == height - 1 ?
				height - 1 : height - 1 - cursorRow +
				(viewOnly ? height - 1 : 0);
			MoveDown(numLines, ccode);
		}
		break;

	case L_MOVE_TOP:
		MoveOperation();
		screenTop = text;
		cursor = text;
		cursorDesiredCol = 0;
		break;

	case L_MOVE_BOTTOM:
		MoveOperation();
		MoveDown(0xFFFF, ccode);
		while (*cursor)
			cursor++;
		cursorDesiredCol = -1;
		break;

	case L_MOVE_BOL:
		MoveOperation();
		Home();
		break;

	case L_MOVE_EOL:
		if (*cursor)
		{
			MoveOperation();
			cursor = AdvanceLine(cursor, 1, cursorCol, &count, width);
			if (count)
				LeftArrow();
			cursorDesiredCol = -1;
		}
		break;

	case L_WORD_TAB_RIGHT:
		if (*cursor)
		{
			MoveOperation();
			nextLine = AdvanceLine(cursor, 1, cursorCol, &count, width);
			while (NonWhiteSpace(*cursor))
				cursor++;
			while (*cursor && WhiteSpace(*cursor))
				cursor++;
			row = cursorRow;
			while (count && cursor >= nextLine)
			{
				row++;
				if (row >= height)
				{
					screenTop = AdvanceLine(screenTop, 1, 0, 0, width);
					row--;
				}
				nextLine = AdvanceLine(nextLine, 1, 0, &count, width);
			}
			cursorDesiredCol = -1;
		}
		break;

	case L_WORD_TAB_LEFT:
		MoveOperation();
		WordTabLeft();
		break;

	case L_INSERT_TOGGLE:
		insertMode = !insertMode;
		break;

	case L_DELETE:
		if (!viewOnly)
		{
			SetMark();
			if (markStart)
				CutBlock();
			else
				DeleteKey();
		}
		break;

	case L_DELETE_EOL:
		if (!viewOnly)
			DeleteEol();
		break;

	case L_DELETE_WORD:
		if (!viewOnly)
			DeleteWord();
		break;

	case L_MARK:
		AllocateUndo(); 	// Remember mark as separate operation.
		markedBlock = (markedBlock) ? 0 : cursor;
		break;

	case L_UNDO:
		{
			TEXT_UNDO *undo = (TEXT_UNDO *) UndoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPush(undoHandle);
			}
		}
		break;

	case L_REDO:
		{
			TEXT_UNDO *undo = (TEXT_UNDO *) UndoRedoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPop(undoHandle);
			}
		}
		break;

	case L_MOVE:
		eventManager->DeviceState(event.type, DM_EDIT);
		reDisplay = FALSE;
		break;

	case L_CUT:
		if (!viewOnly)
		{
			SetMark();	// Set markStart & markTail according to original cursor.
			MoveOperation();
			ComputeMousePosition((event.position.column - region.left) / cellWidth,
				(event.position.line   - region.top)  / cellHeight);
			UpdateCursor(region);
			if (markStart)
				CutBlock();
			break;
		}
		// If view only, continue to L_COPY_MARK.

	case L_COPY_MARK:
		SetMark();
		if (markStart)
			CopyBlock();
		break;

	case L_CUT_PASTE:
		SetMark();
		if (markStart)
		{
			if (viewOnly)
				CopyBlock();
			else
				CutBlock();
			break;
		}
		// Continue to L_PASTE.

	case L_PASTE:
		if (!viewOnly)
		{
			if (event.position.line != -1)
			{
				ComputeMousePosition((event.position.column - region.left) / cellWidth,
					(event.position.line   - region.top)  / cellHeight);
				UpdateCursor(region);
			}
			PasteBlock();
		}
		break;

	case L_END_MARK:
		if (cursor == markedBlock)
			markedBlock = 0;
		reDisplay = FALSE;
		break;

	case L_CONTINUE_MARK:
		if (markedBlock)
		{
			MoveOperation();
			ComputeMousePosition((event.position.column - region.left) / cellWidth,
				(event.position.line - region.top) / cellHeight);
			UpdateCursor(region);
			break;
		}
		// Continue to L_BEGIN_MARK.

	case L_BEGIN_MARK:
		SetMark();	// Set markStart & markTail according to original cursor.
		ComputeMousePosition((event.position.column - region.left) / cellWidth,
			(event.position.line - region.top) / cellHeight);
		UpdateCursor(region);
		AllocateUndo(); 			// Remember mark as separate operation.
		markedBlock = cursor;		// Throw away possible old mark.
		break;

	case S_ERROR_RESPONSE:
		woStatus |= event.rawCode;
		break;

	case S_CURRENT:
		if (NeedsValidation())
			(*Validate)(this, ccode);
		// Clear the WOS_UNANSWERED and WOS_NO_AUTO_CLEAR bits
		woStatus &= ~(WOS_UNANSWERED | WOS_NO_AUTO_CLEAR);
		if (!undoHandle)
			undoHandle = UndoCreate();
		// Continue to S_DISPLAY_ACTIVE.

	case S_DISPLAY_ACTIVE:
		displayCheckRegion = TRUE;
		break;

	case S_NON_CURRENT:
		if (NeedsValidation() && (*Validate)(this, ccode) != 0)
		{
			ccode = S_ERROR;
			woStatus |= WOS_INVALID;
			break;
		}
		woStatus &= ~WOS_INVALID;
		// Continue to S_DISPLAY_INACTIVE.

	case S_DISPLAY_INACTIVE:
		displayCheckRegion = TRUE;
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		reDisplay = FALSE;
		break;
	}

	if (displayCheckRegion)
	{
		if (!UI_WINDOW_OBJECT::NeedsUpdate(event, ccode))
		{
			if ((ccode == S_NON_CURRENT || ccode == S_DISPLAY_INACTIVE) &&
				 markedBlock)
				markedBlock = 0;		 	// Throw away mark and re-display.
			else
				reDisplay = FALSE;
		}
		else
		{
			if (ccode == S_NON_CURRENT || ccode == S_DISPLAY_INACTIVE)
				markedBlock = 0;
			screenInvalid = TRUE;
			ComputeRegion(ccode, region, lastPalette);
		}
	}

	if (reDisplay)
	{
		/* Compute possible new top of screen. */
		for (;;)
		{
			ptr = screenTop;
			row = 0;
			if (width <= 0)
				break;
			while (*ptr && row < height)
			{
				ptr = AdvanceLine(ptr, 1, 0, &count, width);
				row += count;
			}
			if (row >= height && ptr <= cursor)
				screenTop = AdvanceLine(screenTop, 1, 0, 0, width);
			else
				break;
		}

		Redisplay(region, ccode);
	}
	else if (ccode == S_CURRENT)
		UpdateCursor(region);

	if (_scrollCount > 1 && previous)
		ComputeScroll();

	// Return the control code.
	return (ccode);
}

void UIW_TEXT::DataSet(char *newText, short maxLength)
{
	if (newText)
	{
		short oldMaxLength = (short)(textTail + 1 - text);
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			text = newText;
		else
		{
			if (maxLength != -1 && maxLength > oldMaxLength)
			{
				delete text;
				text = new char[maxLength];
			}
			strcpy(text, newText);
		}
		textTail = (maxLength != -1) ?
			text + maxLength - 1 : text + oldMaxLength - 1;
		screenTop = text;
		cursor = text;
		cursorDesiredCol = -1;
	}
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

