//	Program name..	Zinc Interface Library
//	Filename......	WINDOW.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include "ui_win.hpp"

// ----- Static variables ---------------------------------------------------

UI_EVENT_MAP *UI_WINDOW_MANAGER::eventMapTable = _eventMapTable;
int UI_WINDOW_MANAGER::screenID = 1;

// ----- Constructor & Destructor -------------------------------------------

UI_WINDOW_MANAGER::UI_WINDOW_MANAGER(UI_DISPLAY *a_display,
	UI_EVENT_MANAGER *a_eventManager) :
	windowList()
{
	// Initialize the screen manager information.
	display = a_display;
	eventManager = a_eventManager;
}

UI_WINDOW_MANAGER::~UI_WINDOW_MANAGER(void)
{
	// Remove all non-destroy objects from the window manager.
	for (UI_WINDOW_OBJECT *object = First(); object; )
	{
		object->display = 0;
		object->eventManager = 0;
		object->windowManager = 0;
		object = (FlagSet(object->woAdvancedFlags, WOAF_NO_DESTROY)) ?
			(UI_WINDOW_OBJECT *)windowList.Subtract(object) : object->Next();
	}
}

// ----- Member functions ---------------------------------------------------

void UI_WINDOW_MANAGER::Add(UI_WINDOW_OBJECT *object)
{
	// Fix the coordinate system.
	int offset;
	UI_REGION region = object->true;
	if ((display->isText && FlagSet(object->woStatus, WOS_GRAPHICS)) ||
		(!display->isText && !FlagSet(object->woStatus, WOS_GRAPHICS)))
	{
		display->RegionConvert(object->true);
		if (display->isText)
			object->woStatus &= ~WOS_GRAPHICS;
		else
			object->woStatus |= WOS_GRAPHICS;
	}

	// Only new window objects have a screenID of -1.
	if (object->screenID == -1)
	{
		region.left = (region.left >= 0) ? object->true.left :
			display->columns + region.left * display->cellWidth - 1;
		region.top = (region.top >= 0) ? object->true.top :
			display->lines + region.top * display->cellHeight - 1;
		region.right = (region.right > 0) ? object->true.right :
			display->columns + region.right * display->cellWidth - 1;
		region.bottom = (region.bottom > 0) ? object->true.bottom :
			display->lines + region.bottom * display->cellHeight - 1;
		if (region.left < 0)
			offset = -region.left;
		else if (region.right >= display->columns)
			offset = display->columns - region.right;
		else
			offset = 0;
		if (region.left + offset >= 0)
		{
			region.left += offset;
			region.right += offset;
		}
		if (region.top < 0)
			offset = -region.top;
		else if (region.bottom >= display->lines)
			offset = display->lines - region.bottom;
		else
			offset = 0;
		if (region.top + offset >= 0)
		{
			region.top += offset;
			region.bottom += offset;
		}
		object->true = region;
	}

	// Make sure the window object is on the screen.
	region = object->true;
	if (region.left >= display->columns)
	{
		offset = region.left - display->columns + 1;
		region.left -= offset;
		region.right -= offset;
	}
	if (region.top >= display->lines)
	{
		offset = region.top - display->lines + 1;
		region.top -= offset;
		region.bottom -= offset;
	}

	// Initialize the window object and bring it to the screen front.
	if (object->screenID == -1)
		object->screenID = ++screenID;
	object->display = display;
	object->eventManager = eventManager;
	object->windowManager = this;
	object->woFlags |= WOF_NON_FIELD_REGION;
	object->relative = object->true = region;

	// Paint the new window object.
	UI_EVENT event;
	event.type = S_CREATE;
	object->Event(event);
	ToFront(object, TRUE, TRUE);
}

void UI_WINDOW_MANAGER::Erase(UI_WINDOW_OBJECT *object,
	const UI_REGION *newRegion)
{
	// Redefine all the objects on the screen.
	UI_REGION region;
	UI_WINDOW_OBJECT *tObject;
	region.left = region.top = 0;
	region.right = display->columns - 1;
	region.bottom = display->lines - 1;
	display->RegionDefine(ID_SCREEN, region);
	for (tObject = Last(); tObject; tObject = tObject->Previous())
		if (tObject != object)
			display->RegionDefine(tObject->screenID, tObject->true);
		else if (newRegion)
			display->RegionDefine(tObject->screenID, *newRegion);

	// Redraw the affected screen objects.
	UI_EVENT event;
	event.region = object->true;

	// Update the front window object.
	eventManager->DeviceState(E_CURSOR, D_OFF);
	event.type = S_CURRENT;
	display->Fill(ID_SCREEN, object->true, _backgroundPalette);
	if (object == First() && object->next && !newRegion)
	{
		object = object->Next();
		object->Event(event);
		if (object->true.left <= event.region.left &&
			object->true.top <= event.region.top &&
			object->true.right >= event.region.right &&
			object->true.bottom >= event.region.bottom)
			return;
	}

	event.type = S_DISPLAY_INACTIVE;
	for (tObject = object->Next(); tObject; tObject = tObject->Next())
	{
		if (tObject->Overlap(event.region))
			tObject->Event(event);
		if (tObject->true.left <= event.region.left &&
			tObject->true.top <= event.region.top &&
			tObject->true.right >= event.region.right &&
			tObject->true.bottom >= event.region.bottom)
			return;
	}
}

int UI_WINDOW_MANAGER::Event(const UI_EVENT &event)
{
	// Make sure there is an object.
	int ccode = MapEvent(eventMapTable, event, ID_WINDOW_MANAGER, ID_WINDOW_MANAGER);
	UI_WINDOW_OBJECT *firstObject = First();
	if (!firstObject && ccode == L_EXIT)
		return (L_EXIT);
	else if (!firstObject && ccode != S_REDISPLAY)
		return (S_NO_OBJECT);

	// Switch on the event type.
	UI_EVENT tEvent = event;
	UI_WINDOW_OBJECT *object = 0;
	switch (ccode)
	{
	case L_WINDOW_MOVE:
	case L_WINDOW_SIZE:
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
			UI_WINDOW_MANAGER::Subtract(firstObject);
		object = First();
		if (ccode == L_WINDOW_MOVE)
		{
			tEvent.type = S_MOVE;
			tEvent.position.column = object->true.left;
			tEvent.position.line = object->true.top;
		}
		else
		{
			tEvent.type = S_SIZE;
			tEvent.rawCode = M_RIGHT_CHANGE | M_BOTTOM_CHANGE;
			tEvent.position.column = object->true.right;
			tEvent.position.line = object->true.bottom;
		}
		Modify(object, tEvent);
		break;

	case S_MOVE:
	case S_SIZE:
	case S_CHANGE:
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
			UI_WINDOW_MANAGER::Subtract(firstObject);
		object = First();
		Modify(object, event);
		break;

	case S_MAXIMIZE:
	case S_MINIMIZE:
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
			UI_WINDOW_MANAGER::Subtract(firstObject);
		object = First();
		object->Event(event);
		break;

	case S_DELETE:
	case S_DELETE_LEVEL:
		// Delete any temporary windows.
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (ccode == S_DELETE_LEVEL)
				return (ccode);
		}
		// Access a new window if the front can't be removed.
		else if (FlagSet(firstObject->woAdvancedFlags, WOAF_LOCKED))
		{
			for (object = First(); object; object = object->Next())
				if (!FlagSet(object->woAdvancedFlags, WOAF_LOCKED))
				{
					ToFront(object, TRUE, FALSE);
					break;
				}
			break;
		}

		// Get the first unlocked window to delete;
		object = First();
		if (!object)
			return (S_ERROR);
		if (!FlagSet(object->woAdvancedFlags, WOAF_LOCKED))
		{
			UI_WINDOW_MANAGER::Subtract(object);
			if (!FlagSet(object->woAdvancedFlags, WOAF_NO_DESTROY))
				delete object;
			ccode = (First()) ? S_DELETE : S_NO_OBJECT;
		}
		break;

	case S_REDISPLAY:
		eventManager->DeviceState(E_CURSOR, D_OFF);
		display->Fill(ID_SCREEN, 0, 0, display->columns - 1, 
			display->lines - 1, _backgroundPalette);
		for (object = Last(); object; object = object->Previous())
		{
			tEvent.type = (object == First()) ? S_CURRENT : S_DISPLAY_INACTIVE;
			tEvent.region = object->true;
			object->Event(tEvent);
		}
		break;

	case L_EXIT:
		break;

	case L_WINDOW_NEXT:
		// See if it is a modal window.
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL))
		{
			_errorSystem->Beep();
			break;
		}

		// Move to the proper window.
		if (First() != Last())
			ToFront(Last(), TRUE, FALSE);
		break;

	case L_GENERAL_HELP:
		_helpSystem->DisplayHelp(this, NO_HELP_CONTEXT);
		break;

	case L_VIEW:
	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		// Find the proper screen object.
		for (object = First(); object; object = object->Next())
			if (object->Overlap(event.position))
				break;

		// See which object should be current, if any.
		if (object != firstObject &&
			FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL))
		{
			tEvent.rawCode = DM_VIEW;
			eventManager->Event(tEvent);
			if (ccode == L_BEGIN_SELECT)
				_errorSystem->Beep();
			break;
		}
		else if (ccode == L_BEGIN_SELECT && object && object != First())
			ToFront(object, TRUE, FALSE);
		else if (ccode != L_BEGIN_SELECT && event.rawCode != 0)
			object = First();
		if (!object)
		{
			tEvent.rawCode = DM_VIEW;
			eventManager->Event(tEvent);
			break;
		}
		// Continue to default.

	default:
		// Get the front object.
		if (!object)
			object = First();
		ccode = object->Event(event);
		if (ccode != S_UNKNOWN && ccode != S_ERROR)
			break;

		// Check for hot key matches.
		int tHotKey = E_KEY;
		if (!FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL) &&
			event.type == E_KEY && FlagSet(event.key.shiftState, S_ALT) &&
			(tHotKey = MapEvent(_hotKeyMapTable, tEvent, ID_WINDOW_OBJECT)) != E_KEY)
		{
			tEvent.type = S_CHECK_HOT_KEY;
			tEvent.key.value = tHotKey;
			for (object = firstObject->Next(); object; object = object->Next())
				if (object->hotKey != 0 &&
					(tHotKey == object->hotKey ||
					 (object->hotKey == HOT_KEY_SUB_WINDOW && object->Event(tEvent))))
				{
					ToFront(object, TRUE, FALSE);
					ccode = object->Event(event);
					break;
				}
		}
		break;
	}

	// Return the control code.
	return (ccode);
}

void UI_WINDOW_MANAGER::Modify(UI_WINDOW_OBJECT *object,
	const UI_EVENT &event)
{
	int xJump = display->cellWidth;
	int yJump = display->cellHeight;

	// Make sure the object can be sized or moved.
	int ccode = event.type;
	if (ccode == S_SIZE && FlagSet(object->woAdvancedFlags, WOAF_NO_SIZE) ||
		ccode == S_MOVE && FlagSet(object->woAdvancedFlags, WOAF_NO_MOVE))
		return;

	// See if the new coordinates are already selected.
	UI_EVENT tEvent = event;
	if (ccode == S_CHANGE)
	{
		UI_WINDOW_MANAGER::Erase(object, &tEvent.region);
		tEvent.type = S_SIZE;
		object->true = tEvent.region;
		object->Event(tEvent);
		ToFront(object, FALSE, TRUE);
		return;
	}

	// Compute the absolute size of the object.
	int minHeight, minWidth;
	UI_REGION newRegion = object->true;
	object->Information(GET_MINIMUM_HEIGHT, &minHeight);
	object->Information(GET_MINIMUM_WIDTH, &minWidth);
	int height = newRegion.bottom - newRegion.top + 1;
	int width = newRegion.right - newRegion.left + 1;
	UI_REGION absolute = newRegion;
	UI_REGION oldRegion = newRegion;
	USHORT sizeObject = (event.type == S_SIZE) ? event.rawCode : 0;
	int deltaX = (FlagSet(sizeObject, M_RIGHT_CHANGE)) ? 
		newRegion.right - event.position.column :
		event.position.column - newRegion.left;
	int deltaY = (FlagSet(sizeObject, M_BOTTOM_CHANGE)) ?
		newRegion.bottom - event.position.line :
		event.position.line - newRegion.top;
	if (FlagSet(sizeObject, M_LEFT_CHANGE))
		absolute.left = absolute.right + 1 - minWidth;
	else if (FlagSet(sizeObject, M_RIGHT_CHANGE))
		absolute.right = absolute.left + minWidth - 1;
	if (FlagSet(sizeObject, M_TOP_CHANGE))
		absolute.top = absolute.bottom + 1 - minHeight;
	else if (FlagSet(sizeObject, M_BOTTOM_CHANGE))
		absolute.bottom = absolute.top + minHeight - 1;

	// Reverse the window region.
	int move = TRUE;
	int xCount;
	int yCount;
	display->RectangleXOR(newRegion);
	do
	{
		eventManager->Get(tEvent, Q_NORMAL);
		ccode = MapEvent(eventMapTable, tEvent, ID_WINDOW_OBJECT, ID_WINDOW_OBJECT);
		switch (ccode)
		{
		case L_INSERT_TOGGLE:
			xJump = (xJump == 1) ? display->cellWidth : 1;
			yJump = (yJump == 1) ? display->cellHeight : 1;
			break;

		case E_KEY:
			if (tEvent.rawCode != ESCAPE)
				break;
			display->RectangleXOR(oldRegion);
			return;

		case L_SELECT:
		case L_END_SELECT:
			move = FALSE;
			break;

		case L_FIELD_UP:
		case L_FIELD_DOWN:
		case L_FIELD_LEFT:
		case L_FIELD_RIGHT:
			xCount = yCount = 0;
			if (ccode == L_FIELD_UP)
				yCount = -yJump;
			else if (ccode == L_FIELD_DOWN)
				yCount = yJump;
			else if (ccode == L_FIELD_LEFT)
				xCount = -xJump;
			else if (ccode == L_FIELD_RIGHT)
				xCount = xJump;
			if (!sizeObject)
			{
				tEvent.position.column = oldRegion.left + xCount;
				tEvent.position.line = oldRegion.top + yCount;
			}
			else
			{
				tEvent.position.column = oldRegion.right + xCount;
				tEvent.position.line = oldRegion.bottom + yCount;
			}
			// Continue to L_CONTINUE_SELECT

		case L_BEGIN_SELECT:
		case L_CONTINUE_SELECT:
			if (!sizeObject)
			{
				newRegion.left = tEvent.position.column - deltaX;
				newRegion.top = tEvent.position.line - deltaY;
				break;
			}
			if (FlagSet(sizeObject, M_LEFT_CHANGE) &&
				tEvent.position.column <= absolute.left + deltaX)
				newRegion.left = tEvent.position.column - deltaX;
			else if (FlagSet(sizeObject, M_LEFT_CHANGE))
				newRegion.left = absolute.left;
			else if (FlagSet(sizeObject, M_RIGHT_CHANGE) &&
				tEvent.position.column + deltaX >= absolute.right)
				newRegion.right = tEvent.position.column + deltaX;
			else if (FlagSet(sizeObject, M_RIGHT_CHANGE))
				newRegion.right = absolute.right;

			if (FlagSet(sizeObject, M_TOP_CHANGE) &&
				tEvent.position.line <= absolute.top + deltaY)
				newRegion.top = tEvent.position.line - deltaY;
			else if (FlagSet(sizeObject, M_TOP_CHANGE))
				newRegion.top = absolute.top;
			else if (FlagSet(sizeObject, M_BOTTOM_CHANGE) &&
				tEvent.position.line + deltaY >= absolute.bottom)
				newRegion.bottom = tEvent.position.line + deltaY;
			else if (FlagSet(sizeObject, M_BOTTOM_CHANGE))
				newRegion.bottom = absolute.bottom;
			height = newRegion.bottom + 1 - newRegion.top;
			width = newRegion.right + 1 - newRegion.left;
			break;
		}

		// Update the new region.
		if (oldRegion.left != newRegion.left || oldRegion.top != newRegion.top ||
			oldRegion.right != newRegion.right || oldRegion.bottom != newRegion.bottom)
		{
			// Compute the lower-right coordinates.
			newRegion.right = newRegion.left + width - 1;
			newRegion.bottom = newRegion.top + height - 1;

			// Remove the old region and update the new region.
			if (eventManager->Get(tEvent, Q_NO_BLOCK | Q_NO_DESTROY) != 0 ||
				MapEvent(eventMapTable, tEvent, ID_WINDOW_OBJECT, ID_WINDOW_OBJECT) != L_CONTINUE_SELECT)
			{
				display->RectangleXORDiff(oldRegion, newRegion);
				oldRegion = newRegion;
			}
		}
	} while (move);

	// Restore the region.
	display->RectangleXORDiff(newRegion, newRegion);
	display->RectangleXOR(oldRegion);
	if (object->true.left == newRegion.left &&
		object->true.top == newRegion.top &&
		object->true.right == newRegion.right &&
		object->true.bottom == newRegion.bottom)
		return;

	// Deactivate the old window region.
	UI_WINDOW_MANAGER::Erase(object, &newRegion);
	if (!sizeObject)
	{
		tEvent.position.column = newRegion.left - object->true.left;
		tEvent.position.line = newRegion.top - object->true.top;
		tEvent.type = S_MOVE;
	}
	else
		tEvent.type = S_SIZE;
	object->true = object->relative = newRegion;
	object->Event(tEvent);
	ToFront(object, FALSE, TRUE);
}

void UI_WINDOW_MANAGER::Subtract(UI_WINDOW_OBJECT *object)
{
	// Make sure the object is in the window list.
	if (windowList.Index(object) != -1)
	{
		UI_WINDOW_MANAGER::Erase(object, 0);
		windowList.Subtract(object);
	}
	eventManager->DeviceState(E_DEVICE, DM_VIEW);

	UI_EVENT event;
	event.type = S_CLEAR;
	object->Event(event);
	object->display = 0;
	object->eventManager = 0;
	object->windowManager = 0;
}

void UI_WINDOW_MANAGER::ToFront(UI_WINDOW_OBJECT *object, int refreshOld,
	int newWindow)
{
	// Find the maximum update region.
	UI_REGION region = { 0x0FFF, 0x0FFF, 0x0000, 0x0000 };
	for (UI_WINDOW_OBJECT *tObject = First();
		tObject && tObject != object; tObject = tObject->Next())
		if (object->Overlap(tObject->true))
		{
			region.left = min(region.left, tObject->true.left);
			region.top = min(region.top, tObject->true.top);
			region.right = max(region.right, tObject->true.right);
			region.bottom = max(region.bottom, tObject->true.bottom);
		}

	// Delete any temporary windows.
	UI_WINDOW_OBJECT *firstObject = First();
	if (firstObject && FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
	{
		firstObject->woStatus &= ~WOS_CURRENT;
		UI_WINDOW_MANAGER::Erase(firstObject, 0);
		windowList.Subtract(firstObject);
		firstObject = First();
		if (!firstObject)
			return;
	}

	// Bring the object to the front of the object queue.
	if (firstObject != object)
	{
		if (windowList.Index(object) != -1)
			windowList.Subtract(object);
		windowList.Add(firstObject, object);
	}

	// Validate the object's region.
	display->RegionDefine(object->screenID, object->true);

	// Inactivate the old object, then activate the new object.
	UI_EVENT event;
	if (refreshOld && firstObject && firstObject != object &&
		!FlagSet(object->woAdvancedFlags, WOAF_TEMPORARY))
	{
		event.type = S_DISPLAY_INACTIVE;
		event.region.left = event.region.top = event.region.right =
			event.region.bottom = -1;
		firstObject->Event(event);
	}
	object->woStatus |= WOS_CURRENT;
	event.type = S_CURRENT;
	event.region = (newWindow) ? object->true : region;
	object->Event(event);
}

