//	Program name..	Zinc Interface Library
//	Filename......	WINDOW1.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <string.h>
#include "ui_win.hpp"

// ----- Static variables & functions ---------------------------------------

static int _windowHeight[2] = { 2, 28 };
static int _windowWidth[2] = { 15, 140 };

UIW_WINDOW::FindStatus(void *object, void *matchData)
{
	if (FlagSet(((UI_WINDOW_OBJECT *)object)->woStatus, *(USHORT *)matchData))
		return (0);
	return (-1);
}

// ----- Constructor & Destructor -------------------------------------------

UIW_WINDOW::UIW_WINDOW(int left, int top, int width, int height,
	USHORT a_woFlags, USHORT a_woAdvancedFlags, int a_helpContext) :
	UI_WINDOW_OBJECT(left, top, width, height, a_woFlags, a_woAdvancedFlags),
	windowList(),
	regionList()
{
	// Initialize the window information.
	windowID[0] = ID_WINDOW;
	max_woAdvancedFlags = WOAF_NO_FLAGS;
	max_woAdvancedStatus = WOAS_NO_STATUS;
	max_region.left = max_region.top = max_region.right =
		max_region.bottom = -1;
	min_woAdvancedFlags = WOAF_NO_FLAGS;
	min_woAdvancedStatus = WOAS_NO_STATUS;
	min_region.left = min_region.top =  min_region.right =
		min_region.bottom = -1;
	helpContext = a_helpContext;
}

// ----- Member functions ---------------------------------------------------

UIW_WINDOW *UIW_WINDOW::GENERIC(int left, int top, int width, int height,
	USHORT woFlags, USHORT woAdvancedFlags, int helpContext, char *title)
{
	UIW_WINDOW *window = new UIW_WINDOW(left, top, width, height, woFlags,
		woAdvancedFlags, helpContext);

	// Add default window objects.
	*window
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ UIW_SYSTEM_BUTTON::GENERIC()
		+ new UIW_TITLE(title);

	// Return a pointer to the new window.
	return (window);
}

void UIW_WINDOW::Add(UI_WINDOW_OBJECT *object)
{
	// Add the object to the list.
	windowList.Add(object);
	object->InformationSet(screenID, display, eventManager,
		windowManager, paletteMapTable, this);
	woAdvancedStatus |= WOAS_REDISPLAY;
}

int UIW_WINDOW::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	int tcode;
	UI_WINDOW_OBJECT *object = 0, *tObject = 0;
	UI_REGION_ELEMENT *dRegion;
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_WINDOW);
	switch (ccode)
	{
	case S_ERROR_RESPONSE:
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (!object)
			break;
		else if (event.rawCode == 0)
		{
			UI_EVENT tEvent;
			tEvent.type = S_CURRENT;
			tEvent.region = object->true;
			ccode = object->Event(tEvent);
			errorObject = 0;
		}
		else
		{
			ccode = object->Event(event);
			ToFront(errorObject, FALSE);
		}
		break;

	case S_DELETE:
		for (object = First(); object; object = object->Next())
		{
			object->Event(event);
			object->InformationSet(screenID, 0, 0, 0, paletteMapTable, this);
		}
		break;

	case S_CREATE:
	case S_SIZE:
		if (parent)
			UI_WINDOW_OBJECT::Event(event);
		windowList.current = 0;
		for (object = First(); object; object = object->Next())
		{
			if (FlagSet(object->woStatus, WOS_CURRENT))
				windowList.current = object;
			object->InformationSet(screenID, display, eventManager,
				windowManager, paletteMapTable, this);
			object->Event(event);
		}
 		UIW_WINDOW::RegionsCompute();
		break;

	S_CLEAR:
		for (object = First(); object; object = object->Next())
		{
			object->Event(event);
			object->display = 0;
			object->eventManager = 0;
			object->windowManager = 0;
		}
		break;

	case S_MOVE:
		UI_WINDOW_OBJECT::Event(event);
		windowList.current = 0;
		for (object = First(); object; object = object->Next())
		{
			if (FlagSet(object->woStatus, WOS_CURRENT))
				windowList.current = object;
			object->Event(event);
		}
		for (dRegion = regionList.First(); dRegion; dRegion = dRegion->Next())
		{
			dRegion->region.left += event.position.column;
			dRegion->region.top += event.position.line;
			dRegion->region.right += event.position.column;
			dRegion->region.bottom += event.position.line;
		}
		break;

	case S_MAXIMIZE:
		Maximize();
		break;

	case S_MINIMIZE:
		Minimize();
		break;

	case L_WINDOW_RESTORE:
		if (FlagSet(woAdvancedStatus, WOAS_MINIMIZED))
			Minimize();
		else if (FlagSet(woAdvancedStatus, WOAS_MAXIMIZED))
			Maximize();
		break;

	case S_NON_CURRENT:
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (object)
		{
			object->woStatus &= ~WOS_CURRENT;
			object->woAdvancedStatus |= WOAS_NEED_VALIDATE;
			object->Event(event);
		}
		break;

	case S_CURRENT:
	case S_DISPLAY_INACTIVE:
	case S_DISPLAY_ACTIVE:
		// Make sure the window has a valid region.
		woAdvancedStatus &= ~WOAS_REDISPLAY;
		if (FlagSet(woAdvancedStatus, WOAS_INVALID_REGION))
			break;

		// Get the current window object.
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (!object && (!parent || FlagSet(woStatus, WOS_CURRENT)))
		{
			object = (!object) ? First() : object;
			while (object &&
				(FlagSet(object->woFlags, WOF_NON_SELECTABLE) ||
				 FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL) ||
				 FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT)))
				object = object->Next();
			if (object)
			{
				windowList.current = object;
				object->woStatus |= WOS_CURRENT;
			}
		}

		if (!parent)
		{
			eventManager->DeviceState(E_CURSOR, D_OFF);
			eventManager->DevicesHide(true);
		}

		if (FlagSet(woFlags, WOF_BORDER) &&
			!FlagSet(woAdvancedStatus, WOAS_INVALID_REGION) &&
			true.bottom - true.top > 2)
		{
			int logicalPalette;
			if (FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) || FlagSet(woFlags, WOF_VIEW_ONLY))
				logicalPalette = PM_VIEW;
			else if (ccode == S_DISPLAY_INACTIVE)
				logicalPalette = PM_INACTIVE;
			else
				logicalPalette = PM_ACTIVE;
			lastPalette = MapPalette(paletteMapTable, logicalPalette, ID_OUTLINE);
			display->Rectangle(screenID, true, lastPalette);
		}

		lastPalette = UI_WINDOW_OBJECT::LogicalPalette(ccode);
		for (dRegion = regionList.First(); dRegion; dRegion = dRegion->Next())
			if (dRegion->Overlap(event.region))
			{
				UI_REGION tRegion = dRegion->region;
				tRegion.left = max(event.region.left, tRegion.left);
				tRegion.top = max(event.region.top, tRegion.top);
				tRegion.right = min(event.region.right, tRegion.right);
				tRegion.bottom = min(event.region.bottom, tRegion.bottom);
				display->Fill(screenID, tRegion, lastPalette);
			}

		UI_EVENT s_event = event;
		UI_EVENT t_event = event;
		if (ccode == S_CURRENT)
			s_event.type = S_DISPLAY_ACTIVE;
		for (object = First(); object; object = object->Next())
		{
			object->woAdvancedStatus |= WOAS_NEED_VALIDATE;
			if ((ccode == S_DISPLAY_ACTIVE || ccode == S_CURRENT) &&
				!FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION) &&
				FlagSet(object->woStatus, WOS_CURRENT))
				object->Event(t_event);
			else if (!FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION))
				object->Event(s_event);
		}
		if (!parent)
			eventManager->DevicesShow(true);
		break;

	case L_CONTEXT_HELP:
		_helpSystem->DisplayHelp(windowManager, helpContext);
		break;

	case L_VIEW:
	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		// Find the proper window object.
		for (object = Last(); object; object = object->Previous())
			if (!FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL) &&
				!FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
				(FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION) &&
				 (object->Touching(event.position) || !object->Overlap(event.position))  ||
				(!FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION) &&
			 	 object->Overlap(event.position))))
				break;

		// See which object should be current, if any.
		if (object && !FlagSet(object->woStatus, WOS_CURRENT) &&
			(ccode == L_BEGIN_SELECT || (ccode == L_CONTINUE_SELECT &&
			FlagSet(object->woAdvancedFlags, WOAF_HOT_REGION))))
		{
			if (ToFront(object, FALSE) == 0)
				break;
		}
		else if (ccode != L_BEGIN_SELECT && ccode != L_END_SELECT && event.rawCode != 0)
			object = (UI_WINDOW_OBJECT *)windowList.current;
		else if (!object)
		{
			eventManager->DeviceState(event.type, DM_VIEW);
			break;
		}

		// Get the current object.
		if (!object)
			object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (object && !FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
			ccode = object->Event(event);
		else
			ccode = S_UNKNOWN;
		break;

	case L_FIELD_FIRST:
	case L_FIELD_LAST:
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (object && !FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
		{
			int tcode = object->Event(event);
			if (tcode != S_UNKNOWN && tcode != S_ERROR)
				break;
		}
		if (ccode == L_FIELD_FIRST)
			for (tObject = First(); tObject && ToFront(tObject, TRUE) != tObject; )
				tObject = tObject->Next();
		else
			for (tObject = Last(); tObject && ToFront(tObject, TRUE) != tObject; )
				tObject = tObject->Previous();
		ccode = (tObject == object) ? S_UNKNOWN : event.type;
		break;

	case L_FIELD_LEFT:
	case L_FIELD_RIGHT:
	case L_FIELD_UP:
	case L_FIELD_DOWN:
	case L_FIELD_PREVIOUS:
	case L_FIELD_NEXT:
	case L_SELECT:
		tcode = ccode;
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		UI_WINDOW_OBJECT *oldObject = object;
		if (object && !FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
		{
			ccode = object->Event(event);
			if (ccode != S_UNKNOWN && ccode != S_ERROR)
				break;
		}

		if (tcode == L_FIELD_PREVIOUS)
		{
			object = (!object) ? Last() : object->Previous();
			while (object && (tObject = ToFront(object, TRUE)) != object)
				if (tObject)
					object = object->Previous();
				else
					return (ccode);
			if (!object && parent)
				return (S_UNKNOWN);
			if (!object)
			{
				object = (!object) ? Last() : object;
				while (object && (tObject = ToFront(object, TRUE)) != object)
					if (tObject)
						object = object->Previous();
					else
						return (ccode);
			}
		}
		else if (tcode == L_FIELD_NEXT || tcode == L_SELECT)
		{
			object = (!object) ? First() : object->Next();
			while (object && (tObject = ToFront(object, TRUE)) != object)
				if (tObject)
					object = object->Next();
				else
					return (ccode);
			if (!object && parent)
				return (S_UNKNOWN);
			if (!object)
			{
				object = (!object) ? First() : object;
				while (object && (tObject = ToFront(object, TRUE)) != object)
					if (tObject)
						object = object->Next();
					else
						return (ccode);
			}
		}
		else
		{
			UI_WINDOW_OBJECT *nObject = 0;
			for (tObject = First(); tObject; tObject = tObject->Next())
				if (tObject == object ||
					FlagSet(tObject->woFlags, WOF_NON_SELECTABLE) ||
					FlagSet(tObject->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL) ||
					FlagSet(tObject->woAdvancedFlags, WOAF_NON_CURRENT))
					;
				else if (
					(tcode == L_FIELD_LEFT &&
					 (object->true.top == tObject->true.top || object->true.bottom == tObject->true.bottom) &&
					 object->true.left > tObject->true.left &&
					 (!nObject || object->true.left - tObject->true.left < object->true.left - nObject->true.left)) ||
					(tcode == L_FIELD_RIGHT &&
					 (object->true.top == tObject->true.top || object->true.bottom == tObject->true.bottom) &&
					 object->true.left < tObject->true.left &&
					 (!nObject || tObject->true.left - object->true.left < nObject->true.left - object->true.left)) ||
					(tcode == L_FIELD_UP &&
					 (object->true.left == tObject->true.left || object->true.right == tObject->true.right) &&
					 object->true.top > tObject->true.top &&
					 (!nObject || object->true.top - tObject->true.top < object->true.top - nObject->true.top)) ||
					(tcode == L_FIELD_DOWN &&
					 (object->true.left == tObject->true.left || object->true.right == tObject->true.right) &&
					 object->true.top < tObject->true.top &&
					 (!nObject || tObject->true.top - object->true.top < nObject->true.top - object->true.top)))
					nObject = tObject;
			if (nObject)
				object = ToFront(nObject, FALSE);
		}
		ccode = (object == oldObject) ? S_UNKNOWN : tcode;
		break;

	case S_CHECK_HOT_KEY:
		for (object = First(); object; object = object->Next())
			if (object->hotKey != 0 && event.key.value == object->hotKey)
				return (TRUE);
		return (FALSE);

	case S_ALT_KEY:
		for (object = First(); object; object = object->Next())
			if (object->hotKey == HOT_KEY_SUB_WINDOW)
			{
				ToFront(object, FALSE);
				return (ccode);
			}
		// Continue to default.

	default:
		UI_EVENT tEvent = event;

		// Check for hot key matches.
		int tHotKey = E_KEY;
		if (ccode == E_KEY && FlagSet(event.key.shiftState, S_ALT))
			tHotKey = MapEvent(_hotKeyMapTable, tEvent, ID_WINDOW_OBJECT);
		else if (ccode == E_KEY && (event.rawCode & 0xFF) &&
			FlagSet(woAdvancedFlags, WOAF_NORMAL_HOT_KEYS))
			tHotKey = toupper(event.key.value);
		if (ccode == E_KEY && tHotKey != E_KEY && tHotKey == hotKey)
			tEvent.type = L_SELECT;
		else if (ccode == E_KEY && tHotKey != E_KEY)
		{
			tEvent.type = S_CHECK_HOT_KEY;
			tEvent.key.value = tHotKey;
			for (object = First(); object; object = object->Next())
				if (object->hotKey != 0 &&
					(tHotKey == object->hotKey ||
					 (object->hotKey == HOT_KEY_SUB_WINDOW && object->Event(tEvent))))
				{
					if (ToFront(object, FALSE) == object)
						tEvent.type = (object->hotKey == HOT_KEY_SUB_WINDOW) ?
							E_KEY : L_SELECT;
					break;
				}
			if (!object)
				tEvent.type = E_KEY;
		}

		// Get the current object.
		tcode = ccode;
		if (!object)
			object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (object && !FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
			ccode = object->Event(tEvent);
		else
			ccode = S_UNKNOWN;
		break;
	}

	// Return the control code.
	return (ccode);
}

void UIW_WINDOW::Information(INFORMATION_REQUEST request, void *data)
{
	// Switch on the request type.
	switch(request)
	{
	case GET_MINIMUM_HEIGHT:
		*(int *)data = (display->isText) ? _windowHeight[0] : _windowHeight[1];
		break;

	case GET_MINIMUM_WIDTH:
		*(int *)data = (display->isText) ? _windowWidth[0] : _windowWidth[1];
		break;

	default:
		UI_WINDOW_OBJECT::Information(request, data);
		break;
	}
}

void UIW_WINDOW::Maximize(void)
{
	UI_EVENT event;

	// Maximize or restore the window.
	if (!FlagSet(woAdvancedStatus, WOAS_MAXIMIZED | WOAS_MINIMIZED))
	{
		// Set the save values.
		max_woAdvancedFlags = woAdvancedFlags;
		max_woAdvancedStatus = woAdvancedStatus;
		max_region = true;

		// Set the new window values.
		woAdvancedFlags |= WOAF_NO_SIZE | WOAF_NO_MOVE;
		woAdvancedStatus |= WOAS_MAXIMIZED;
		event.type = S_CHANGE;
		event.region.left = 0;
		event.region.top = 0;
		event.region.right = display->columns - 1;
		event.region.bottom = display->lines - 1;
		eventManager->Put(event, Q_BEGIN);
	}
	else if (FlagSet(woAdvancedStatus, WOAS_MAXIMIZED))
	{
		// Set the new window values.
		woAdvancedFlags = max_woAdvancedFlags;
		woAdvancedStatus = max_woAdvancedStatus;
		event.type = S_CHANGE;
		event.region = max_region;
		eventManager->Put(event, Q_BEGIN);
	}
}

void UIW_WINDOW::Minimize(void)
{
	UI_EVENT event;
	UI_REGION region;
	int height;
	int width;

	// See if the region has ever been set.
	if (min_region.left == -1 && min_region.right == -1)
	{
		UIW_WINDOW::Information(GET_MINIMUM_HEIGHT, &height);
		UIW_WINDOW::Information(GET_MINIMUM_WIDTH, &width);
		min_region.left = 0;
		min_region.top = display->lines - height;
		min_region.right = width - 1;
		min_region.bottom = display->lines - 1;
	}

	// Form the new window region.
	if (!FlagSet(woAdvancedStatus, WOAS_MAXIMIZED | WOAS_MINIMIZED))
	{
		// Set the save values.
		region = min_region;
		min_woAdvancedFlags = woAdvancedFlags;
		min_woAdvancedStatus = woAdvancedStatus;
		min_region = true;

		// Set the new window values.
		woAdvancedFlags |= WOAF_NO_SIZE;
		woAdvancedStatus |= WOAS_MINIMIZED;
		event.type = S_CHANGE;
		event.region = region;
		eventManager->Put(event, Q_BEGIN);
	}
	else if (FlagSet(woAdvancedStatus, WOAS_MINIMIZED))
	{
		// Set the new window values.
		woAdvancedFlags = min_woAdvancedFlags;
		woAdvancedStatus = min_woAdvancedStatus;
		event.type = S_CHANGE;
		region = true;
		event.region = min_region;
		min_region = region;
		eventManager->Put(event, Q_BEGIN);
	}
}

void UIW_WINDOW::RegionsCompute(void)
{
	// Compute the window update regions.
	regionList.Destroy();
	UI_REGION region = true;
	UI_WINDOW_OBJECT::Border(S_CREATE, region, 0);
	regionList + new UI_REGION_ELEMENT(screenID, &region);
	for (UI_WINDOW_OBJECT *object = First(); object; object = object->Next())
		if (FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION))
		{
			regionList.Destroy();
			regionList + new UI_REGION_ELEMENT(screenID, object->true.left + 1,
				object->true.top + 1, object->true.right - 1,
				object->true.bottom - 1);
		}
		else if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION) &&
			!FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION))
			regionList.Split(screenID, object->true);
}

void UIW_WINDOW::Subtract(UI_WINDOW_OBJECT *object)
{
	// Remove the object from the list.
	windowList.Subtract(object);
	woAdvancedStatus |= WOAS_REDISPLAY;
}

UI_WINDOW_OBJECT *UIW_WINDOW::ToFront(UI_WINDOW_OBJECT *object, 
	int checkCurrent)
{
	// Make sure the object can be made current.
	if (FlagSet(object->woFlags, WOF_NON_SELECTABLE) ||
		FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL) ||
		(checkCurrent && FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT)) ||
		(FlagSet(object->woStatus, WOS_CURRENT) &&
		 object == (UI_WINDOW_OBJECT *)windowList.current))
		return ((UI_WINDOW_OBJECT *)windowList.current);
	else if (FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
	{
		windowList.current = object;
		return (object);
	}

	// Get the current object.
	int ccode = 0;
	UI_WINDOW_OBJECT *t_object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
	if (t_object != object)
	{
		UI_EVENT event;
		event.type = S_NON_CURRENT;
		if (t_object && FlagSet(t_object->woStatus, WOS_CURRENT) &&
			errorObject == object)
		{
			t_object->woStatus &= ~WOS_CURRENT;
			event.type = S_DISPLAY_ACTIVE;
			event.region.left = event.region.top = event.region.right =
				event.region.bottom = -1;
			ccode = t_object->Event(event);
		}
		else if (t_object && FlagSet(t_object->woStatus, WOS_CURRENT))
		{
			t_object->woAdvancedStatus |= WOAS_NEED_VALIDATE;
			if (!FlagSet(t_object->woAdvancedStatus, WOAS_INVALID_REGION))
				ccode = t_object->Event(event);
			t_object->woStatus &= ~WOS_CURRENT;
		}
		if (ccode == S_ERROR)
		{
			t_object->woStatus |= WOS_CURRENT;
			errorObject = object;
			object = 0;
		}
		else
		{
			eventManager->DeviceState(E_CURSOR, D_OFF);
			event.type = S_CURRENT;
			object->woAdvancedStatus |= WOAS_NEED_VALIDATE;
			object->woStatus |= WOS_CURRENT;
			object->Event(event);
			windowList.current = object;
			errorObject = 0;
		}
	}
	return (object);
}

