//	Program name..	Zinc Interface Library
//	Filename......	WINDOW2.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

// ----- Static variables ---------------------------------------------------

int UI_WINDOW_OBJECT::defaultDepth = 1;
UI_EVENT_MAP *UI_WINDOW_OBJECT::eventMapTable = _eventMapTable;

// Constructor & Destructor -------------------------------------------------

UI_WINDOW_OBJECT::UI_WINDOW_OBJECT(int left, int top, int width, int height,
	USHORT a_woFlags, USHORT a_woAdvancedFlags)
{
	/* Intialize the screen object information */
	screenID = -1;
	display = 0;
	eventManager = 0;
	windowManager = 0;
	paletteMapTable = _normalPaletteMapTable;
	lastPalette = &paletteMapTable[0].palette;
	parent = 0;
	Validate = 0;
	hotKey = 0;
	windowID[0] = windowID[1] = windowID[2] = windowID[3] = ID_WINDOW_OBJECT;

	woFlags = a_woFlags;
	woAdvancedFlags = a_woAdvancedFlags;
	woStatus = (a_woFlags & WOF_UNANSWERED) ? WOS_UNANSWERED : 0;
	if (woFlags & WOF_INVALID) woStatus |= WOS_INVALID;
	woAdvancedStatus = 0;
	true.left = relative.left = left;
	true.top = relative.top = top;
	true.right = relative.right = (width > 0) ? left + width - 1 : width;
	true.bottom = relative.bottom = (height > 0) ? top + height - 1 : height;
}

UI_WINDOW_OBJECT::~UI_WINDOW_OBJECT(void)
{
}

// Member functions ---------------------------------------------------------

void UI_WINDOW_OBJECT::Border(int ccode, UI_REGION &region,
	const UI_PALETTE *palette)
{
	// Compute the display region.
	region = true;

	// Make sure the object needs a border.
	if (!FlagSet(woFlags, WOF_BORDER))
		return;

	// Determine the border and update the region.
	int displayBorder = (ccode == S_DISPLAY_ACTIVE || 
		ccode == S_NON_CURRENT || ccode == S_DISPLAY_INACTIVE ||
		ccode == S_CURRENT) ? TRUE : FALSE;
	int logicalPalette;
	if (FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) || FlagSet(woFlags, WOF_VIEW_ONLY))
		logicalPalette = PM_VIEW;
	else if (ccode == S_DISPLAY_INACTIVE)
		logicalPalette = PM_INACTIVE;
	else
		logicalPalette = PM_ACTIVE;
	UI_PALETTE *outlinePalette = MapPalette(paletteMapTable, logicalPalette,
		ID_OUTLINE);
	if (display->isText && (region.top == region.bottom))
	{
		if (region.left <= region.right - 2 * display->cellWidth)
		{
			if (displayBorder)
				display->Text(screenID, region.left, region.top, "[", outlinePalette, 1);
			region.left++;
		}
		if (region.left <= region.right - display->cellWidth)
		{
			if (displayBorder)
				display->Text(screenID, region.right, region.top, "]", outlinePalette, 1);
			region.right--;
		}
	}
	if (region.top + 1 >= region.bottom || region.left + 1 >= region.right)
		return;
	if (displayBorder)
		display->Rectangle(screenID, region, outlinePalette);
	region.left++;
	region.top++;
	region.right--;
	region.bottom--;
	if (FlagSet(woFlags, WOF_NON_FIELD_REGION) || ccode == S_CREATE ||
		display->isText)
		return;

	if (region.top + 1 >= region.bottom || region.left + 1 >= region.right)
		return;
	if (displayBorder && palette)
		display->Rectangle(screenID, region, palette);
	region.left++;
	region.top++;
	region.right--;
	region.bottom--;
	if (region.left + 1 >= region.right)
		return;
	if (displayBorder && palette)
	{
		display->Line(screenID, region.left, region.top, region.left, region.bottom, palette);
		display->Line(screenID, region.right, region.top, region.right, region.bottom, palette);
	}
	region.left++;
	region.right--;
}
		
int UI_WINDOW_OBJECT::Event(const UI_EVENT &event)
{
	/* Switch on the event type */
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_WINDOW_OBJECT);
	switch (ccode)
	{
	case S_CREATE:
	case S_SIZE:
		if ((display->isText && FlagSet(woStatus, WOS_GRAPHICS)) ||
			(!display->isText && !FlagSet(woStatus, WOS_GRAPHICS)))
			display->RegionConvert(relative);
		if (display->isText)
			woStatus &= ~WOS_GRAPHICS;
		else
			woStatus |= WOS_GRAPHICS;
		UI_WINDOW_OBJECT::RegionMax(TRUE);
		break;

	case S_MOVE:
		if (!parent || FlagSet(woAdvancedFlags, WOAF_TEMPORARY) ||
			FlagSet(woAdvancedStatus, WOAS_INVALID_REGION))
			break;
		true.left += event.position.column;
		true.top += event.position.line;
		true.right += event.position.column;
		true.bottom += event.position.line;
		UI_REGION region = parent->true;
		if (true.left < region.left || true.top < region.top ||
			true.right > region.right || true.bottom > region.bottom)
			woAdvancedStatus |= WOAS_INVALID_REGION;
		else
			woAdvancedStatus &= ~WOAS_INVALID_REGION;
		break;

	case L_VIEW:
		{
		UI_EVENT t_event = event;
		t_event.rawCode = DM_VIEW;
		eventManager->Event(t_event);
		}
		break;

	case L_SELECT:
		{
		UI_EVENT t_event = event;
		t_event.type = L_FIELD_NEXT;
		eventManager->Event(t_event);
		}

	default:
		ccode = S_UNKNOWN;
		break;
	}

	/* Return the control code */
	return (ccode);
}

void UI_WINDOW_OBJECT::Information(INFORMATION_REQUEST request, void *data)
{
	/* Switch on the request type */
	switch(request)
	{
	case GET_DEFAULT_HEIGHT:
		*(int *)data = (display->isText) ? display->cellHeight : 16;
		break;

	case GET_DEFAULT_WIDTH:
		*(int *)data = (display->isText) ? display->cellWidth : 32;
		break;
	}
}

UI_WINDOW_OBJECT::NeedsUpdate(const UI_EVENT &event, int ccode)
{
	UI_PALETTE *palette = UI_WINDOW_OBJECT::LogicalPalette(ccode);
	if ((palette == lastPalette && !FlagSet(woAdvancedFlags, WOAF_OUTSIDE_REGION) && 
		 !UI_WINDOW_OBJECT::Overlap(event.region)) ||
		(palette == lastPalette && FlagSet(woAdvancedFlags, WOAF_OUTSIDE_REGION) && 
		 !parent->Overlap(event.region)))
		return (FALSE);
	lastPalette = palette;
	return(TRUE);
}

int UI_WINDOW_OBJECT::NeedsValidation(void)
{
	int needsValidation = 
		(Validate && !FlagSet(woStatus, WOS_UNANSWERED) &&
		FlagSet(woAdvancedStatus, WOAS_NEED_VALIDATE)) ? TRUE : FALSE;
	woAdvancedStatus &= ~WOAS_NEED_VALIDATE;
	return (needsValidation);
}

int UI_WINDOW_OBJECT::Overlap(const UI_REGION &region)
{
	return (max(region.left, true.left) <= min(region.right, true.right) &&
		max(region.top, true.top) <= min(region.bottom, true.bottom));
}

void UI_WINDOW_OBJECT::RegionMax(int leftTop)
{
	UI_REGION_LIST regionList;
	UI_REGION_ELEMENT *dRegion;
	UI_WINDOW_OBJECT *object;
	UI_REGION region;

	/* Make sure the object has a parent */
	if (!parent || FlagSet(woAdvancedFlags, WOAF_TEMPORARY))
	{
		true = relative;
		return;
	}

	/* Compute the update regions then make an update request */
	region = parent->true;
	parent->Border(S_CREATE, region, 0);
	regionList.Add(0, new UI_REGION_ELEMENT(screenID, &region));
	/* Get the first object */
	for (object = this; object->previous; object = object->Previous())
		;
	for (; object && object != this; object = object->Next())
	{
		if (FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION))
		{
			regionList.Destroy();
			regionList.Add(0, new UI_REGION_ELEMENT(screenID,
				object->true.left + 1, object->true.top + 1,
				object->true.right - 1, object->true.bottom - 1));
		}
		else if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION) &&
			!FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION))
			regionList.Split(screenID, object->true);
	}

	/* Get the region then destroy the region list */
	if (leftTop)
	{
		region.left = parent->true.right;
		region.top = parent->true.bottom;
		region.right = region.left - 1;
		region.bottom = region.top - 1;
		for (dRegion = regionList.First(); dRegion; dRegion = dRegion->Next())
			if (dRegion->region.top <= region.top &&
				(!FlagSet(woFlags, WOF_NON_FIELD_REGION) ||
				 dRegion->region.right - dRegion->region.left >=
				 relative.right - relative.left))
				region = dRegion->region;
	}
	else
	{
		region.right = parent->true.left;
		region.bottom = parent->true.top;
		region.left = region.right + 1;
		region.top = region.bottom + 1;
		for (dRegion = regionList.First(); dRegion; dRegion = dRegion->Next())
			if (dRegion->region.bottom >= region.bottom &&
				(!FlagSet(woFlags, WOF_NON_FIELD_REGION) ||
				 dRegion->region.right - dRegion->region.left >=
				 relative.right - relative.left))
				region = dRegion->region;
	}

	/* Compute relative coordinates */
	if (!FlagSet(woFlags, WOF_NON_FIELD_REGION))
	{
		if (relative.left < 0 || relative.top < 0)
		{
			woAdvancedStatus |= WOAS_INVALID_REGION;
			return;
		}
		true.left = region.left + relative.left;
		true.top = region.top + relative.top;
		true.right = region.left + relative.right;
		true.bottom = region.top + relative.bottom;
		woAdvancedStatus &= ~WOAS_TOO_SMALL;
		if (true.right > region.right)
		{
			if (true.right > region.right + 2)
				woAdvancedStatus |= WOAS_TOO_SMALL;
			true.right = region.right;
		}
		if (true.bottom > region.bottom)
		{
			if (true.bottom > region.bottom + 2)
				woAdvancedStatus |= WOAS_TOO_SMALL;
			true.bottom = region.bottom;
		}
	}
	else
		true = region;

	/* Check for the validity of the region */
	if (true.left > true.right || true.top > true.bottom ||
		true.left < parent->true.left || true.top < parent->true.top)
		woAdvancedStatus |= WOAS_INVALID_REGION;
	else
		woAdvancedStatus &= ~WOAS_INVALID_REGION;
	if (FlagSet(woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
		woStatus &= ~WOS_CURRENT;
}

void UI_WINDOW_OBJECT::Redisplay(int fromRoot)
{
	// Find the root then make sure the window manager is attached.
	UI_WINDOW_OBJECT *root = this;
	while (root->parent)
		root = root->parent;
	if (FlagSet(root->woAdvancedStatus, WOAS_REDISPLAY) ||
		!display || !eventManager || !windowManager)
		fromRoot = TRUE;
	if (!root->display || !root->eventManager || !root->windowManager)
	{
		root->woAdvancedStatus |= WOAS_REDISPLAY;
		return;
	}

	// Redisplay the window by sending a create, then redisplay message.
	UI_EVENT event;
	event.type = S_CREATE;
	if (fromRoot)
		root->Event(event);
	else
		Event(event);
	event.region = true;
	event.type = (root == windowManager->First()) ?
		S_DISPLAY_ACTIVE : S_DISPLAY_INACTIVE;
	if (fromRoot)
		root->Event(event);
	else
		Event(event);
}

void UI_WINDOW_OBJECT::Shadow(UI_REGION &region, int depth)
{
	/* Make sure the region is large enough to shadow */
	if (depth == 0)
		return;
	int left, top, right, bottom;
	UI_REGION tRegion;
	tRegion = region;
	left = region.left + 1;
	top = region.top + 1;
	right = region.right - 2;
	bottom = region.bottom - 2;
	if (left + depth > right || top + depth > bottom)
	{
		region = true;
		region.left += 1;
		region.top += 1;
		region.right -= 1;
		region.bottom -= 1;
	}

	/* Hide the screen devices */
	if (eventManager)
		eventManager->DevicesHide(tRegion);

	/* Left column and top lines */
	UI_PALETTE *palette = MapPalette(paletteMapTable, PM_ANY,
		(depth > 0) ? ID_WHITE_SHADOW : ID_LIGHT_SHADOW);
	display->Line(screenID, left, top, left, bottom, palette);
	display->Line(screenID, left, top, right--, top, palette);
	top++;
	if (depth == 2 || depth == -2)
		display->Line(screenID, left, top, right++, top, palette);
	else
		right++;

	/* Right columns and bottom line */
	palette = MapPalette(paletteMapTable, PM_ANY,
		(depth > 0) ? ID_LIGHT_SHADOW : ID_DARK_SHADOW);
	if (depth == 2 || depth == -2)
		display->Line(screenID, right, top--, right, bottom++, palette);
	else
	{
		top--;
		bottom++;
	}
	right++;
	display->Line(screenID, right, top, right, bottom, palette);
	display->Line(screenID, left, bottom, right, bottom, palette);

	/* Show the screen devices */
	if (eventManager)
		eventManager->DevicesShow(tRegion);

	/* Compute the new region coordinates */
	tRegion = region;
	region.left = tRegion.left + 2;
	region.bottom = tRegion.bottom - 2;
	if (depth == 2 || depth == -2)
	{
		region.top = tRegion.top + 3;
		region.right = tRegion.right - 3;
	}
	else
	{
		region.top = tRegion.top + 2;
		region.right = tRegion.right - 2;
	}
}

void UI_WINDOW_OBJECT::Text(char *string, int depth, int ccode,
	const UI_PALETTE *palette)
{
	/* Shadow the region if needed */
	UI_REGION region = true;
	if (depth != 0 && !display->isText)
	{
		UI_WINDOW_OBJECT::Border(ccode, region, palette);
		region = true;
		UI_WINDOW_OBJECT::Shadow(region, depth);
		display->Fill(screenID, region, palette);
		if (!FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER))
			region.left += depth;
	}
	else
	{
		UI_WINDOW_OBJECT::Border(ccode, region, palette);
		display->Fill(screenID, region, palette);
	}

	/* Make sure it is a valid string */
	if (string == 0 || string[0] == '\0')
		return;

	/* Display the text */
	char scrapBuffer[128];
	strcpy(scrapBuffer, string);
	int hotKey = -1;
	for (int i = 0; scrapBuffer[i]; i++)
		if (scrapBuffer[i] == '~')
		{
			hotKey = i;
			strcpy(&scrapBuffer[i], &scrapBuffer[i+1]);
			break;
		}
	int length1 = strlen(scrapBuffer);
	int width1 = display->TextWidth(scrapBuffer);
	int width2 = region.right - region.left + 1;
	int height1 = display->TextHeight(scrapBuffer);
	int height2 = region.bottom - region.top + 1;
	while (width1 > width2)
	{
		scrapBuffer[--length1] = '\0';
		width1 = display->TextWidth(scrapBuffer);
	}
	if (width1 > 0 && height2 >= height1)
	{
		int column;
		int line = region.top + (height2 - height1 + 1) / 2;
		if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
			column = region.left + width2 - width1;
		else if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
			column = region.left + (width2 - width1) / 2;
		else
			column = region.left;
		display->Text(screenID, column, line, scrapBuffer, palette, length1, FALSE);
		if (hotKey >= 0)
		{
			column += hotKey * display->cellWidth;
			if (!display->isText)
			{
				UI_PALETTE tPalette = *palette;
				// tPalette.background = tPalette.foreground;
				tPalette.color <<= 4;
				tPalette.bwColor <<= 4;
				tPalette.grayScaleColor <<= 4;
				line += display->TextHeight("X");
				display->Line(screenID, column, line, 
					column + display->cellWidth - 1, line, &tPalette);
			}
			else if (!FlagSet(woFlags, WOF_NON_SELECTABLE) &&
				(!FlagSet(woStatus, WOS_CURRENT) || ccode == S_DISPLAY_INACTIVE))
			{
				UI_PALETTE *tPalette = MapPalette(paletteMapTable, 
					PM_HOT_KEY, windowID[0], windowID[1], windowID[2], 
					windowID[3]);
				display->Text(screenID, column, line, &scrapBuffer[hotKey], tPalette, 1, FALSE);
			}
		}
	}
}

void UI_WINDOW_OBJECT::InformationSet(int a_screenID, UI_DISPLAY *a_display,
	UI_EVENT_MANAGER *a_eventManager, UI_WINDOW_MANAGER *a_windowManager,
	UI_PALETTE_MAP *a_paletteMapTable, UI_WINDOW_OBJECT *a_parent)
{
	screenID = a_screenID;
	display = a_display;
	eventManager = a_eventManager;
	windowManager = a_windowManager;
	if (paletteMapTable == _normalPaletteMapTable)
		paletteMapTable = a_paletteMapTable;
	lastPalette = &paletteMapTable[0].palette;
	parent = a_parent;
}

UI_PALETTE *UI_WINDOW_OBJECT::LogicalPalette(int ccode)
{
	USHORT logicalPalette = PM_ACTIVE;
	if (FlagSet(woFlags, WOF_NON_SELECTABLE))
		logicalPalette = PM_NON_SELECTABLE;
	else if (FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) || FlagSet(woFlags, WOF_VIEW_ONLY))
		logicalPalette = PM_VIEW;
	else if (ccode == S_DISPLAY_INACTIVE)
		logicalPalette = PM_INACTIVE;
	else if (ccode == S_NON_CURRENT && FlagSet(woStatus, WOS_SELECTED))
		logicalPalette = PM_SELECTED;
	else if (ccode == S_NON_CURRENT)
		logicalPalette = PM_ACTIVE;
	else if (FlagSet(woStatus, WOS_CURRENT) &&
		(!parent || FlagSet(parent->woStatus, WOS_CURRENT)))
		logicalPalette = PM_CURRENT;
	else if (FlagSet(woStatus, WOS_SELECTED))
		logicalPalette = PM_SELECTED;

	return (MapPalette(paletteMapTable, logicalPalette, windowID[0],
		windowID[1], windowID[2], windowID[3]));
}

