//	Program name..	Phone Book
//	Filename......	PHONEBK.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <fcntl.h>
#include <sys\stat.h>
#include <io.h>
#include <string.h>
#include "ui_win.hpp"
#include "phonebk.hlh"

typedef struct
{
	char name[40];
	char address1[40];
	char address2[40];
	char phone[20];
} PHONE_RECORD;

class PHONE_BOOK : public UIW_WINDOW
{
public:
	static int fileHandle;

	PHONE_BOOK(char *filename, int left, int top);
	~PHONE_BOOK(void);

	static void PHONE_BOOK::AddRecord(void *item, UI_EVENT &event);
	static void PHONE_BOOK::NextRecord(void *item, UI_EVENT &event);
	static void PHONE_BOOK::PrevRecord(void *item, UI_EVENT &event);

private:
	static int newRecord;
	static long recordNumber;
	static PHONE_RECORD record;
	static PHONE_RECORD tmpRecord;

	static void PHONE_BOOK::ReadRecord(void);
	static void PHONE_BOOK::WriteRecord(void);
};

UI_DISPLAY *_display;
UI_EVENT_MANAGER *_eventManager;
UI_WINDOW_MANAGER *_windowManager;
UIW_PULL_DOWN_MENU *controlMenu;
UIW_POP_UP_ITEM *closeOption;
PHONE_BOOK *book = 0;

// Phone book constuctor.
PHONE_BOOK::PHONE_BOOK(char *filename, int left, int top) :
	UIW_WINDOW(left - 23, top - 5, 46, 9, WOF_NO_FLAGS,
	WOAF_NO_FLAGS, HELP_RECORD)
{
	newRecord = TRUE;
	record.name[0] =
		record.address1[0] =
		record.address2[0] = '\0';
	strcpy(record.phone, "..........");
	tmpRecord = record;

	// Open the file and read the first record.
	fileHandle = open(filename, O_RDWR);
	if (fileHandle < 0)
		fileHandle = open(filename, O_CREAT, S_IREAD | S_IWRITE);
	if (fileHandle < 0)
		return;
	ReadRecord();

	// Create the window menu.
	UIW_PULL_DOWN_MENU *menu = new UIW_PULL_DOWN_MENU(0, WOF_NO_FLAGS, WOAF_NO_FLAGS);
	*menu
		+ new UIW_PULL_DOWN_ITEM(" ~Previous ", MNIF_NO_FLAGS,
			PHONE_BOOK::PrevRecord)
		+ new UIW_PULL_DOWN_ITEM(" ~Next ", MNIF_NO_FLAGS,
			PHONE_BOOK::NextRecord)
		+ new UIW_PULL_DOWN_ITEM(" ~Add ", MNIF_NO_FLAGS,
			PHONE_BOOK::AddRecord);

	// Create the phone book record entry window.
	*this
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ new UIW_SYSTEM_BUTTON
		+ new UIW_TITLE(filename, WOF_JUSTIFY_CENTER)
		+ menu

		+ new UIW_PROMPT(38, 0, "#", WOF_NO_FLAGS)
		+ new UIW_NUMBER(40, 0, 6, &recordNumber, "", NMF_NO_FLAGS,
			WOF_NO_ALLOCATE_DATA | WOF_NON_SELECTABLE)

		+ new UIW_PROMPT(2, 1, "Name......", WOF_NO_FLAGS)
		+ new UIW_STRING(15, 1, 27, tmpRecord.name, 26, STF_NO_FLAGS,
			WOF_BORDER | WOF_NO_ALLOCATE_DATA)

		+ new UIW_PROMPT(2, 2, "Address...", WOF_NO_FLAGS)
		+ new UIW_STRING(15, 2, 27, tmpRecord.address1, 26, STF_NO_FLAGS,
			WOF_BORDER | WOF_NO_ALLOCATE_DATA)
		+ new UIW_STRING(15, 3, 27, tmpRecord.address2, 26, STF_NO_FLAGS,
			WOF_BORDER | WOF_NO_ALLOCATE_DATA)

		+ new UIW_PROMPT(2, 4, "Phone.....", WOF_NO_FLAGS)
		+ new UIW_FORMATTED_STRING(15, 4, 27, tmpRecord.phone, "LNNNLLNNNLCCCC",
			"(...) ...-....", WOF_BORDER | WOF_NO_ALLOCATE_DATA);

	// Make the ( File | Close ) option active and redisplay menu.
	closeOption->woFlags &= ~WOF_NON_SELECTABLE;
}

// Phone book destructor.
PHONE_BOOK::~PHONE_BOOK(void)
{
	WriteRecord();
	close(fileHandle);
	book = 0;

	// Make the ( File | Close ) option inactive and redisplay menu.
	closeOption->woFlags |= WOF_NON_SELECTABLE;
}

// Read the next record from the current phone book file.
void PHONE_BOOK::ReadRecord(void)
{
	if (read(fileHandle, &tmpRecord, sizeof(PHONE_RECORD)) == sizeof(PHONE_RECORD))
	{
		record = tmpRecord;
		newRecord = FALSE;
	}
	recordNumber = lseek(fileHandle, 0L, SEEK_CUR) / sizeof(PHONE_RECORD);
	if (newRecord)
		recordNumber++;

	// Update the phone book window if needed.
	if (book)
		*_windowManager + book;
}

// Write the record to the current phone book file.
void PHONE_BOOK::WriteRecord(void)
{
	// Write information if changed.
	if (strcmp(record.name, tmpRecord.name) ||
		strcmp(record.address1, tmpRecord.address1) ||
		strcmp(record.address2, tmpRecord.address2) ||
		strcmp(record.phone, tmpRecord.phone))
	{
		if (newRecord)
			lseek(fileHandle, 0L, SEEK_END);
		else if (lseek(fileHandle, sizeof(PHONE_RECORD) * -1L, SEEK_CUR) < 0)
			lseek(fileHandle, 0L, SEEK_SET);
		write(fileHandle, &tmpRecord, sizeof(PHONE_RECORD));
		newRecord = FALSE;
	}
	record = tmpRecord;
}

// Control menu ( Edit | Add ) option to add a new record entry.
#pragma argsused
void PHONE_BOOK::AddRecord(void *item, UI_EVENT &event)
{
	WriteRecord();
	long i = lseek(fileHandle, 0L, SEEK_END);
	if (i != 0L)
		newRecord = TRUE;
	tmpRecord.name[0] =
		tmpRecord.address1[0] =
		tmpRecord.address2[0] = '\0';
	strcpy(tmpRecord.phone, "..........");
	ReadRecord();
	record = tmpRecord;
}

// Control menu ( Edit | Next ) option to view/modify the next record.
#pragma argsused
void PHONE_BOOK::NextRecord(void *item, UI_EVENT &event)
{
	WriteRecord();
	ReadRecord();
}

// Control menu ( Edit | Previous ) option to view/modify the previous record.
#pragma argsused
void PHONE_BOOK::PrevRecord(void *item, UI_EVENT &event)
{
	WriteRecord();
	if (lseek(fileHandle, sizeof(PHONE_RECORD) * -2L, SEEK_CUR) < 0)
		lseek(fileHandle, 0L, SEEK_SET);
	ReadRecord();
}

// Button validate procedure for OK button on "Open book.." window.
#pragma argsused
static void OKButton(void *object, UI_EVENT &event)
{
	// Put an "exit current window" message on the event queue.
	event.type = S_DELETE_LEVEL;
	_eventManager->Put(event, Q_BEGIN);
}

// Control menu ( File | Close ) option to close the current phone book.
#pragma argsused
static void CloseBook(void *item, UI_EVENT &event)
{
	// Remove and delete the current phone book window from the screen.
	*_windowManager - book;
	delete book;
}

// Control menu ( File | Open ) option to open a phone book.
#pragma argsused
static void OpenBook(void *item, UI_EVENT &event)
{
	char filename[20];

	filename[0] = '\0';
	if (book)
		CloseBook(item, event);

	// Create a window in the screen center to get phone book name.
	int left = _display->columns / _display->cellWidth / 2;
	int top = _display->lines / _display->cellHeight / 2;
	*_windowManager
		+ &(*new UIW_WINDOW(left - 20, top - 4, 40, 7, WOF_NO_FLAGS,
			WOAF_MODAL | WOAF_NO_SIZE, HELP_OPEN)
			+ new UIW_BORDER
			+ new UIW_TITLE("Open new phone book...", WOF_JUSTIFY_CENTER)
			+ new UIW_PROMPT(2, 1, "File name...", WOF_NO_FLAGS)
			+ new UIW_STRING(15, 1, 20, filename, 12, STF_NO_FLAGS,
				WOF_BORDER | WOF_NO_ALLOCATE_DATA)
			+ new UIW_BUTTON(18, 3, 5, "OK", BTF_NO_FLAGS,
				WOF_JUSTIFY_CENTER, OKButton));

	// Wait for user response.
	int ccode;
	do
	{
		_eventManager->Get(event, Q_NORMAL);

		// Send an "exit current window" message if ENTER key.
		if (event.type == E_KEY &&
			(event.rawCode == ENTER || event.rawCode == GRAY_ENTER))
			event.type = S_DELETE_LEVEL;
		ccode = _windowManager->Event(event);
	} while (ccode != L_EXIT && event.type != S_DELETE_LEVEL);

	// Open/create the new phone book if possible.
	if(filename[0] != '\0')
	{
		book = new PHONE_BOOK(filename, left, top);

		// Add the new phone book to the window manager if no error.
		if (book->fileHandle < 0)
		{
			_errorSystem->ReportError(_windowManager, -1, "Error opening file.");
			delete book;
		}
		else
			*_windowManager + book;
	}
}

// Control menu ( File | Help ) option to _display general help.
#pragma argsused
static void Help(void *item, UI_EVENT &event)
{
	// Call the help system to _display general help.
	_helpSystem->DisplayHelp(_windowManager, HELP_GENERAL);
}

// Control menu ( File | Exit ) option to exit the program.
#pragma argsused
static void Exit(void *item, UI_EVENT &event)
{
	// Put an EXIT message on the event queue.
	event.type = L_EXIT;
	_eventManager->Put(event, Q_BEGIN);
}

static void CreateMenu(void)
{
	// Create the main control menu.
	controlMenu = new UIW_PULL_DOWN_MENU(0, WOF_NO_FLAGS, WOAF_NO_FLAGS);
	controlMenu->woAdvancedFlags |= WOAF_LOCKED | WOAF_NON_CURRENT;

	// ( File ) option pull down menu (Close is inactive).
	UIW_PULL_DOWN_ITEM *fileOption = new UIW_PULL_DOWN_ITEM(" ~File ", MNF_NO_FLAGS, 0);
	closeOption = new UIW_POP_UP_ITEM("~Close book...", MNIF_NO_FLAGS,
		BTF_NO_TOGGLE, WOF_NO_FLAGS, CloseBook);
	closeOption->woFlags |= WOF_NON_SELECTABLE;
	*fileOption
		+ new UIW_POP_UP_ITEM("~Open book...", MNIF_NO_FLAGS,
			BTF_NO_TOGGLE, WOF_NO_FLAGS, OpenBook)
		+ closeOption
		+ new UIW_POP_UP_ITEM("~Help...", MNIF_NO_FLAGS,
			BTF_NO_TOGGLE, WOF_NO_FLAGS, Help)
		+ new UIW_POP_UP_ITEM
		+ new UIW_POP_UP_ITEM("E~xit", MNIF_NO_FLAGS, BTF_NO_TOGGLE,
			WOF_NO_FLAGS, Exit);

	// Add the option menus to the control menu.
	*controlMenu + fileOption;
	*_windowManager + controlMenu;
}

main()
{
	// Initialize the _display, trying for graphics first.
	_display = new UI_DOS_BGI_DISPLAY;
	if (!_display->installed)
	{
		delete _display;
		_display = new UI_DOS_TEXT_DISPLAY;
	}

	// Initialize the event and window managers.
	_eventManager = new UI_EVENT_MANAGER(100, _display);
	*_eventManager + new UI_BIOS_KEYBOARD + new UI_MS_MOUSE + new UI_CURSOR;
	_windowManager = new UI_WINDOW_MANAGER(_display, _eventManager);

	// Initialize the help and error window systems.
	_helpSystem = new UI_HELP_WINDOW_SYSTEM("phonebk.hlp",
		_windowManager, HELP_GENERAL);
	_errorSystem = new UI_ERROR_WINDOW_SYSTEM;

	CreateMenu();

	// Wait for user response.
	int ccode;
	UI_EVENT event;
	do
	{
		_eventManager->Get(event, Q_NORMAL);
		ccode = _windowManager->Event(event);
	} while (ccode != L_EXIT);

	// Clean up.
	delete _windowManager;
	delete _eventManager;
	delete _display;
}
