//	Program name..	Zinc Interface Library
//	Filename......	UI_GEN.HPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef UI_GEN_HPP
#define UI_GEN_HPP

typedef unsigned char UCHAR;
typedef unsigned short USHORT;
typedef unsigned long ULONG;

#ifndef NULL
	#define NULL 0
#endif

/* ----- Definition of the BOOLEAN constants ----- */
#ifndef TRUE
	const int TRUE = 1;
	const int FALSE = 0;
#endif

/* ----- Definition of the POSITION type ----- */
struct UI_POSITION
{
	int column;
	int line;
};

/* ----- Definition of the REGION type ----- */
struct UI_REGION
{
	int left;
	int top;
	int right;
	int bottom;
};

/* ----- Definition of the SCROLL_INFORMATION type ----- */
struct UI_SCROLL_INFORMATION
{
	int current;
	int showing;
	int maximum;
	int delta;
};

/* ----- Other misellaneous definitions ----- */
#define FlagSet(flag1, flag2) ((flag1) & (flag2))
#define FlagsSet(flag1, flag2) (((flag1) & (flag2)) == (flag2))
#ifndef max
#define max(arg1, arg2) (((arg1) > (arg2)) ? (arg1) : (arg2))
#define min(arg1, arg2) (((arg1) < (arg2)) ? (arg1) : (arg2))
#endif

/* ----- Definition of the UI_ELEMENT class ----- */
class UI_ELEMENT
{
public:
	UI_ELEMENT *previous;
	UI_ELEMENT *next;

	UI_ELEMENT(void) : previous(0), next(0) { }
	virtual ~UI_ELEMENT(void) { }
};

/* ----- Definition of the UI_LIST class ----- */
class UI_LIST
{
public:
	UI_ELEMENT *first;
	UI_ELEMENT *last;
	UI_ELEMENT *current;
	int (*compareFunction)(void *element1, void *element2);

	UI_LIST(int (*a_compareFunction)(void *element1, void *element2) = 0) :
		first(0), last(0), current(0),
		compareFunction(a_compareFunction) { }
	virtual ~UI_LIST(void) { Destroy(); }

	UI_ELEMENT *Add(UI_ELEMENT *newElement);
	UI_ELEMENT *Add(UI_ELEMENT *element, UI_ELEMENT *newElement);
	int Count(void);
	UI_ELEMENT *Subtract(UI_ELEMENT *element);
	void Destroy(void);
	UI_ELEMENT *Get(int index);
	UI_ELEMENT *Get(int (*findFunction)(void *element1, void *matchData), void *matchData);
	int Index(UI_ELEMENT const *element);
	void Sort(void);

	UI_LIST &operator + (UI_ELEMENT *element)
		{ Add(element); return (*this); }
	UI_LIST &operator - (UI_ELEMENT *element)
		{ Subtract(element); return (*this); };
};

/* ----- Definition of the UI_LIST_BLOCK class ----- */
class UI_LIST_BLOCK : public UI_LIST
{
public:
	UI_LIST_BLOCK(int a_noOfElements,
		int (*a_compareFunction)(void *element1, void *element2) = 0) :
		UI_LIST(a_compareFunction), noOfElements(a_noOfElements),
		elementArray(0), freeList() { }
	virtual ~UI_LIST_BLOCK(void);

	UI_ELEMENT *Add(void);
	UI_ELEMENT *Add(UI_ELEMENT *element);
	UI_ELEMENT *Subtract(UI_ELEMENT *element);

protected:
	int noOfElements;
	void *elementArray;
	UI_LIST freeList;
};

/* ----- Definition of the UI_DATE class ----- */

// Flags applying to both UI_DATE::Import and UI_DATE::Export string routines.
const USHORT DTF_NO_FLAGS			= 0x0000;
const USHORT DTF_US_FORMAT		 	= 0x0001;
const USHORT DTF_EUROPEAN_FORMAT	= 0x0002;
const USHORT DTF_JAPANESE_FORMAT	= 0x0004;
const USHORT DTF_MILITARY_FORMAT	= 0x0008;

// Flags applying only to string UI_DATE::Export.
const USHORT DTF_DASH				= 0x0010;
const USHORT DTF_SLASH				= 0x0020;
const USHORT DTF_ALPHA_MONTH		= 0x0040;
const USHORT DTF_DAY_OF_WEEK		= 0x0080;
const USHORT DTF_UPPER_CASE			= 0x0100;
const USHORT DTF_SHORT_YEAR			= 0x0200;
const USHORT DTF_SHORT_MONTH		= 0x0400;
const USHORT DTF_SHORT_DAY			= 0x0800;
const USHORT DTF_ZERO_FILL			= 0x1000;

// Flags applying only to string UI_DATE::Import.
const USHORT DTF_FORCE_ALPHA_MONTH  = 0x2000;
const USHORT DTF_SYSTEM				= 0x4000;

enum DTI_RESULT
{
	DTI_OK = 0,					// Date successfully imported.
	DTI_INVALID, 				// Invalid date value or format.
	// Error codes from here down apply only to string Import().
	DTI_AMBIGUOUS,				// Ambiguous month name.
	DTI_INVALID_NAME,			// Invalid month or day-of-week name.
	DTI_VALUE_MISSING,			// Required date value not present.
	DTI_OUT_OF_RANGE			// Date out of range.
};

struct DATE_TEXT
{
	char *shortText;
	char *text;
};

class UI_DATE
{
public:
	UI_DATE(void)
		{ Import();	}		// Construct from current system date.
	UI_DATE(int packedDate)
		{ Import(packedDate); }
	UI_DATE(int year, int month, int day)
		{ Import(year, month, day); }
	UI_DATE(const char *string, USHORT dtFlags = DTF_NO_FLAGS)
		{ Import(string, dtFlags); }

	void Export(void);			// Export to (set) the system date.
	void Export(int *packedDate);
	void Export(int *year, int *month, int *day, int *dayOfWeek = 0);
	void Export(char *string, int maxLength, USHORT dtFlags = DTF_NO_FLAGS);
	DTI_RESULT Import(void);	  	// Import from (get) the system date.
	DTI_RESULT Import(int date);
	DTI_RESULT Import(int year, int month, int day);
	DTI_RESULT Import(const char *string, USHORT dtFlags = DTF_NO_FLAGS);
	void NameTablesSet(DATE_TEXT *monthTablePtr = 0, DATE_TEXT *dayTablePtr = 0);
	int DayOfWeek(void);
	int DaysInMonth(void);
	int LeapYear(void);

	int operator == (UI_DATE& argument)
		{ return (dYear == argument.dYear && dMonth == argument.dMonth &&
		  dDay == argument.dDay); }
	int operator > (UI_DATE& argument)
		{ return (dYear * 365L + dMonth * 31L + dDay >
		  argument.dYear * 365L + argument.dMonth * 31L + argument.dDay); }
	int operator < (UI_DATE& argument)
		{ return (dYear * 365L + dMonth * 31L + dDay <
		  argument.dYear * 365L + argument.dMonth * 31L + argument.dDay); }

protected:
	int dYear;
	UCHAR dMonth;
	UCHAR dDay;
	static DATE_TEXT *monthTable;
	static DATE_TEXT *dayTable;

	int TableLookup(DATE_TEXT *table, int tableLength, const char *token);
};

/* ----- Definition of the UI_TIME class ----- */

// Flags applying to both UI_TIME::Import and UI_TIME::Export string routines.
const USHORT TMF_NO_FLAGS			= 0x0000;
const USHORT TMF_NO_HOURS			= 0x0004;
const USHORT TMF_NO_MINUTES			= 0x0008;

// Flags applying only to string UI_TIME::Export
const USHORT TMF_SECONDS 			= 0x0001;
const USHORT TMF_HUNDREDTHS			= 0x0002;
const USHORT TMF_TWELVE_HOUR		= 0x0010;
const USHORT TMF_TWENTY_FOUR_HOUR	= 0x0020;
const USHORT TMF_ZERO_FILL			= 0x0040;
const USHORT TMF_COLON_SEPARATOR	= 0x0080;
const USHORT TMF_NO_SEPARATOR		= 0x0100;
const USHORT TMF_UPPER_CASE			= 0x0200;
const USHORT TMF_LOWER_CASE			= 0x0400;

// Flags applying only to string UI_TIME::Import
const USHORT TMF_SYSTEM				= 0x0800;

enum TMI_RESULT
{
	TMI_OK = 0,				// Time successfully imported.
	TMI_INVALID,			// Invalid time value or format.
	TMI_RANGE,				// Time out of range (used by UIW_TIME)
	TMI_VALUE_MISSING		// Required value not present.
};

class UI_TIME
{
public:
	USHORT tmFlags;

	UI_TIME(void)
		{ Import(); }		// Construct from current system time.
	UI_TIME(int packedTime)
		{ Import(packedTime); }
	UI_TIME(int hour, int minute, int second = 0, int hundredth = 0)
		{ Import(hour, minute, second, hundredth); }
	UI_TIME(const char *string, USHORT tmFlags = TMF_NO_FLAGS)
		{ Import(string, tmFlags); }

	void Export(void);
	void Export(int *packedTime);
	void Export(int *hour, int *minute, int *second = 0, int *hundredth = 0);
	void Export(char *string, USHORT tmFlags);
	TMI_RESULT Import(void);
	TMI_RESULT Import(int packedTime);
	TMI_RESULT Import(int hour, int minute, int second = 0, int hundredth = 0);
	TMI_RESULT Import(const char *string, USHORT tmFlags);
	void SetAmPm(char *a_amPtr = 0, char *a_pmPtr = 0);

	int operator == (UI_TIME& argument)
		{ return (tHour == argument.tHour && tMinute == argument.tMinute &&
		  tSecond == argument.tSecond && tHundredth == argument.tHundredth); }
	int operator > (UI_TIME& argument)
		{ return (tHour * 1000000L + tMinute * 10000L + tSecond * 100L + tHundredth >
		  argument.tHour * 1000000L + argument.tMinute * 10000L + argument.tSecond * 100L + argument.tHundredth); }
	int operator < (UI_TIME& argument)
		{ return (tHour * 1000000L + tMinute * 10000L + tSecond * 100L + tHundredth <
		  argument.tHour * 1000000L + argument.tMinute * 10000L + argument.tSecond * 100L + argument.tHundredth); }

protected:
	UCHAR tHour;
	UCHAR tMinute;
	UCHAR tSecond;
	UCHAR tHundredth;
	static char *amPtr;
	static char *pmPtr;

	int LookupAmPm(const char *string);
};


class UI_PATH
{
public:
	UI_PATH(void);
	UI_PATH(char *programPath, int rememberCwd);
	~UI_PATH(void);

	void SetFileName(const char *fileName);
	char *NextPathName(void);

protected:
	char *envPath;
	char *prgPath;
	char *cwdPath;
	char *fileName;
	char *pathName;
	char *pathPtr;
	UCHAR pathCnt;
};

extern UI_PATH *_path;

extern int OpenPath(const char *file, unsigned int mode);

/* ------- Country info stuff ---------- */

struct countryInfo
{
	int co_date;
	char co_curr[5];
	char co_thsep[2];
	char co_desep[2];
	char co_dtsep[2];
	char co_tmsep[2];
	char co_currstyle;
	char co_digits;
	char co_time;
	long co_case;
	char co_dasep[2];
	char co_fill[10];
};

extern struct countryInfo near _countryInfo;
extern int countryCode;

extern void GetCountryInfo(void);
extern int SetCountryCode(int desiredCountryCode);

// Miscellaneous functions.
extern void ui_dumpheap(int showFree);
extern char *ui_strdup(const char *string);	// uses new instead of malloc.
extern int ui_strlen(const char *string);	// checks for null.
extern void ui_strrepc(char *str, int c, int repc);

#endif /* UI_GEN_HPP */
