//	Program name..	Zinc Interface Library
//	Filename......	DATEWIN.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

// ----- Constructor & Destructor -------------------------------------------

UIW_DATE::UIW_DATE(int left, int top, int width, UI_DATE *a_datePtr,
	const char *a_range, USHORT a_dtFlags, USHORT a_woFlags,
	int (*validate)(void *object, int ccode)) :
	UIW_STRING(left, top, width, 0, 64, STF_NO_FLAGS,
		a_woFlags & ~WOF_NO_ALLOCATE_DATA, validate)
{
	if (a_datePtr)
		date = *a_datePtr;
	windowID[0] = ID_STRING;
	dtFlags = a_dtFlags;
	datePtr = FlagSet(a_woFlags, WOF_NO_ALLOCATE_DATA) ? a_datePtr : 0;
	dateWidth = width;
	range = (a_range) ? ui_strdup(a_range) : 0;
}

UIW_DATE::~UIW_DATE(void)
{
	if (range)
		delete range;
}

// ----- Member functions ---------------------------------------------------

const UI_DATE *UIW_DATE::DataGet(void)
{
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		return (datePtr);
	return (&date);
}

void UIW_DATE::DataSet(UI_DATE *newDatePtr)
{
	if (newDatePtr)
	{
		date = *newDatePtr;
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			datePtr = newDatePtr;
	}
	date.Export(text, dateWidth, dtFlags);
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

int UIW_DATE::Event(const UI_EVENT &event)
{
	/* Switch on the event type */
	int needValidate = NeedsValidation();
	int ccode = event.type;
	if (ccode == S_CREATE)
	{
		if (woStatus & WOS_UNANSWERED)
			date.Import(0, 0, 0);
		else if (datePtr)
			date = *datePtr;
		date.Export(text, dateWidth, dtFlags);
	}
	else if (ccode == S_NON_CURRENT)
	{
		DTI_RESULT result = date.Import(text, dtFlags);
		if (result == DTI_OK)
			result = DateRangeCheck();
		if (result != DTI_OK)
		{
			DateError(result);
			ccode = S_ERROR;
		}
		else
		{
			if (needValidate && (*Validate)(this, ccode) != 0)
				ccode = S_ERROR;
			else
			{
				if (datePtr)
					*datePtr = date;
				date.Export(text, dateWidth, dtFlags);
			}
		}
	}
	else if (ccode == S_CURRENT)
	{
		if (needValidate)
			(*Validate)(this, ccode);
		woStatus &= ~WOS_UNANSWERED;
		if (!(woStatus & WOS_CURRENT))
			date.Export(text, dateWidth, dtFlags & ~(DTF_DAY_OF_WEEK | DTF_SHORT_DAY));
	}
	if (ccode != S_ERROR)
	{
		woStatus &= ~WOS_INVALID;
		ccode = UIW_STRING::Event(event);
	}
	else
		woStatus |= WOS_INVALID;
	return(ccode);
}

int ParseRange(char *buffer, int offset, char *minValue, char *maxValue)
{
	/* Declaration of local variables */
	UCHAR position;

	/* Get the minimum value */
	position = 0;
	while (buffer[offset] != '\0' &&
		(buffer[offset] != '.' || buffer[offset+1] != '.') &&
		buffer[offset] != '/')
		minValue[position++] = buffer[offset++];
	minValue[position] = '\0';

	/* See if it is a standalone value */
	if (buffer[offset] == '/' || buffer[offset] == '\0')
	{
		strcpy(maxValue, minValue);
		return(++offset);
	}

	/* Increment the offset */
	while (buffer[offset] && buffer[offset] == '.')
		offset++;

	/* Get the maximum value */
	position = 0;
	while (buffer[offset] != '\0' && buffer[offset] != '/')
		maxValue[position++] = buffer[offset++];
	maxValue[position] = '\0';
	/* Return the offset */
	return(++offset);
}

void UIW_DATE::DateError(DTI_RESULT errorCode)
{
	static struct
	{
		DTI_RESULT code;
		char *msg;
	} errorTable[] =
	{
		{ DTI_INVALID,		"The date %s, is in an invalid format." 		},
		{ DTI_AMBIGUOUS,	"The date %s, has an ambiguous month name."		},
		{ DTI_INVALID_NAME,	"The date %s, has an invalid month name."		},
		{ DTI_VALUE_MISSING,"A date value must be entered."					},
		{ DTI_OK,			0												}
	};
	char formattedString[512];

	if (errorCode == DTI_OUT_OF_RANGE)
		DateRangeError(formattedString);
	else
	{
		strcpy(formattedString, "An unknown date error was received.");
		for (int i = 0; errorTable[i].msg; i++)
			if (errorTable[i].code == errorCode)
			{
				char dateString[64];
				date.Export(dateString, 64, dtFlags);
				sprintf(formattedString, errorTable[i].msg, dateString);
				break;
			}
	}
	_errorSystem->ReportError(windowManager,
		woFlags & (WOF_NO_UNANSWERED | WOF_NO_INVALID), formattedString);
}

DTI_RESULT UIW_DATE::DateRangeCheck(void)
{
	char minValue[20];
	char maxValue[20];
	DTI_RESULT errorCode = DTI_OK;
	int rangeLength;
	int offset;

	/* See if a range exists */
	if (!range || range[0] == '\0')
		return (DTI_OK);

	offset = 0;
	rangeLength = (range) ? strlen(range) : 0;
	while (offset < rangeLength)
	{
		offset = ParseRange(range, offset, minValue, maxValue);
		UI_DATE minDate(0, 0, 0);
		UI_DATE maxDate(0, 0, 0);
		DTI_RESULT minValid = minDate.Import(minValue);
		DTI_RESULT maxValid = maxDate.Import(maxValue);
		if (minValid != DTI_OK || maxValid != DTI_OK ||
			date < minDate || date > maxDate)
			errorCode = DTI_OUT_OF_RANGE;
		else
		{
			errorCode = DTI_OK;
			break;
		}
	}
	return errorCode;
}

void UIW_DATE::DateRangeError(char *fmtStr)
{
	char fmtMin[40];
	char fmtMax[40];
	char tempStr[80];
	static char thruStr[] = "through";
	static char orStr[] = "or";
	int position;
	int offset = 0;
	int count = 0;
	char sep = ',';							// Separator character.
	char badText[64];
	int rangeLength = (range) ? strlen(range) : 0;

	date.Export(badText, sizeof(badText), dtFlags);
	sprintf(fmtStr,
		"The date %s is not valid.  The date must be in the range ", badText);
	while (offset < rangeLength)
	{
		/* Format the current minimum and maximum values */
		if (count == 1)
		{
			if (!strcmp(fmtMin, fmtMax))
				strcpy(tempStr, fmtMin);
			else
				sprintf(tempStr, "%s %s %s", fmtMin, thruStr, fmtMax);
			strcat(fmtStr, tempStr);
		}
		else if (count != 0)
		{
			if (!strcmp(fmtMin, fmtMax))
				sprintf(tempStr, "%c %s", sep, fmtMin);
			else
				sprintf(tempStr, "%c %s %s %s", sep, fmtMin, thruStr, fmtMax);
			strcat(fmtStr, tempStr);
		}

		/* Get a new set of minimum and maximum values */
		count++;
		offset = ParseRange(range, offset, fmtMin, fmtMax);
	}

	/* Append the final minimum and maximum values on the string */
	if (count > 2 && !strcmp(fmtMin, fmtMax))
		sprintf(tempStr, "%c %s %s.", sep, orStr, fmtMin);
	else if (count > 2)
		sprintf(tempStr, "%c %s %s %s %s.", sep, orStr, fmtMin, thruStr, fmtMax);
	else if (count == 2 && !strcmp(fmtMin, fmtMax))
		sprintf(tempStr, " %s %s.", orStr, fmtMin);
	else if (count == 2)
		sprintf(tempStr, " %s %s %s %s.", orStr, fmtMin, thruStr, fmtMax);
	else if (count > 0)
		sprintf(tempStr, "%s %s %s.", fmtMin, thruStr, fmtMax);
	else
	{
		position = 0;		
		while (fmtStr[position] != '\0' && fmtStr[position] != '.')
			position++;
		fmtStr[++position] = '\0';
	}
	strcat(fmtStr, tempStr);
}

