//	Program name..	Zinc Interface Library
//	Filename......	DISPLAY.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#pragma inline

#include <dos.h>
#include <conio.h>
#include <stdio.h>
#include "ui_dsp.hpp"

#pragma warn -use

// VideoInt executes BIOS interrupt 10H and insures BP is preserved.
void VideoInt(void)
{
	asm		push	bp				// Save BP
	asm		int		10H				// because INT 10H may clobber it
	asm		pop		bp				// due to bug in older BIOS versions
}

// ----- Member functions ---------------------------------------------------

int UI_REGION_ELEMENT::Overlap(const UI_REGION &a_region)
{
	return (max(a_region.left, region.left) <= min(a_region.right, region.right) &&
		max(a_region.top, region.top) <= min(a_region.bottom, region.bottom));
}

int UI_REGION_ELEMENT::Overlap(const UI_REGION &sRegion, UI_REGION &tRegion)
{
	// Use a comparison formula to see if the regions overlap.
	tRegion.left = max(region.left, sRegion.left);
	tRegion.top = max(region.top, sRegion.top);
	tRegion.right = min(region.right, sRegion.right);
	tRegion.bottom = min(region.bottom, sRegion.bottom);
	if (tRegion.left > tRegion.right || tRegion.top > tRegion.bottom)
		return(FALSE);

	// Make sure the region is on the screen.
	if (tRegion.left < 0)
		tRegion.left = 0;
	if (tRegion.top < 0)
		tRegion.top = 0;
	return(TRUE);
}

void UI_REGION_LIST::Split(int screenID, const UI_REGION &region)
{
	UI_REGION tRegion, sRegion;
	UI_REGION_ELEMENT *dRegion, *t_dRegion;

	// Split any overlapping regions.
	for (dRegion = First(); dRegion; dRegion = t_dRegion)
	{
		// Preset the previous region element.
		t_dRegion = dRegion->Next();

		// Object encompasses the whole region.
		if (dRegion->Encompassed(region))
		{
			UI_LIST::Subtract(dRegion);
			delete dRegion;
		}

		// Object overlaps the region.
		else if (dRegion->Overlap(region, tRegion))
		{
			screenID = dRegion->screenID;
			tRegion = dRegion->region;
			sRegion = region;

			// Region is split at a maximum shown by the following set
			// of regions:
			//		���������Ŀ		1
		 	//		���������Ĵ		2,3,4
		 	//		���������Ĵ		5
		 	//		�����������
		 	//

			// Check for a top region (region 1 above).
			if (region.top > tRegion.top)
				UI_LIST::Add(0, new UI_REGION_ELEMENT(screenID, tRegion.left,
					tRegion.top, tRegion.right, region.top - 1));
			else
				sRegion.top = tRegion.top;

			// Check for a bottom region (region 5 above).
			if (region.bottom < tRegion.bottom)
				UI_LIST::Add(0, new UI_REGION_ELEMENT(screenID, tRegion.left,
					region.bottom + 1, tRegion.right, tRegion.bottom));
			else
				sRegion.bottom = tRegion.bottom;

			// Check for a left region (region 2 above).
			if (region.left > tRegion.left)
				UI_LIST::Add(0, new UI_REGION_ELEMENT(screenID, tRegion.left,
					sRegion.top, region.left - 1, sRegion.bottom));

			// Check for a right region (region 4 above).
			if (region.right < tRegion.right)
				UI_LIST::Add(0, new UI_REGION_ELEMENT(screenID, region.right + 1,
					sRegion.top, tRegion.right, sRegion.bottom));

			// Region 3 is the object's region.
			UI_LIST::Subtract(dRegion);
			delete dRegion;
		}
	}
}

static int VideoPresent(char far *videoBuffer)
{
	char oldContents = *videoBuffer;
	*videoBuffer = ~oldContents;
	int  result = (*videoBuffer == ~oldContents) ? TRUE : FALSE;
	*videoBuffer = oldContents;
	return result;
}

UCHAR UI_DISPLAY::usingActiveDisplay = FALSE;
UCHAR UI_DISPLAY::usingAlternateDisplay = FALSE;

UCHAR UI_DISPLAY::activeDisplayCode = DC_NO_DISPLAY;
UCHAR UI_DISPLAY::alternateDisplayCode = DC_NO_DISPLAY;
UCHAR UI_DISPLAY::oldEquipmentFlags = 0;
UCHAR UI_DISPLAY::oldVideoMode = 0;

UI_DISPLAY::UI_DISPLAY(UCHAR a_isText, UCHAR a_cellWidth, UCHAR a_cellHeight) :
	isText(a_isText),
	cellWidth(a_cellWidth),
	cellHeight(a_cellHeight)
{
	installed = FALSE;
	eventManager = 0;
	if (activeDisplayCode == DC_NO_DISPLAY)
	{
		// Save the current video state.
		oldEquipmentFlags = *((UCHAR far *)0x410L);
		_AX = 0x1A00;
		VideoInt();
		if (_AL != 0x1A)		// Display Combination Code call not supported.
		{
			int hasColor;
			int hasMono;
			UCHAR egaCode;
			_AH = 0x12;
			_BL = 0x10;
			VideoInt();
			egaCode = (_BL == 0x10) ? 0 : DC_EGA_COLOR + _BH;
			hasColor = VideoPresent((char far *) 0xB8000000L);
			hasMono = VideoPresent((char far *) 0xB0000000L);
			if ((oldEquipmentFlags & 0x30) == 0x30)		// Primary mono
			{
				activeDisplayCode = DC_MONOCHROME;
				if (egaCode == DC_EGA_COLOR)
					alternateDisplayCode = egaCode;
				else if (egaCode == DC_EGA_MONO)
					activeDisplayCode = DC_EGA_MONO;
				else if (hasColor)
					alternateDisplayCode = DC_CGA;
			}
			else
			{
				activeDisplayCode = egaCode ? egaCode : DC_CGA;
				if (hasMono)
					alternateDisplayCode = DC_MONOCHROME;
			}
		}
		else
		{
	    	activeDisplayCode = _BL;
			alternateDisplayCode = _BH;
		}
		displayCode = activeDisplayCode;
		_AH = 0x0F;
		VideoInt();
		_DL = _AL;
		oldVideoMode = _DL;
		// oldVideoMode = _AL;		// Borland Bug: Doesn't work in large model
	}
	if (!usingActiveDisplay)
	{
		isActiveDisplay = TRUE;
		usingActiveDisplay = TRUE;
		// Save the current video state.
		equipmentFlags = oldEquipmentFlags;
		displayCode = activeDisplayCode;
	}
	else
	{
		isActiveDisplay = FALSE;
		usingAlternateDisplay = TRUE;
		equipmentFlags = (alternateDisplayCode == DC_MONOCHROME) ?
			oldEquipmentFlags | 0x30 :
			(oldEquipmentFlags & ~0x30) | 0x20;
		displayCode = alternateDisplayCode;
	}
}

UI_DISPLAY::~UI_DISPLAY(void)
{
	*((UCHAR far *)0x410L) = equipmentFlags;
	_AX = isActiveDisplay ? (USHORT)oldVideoMode :
		(displayCode == DC_MONOCHROME ? 7 : 3);
	VideoInt();
	if (isActiveDisplay)
		usingActiveDisplay = FALSE;
	else
		usingAlternateDisplay = FALSE;
}

void UI_DISPLAY::Fill(int screenID, int left, int top, int right, int bottom,
	const UI_PALETTE *palette)
{
	UI_REGION region;

	region.left = left;
	region.top = top;
	region.right = right;
	region.bottom = bottom;
	Fill(screenID, region, palette);
}

void UI_DISPLAY::RegionDefine(int screenID, const UI_REGION &a_region)
{
	// See if it is a full screen definition.
	UI_REGION region = a_region;
	if (region.left <= 0 && region.top <= 0 && region.right >= columns - 1 && region.bottom >= lines - 1)
	{
		regionList.Destroy();
		regionList.Add(0, new UI_REGION_ELEMENT(screenID, 0, 0, columns - 1, lines - 1));
		return;
	}

	// Throw away regions with a negative width or height.
	if (region.right < region.left || region.bottom < region.top)
		return;

	// Throw away regions that are completely off the screen.
	if (region.left >= columns || region.right < 0 ||
		region.top >= lines || region.bottom < 0)
		return;

	// Clip regions partially off the screen to fit on the screen.
	if (region.left < 0)
		region.left = 0;
	if (region.right >= columns)
		region.right = columns - 1;
	if (region.top < 0)
		region.top = 0;
	if (region.bottom >= lines)
		region.bottom = lines - 1;

	// Split any overlapping regions.
	regionList.Split(screenID, region);

	// Define the new display region.
	regionList.Add(0, new UI_REGION_ELEMENT(screenID, &region));
}

