//	Program name..	Zinc Interface Library
//	Filename......	EVENT.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include "ui_evt.hpp"

// ----- Constructor & Destructor -------------------------------------------

UI_EVENT_MANAGER::UI_EVENT_MANAGER(int a_noOfElements, UI_DISPLAY *a_display) :
	queueBlock(a_noOfElements),
	deviceList(UI_DEVICE::CompareFunction)
{
	display = a_display;
	display->eventManager = this;
	level = 1;
}

// Member functions ---------------------------------------------------------

void UI_EVENT_MANAGER::Add(UI_DEVICE *device)
{
	// Add the device to the object list.
	deviceList.Add(device);

	device->display = display;
	device->eventManager = this;
	UI_EVENT event;
	event.type = device->type;
	event.rawCode = D_INITIALIZE;
	device->Event(event);
}

void UI_EVENT_MANAGER::DevicesHide(const UI_REGION &region)
{
	// Hide devices when at the highest level.
	if (level-- <= 0)
		return;

	disable();
	UI_EVENT event;
	event.type = E_DEVICE;
	event.rawCode = D_HIDE;
	event.region = region;
	for (UI_DEVICE *device = First(); device; device = device->Next())
	 	device->Event(event);
}

void UI_EVENT_MANAGER::DevicesShow(const UI_REGION &region)
{
	// Show devices only when at the highest level.
	if (level > -1)
	{
		UI_EVENT event;
		event.type = E_DEVICE;
		event.rawCode = D_SHOW;
		event.region = region;
		for (UI_DEVICE *device = Last(); device; device = device->Previous())
	 		device->Event(event);
		enable();
	}
	level++;
}

void UI_EVENT_MANAGER::DevicePosition(int deviceType, int column, int line)
{
	UI_EVENT event;
	event.type = deviceType;
	event.rawCode = D_POSITION;
	event.position.column = column;
	event.position.line = line;
	UI_EVENT_MANAGER::Event(event);
}

int UI_EVENT_MANAGER::DeviceState(int deviceType, USHORT deviceState)
{
	UI_EVENT event;
	event.type = deviceType;
	event.rawCode = deviceState;
	return (UI_EVENT_MANAGER::Event(event));
}

int UI_EVENT_MANAGER::Event(const UI_EVENT &event)
{
	// Process the event.
	int ccode = event.type;
	int status = D_OFF;
	for (UI_DEVICE *device = First(); device; device = device->Next())
		if (ccode == E_DEVICE || ccode == device->type)
		 	status = device->Event(event);
	return (status);
}

int UI_EVENT_MANAGER::Get(UI_EVENT &event, USHORT flags)
{
	UI_DEVICE *device;
	UI_QUEUE_ELEMENT *element;
	int error = -1;

	// Stay in loop while no event conditions are met.
	do
	{
		// Call all the polled devices.
		if (!FlagSet(flags, Q_NO_POLL))
			for (device = First(); device; device = device->Next())
			 	device->Poll();

		// Get the event.
		disable();
		element = (FlagSet(flags, Q_END)) ?
			queueBlock.Last() : queueBlock.First();
		if (element)
		{
			event = element->event;
			if (!FlagSet(flags, Q_NO_DESTROY))
				queueBlock.Subtract((UI_ELEMENT *)element);

			if (!FlagSet(flags, Q_NO_POLL))
			{
				for (device = First(); device; device = device->Next())
					if (device->type == E_DEVICE)
					 	device->Event(event);
			}

			enable();
			error = 0;
		}
		else if (FlagSet(flags, Q_NO_BLOCK))
		{
			enable();
			return (-2);
		}
		enable();

	} while (error);

	// Return the error status.
	return (error);
}

void UI_EVENT_MANAGER::Put(const UI_EVENT &event, USHORT flags)
{
	// Place the event back in the event queue.
	disable();
	UI_QUEUE_ELEMENT *element = (FlagSet(flags, Q_END)) ?
		(UI_QUEUE_ELEMENT *)queueBlock.Add(0) :
		(UI_QUEUE_ELEMENT *)queueBlock.Add(queueBlock.first);
	if (element)
		element->event = event;
	enable();
}

void UI_EVENT_MANAGER::Subtract(UI_DEVICE *device)
{
	UI_EVENT event;
	event.type = device->type;
	event.rawCode = D_RESTORE;
	device->Event(event);
	deviceList.Subtract(device);
}

int UI_DEVICE::CompareFunction(void *device1, void *device2)
{
	return (((UI_DEVICE *)device1)->type - ((UI_DEVICE *)device2)->type);
}

UI_QUEUE_BLOCK::UI_QUEUE_BLOCK(int a_noOfElements) :
	UI_LIST_BLOCK(a_noOfElements, 0)
{
	// Initialize the queue block.
	UI_QUEUE_ELEMENT *queueBlock = new UI_QUEUE_ELEMENT[a_noOfElements];
	elementArray = queueBlock;
	for (int i = 0; i < a_noOfElements; i++)
		freeList.Add(0, &queueBlock[i]);
}

UI_QUEUE_BLOCK::~UI_QUEUE_BLOCK(void)
{
	// Free the queue block.
	UI_QUEUE_ELEMENT *queueBlock = (UI_QUEUE_ELEMENT *)elementArray;
	delete [noOfElements]queueBlock;
}
