//	Program name..	Zinc Interface Library
//	Filename......	FMTSTR.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

// ----- Constructor --------------------------------------------------------

FS_STATE::FS_STATE(FS_STATE& src)
{
	this->isMarked = src.isMarked;
	this->cursor = src.cursor;
	int srcTextLen = (src.text) ? strlen(src.text) : 0;
	this->text = new char[srcTextLen + 1];
	strcpy(this->text, src.text);
}

// ----- Member functions ---------------------------------------------------

void FS_STATE::operator = (FS_STATE& src)
{
	this->isMarked = src.isMarked;
	this->cursor = src.cursor;
	strcpy(this->text, src.text);
}

int FS_STATE::operator == (FS_STATE& rightOperand)
{
	return (this->isMarked == rightOperand.isMarked &&
		this->cursor == rightOperand.cursor &&
		!strcmp(this->text, rightOperand.text));
}

int FS_STATE::operator != (FS_STATE& rightOperand)
{
	return this->isMarked != rightOperand.isMarked ||
		this->cursor != rightOperand.cursor ||
		strcmp(this->text, rightOperand.text);
}

// ----- Constructor & Destructor -------------------------------------------

UIW_FORMATTED_STRING::UIW_FORMATTED_STRING(int left, int top, int width,
	char *a_text, char *a_editMask, char *a_literalMask, USHORT a_woFlags,
	int (*a_Validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, a_woFlags, WOAF_NO_FLAGS)
{
	windowID[0] = ID_STRING;

	Validate = a_Validate;
	int editMaskLen = strlen(a_editMask);
	editMaskLen = min(editMaskLen, width);
	int litMaskLen = strlen(a_literalMask);
	editMask = new char[editMaskLen];
	memcpy(editMask, a_editMask, editMaskLen);
	literalMask = new char[editMaskLen];
	memcpy(literalMask, a_literalMask, editMaskLen);
	if (litMaskLen < editMaskLen)
		memset(literalMask + litMaskLen, '?', editMaskLen - litMaskLen);
	maskLen = editMaskLen;
	fieldWidth = width;
	int textLen = TextLength();
	int passedLength = ui_strlen(a_text);
	if (FlagSet(a_woFlags, WOF_NO_ALLOCATE_DATA))
		state.text = a_text;
	else
	{
		state.text = new char[textLen + 1];
		int actualLen = min(textLen, passedLength);
		if (a_text)
			memcpy(state.text, a_text, actualLen);
		state.text[actualLen] = '\0';
	}
}

UIW_FORMATTED_STRING::~UIW_FORMATTED_STRING(void)
{
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		state.text = 0; 				 	// So FS_STATE destructor doesn't free it.
	delete editMask;
	delete literalMask;
}

// ----- Member functions ---------------------------------------------------

void UIW_FORMATTED_STRING::DataSet(char *newText)
{
	if (newText)
	{
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			state.text = newText;
		else
		{
			delete state.text;
			int textLen = TextLength();
			state.text = new char[textLen + 1];
			int actualLen = strlen(newText);
			actualLen = min(textLen, actualLen);
			memcpy(state.text, newText, actualLen);
			state.text[actualLen] = '\0';
		}
	}
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

int UIW_FORMATTED_STRING::Event(const UI_EVENT &event)
{
	int viewOnly = FlagSet(woFlags, WOF_VIEW_ONLY) ? TRUE : FALSE;
 	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_STRING);
	short cellWidth = display->TextWidth("W");
	UI_REGION region;
	UI_WINDOW_OBJECT::Border(0, region, 0);
	short width = (region.right + 1 - region.left) / cellWidth;
	USHORT key;
	if (ccode == S_CREATE)
	{
		UndoDestroy(undoHandle);
		undoHandle = 0;
		state.isMarked = FALSE;
		state.cursor = 0;
		FixText();
	}
	FS_STATE oldStateInfo(state);
	UCHAR forceDisplay = FALSE;
	UCHAR reDisplay = TRUE;
	UCHAR displayCheckRegion = FALSE;
	int needValidate = NeedsValidation();

	if (!FlagSet(woStatus, WOS_CURRENT))
		state.isMarked = FALSE;
	switch (ccode)
	{
	case E_KEY:
		key = event.rawCode & 0xFF;
		if ( (key == '\b' || key >= ' ') && !viewOnly)
		{
			if (key == '\b')
				BackspaceKey();
			else
				RegularKey(key);
		}
		else
		{
			reDisplay = FALSE;
			ccode = S_UNKNOWN;
		}
		break;

	case L_MOVE_LEFT:
		if (state.cursor > 0)
			state.cursor--;
		else
			ccode = S_UNKNOWN;
		break;

	case L_MOVE_RIGHT:
		if (state.text[state.cursor + 1])
			state.cursor++;
		else
			ccode = S_UNKNOWN;
		break;

	case L_MOVE_BOL:
		state.cursor = 0;
		break;

	case L_MOVE_EOL:
		state.cursor = ui_strlen(state.text) - 1;
		break;

	case L_DELETE:
		if (!viewOnly)
		{
			if (state.isMarked)
				CutBlock();
			else
				DeleteKey();
		}
		break;

	case L_MARK:
		state.isMarked = !state.isMarked;
		break;

	case L_MOVE:
		eventManager->DeviceState(event.type, DM_EDIT);
		reDisplay = FALSE;
		break;

	case L_CUT:
		if (!viewOnly)
		{
			if (state.isMarked)
				CutBlock();
			else
				reDisplay = FALSE;
			break;
		}
		// If view only, treat a CUT as a copy.

	case L_COPY_MARK:
		if (state.isMarked)
			CopyBlock();
		else
			reDisplay = FALSE;
		break;

	case L_CUT_PASTE:
		if (state.isMarked)
		{
			if (viewOnly)
				CopyBlock();
			else
				CutBlock();
			break;
		}
		// Continue to L_PASTE.

	case L_PASTE:
		if (!viewOnly)
		{
			if (event.position.line != -1)
				SetCursor((event.position.column - region.left) / cellWidth, width);
			PasteBlock();
		}
		break;

	case L_UNDO:
		{
			FS_UNDO *undo = (FS_UNDO *) UndoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPush(undoHandle);
			}
		}
		break;

	case L_REDO:
		{
			FS_UNDO *undo = (FS_UNDO *) UndoRedoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPop(undoHandle);
			}
		}
		break;

	case L_CONTINUE_MARK:
		if (markColumn != (event.position.column - region.left) / cellWidth)
			state.isMarked = TRUE;
		break;

	case L_BEGIN_MARK:
		state.isMarked = FALSE;
		markColumn = (event.position.column - region.left) / cellWidth;
		SetCursor(markColumn, width);
		forceDisplay = TRUE;			// Force cursor to get updated.
		break;

	case S_CURRENT:
		woStatus &= ~WOS_UNANSWERED;
		if (needValidate)
			(*Validate)(this, ccode);
		if (!undoHandle)
			undoHandle = UndoCreate();
		displayCheckRegion = TRUE;
		break;

	case S_DISPLAY_ACTIVE:
		displayCheckRegion = TRUE;
		break;

	case S_NON_CURRENT:
		if ( needValidate && (*Validate)(this, ccode) != 0 )
		{
			ccode = S_ERROR;
			woStatus |= WOS_INVALID;
			reDisplay = FALSE;
			break;
		}
		woStatus &= ~WOS_INVALID;
		// else fall through to S_DISPLAY_INACTIVE.
	case S_DISPLAY_INACTIVE:
		state.isMarked = FALSE;
		displayCheckRegion = TRUE;
		break;

	case S_ERROR_RESPONSE:
		woStatus |= event.rawCode;
		forceDisplay = TRUE;
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		reDisplay = FALSE;
		break;
	}

	if (displayCheckRegion)
	{
		forceDisplay = TRUE;
		if ( !UI_WINDOW_OBJECT::NeedsUpdate(event, ccode) &&
			  oldStateInfo == state )
			reDisplay = FALSE;
		else
		{
			UI_WINDOW_OBJECT::Border(ccode, region, lastPalette);
		}
	}

	char dText[256];
	int cursorColumn;
	if (reDisplay)
	{
		if (FlagSet(woStatus, WOS_UNANSWERED))
			FixText();
		if (ccode != L_UNDO && ccode != L_REDO)
		{
			if (oldStateInfo != state)
			{
				FS_UNDO *undo = new FS_UNDO(oldStateInfo);
				UndoAdd(undoHandle, undo, sizeof(FS_UNDO));
			}
		}
		if (forceDisplay || oldStateInfo != state)
		{
			cursorColumn = Expand(dText);
			Redisplay(region, width, dText);
			UpdateCursor(region, width, dText, cursorColumn);
		}
	}
	else if (ccode == S_CURRENT)
	{
		cursorColumn = Expand(dText);
		UpdateCursor(region, width, dText, cursorColumn);
	}
	return (ccode);
}

void UIW_FORMATTED_STRING::BackspaceKey(void)
{
	if (state.cursor > 0)
	{
		state.cursor--;
		DeleteKey();
	}
}

int UIW_FORMATTED_STRING::CalcLeading(int width, char *dText)
{
	int textLen = strlen(dText);
	int leading = 0;
	if (woFlags & WOF_JUSTIFY_CENTER)
		leading = min((fieldWidth - textLen) / 2, width);
	else if (woFlags & WOF_JUSTIFY_RIGHT)
		leading = min(fieldWidth - textLen, width);
	if (leading < 0)
		leading = 0;
	return leading;
}

void UIW_FORMATTED_STRING::CopyBlock(void)
{
	char dText[256];

	Expand(dText);
	pasteLength = (USHORT) strlen(dText);
	if (pasteBuffer)
		delete pasteBuffer;
	pasteBuffer = new char[pasteLength];
	if (pasteBuffer)
		memcpy(pasteBuffer, dText, pasteLength);
	state.isMarked = FALSE;
}

void UIW_FORMATTED_STRING::CutBlock(void)
{
	CopyBlock();
	int textIndex = 0;
	for (int i = 0; state.text[textIndex] && i < maskLen; i++)
	{
		if (editMask[i] != 'L')
			state.text[textIndex++] = literalMask[i];
	}
	state.cursor = 0;
}

void UIW_FORMATTED_STRING::DeleteKey(void)
{
	int column = ExpandedColumn();
	// ## Borland bug: large model code generation problem on following line:
	// state.text[state.cursor] = literalMask[column];
	char literalChar = literalMask[column];
	state.text[state.cursor] = literalChar;
}

int UIW_FORMATTED_STRING::Expand(char *destText)
{
	char *src = state.text;
	int column = maskLen - 1;
	for (int i = 0; i < maskLen; i++)
	{
		if (src - state.text == state.cursor)
			column = i;
		if ( editMask[i] == 'L' )
			*destText++ = literalMask[i];
		else
			*destText++ = *src++;
	}
	*destText = 0;
	return column;
}

int UIW_FORMATTED_STRING::ExpandedColumn(void)
{
	char *src = state.text;
	int column = maskLen - 1;
	for (int i = 0; i < maskLen; i++)
	{
		if (src - state.text == state.cursor)
			column = i;
		if (editMask[i] != 'L')
			src++;
	}
	return column;
}

void UIW_FORMATTED_STRING::FixText(void)
{
	int textLen = TextLength();
	if (FlagSet(woStatus, WOS_UNANSWERED))
		memset(state.text, '\0', textLen + 1);
	int copyLen = strlen(state.text);
	int t = 0;
	for (int i = 0; i < maskLen; i++)
	{
		if (editMask[i] != 'L')
		{
			int textValue = ValidateMask(editMask[i], state.text[t]);
			if (t >= copyLen || !textValue)
				state.text[t] = literalMask[i];
			else
				state.text[t] = textValue;
			t++;
		}
	}
	state.text[textLen] = '\0';
}

void UIW_FORMATTED_STRING::PasteBlock(void)
{
	if (pasteBuffer && !state.isMarked)
	{
		int textIndex = 0;
		int p = 0;
		for (int i = 0; i < maskLen && p < pasteLength; i++)
		{
			if (editMask[i] != 'L')
			{
				int textValue;
				do
				{
					textValue = ValidateMask(editMask[i], pasteBuffer[p]);
					p++;
				} while (!textValue && p < pasteLength);
				if (textValue)
					state.text[textIndex++] = textValue;
			}
		}
		state.text[textIndex] = '\0';
		FixText();
		state.cursor = 0;
	}
}

void UIW_FORMATTED_STRING::Redisplay(UI_REGION &region, short width,
	char *dText)
{
	short cellHeight = display->TextHeight("W");
	short cellWidth = display->TextWidth("W");
	if ( !display->isText || (woStatus & WOS_UNANSWERED) )
	{							// Pre-fill for graphics or unanswered.
		display->Fill(screenID,	region.left, region.top,
			region.right, region.bottom, lastPalette);
	}
	int text_len = strlen(dText);
	int leading = CalcLeading(width, dText);
	text_len = min(text_len, width - leading);
	int trailing = width - text_len - leading;
	int bottom = region.top + display->TextHeight("W") - 1;
	if (bottom <= region.bottom && (woStatus & WOS_UNANSWERED) == 0)
	{
		if (!display->isText)			// Center text vertically.
		{
			int vertRoom = region.bottom + 1 - region.top - cellHeight;
			region.top 	  += (vertRoom + 1) / 2;
			region.bottom += (vertRoom + 1) / 2;
		}
		if (leading && display->isText)
		{
			display->Fill(screenID,
				region.left, region.top,
				region.left + leading - 1, region.bottom, lastPalette);
		}
		display->Text(screenID,
			region.left + leading * cellWidth,
			region.top, dText, lastPalette, text_len, FALSE);
		if ( trailing && display->isText )
		{
			display->Fill(screenID,
				region.left + leading + text_len, region.top,
				region.left + leading + text_len + trailing - 1,
				region.bottom, lastPalette);
		}
		if (state.isMarked)
			display->TextXOR(region.left, region.top, width);
	}
}

void UIW_FORMATTED_STRING::RegularKey(USHORT key)
{
	int textValue = ValidateMask(editMask[ExpandedColumn()], key);
	if (textValue)
	{
		state.text[state.cursor] = textValue;
		if (state.text[state.cursor + 1])
			state.cursor++;
	}
}

void UIW_FORMATTED_STRING::SetCursor(int column, int width)
{
	char expandedText[256];
	Expand(expandedText);
	int leading = 0;
	int textLen = strlen(expandedText);
	if (woFlags & WOF_JUSTIFY_CENTER)
		leading = min((fieldWidth - textLen) / 2, width);
	else if (woFlags & WOF_JUSTIFY_RIGHT)
		leading = min(fieldWidth - textLen, width);
	column -= leading;
	column = min(column, maskLen);
	int match = 0;
	for (int i = 0; i < column; i++)
	{
		if (editMask[i] != 'L')
			match++;
	}
	state.cursor = match;
}

int UIW_FORMATTED_STRING::TextLength(void)
{
	int textLen = 0;
	for (int i = 0; i < maskLen; i++)
	{
		if (editMask[i] != 'L')
			textLen++;
	}
	return textLen;
}

void UIW_FORMATTED_STRING::UndoOperation(FS_UNDO *undo)
{
	FS_STATE stateInfo(state);

	state = undo->info;
	undo->info = stateInfo;
}

void UIW_FORMATTED_STRING::UpdateCursor(UI_REGION &region, int width,
	char *dText, int cursorColumn)
{
	short cellWidth = display->TextWidth("W");
	int leading = CalcLeading(width, dText);
	if (  FlagSet(woStatus, WOS_CURRENT) &&
		 !FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) &&
		 !FlagSet(woFlags, WOF_VIEW_ONLY) &&
		 (leading + cursorColumn < width) )
	{
		int trueColumn = region.left +
			(leading + cursorColumn) * cellWidth;
		if (trueColumn >= 0 && trueColumn < display->columns &&
			region.top >= 0 && region.top < display->lines )
		{
			eventManager->DevicePosition(E_CURSOR, trueColumn, region.top);
			eventManager->DeviceState(E_CURSOR, DC_OVERTYPE);
		}
	}
}

int UIW_FORMATTED_STRING::ValidateMask(int editValue, int textValue)
{
	int isAlpha = (textValue == ' ' || isalpha(textValue));
	int isDigit = isdigit(textValue);
	if (isupper(editValue))
		textValue = toupper(textValue);
	editValue == toupper(editValue);
	if ((editValue == 'X' && textValue >= ' ' && textValue <= '~') ||
		(editValue == 'N' && isDigit) ||
		(editValue == 'A' && isAlpha) ||
		(editValue == 'C' && (isDigit || isAlpha)))
		return (textValue);
	else
		return (0);
}

