//	Program name..	Zinc Interface Library
//	Filename......	HELPWIN.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <io.h>
#include <fcntl.h>
#include <string.h>
#include "ui_win.hpp"

// Static variables & functions ---------------------------------------------

static char *_defaultTitle = "No Help";

static void ItemClose(void *item, UI_EVENT &event)
{
 	event.type = S_DELETE;
 	((UIW_PULL_DOWN_ITEM *)item)->eventManager->Put(event, Q_BEGIN);
}

// Constructor & Destructor -------------------------------------------------

UI_HELP_WINDOW_SYSTEM::UI_HELP_WINDOW_SYSTEM(char *helpFileName,
	UI_WINDOW_MANAGER *a_windowManager, int a_defaultHelpContext)
{
	defaultHelpContext = a_defaultHelpContext;
	installed = FALSE;
	index = 0;
	strings = 0;
	helpWindow = 0;
	message = 0;
	windowManager = a_windowManager;
	char *helpName = new char[strlen(helpFileName) + 4];
	strcpy(helpName, helpFileName);
	if (!strchr(helpName, '.'))
		strcat(helpName, ".hlp");
	helpFile = OpenPath(helpName, O_RDONLY);
	delete helpName;
	if (helpFile < 0)
		return;

	if (!Read(&header, sizeof(header)) ||
		header.magicWord != HELP_MAGIC_WORD ||
		header.helpVersion != HELP_VERSION ||
		header.numContexts > MAX_HELP_CONTEXTS ||
		header.indexPtr < (long)sizeof(header))
		return;
	if (!Seek(header.indexPtr))
		return;
	index = new TEXT_INDEX[header.numContexts];
	if (!Read(index, sizeof(TEXT_INDEX) * header.numContexts))
		return;
	int stringSize = (int)(filelength(helpFile) - lseek(helpFile, 0L, 1));
	strings = new char[stringSize];
	if (!Read(strings, stringSize))
		return;

	// Create the help window.
	title = _defaultTitle;
	message = ui_strdup("");
	helpWindow = new UIW_WINDOW(10, 5, -10, -5, WOF_NO_FLAGS,
		WOAF_NO_DESTROY, NO_HELP_CONTEXT);
	*helpWindow
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ &(*new UIW_SYSTEM_BUTTON
			+ new UIW_POP_UP_ITEM("~Restore", MNIF_RESTORE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("~Move", MNIF_MOVE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("~Size", MNIF_SIZE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("Mi~nimize", MNIF_MINIMIZE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("Ma~ximize", MNIF_MAXIMIZE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM
			+ new UIW_POP_UP_ITEM("~Close", MNIF_CLOSE, BTF_NO_TOGGLE,
				WOF_NO_FLAGS, 0))
		+ (titleField = new UIW_TITLE(title,
			WOF_NO_ALLOCATE_DATA | WOF_JUSTIFY_CENTER))
		+ &(*new UIW_PULL_DOWN_MENU(0, WOF_NO_FLAGS, WOAF_NO_FLAGS)
			+ new UIW_PULL_DOWN_ITEM(" ~Close ", MNF_NO_FLAGS, ItemClose))
		+ new UIW_SCROLL_BAR(0, 0, 0, 0, SBF_VERTICAL, WOF_NON_FIELD_REGION)
		+ (textField = new UIW_TEXT(0, 0, 0, 0, message, 20480,
			TXF_NO_FLAGS, WOF_NO_ALLOCATE_DATA | WOF_VIEW_ONLY |
			WOF_NON_FIELD_REGION, 0));
	helpWindow->paletteMapTable = _helpPaletteMapTable;
	installed = TRUE;
}

UI_HELP_WINDOW_SYSTEM::~UI_HELP_WINDOW_SYSTEM(void)
{
	Close();
	if (index)
		delete index;
	if (strings)
		delete strings;
	if (message)
		delete message;
	if (helpWindow)
	{
		if (helpWindow->windowManager)
			*(helpWindow->windowManager) - helpWindow;
		delete helpWindow;
	}
}

// Member functions ---------------------------------------------------------

void UI_HELP_WINDOW_SYSTEM::Close(void)
{
	// Close the help file.
	if (helpFile != -1)
	{
		_close(helpFile);
		helpFile = -1;
	}
}

void UI_HELP_WINDOW_SYSTEM::DisplayHelp(UI_WINDOW_MANAGER *a_windowManager,
	int helpContext)
{
	if (helpFile < 0 || !installed)
		return;

	// For now, grab first context.
	windowManager = a_windowManager;
	if (helpContext == NO_HELP_CONTEXT)
		helpContext = defaultHelpContext;

	// Construct the new help message and title.
	title = _defaultTitle;
	delete message;
	if (helpContext == NO_HELP_CONTEXT)
		message = ui_strdup("\nNo help available at this point.");
	else if (helpContext < 1 || helpContext > header.numContexts)
		message = ui_strdup("\nInvalid help context.");
	else
	{
		int length;
		title = "Help Error";
		if (Seek(index[helpContext - 1].offset))
		{
			length = index[helpContext - 1].length;
			message = new char[length + 1];
			if (!Read(&message[0], length))
			{
				delete message;
				message = new char[128];
				strcpy(message, "\nError reading help file.");
			}
			else
			{
				message[length] = '\0';
				title = &strings[index[helpContext - 1].titleIndex];
			}
		}
		else
		{
			message = new char[128];
			strcpy(message, "\nError positioning in help file.");
		}
	}

	// Show the help window.
	for (UI_WINDOW_OBJECT *object = helpWindow->First(); object;
		object = object->Next())
		object->woStatus &= ~WOS_CURRENT;
	textField->woStatus |= WOS_CURRENT;
	helpWindow->display = 0;
	helpWindow->eventManager = 0;
	helpWindow->windowManager = 0;
	titleField->DataSet(title);
	textField->DataSet(message);
	*windowManager + helpWindow;
}

int UI_HELP_WINDOW_SYSTEM::Read(void *buffer, unsigned length)
{
	// Read data from the help file.
	if (helpFile < 0 || _read(helpFile, buffer, length) != length)
		return (FALSE);
	return (TRUE);
}

int UI_HELP_WINDOW_SYSTEM::Seek(long fileOffset)
{
	// Position inside the help file.
	if (helpFile < 0 || lseek(helpFile, fileOffset, 0) != fileOffset)
		return (FALSE);
	return (TRUE);
}
