//	Program name..	Zinc Interface Library
//	Filename......	MATRIX.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <math.h>
#include "ui_win.hpp"

// Constructor & Destructor -------------------------------------------------

UIW_MATRIX::UIW_MATRIX(int left, int top, int width, int height,
	int a_maxRowsColumns, int a_cellWidth, int a_cellHeight,
	int (*a_compareFunction)(void *element1, void *element2),
	USHORT a_mxFlags, USHORT a_woFlags, USHORT a_woAdvancedFlags) :
	UIW_WINDOW(left, top, width, height, a_woFlags, a_woAdvancedFlags)
{
	// Initialize the matrix information.
	windowID[0] = ID_MATRIX;
	windowID[1] = ID_WINDOW;
	windowList.compareFunction = a_compareFunction;
	mxFlags = a_mxFlags;
	maxRowsColumns = a_maxRowsColumns;
	cellWidth = a_cellWidth;
	cellHeight = a_cellHeight;
}

// Member functions ---------------------------------------------------------

UIW_MATRIX::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	int height = 0;
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_MATRIX);
	UI_WINDOW_OBJECT *object;
	UI_EVENT tEvent;

	UI_WINDOW_OBJECT *topObject, *bottomObject;
	UI_WINDOW_OBJECT *tObject = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);

	int scrollRecompute = FALSE;
	int tcode;
	switch (ccode)
	{
	case L_PGUP:
	case L_PGDN:
	case L_FIELD_PREVIOUS:
	case L_FIELD_NEXT:
	case L_FIELD_UP:
	case L_FIELD_DOWN:
		topObject = bottomObject = First();
		for (object = First(); object; object = object->Next())
		{
			if (object->relative.top < topObject->relative.top)
				topObject = object;
			if (object->relative.top > bottomObject->relative.top)
				bottomObject = object;
		}
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (((ccode == L_FIELD_PREVIOUS || ccode == L_PGUP) && !object->previous) ||
			((ccode == L_FIELD_NEXT || ccode == L_PGDN) && !object->next) ||
			(ccode == L_FIELD_UP && object->relative.top == topObject->relative.top) ||
			(ccode == L_FIELD_DOWN && object->relative.bottom == bottomObject->relative.bottom))
			return (S_UNKNOWN);

		scrollRecompute = TRUE;
		if (ccode != L_PGDN && ccode != L_PGUP)
			tcode = UIW_WINDOW::Event(event);
		if (tcode != S_UNKNOWN && tcode != S_ERROR && ccode != L_PGDN && ccode != L_PGUP)
//		if (tcode != S_UNKNOWN && tcode != S_ERROR)
			break;
		if (ccode == L_FIELD_UP || ccode == L_FIELD_PREVIOUS)
			height = display->cellHeight;
		else if (ccode == L_FIELD_DOWN || ccode == L_FIELD_NEXT)
			height = -display->cellHeight;
		else if (ccode == L_PGUP)
			height = ((true.bottom - true.top + 1) / display->cellHeight) * display->cellHeight;
		else
			height = -((true.bottom - true.top - 1) / display->cellHeight) * display->cellHeight;
		// Continue to S_SCROLL_HORIZONTAL and S_SCROLL_VERTICAL.

	case S_SCROLL_VERTICAL:
		// Get the delta scroll values.
		if (ccode == S_SCROLL_VERTICAL)
			height = event.scroll.delta * display->cellHeight;
		topObject = bottomObject = First();
		for (object = First(); object; object = object->Next())
		{
			if (object->relative.top < topObject->relative.top)
				topObject = object;
			if (object->relative.top > bottomObject->relative.top)
				bottomObject = object;
		}
		if ((height > 0 && FlagSet(topObject->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL)) ||
			(height < 0 && FlagSet(bottomObject->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL)))
		{
			if (height < 0 &&
				bottomObject->relative.bottom < true.bottom - true.top - 1 /* - display->cellHeight */)
				height = ((true.bottom - true.top - bottomObject->relative.bottom) /
					display->cellHeight) * display->cellHeight;
			else if (height > 0 &&
				topObject->relative.top + height > 0)
				height = -topObject->relative.top;
			for (object = First(); object; object = object->Next())
			{
				object->relative.top += height;
				object->relative.bottom += height;
			}
			tEvent.type = S_SIZE;
			UIW_WINDOW::Event(tEvent);
 			UIW_WINDOW::RegionsCompute();
		}

		// Change the current field.
		if (!tObject)
			break;
		height = (ccode == S_SCROLL_VERTICAL) ? event.scroll.delta : height / display->cellHeight;
		topObject = tObject;
		for (object = tObject->Previous(); object && height > 0; object = object->Previous())
			if (topObject->relative.top > object->relative.top)
			{
				topObject = object;
				height--;
			}
			else if (topObject->relative.top == object->relative.top)
				topObject = object;
		for (object = topObject->Previous(); object; object = object->Previous())
			if (object->relative.top == topObject->relative.top)
				topObject = object;
			else
				break;
		for (object = tObject->Next(); object && height < 0; object = object->Next())
			if (topObject->relative.top < object->relative.top)
			{
				topObject = object;
				height++;
			}
		if (topObject)
		{
			object = topObject;
			while (object->relative.top == topObject->relative.top)
				if (object->relative.left == tObject->relative.left)
					break;
				else
				{
					topObject = topObject->Next();
					if (!topObject || topObject->relative.top != object->relative.top)
						break;
					object = topObject;
				}
			tObject->woStatus &= ~WOS_CURRENT;
			object->woStatus |= WOS_CURRENT;

			scrollRecompute = TRUE;
		}

		tEvent.type = S_CURRENT;
		tEvent.region = true;
		ccode = UIW_WINDOW::Event(tEvent);
		break;

	case S_CREATE:
		{
		UI_WINDOW_OBJECT::Event(event);

		if (!FlagSet(mxFlags, MXF_ROWS_FILL | MXF_COLUMNS_FILL))
		{
			ccode = UIW_WINDOW::Event(event);
			break;
		}

		int top = 0;
		int left = 0;
		int count = maxRowsColumns;
		for (object = First(); object; object = object->Next())
		{
			object->relative.left = left * cellWidth;
			object->relative.top = top * cellHeight;
			object->relative.right = object->relative.left + cellWidth - 1;
			object->relative.bottom = object->relative.top + cellHeight - 1;
			object->woStatus &= ~WOS_GRAPHICS;
			if (FlagSet(mxFlags, MXF_ROWS_FILL) && ++top == count)
			{
				left++;
				top = 0;
			}
			else if (!FlagSet(mxFlags, MXF_ROWS_FILL) && ++left == count)
			{
				top++;
				left = 0;
			}
		}
		}
		ccode = UIW_WINDOW::Event(event);
		break;

	case S_CURRENT:
	case S_DISPLAY_INACTIVE:
	case S_DISPLAY_ACTIVE:
		scrollRecompute = TRUE;
		// Continue to default.

	default:
		ccode = UIW_WINDOW::Event(event);
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (ccode != S_CREATE && ccode != S_SIZE && tObject != object)
			scrollRecompute = TRUE;

		// Unhighlight the remaining items.
		if (!object || !FlagSet(mxFlags, MXF_SELECT_ONE) ||
			!FlagsSet(object->woStatus, WOS_SELECTED))
			break;
		tEvent.type = S_DISPLAY_ACTIVE;
		for (UI_WINDOW_OBJECT *tObject = First(); tObject; tObject = tObject->Next())
			if (tObject != object && FlagSet(tObject->woStatus, WOS_SELECTED))
			{
				tObject->woStatus &= ~WOS_SELECTED;
				tEvent.region = tObject->true;
				if (!FlagSet(tObject->woAdvancedStatus, WOAS_INVALID_REGION))
					tObject->Event(tEvent);
			}
		break;
	}

	if (scrollRecompute && previous)
	{
		tEvent.type = S_SCROLL_VERTICAL_SET;
		topObject = bottomObject = First();
		for (object = First(); object; object = object->Next())
		{
			if (object->relative.top < topObject->relative.top)
				topObject = object;
			if (object->relative.top > bottomObject->relative.top)
				bottomObject = object;
		}
		object = (UI_WINDOW_OBJECT *)windowList.Get(UIW_WINDOW::FindStatus, &WOS_CURRENT);
		if (!object)
			object = First();
		while (object && FlagSet(object->woAdvancedStatus, WOAS_INVALID_REGION | WOAS_TOO_SMALL))
			object = object->Next();
		tEvent.scroll.current = object ? (object->relative.top - topObject->relative.top) / (cellHeight * display->cellHeight) : 0;
		tEvent.scroll.showing = (true.bottom - true.top) / (cellHeight * display->cellHeight);
		tEvent.scroll.maximum = (bottomObject->relative.top - topObject->relative.top) / (cellHeight * display->cellHeight);
		object = Previous();

		if (object->Event(tEvent) != S_UNKNOWN)
		{
			tEvent.type = S_DISPLAY_ACTIVE;
			tEvent.region = object->true;
			object->Event(tEvent);
		}
	}

	// Return the event control code.
	return (ccode);
}
