//	Program name..	Zinc Interface Library
//	Filename......	STRING.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#pragma inline

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

#if 0

// Original C version of the "LineChanged" routine below.
USHORT LineChanged(char *newLine, char *oldLine, int width)
{
	if (width > 0)
	{
		int leftCol = 0;
		int rightCol = width - 1;
		while (leftCol <= rightCol && newLine[leftCol] == oldLine[leftCol])
			leftCol++;
		while (leftCol <= rightCol && newLine[rightCol] == oldLine[rightCol])
			rightCol--;
		if (leftCol <= rightCol)
		{
			memcpy(&oldLine[leftCol], &newLine[leftCol], rightCol + 1 - leftCol);
			return leftCol + (rightCol << 8);
		}
	}
	return 0xFFFF;
}
#endif

/* LineChanged -
 *		Compare a line of text about to be written to the screen with the
 *		saved copy of that line to see if there are any changes.  If so,
 *		return the leftmost column of the changed area in the low order
 *		byte, and the rightmost column of the changed area in the high order
 *		byte.  If the line has not changed, return -1 (0xFFFF).
 */
USHORT LineChanged(char *newLine, char *oldLine, int width)
{
#if sizeof(newLine) == 4
	asm		push	ds
	asm		lds		si, newLine
	asm		les		di, oldLine
#else
	asm		mov		si, newLine
	asm		push	ds
	asm		pop		es
	asm		mov		di, oldLine
#endif
	asm		mov		ax, 0FFFFH		// Preset for "no change" (-1) return.
	asm		mov		cx, width
	asm		cmp		cx, 0
	asm		jle		done

	asm		repe	cmpsb
	asm		je		done			// Return -1 if no change in the line.

	asm		dec		si				// Point back to first non-match
	asm		dec		di				// ""
	asm		mov		bx, si			// Save copy of ptr to first non-match
	asm		add		si, cx			// Point to last character
	asm		add		di, cx			//   "   "   "       "
	asm		mov		cx, width
	asm		std
	asm		repe	cmpsb
	asm		inc		si				// Point to last non-match
	asm		inc		di				//   "   "    "     "
	asm		mov		ah, cl			// Save copy of rightCol in AH
	asm		mov		cx, si
	asm		inc		cx
	asm		sub		cx, bx			// CX has number of different bytes.
	asm		mov		al, ah		
	asm		inc		al
	asm		sub		al, cl			// AL = leftCol = rightCol + 1 - count
	asm		rep		movsb			// Update copy of old line
	asm		cld

done:
#if sizeof(newLine) == 4
	asm		pop		ds
#endif
	return _AX;
}

STRING_UNDO::STRING_UNDO(void)
{
	textAdd = 0;
	operation = MOVE_POSITION;
}

STRING_UNDO::~STRING_UNDO(void)
{
	if (textAdd)
		delete textAdd;
}
 
// Constructor & Destructor -------------------------------------------------

UIW_STRING::UIW_STRING(int left, int top, int width, char *a_text,
	short maxLength, USHORT a_stFlags, USHORT woFlags,
	int (*a_Validate)(void *object, int ccode)) :
	UI_WINDOW_OBJECT(left, top, width, 1, woFlags, WOAF_NO_FLAGS)
{
	windowID[0] = ID_STRING;
	stFlags = a_stFlags;

	Validate = a_Validate;
	originalWidth = width;
	// NOTE: in-field scrolling is not currently supported for right-justified
	// and center-justified strings.  The following line enforces that by
	// disallowing the "maxLength" from being larger than the field width in
	// those cases.
	if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER))
		maxLength = min(maxLength, width);
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = a_text;
	else
	{
		text = new char[maxLength];
		if (a_text)
		{
			int srcLen = strlen(a_text) + 1;
			memcpy(text, a_text, min(srcLen, maxLength));
		}
		else
			text[0] = '\0';
	}
	insertMode = TRUE;
	textTail = text + maxLength - 1;
	*textTail = '\0';						// Insure trailing null.
	screen = 0;
	screenMarkStart = 0;
	screenMarkTail = 0;
}

UIW_STRING::~UIW_STRING(void)
{
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
	if (screen)
		delete screen;
}

// Member functions ---------------------------------------------------------

int UIW_STRING::Event(const UI_EVENT &event)
{
	int viewOnly = FlagSet(woFlags, WOF_VIEW_ONLY) ||
		FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) ? TRUE : FALSE;
	int ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_STRING);
	UI_REGION region;
	UI_WINDOW_OBJECT::Border(0, region, 0);
	USHORT key;
	short cellWidth = display->TextWidth("W");
	short width = (region.right + 1 - region.left) / cellWidth;
	UCHAR reDisplay = TRUE;
	UCHAR displayCheckRegion = FALSE;
	int needValidate = NeedsValidation();

	if (ccode == S_CREATE)
	{
		UndoDestroy(undoHandle);
		undoHandle = 0;
		markedBlock = 0;
		screenTop = text;
		cursor = text;
		if (FlagSet(woFlags, WOF_BORDER))
			fieldWidth = display->isText ? originalWidth - 2 : originalWidth - 1;
		else
			fieldWidth = originalWidth;
		// NOTE: Prevent in-field scrolling in right-justified and center-
		// justified fields by limiting maximum size of string to fit in
		// the field width in those cases.
		if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER) &&
			textTail + 1 - text > fieldWidth)
			textTail = text + fieldWidth - 1;
	}
	textNull = strchr(text, '\0');
	if (textNull < cursor)
		cursor = textNull;

	/* Switch on the event type */
	switch (ccode)
	{
	case E_KEY:
		key = event.rawCode & 0xFF;
		if ( (key == '\b' || key >= ' ') && !viewOnly)
		{
			if (key == '\b')
				BackspaceKey();
			else
				RegularKey(key);
		}
		else
		{
			reDisplay = FALSE;
			ccode = S_UNKNOWN;
		}
		break;

	case L_MOVE_LEFT:
		if (cursor == text)
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		else
		{
			MoveOperation();
			LeftArrow();
			CheckLeftScroll();
		}
		break;

	case L_MOVE_RIGHT:
		if (*cursor)
		{
			MoveOperation();
			cursor++;
			CheckRightScroll();
		}
		else
		{
			ccode = S_UNKNOWN;
			reDisplay = FALSE;
		}
		break;

	case L_MOVE_BOL:
		MoveOperation();
		screenTop = text;
		cursor    = text;
		break;

	case L_MOVE_EOL:
		if (*cursor)
		{
			MoveOperation();
			cursor = textNull;
			if (textNull < text + fieldWidth)
				screenTop = text;
			else
				screenTop = textNull - fieldWidth + 1;
		}
		break;

	case L_WORD_TAB_RIGHT:
		if (*cursor)
		{
			MoveOperation();
			while ( NonWhiteSpace(*cursor) )
				cursor++;
			while (*cursor && WhiteSpace(*cursor) )
				cursor++;
			CheckRightScroll();
		}
		break;

	case L_WORD_TAB_LEFT:
		MoveOperation();
		WordTabLeft();
		CheckLeftScroll();
		break;

	case L_INSERT_TOGGLE:
		insertMode = !insertMode;
		break;

	case L_DELETE:
		if (!viewOnly)
		{
			SetMark();
			if (markStart)
			{
				CutBlock();
			}
			else
				DeleteKey();
		}
		break;

	case L_DELETE_EOL:
		if (!viewOnly)
			DeleteEol();
		break;

	case L_DELETE_WORD:
		if (!viewOnly)
			DeleteWord();
		break;

	case L_MARK:
		AllocateUndo(); 			// Remember mark as separate operation.
		if (markedBlock)
			markedBlock = 0;		// Toggle mark off if its on.
		else
			markedBlock = cursor;	// Or turn it on if its off.
		break;

	case L_MOVE:
		eventManager->DeviceState(event.type, DM_EDIT);
		reDisplay = FALSE;
		break;

	case L_CUT:
		if (!viewOnly)
		{
			SetMark();
			if (markStart)
				CutBlock();
			else
				reDisplay = FALSE;
			break;
		}
		// If view only, treat a CUT as a copy.

	case L_COPY_MARK:
		SetMark();
		if (markStart)
			CopyBlock();
		else
			reDisplay = FALSE;
		break;

	case L_CUT_PASTE:
		SetMark();
		if (markStart)
		{
			if (viewOnly)
				CopyBlock();
			else
				CutBlock();
			break;
		}
		/*** else fall through to paste code. ***/
	case L_PASTE:
		if (!viewOnly)
		{
			if (event.position.line != -1)
			{
				MoveOperation();
				int leading = CalcLeading(width);
				int fieldColumn = (event.position.column - region.left) / cellWidth;
				cursor = screenTop + fieldColumn - leading;
				if (cursor > textNull)
					cursor = textNull;
				else if (cursor < text)
					cursor = text;
				UpdateCursor(region, width);
			}
			PasteBlock();
		}
		break;

	case L_UNDO:
		{
			STRING_UNDO *undo = (STRING_UNDO *) UndoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPush(undoHandle);
			}
		}
		break;

	case L_REDO:
		{
			STRING_UNDO *undo = (STRING_UNDO *) UndoRedoCurrent(undoHandle);
			if (undo)
			{
				UndoOperation(undo);
				UndoRedoPop(undoHandle);
			}
		}
		break;

	case L_END_MARK:
		if (cursor == markedBlock)
			markedBlock = 0;			// Eliminate zero-length mark.
		reDisplay = FALSE;
		break;

	case L_CONTINUE_MARK:
		if (markedBlock)
		{
			MoveOperation();
			int leading = CalcLeading(width);
			if (event.position.column > region.left)
				cursor = screenTop + (event.position.column - region.left) / cellWidth - leading;
			else
				cursor = screenTop;
			if (cursor > textNull)
				cursor = textNull;
			else if (cursor < text)
				cursor = text;
			UpdateCursor(region, width);
			break;
		}
		/*** Else fall through to code that begins the mark. */
	case L_BEGIN_MARK:
		{
			AllocateUndo(); 			// Remember mark as separate operation.
			int leading = CalcLeading(width);
			int fieldColumn = (event.position.column - region.left) / cellWidth;
			cursor = screenTop + fieldColumn - leading;
			if (cursor > textNull)
				cursor = textNull;
			else if (cursor < text)
				cursor = text;
			if (fieldColumn == 0)
				CheckLeftScroll();
			else if (fieldColumn >= fieldWidth - 1 && cursor < textNull)
			{
				cursor++;
				CheckRightScroll();
			}
			UpdateCursor(region, width);
			markedBlock = cursor;			// Throw away possible old mark.
		}
		break;

	case S_ERROR_RESPONSE:
		woStatus |= event.rawCode;
		break;

	case S_CURRENT:
		if (needValidate)
			(*Validate)(this, ccode);
		// Clear the WOS_UNANSWERED and WOS_NO_AUTO_CLEAR bits
		woStatus &= ~(WOS_UNANSWERED | WOS_NO_AUTO_CLEAR);
		if (!undoHandle)
			undoHandle = UndoCreate();
		// Fall through.
	case S_DISPLAY_ACTIVE:
		displayCheckRegion = TRUE;
		break;

	case S_NON_CURRENT:
		if (needValidate && (*Validate)(this, ccode) != 0 )
		{
			ccode = S_ERROR;
			woStatus |= WOS_INVALID;
			break;
		}
		woStatus &= ~WOS_INVALID;			// Clear invalid status bit.
		// else fall through.
	case S_DISPLAY_INACTIVE:
		markedBlock = 0;					// Throw away marked block.
		displayCheckRegion = TRUE;
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		reDisplay = FALSE;
		break;
	}
	if (displayCheckRegion)
	{
		if ( !UI_WINDOW_OBJECT::NeedsUpdate(event, ccode) )
		{
			reDisplay = FALSE;
			if (ccode == S_CURRENT)
				UpdateCursor(region, width);
		}
		else
		{
			screenInvalid = TRUE;
			UI_WINDOW_OBJECT::Border(ccode, region, lastPalette);
		}
	}
	if (reDisplay)
	{
		SetMark();
		if (cursor - screenTop >= fieldWidth)
			screenTop = cursor - fieldWidth + 1;
		else if (cursor < screenTop)
			screenTop = cursor;
		Redisplay(ccode);
	}
	return (ccode);
}

void UIW_STRING::DataSet(char *newText, short maxLength)
{
	if (newText)
	{
		short oldMaxLength = (short)(textTail + 1 - text);
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			text = newText;
		else
		{
			if (maxLength != -1 && maxLength > oldMaxLength)
			{
				delete text;
				text = new char[maxLength];
			}
			int srcLen = strlen(newText) + 1;
			int realMaxLength = (maxLength == -1) ? oldMaxLength : maxLength;
			memcpy(text, newText, min(srcLen, realMaxLength));
		}
		if (maxLength != -1)
			textTail = text + maxLength - 1;
		else
			textTail = text + oldMaxLength - 1;
		*textTail = '\0';						// Insure trailing null.
		screenTop = text;
		cursor = text;
		markedBlock = 0;
	}
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

STRING_UNDO * UIW_STRING::AllocateUndo(void)
{
	STRING_UNDO *undo = new STRING_UNDO;
	StateInfoSave(&undo->info);
	UndoAdd(undoHandle, undo, sizeof(STRING_UNDO));
	return undo;
}

void UIW_STRING::BackspaceKey(void)
{
	STRING_UNDO *undo;

	if (cursor > text)
	{
		undo = AllocateUndo();
		LeftArrow();
		DeleteChar(undo);
		CheckLeftScroll();
	}
}

int UIW_STRING::CalcLeading(int width)
{
	int textLen = strlen(screenTop);
	int leading = 0;
	if (textLen < fieldWidth - 1)
	{
		if (woFlags & WOF_JUSTIFY_CENTER)
			leading = min((fieldWidth - 1 - textLen) / 2, width);
		else if (woFlags & WOF_JUSTIFY_RIGHT)
			leading = min(fieldWidth - 1 - textLen, width);
	}
	return leading;
}

void UIW_STRING::CheckLeftScroll(void)
{
	if (cursor <= screenTop && screenTop > text)
		screenTop = max(text, cursor - fieldWidth / 3);
}

void UIW_STRING::CheckRightScroll(void)
{
	if (cursor - screenTop >= fieldWidth)
	{
		char *maxScreenTop = textTail + 1 - fieldWidth;
		screenTop = min(maxScreenTop, cursor + 1 - fieldWidth + fieldWidth / 3);
	}
}

STRING_UNDO *UIW_STRING::CopyBlock(void)
{
	STRING_UNDO *undo;

	if (screenTop >= markStart && screenTop > text)
		screenTop = markStart;
	pasteLength = (USHORT) (markTail - markStart);
	if (pasteBuffer)
		delete pasteBuffer;
	pasteBuffer = new char[pasteLength];
	memcpy(pasteBuffer, markStart, pasteLength);
	undo = AllocateUndo();
	markedBlock = 0;
	return undo;
}

void UIW_STRING::CutBlock(void)
{

	STRING_UNDO *undo;
	undo = CopyBlock();
	DeleteBlock(undo, markStart, pasteLength);
}

void UIW_STRING::DeleteBlock(STRING_UNDO *undo, char *block, USHORT length)
{
	if (!length)
		return;
	woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
	if (undo)
	{
		undo->operation = ADD_TEXT;				/* Insert the block. */
		undo->textAdd = new char[length];
		if (undo->textAdd)
			memcpy(undo->textAdd, block, length);
		undo->textLength = length;
		undo->buff = block;
	}
	memmove(block, block + length, (size_t)(textNull + 1 - (block + length)));
	textNull -= length;
	cursor = block;
}

void UIW_STRING::DeleteChar(STRING_UNDO *undo)
{
	DeleteBlock(undo, cursor, 1);
}

void UIW_STRING::DeleteEol(void)
{
	char *rover;
	USHORT count;
	STRING_UNDO *undo;

	rover = strchr(cursor, '\0');
	count = (USHORT) (rover - cursor);
	if (count)
	{
		undo = AllocateUndo();
		DeleteBlock(undo, cursor, count);
	}
}

void UIW_STRING::DeleteKey(void)
{
	STRING_UNDO *undo;

	if (*cursor)
	{
		undo = AllocateUndo();
		DeleteChar(undo);
	}
}

void UIW_STRING::DeleteWord(void)
{
	STRING_UNDO *undo;
	char *ptr;

	if (*cursor)
	{
		undo = AllocateUndo();
		if (*cursor == ' ')
		{
			if (cursor == text || WhiteSpace(*(cursor - 1) ) )
			{
				ptr = cursor;
				while (*ptr == ' ')
					ptr++;
				DeleteBlock(undo, cursor, (USHORT) (ptr - cursor) );
				return;					/* NOTE: second return point here. */
			}
			WordTabLeft();
		}
		else
		{
			while ( cursor > text && NonWhiteSpace(*(cursor - 1) ) )
				LeftArrow();
		}
		ptr = cursor;
		while ( NonWhiteSpace(*ptr) )
			ptr++;
		while (*ptr == ' ')
			ptr++;
		DeleteBlock(undo, cursor, (USHORT) (ptr - cursor) );
	}
}

void UIW_STRING::InsertBlock(STRING_UNDO *undo, char *insert_point,
	char *block, USHORT length)
{
	if (length)
	{
		woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
		if (undo)
		{
			undo->operation = DELETE_TEXT;				// Delete the block.
			undo->textLength = length;
			undo->buff = insert_point;
		}
		memmove(insert_point + length, insert_point,
			(size_t) (textNull + 1 - insert_point) );
		memcpy(insert_point, block, length);
		textNull += length;
		if (insert_point == cursor)
			cursor += length;
	}
}

void UIW_STRING::LeftArrow(void)
{
	if (cursor > text)
	{
		cursor--;
		if (cursor < screenTop)
			screenTop--;
	}
}

void UIW_STRING::MoveOperation(void)
{
	STRING_UNDO *undo = (STRING_UNDO *) UndoCurrent(undoHandle);

	woStatus |= WOS_NO_AUTO_CLEAR;
	if (!undo || undo->operation != MOVE_POSITION)
		AllocateUndo();					// Allocate positional undo object.
}

int UIW_STRING::NonWhiteSpace(int value)
{
	if (value && !WhiteSpace(value))
		return TRUE;
	else
		return FALSE;
}

void UIW_STRING::PasteBlock(void)
{
	STRING_UNDO *undo;

	if (pasteBuffer && !markedBlock &&
		pasteLength <= textTail - textNull)
	{
		undo = AllocateUndo();
		InsertBlock(undo, cursor, pasteBuffer, pasteLength);
	}
}

void UIW_STRING::Redisplay(int ccode)
{
	UI_REGION region;
	UI_WINDOW_OBJECT::Border(0, region, 0);
	short cellWidth = display->TextWidth("W");
	short cellHeight = display->TextHeight("W");
	short width = (region.right + 1 - region.left) / cellWidth;
	char line[133];
	char xor[133];
	if (FlagSet(woStatus, WOS_UNANSWERED))
	{
		text[0] = '\0';
		cursor = text;
		screenTop = text;
		textNull = text;
		markedBlock = 0;
	}
	int newScreenSize = width;
	if (newScreenSize > screenSize)
	{
		if (screen)
			delete screen;
		screen = 0;
	}
	if (!screen)
	{
		screen = new char[newScreenSize];
		screenSize = newScreenSize;
		screenInvalid = TRUE;
	}
	if (screenInvalid)
		memset(screen, '\1', screenSize);
	char *newMarkStart = 0;
	char *newMarkTail = 0;
	char *ptr = screenTop;
	int markLine = FALSE;
	int markLeft = width - 1;
	int textLen = strlen(screenTop);
	int leading = CalcLeading(width);
	textLen = min(textLen, width - leading);
	memset(line, ' ', leading);
	int col = leading;
	int markRight = col;
	if (width <= 0)
	{
		if (!display->isText)
		{
			display->Fill(screenID,
				region.left, region.top,
				region.right, region.bottom, lastPalette);
		}
	}
	else
	{
		while (*ptr && col < width)
		{
			if (markedBlock)
			{
				if (ptr >= markStart && ptr < markTail)
				{
					if (ptr == markStart)
						newMarkStart = screen + col;
					markLine = TRUE;
					if (col < markLeft)
						markLeft = col;
					if (col > markRight)
						markRight = col;
				}
				else if (ptr == markTail)
					newMarkTail = screen + col;
			}
			line[col++] = *ptr++;
		}
		memset(&line[col], '\0', width - col);
		if (ptr == markStart && !newMarkStart)
			newMarkStart = screen + col;
		else if (ptr == markTail && !newMarkTail)
			newMarkTail  = screen + col;
		memset(xor,  '\0', width);
		if (screenMarkStart && screenMarkStart < screen + width &&
			screenMarkTail > screen)
		{
			int lineStart = 0;
			if (screenMarkStart > screen)
				lineStart = (int) (screenMarkStart - screen);
			int lineTail = min(col, width);
			if (screenMarkTail < screen + width)
				lineTail = (int) (screenMarkTail - screen);
			memset(&xor[lineStart], '\1', lineTail - lineStart);
		}
		int vertOffset = VerticalCenterOffset(&region);
		if (region.bottom >= region.top + cellHeight - 1)
		{
			_DX = LineChanged(line, screen, width);
			if (_DX != 0xFFFF)
			{
				int leftCol  = _DL;
				int rightCol = _DH;
				// Re-displaying will wipe out mark.
				memset(&xor[leftCol], '\0', rightCol + 1 - leftCol);
				if ( !display->isText )		// Pre-fill for graphics.
				{
					display->Fill(screenID,
						region.left + leftCol * cellWidth, region.top,
						rightCol == width - 1 ? region.right :
						region.left + (rightCol + 1) * cellWidth - 1,
						region.bottom, lastPalette);
				}
				display->Text(screenID,
					region.left + leftCol * cellWidth,
					region.top + vertOffset, &line[leftCol],
					lastPalette, rightCol + 1 - leftCol, FALSE);
			}
			if (markLine)
			{
				for (int i = markLeft; i <= markRight; i++)
					xor[i] ^= 1;
			}
			char *xorStart = xor;
			while ( (xorStart = (char *) memchr(xorStart, '\1', (int) (xor + width - xorStart))) != 0 )
			{
				int leftXor = (int) (xorStart - xor);
				int rightXor = leftXor;
				while (rightXor < width - 1 && xor[rightXor + 1])
					rightXor++;
				display->TextXOR(region.left + leftXor * cellWidth,
					region.top + vertOffset,
					rightXor + 1 - leftXor);
				xorStart = xor + rightXor + 1;
			}
			if (!newMarkStart && ptr == markStart)
				newMarkStart = screen + col;
			else if (!newMarkTail && ptr == markTail)
				newMarkTail  = screen + col;
		}
		else if (screenInvalid)
		{
			// We are in graphics mode with not enough height to display text.
			display->Fill(screenID,	region.left, region.top,
				region.right, region.bottom, lastPalette);
		}
		screenInvalid = FALSE;
		screenMarkStart = newMarkStart;
		screenMarkTail  = newMarkTail;
		if (ccode != S_DISPLAY_INACTIVE)
			UpdateCursor(region, width);
	}
}

void UIW_STRING::RegularKey(USHORT key)
{
	STRING_UNDO *undo;

	if (FlagSet(woFlags, WOF_AUTO_CLEAR) &&
		!FlagSet(woStatus, WOS_NO_AUTO_CLEAR) && cursor == text && *text)
	{
		undo = AllocateUndo();
		DeleteBlock(undo, text, strlen(text));
		textNull = text;
	}
	if (insertMode || cursor == textNull)
	{
		if (textNull < textTail)
		{
			undo = (STRING_UNDO *) UndoCurrent(undoHandle);
			if ( undo && (undo->operation == DELETE_TEXT) &&
				 (undo->buff + undo->textLength == cursor) &&
				 ( (isalnum(*(cursor - 1)) ? 1 : 0) ==
				   (isalnum(key) ? 1 : 0) ) )
			{
				undo->textLength++;
				undo = 0;		// Set 0 for insert_block() call.
			}
			else
			{
				undo = AllocateUndo();
			}
			InsertBlock(undo, cursor, (char *) &key, 1);
		}
	}
	else if (cursor < textTail)
	{
		undo = AllocateUndo();
		ReplaceChar(undo, cursor, key);
	}
	CheckRightScroll();
}

void UIW_STRING::ReplaceChar(STRING_UNDO *undo, char *buff, USHORT key)
{
	woStatus |= (WOS_CHANGED | WOS_NO_AUTO_CLEAR);
	if (undo)
	{
		undo->operation = REPLACE_CHAR;
		undo->buff = buff;
		undo->textLength = *buff;
	}
	*buff = key;
	if (buff == cursor)
		cursor++;
}

void UIW_STRING::SetMark(void)
{
	markStart = markTail = 0;
	if (markedBlock)
	{
		if (markedBlock > cursor)
		{
			markStart = cursor;
			markTail  = markedBlock;
		}
		else if (markedBlock < cursor)
		{
			markStart = markedBlock;
			markTail  = cursor;
		}
	}
}

int UIW_STRING::StateInfoChanged(STRING_STATE *state)
{
	if (screenTop == state->screenTop &&
		cursor == state->cursor &&
		markedBlock == state->markedBlock)
		return FALSE;
	else
		return TRUE;
}

void UIW_STRING::StateInfoRestore(STRING_STATE *state)
{
	if (StateInfoChanged(state))
	{
		screenTop = state->screenTop;
		cursor = state->cursor;
		markedBlock = state->markedBlock;
	}
}

void UIW_STRING::StateInfoSave(STRING_STATE *state)
{
	state->screenTop = screenTop;
	state->cursor = cursor;
	state->markedBlock = markedBlock;
}

void UIW_STRING::UndoOperation(STRING_UNDO *undo)
{
	STRING_STATE stateInfo;

	StateInfoSave(&stateInfo);
	if (undo->operation == ADD_TEXT)
	{
		if (undo->textAdd)
		{
			InsertBlock(undo, undo->buff, undo->textAdd, undo->textLength);
			delete undo->textAdd;
			undo->textAdd = 0;
		}
	}
	else if (undo->operation == DELETE_TEXT)
		DeleteBlock(undo, undo->buff, undo->textLength);
	else if (undo->operation == REPLACE_CHAR)
		ReplaceChar(undo, undo->buff, (UCHAR) undo->textLength);
	StateInfoRestore(&undo->info);
	undo->info = stateInfo;
}

void UIW_STRING::UpdateCursor(UI_REGION &region, int width)
{
	int cursorColumn = (int)(cursor - screenTop);
	int leading = CalcLeading(width);
	int trueColumn = region.left + (leading + cursorColumn) * display->TextWidth("W");
	if ( FlagSet(woStatus, WOS_CURRENT) && !FlagSet(woFlags, WOF_VIEW_ONLY) &&
		 !FlagSet(woAdvancedStatus, WOAS_TOO_SMALL) &&
		trueColumn >= 0 && 
		trueColumn < display->columns && region.top >= 0 && 
		region.top  < display->lines && leading + cursorColumn < width)
	{
		eventManager->DevicePosition(E_CURSOR, trueColumn, region.top + VerticalCenterOffset(&region) );
		eventManager->DeviceState(E_CURSOR, (insertMode ? DC_INSERT : DC_OVERTYPE));
	}
}

int UIW_STRING::VerticalCenterOffset(UI_REGION *region)
{
	if (display->isText)			// Center text vertically (if graphics).
		return 0;
	else
		return (region->bottom + 2 - region->top - display->TextHeight("W")) / 2;
}

int UIW_STRING::WhiteSpace(int value)
{
	if (value == ' ')
		return TRUE;
	else
		return FALSE;
}

void UIW_STRING::WordTabLeft(void)
{
	while ( cursor > text && WhiteSpace( *(cursor - 1) ) )
		cursor--;
	while ( cursor > text && NonWhiteSpace( *(cursor - 1) ) )
		cursor--;
	if (cursor < screenTop)
		screenTop = cursor;
}

