//	Program name..	Zinc Interface Library
//	Filename......	TIMEWIN.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"

extern int ParseRange(char *buff, int offset, char *min, char *max);

// ----- Constructor & Destructor -------------------------------------------

UIW_TIME::UIW_TIME(int left, int top, int width, UI_TIME *a_timePtr,
	char *a_range, USHORT a_tmFlags, USHORT a_woFlags,
	int (*validate)(void *object, int ccode)) :
	UIW_STRING(left, top, width, 0, 64, STF_NO_FLAGS,
		a_woFlags & ~WOF_NO_ALLOCATE_DATA, validate)
{
	if (a_timePtr)
		time = *a_timePtr;
	windowID[0] = ID_STRING;
	tmFlags = a_tmFlags;
	timePtr = (FlagSet(a_woFlags, WOF_NO_ALLOCATE_DATA)) ? a_timePtr : 0;
	range = (a_range) ? ui_strdup(a_range) : 0;
}

UIW_TIME::~UIW_TIME(void)
{
	if (range)
		delete range;
}

// ----- Member functions ---------------------------------------------------

const UI_TIME *UIW_TIME::DataGet(void)
{
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		return (timePtr);
	return (&time);
}

void UIW_TIME::DataSet(UI_TIME *newTimePtr)
{
	if (newTimePtr)
	{
		time = *newTimePtr;
		if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
			timePtr = newTimePtr;
	}
	time.Export(text, tmFlags);
	UI_WINDOW_OBJECT::Redisplay(FALSE);
}

int UIW_TIME::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	int needValidate = NeedsValidation();
	int ccode = event.type;
	if (ccode == S_CREATE)
	{
		if (FlagSet(woStatus, WOS_UNANSWERED))
		{
			time.Import(0, 0, 0, 0);
			*text = '\0';
		}
		else
		{
			if (timePtr)
				time = *timePtr;
			time.Export(text, tmFlags);
		}
	}
	if (ccode == S_CURRENT)
	{
		if (needValidate)
			(*Validate)(this, ccode);
		woStatus &= ~WOS_UNANSWERED;
	}
	else if (ccode == S_NON_CURRENT)
	{
		TMI_RESULT result = time.Import(text, tmFlags);
		if (result == TMI_OK)
			result = TimeRangeCheck();
		if (result != TMI_OK)
		{
			TimeError(result);
			ccode = S_ERROR;
		}
		else
		{
			if (needValidate && (*Validate)(this, ccode) != 0)
				ccode = S_ERROR;
			else
			{
				if (*text == '\0' && !FlagSet(tmFlags, TMF_SYSTEM))
					woStatus |= WOS_UNANSWERED;		// Leave unanswered.
				if (timePtr)
					*timePtr = time;
				time.Export(text, tmFlags);
				if (FlagSet(woStatus, WOS_UNANSWERED))
					*text = '\0';
			}
		}
	}
	if (ccode != S_ERROR)
	{
		woStatus &= ~WOS_INVALID;
		ccode = UIW_STRING::Event(event);
	}
	else
		woStatus |= WOS_INVALID;
	return(ccode);
}

void UIW_TIME::TimeError(TMI_RESULT errorCode)
{
	static struct
	{
		TMI_RESULT code;
		char *msg;
	} errorTable[] =
	{
		{ TMI_INVALID,			"The time %s, is in an invalid format."		},
		{ TMI_VALUE_MISSING,	"A time value must be entered."				},
		{ TMI_OK,				0											}
	};
	char formattedString[512];

	if (errorCode == TMI_RANGE)
		TimeRangeError(formattedString);
	else
	{
		strcpy(formattedString, "An unknown time error was received.");
		for (int i = 0; errorTable[i].msg; i++)
			if (errorTable[i].code == errorCode)
			{
				char timeString[64];
				time.Export(timeString, tmFlags);
				sprintf(formattedString, errorTable[i].msg, timeString);
				break;
			}
	}
	_errorSystem->ReportError(windowManager,
		woFlags & (WOF_NO_UNANSWERED | WOF_NO_INVALID), formattedString);
}

TMI_RESULT UIW_TIME::TimeRangeCheck(void)
{
	char minValue[20];
	char maxValue[20];
	TMI_RESULT errorCode = TMI_OK;

	/* See if a range exists */
	if (!range || range[0] == '\0')
		return (TMI_OK);

	int offset = 0;
	int rangeLength = (range) ? strlen(range) : 0;
	while (offset < rangeLength)
	{
		offset = ParseRange(range, offset, minValue, maxValue);
		UI_TIME minTime;
		UI_TIME maxTime;
		TMI_RESULT minValid = minTime.Import(minValue, tmFlags);
		TMI_RESULT maxValid = maxTime.Import(maxValue, tmFlags);
		if (minValid != TMI_OK || maxValid != TMI_OK ||
			time < minTime || time > maxTime)
			errorCode = TMI_RANGE;
		else
		{
			errorCode = TMI_OK;
			break;
		}
	}
	return (errorCode);
}

void UIW_TIME::TimeRangeError(char *formattedString)
{
	char formattedMin[40];
	char formattedMax[40];
	char tempString[80];
	static char throughString[] = "through";
	static char orString[] = "or";
	int position;
	int offset = 0;
	int count = 0;
	char badText[64];

	/* Set up the initial variables */
	int rangeLength = (range) ? strlen(range) : 0;
	char separator = ',';

	time.Export(badText, tmFlags);
	sprintf(formattedString,
		"%s is not valid.  The time must be in the range ", badText);
	while (offset < rangeLength)
	{
		/* Format the current minimum and maximum values */
		if (count == 1)
		{
			if (!strcmp(formattedMin, formattedMax))
				strcpy(tempString, formattedMin);
			else
				sprintf(tempString, "%s %s %s", formattedMin, throughString,
				formattedMax);
			strcat(formattedString, tempString);
		}
		else if (count != 0)
		{
			if (!strcmp(formattedMin, formattedMax))
				sprintf(tempString, "%c %s", separator, formattedMin);
			else
				sprintf(tempString, "%c %s %s %s", separator,
					formattedMin, throughString, formattedMax);
			strcat(formattedString, tempString);
		}

		/* Get a new set of minimum and maximum values */
		count++;
		offset = ParseRange(range, offset, formattedMin, formattedMax);
	}

	/* Append the final minimum and maximum values on the string */
	if (count > 2 && !strcmp(formattedMin, formattedMax))
		sprintf(tempString, "%c %s %s.", separator, orString, formattedMin);
	else if (count > 2)
		sprintf(tempString, "%c %s %s %s %s.", separator, orString,
			formattedMin, throughString, formattedMax);
	else if (count == 2 && !strcmp(formattedMin, formattedMax))
		sprintf(tempString, " %s %s.", orString, formattedMin);
	else if (count == 2)
		sprintf(tempString, " %s %s %s %s.", orString, formattedMin,
			throughString, formattedMax);
	else if (count > 0)
		sprintf(tempString, "%s %s %s.", formattedMin, throughString,
			formattedMax);
	else
	{
		position = 0;		
		while (formattedString[position] != '\0' &&
			formattedString[position] != '.')
			position++;
		formattedString[++position] = '\0';
	}
	strcat(formattedString, tempString);
}

