//	Program name..	Zinc Interface Library
//	Filename......	GENHELP.CPP
//	Version.......	1.0
//	
//	COPYRIGHT (C) 1990.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <ctype.h>
#include "ui_win.hpp"

char text_file[65];
char h_file[65];
char hlp_file[65];

static char valueMap[MAX_HELP_CONTEXTS];

class HELP_CONTEXT
{
public:
	char *name;							// Name of the help context.
	char *title;						// Title of the help context.
	int value;							// Number associated with the name.
	TEXT_INDEX index;
	HELP_CONTEXT *next;
	static int textIndex;
	static int maxValue;
	static int maxNameLength;
	static HELP_CONTEXT *head;
	static HELP_CONTEXT *tail;
	static int count;

	HELP_CONTEXT(char *helpName, char *helpTitle, int helpValue);
};

int HELP_CONTEXT::count    = 0;
HELP_CONTEXT *HELP_CONTEXT::head = 0;
HELP_CONTEXT *HELP_CONTEXT::tail = 0;
int HELP_CONTEXT::maxValue = 0;
int HELP_CONTEXT::maxNameLength = 0;
int HELP_CONTEXT::textIndex = 0;

void strrepc(
	char *str,				// Pointer to target string.
	int  c,					// Character in the string to replace.
	int  repc)				// Replacement character.
{
    while (*str)
	{
		if (*str == c)
			*str = repc;
		str++;
    }
}

HELP_CONTEXT::HELP_CONTEXT(
	char *helpName,
	char *helpTitle,
	int helpValue)
{
	if (count >= MAX_HELP_CONTEXTS)
	{
		printf("Too many help contexts.  Maximum = %d.\n", MAX_HELP_CONTEXTS);
		exit(5);
	}
	count++;
	int nameLength = strlen(helpName);
	name = new char [nameLength + 1];
	if (nameLength > maxNameLength)
		maxNameLength = nameLength;
	strcpy(name, helpName);
	strupr(name);
	strrepc(name, ' ', '_');
	int titleLength = strlen(helpTitle);
	title = new char[titleLength + 1];
	strcpy(title, helpTitle);
	if (helpValue < 0 || helpValue > MAX_HELP_CONTEXTS)
	{
		printf("Bad help context, value = %d.\n", helpValue);
		exit(3);
	}
	value = helpValue;
	if (helpValue > 0 && valueMap[helpValue - 1])
	{
		printf("Multiple help contexts have the same numeric value: %d.\n",
			helpValue);
		exit(4);
	}
	valueMap[helpValue - 1] = 1;
	index.offset = 0L;
	index.length = 0;
	index.titleIndex = textIndex;
	textIndex += titleLength + 1;
	next     = 0;
	if (!head)
		head = tail = this;
	else
	{
		for (HELP_CONTEXT *help = head; help; help = help->next)
		{
			if ( strcmp(name, help->name) == 0 )
			{
				printf("Multiple help contexts have the same variable name: %s.\n",
					name);
				exit(5);
			}
			if ( strcmp(title, help->title) == 0 )
			{
				printf("Multiple help contexts have the same title: %s.\n",
					title);
				exit(6);
			}
		}
		tail->next = this;
		tail = this;
	}
}

void GenerateIncludeFile(void)
{
	FILE *h_fp = fopen(h_file, "w+");
	if (!h_fp)
	{
		printf("Could not open file %s for write access.\n", h_file);
		exit(7);
	}
	fprintf(h_fp, "// This file was created by the genhelp utility.\n");
	fprintf(h_fp, "// PLEASE DO NOT MODIFY WITH AN EDITOR!.\n\n");
	HELP_CONTEXT *rover = HELP_CONTEXT::head;
	while (rover)
	{
		int maxWidth;
		if (HELP_CONTEXT::maxValue > 9999)
			maxWidth = 5;
		else if (HELP_CONTEXT::maxValue > 999)
			maxWidth = 4;
		else if (HELP_CONTEXT::maxValue > 99)
			maxWidth = 3;
		else if (HELP_CONTEXT::maxValue > 9)
			maxWidth = 2;
		else
			maxWidth = 1;
		if (rover->index.offset)
			fprintf(h_fp, "const int %-*s = %*d; // %s\n",
				HELP_CONTEXT::maxNameLength, rover->name,
				maxWidth, rover->value, rover->title);
		rover = rover->next;
	}
	fclose(h_fp);
}

void RegenerateTextFile(void)
{
	static char temp_file[] = "genhelp.$$$";
	FILE *text_fp = fopen(text_file, "rb");
	if (!text_fp)
	{
		printf("Could not open text file: %s.\n", text_file);
		exit(8);
	}
	FILE *temp_fp = fopen(temp_file, "w+b");
	if (!temp_fp)
	{
		printf("Could not create temporary file.\n");
		exit(8);
	}
	char line[255];
	HELP_CONTEXT *help = HELP_CONTEXT::head;
	while ( fgets(line, sizeof(line), text_fp) )
	{
		char helpName[128];
		int  helpValue;
		int numScanned = sscanf(line, "--- %[a-zA-Z0-9_] %d", helpName, &helpValue);
		if (numScanned >= 1)
		{
			if (numScanned == 1) helpValue = 0;
			if ( !help || strcmp(helpName, help->name) != 0 ||
				 (helpValue && helpValue != help->value) )
			{
				printf("Lost synchronization re-creating text file.\n");
				exit(10);
			}
			fprintf(temp_fp, "--- %s %d ---\r\n", helpName, help->value);
			help = help->next;
		}
		else
			fwrite(line, strlen(line), 1, temp_fp);
	}
	fclose(text_fp);
	fclose(temp_fp);
	unlink(text_file);
	rename(temp_file, text_file);
}

void GenerateHelpFiles(void)
{
	FILE *text_fp = fopen(text_file, "rb");
	if (!text_fp)
	{
		printf("Could not open text file: %s.\n", text_file);
		exit(8);
	}
	FILE *help_fp = fopen(hlp_file, "wb+");
	if (!help_fp)
	{
		printf("Could not create help file: %s.\n", hlp_file);
		exit(8);
	}
	HELP_HEADER header = { HELP_MAGIC_WORD, HELP_VERSION, 0, 0 };
	fwrite(&header, sizeof(header), 1, help_fp);
	int foundHelp = FALSE;
	char line[255];
	HELP_CONTEXT *lastHelp = 0;
	int sawLoneCr = FALSE;
	while ( fgets(line, sizeof(line), text_fp) )
	{
		char helpName[128];
		int  helpValue;
		int numScanned = sscanf(line, "--- %[a-zA-Z0-9_] %d", helpName, &helpValue);
		if (numScanned >= 1)
		{
			if (numScanned == 1) helpValue = 0;
			foundHelp = TRUE;
			if (lastHelp)
				lastHelp->index.length = (int)
					(ftell(help_fp) - lastHelp->index.offset);
			if ( !fgets(line, sizeof(line), text_fp) ) line[0] = '\0';
			strrepc(line, '\r', '\0');
			strrepc(line, '\n', '\0');
			HELP_CONTEXT *help = new HELP_CONTEXT(helpName, line, helpValue);
			help->index.offset = ftell(help_fp);
			lastHelp = help;
		}
		else if (foundHelp)
		{
			char *lineTail = strchr(line, '\0');
			while (lineTail > line &&
			  	(*(lineTail - 1) == '\r' || *(lineTail - 1) == '\n') )
				lineTail--;
			*lineTail = '\0';
			if (line[0] == '\0')
			{
				fwrite("\r\n\r\n", sawLoneCr ? 2 : 4, 1, help_fp);
				sawLoneCr = TRUE;
			}
			else
			{
				sawLoneCr = FALSE;
				if ( *(lineTail - 1) == '\\' )
				{
					lineTail--;			  			// Point at the slash.
					while (lineTail > line && isspace(*(lineTail - 1)) )
						lineTail--;		  			// Strip trailing white.
					strcpy(lineTail, "\r\n");
				}
				else if ( *(lineTail - 1) == '\' )
					strcpy(lineTail, "\r\n");
				else
					strcpy(lineTail, " ");
				fwrite(line, strlen(line), 1, help_fp);
			}
		}
	}
	if (lastHelp)
		lastHelp->index.length = (int) (ftell(help_fp) - lastHelp->index.offset);
	HELP_CONTEXT *hc = HELP_CONTEXT::head;
	while (hc)
	{
		if (!hc->value)
		{
			hc->value = (int) (strchr(valueMap, '\0') - valueMap) + 1;
			valueMap[hc->value - 1] = 1;
		}
		if (hc->value > HELP_CONTEXT::maxValue)
			HELP_CONTEXT::maxValue = hc->value;
		hc = hc->next;
	}
	header.numContexts = HELP_CONTEXT::maxValue;
	header.indexPtr    = ftell(help_fp);
	TEXT_INDEX *tndx = new TEXT_INDEX[header.numContexts];
	hc = HELP_CONTEXT::head;
	while (hc)
	{
		if (hc->index.offset)
		{
			tndx[hc->value - 1] = hc->index;
		}
		hc = hc->next;
	}
	fwrite(tndx, sizeof(TEXT_INDEX), header.numContexts, help_fp);
	hc = HELP_CONTEXT::head;
	while (hc)
	{
		if (hc->index.offset)
		{
			fwrite(hc->title, strlen(hc->title) + 1, 1, help_fp);
		}
		hc = hc->next;
	}
	rewind(help_fp);
	fwrite(&header, sizeof(header), 1, help_fp);
	fclose(text_fp);
	fclose(help_fp);
	RegenerateTextFile();
	GenerateIncludeFile();
}

void FixExtension(
	char *str,
	char *ext)
{
	char *tail = strchr(str, '.');
	if (!tail) tail = strchr(str, '\0');
	strcpy(tail, ext);
}

void cdecl FreeStoreException(void)
{
	printf("Out of memory.\n");
	abort();
}

void main(
	int argc,
	char *argv[])
{
	extern void (*_new_handler)();

	_new_handler = FreeStoreException;
	if (argc != 2)
	{
		printf("Usage: genhelp <text file name>\n");
		exit(1);
	}
	strcpy(text_file, argv[1]);
	if ( !strchr(text_file, '.') )
		FixExtension(text_file, ".txt");
	strcpy(h_file, argv[1]);
	FixExtension(h_file, ".hlh");
	strcpy(hlp_file, argv[1]);
	FixExtension(hlp_file, ".hlp");
	GenerateHelpFiles();
}

