//	CALC.CPP (CALC) - Sample calculator
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include <ui_win.hpp>

// Definition of the calculator class.
class CALCULATOR : public UIW_WINDOW
{
public:
	CALCULATOR(int left, int top, char *title);

private:
	UIW_REAL *numberField;
	UIW_STRING *memField;
	double operand1;
	double operand2;
	int operatorLast;
	UCHAR operation;
	double decimal;
	double memory;

	static EVENT_TYPE ButtonFunction(UI_WINDOW_OBJECT *button, UI_EVENT &event, EVENT_TYPE ccode);
	void Display(UIW_BUTTON *button);
};

CALCULATOR::CALCULATOR(int left, int top, char *title) :
	UIW_WINDOW(left, top, 25, 11, WOF_NO_FLAGS, WOAF_NO_SIZE)
{
	UIW_BUTTON *mpButton;
	
	// Read in the icon from storage
	UI_STORAGE storage("CALCICO.DAT", UIS_READ);
	UI_WINDOW_OBJECT::defaultStorage = &storage;

	// Initialize the calculator values.
	operand1 = 0.0;
	operand2 = 0.0;
	operatorLast = TRUE;
	operation = '=';
	decimal = 0.0;
	memory = 0.0;


	// Create the memory and number display fields.
	memField = new UIW_STRING(1, 1, 3, "", 4, STF_NO_FLAGS,
		WOF_JUSTIFY_CENTER | WOF_NON_SELECTABLE);
	
	numberField = new UIW_REAL(4, 1, 17, &operand2, NULL, NMF_NO_FLAGS,
		WOF_BORDER | WOF_JUSTIFY_RIGHT | WOF_NON_SELECTABLE | WOF_NO_ALLOCATE_DATA);
	

   	// Add the buttons and other objects.
	*this
		+ new UIW_BORDER
		+ new UIW_MINIMIZE_BUTTON

		+ &(*new UIW_SYSTEM_BUTTON
			+ new UIW_POP_UP_ITEM("~Restore", MNIF_RESTORE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("~Move", MNIF_MOVE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("Mi~nimize", MNIF_MINIMIZE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM
			+ new UIW_POP_UP_ITEM("~Close", MNIF_CLOSE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0))
		+ new UIW_TITLE(title, WOF_JUSTIFY_CENTER)
		+ memField
		+ numberField
		+ new UIW_BUTTON(2, 3, 4, "~C", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
	  	+ new UIW_BUTTON(6, 3, 4, "C~E", BTF_NO_TOGGLE, WOF_NO_FLAGS, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(2, 4, 4, "~MC", BTF_NO_TOGGLE, WOF_NO_FLAGS, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(6, 4, 4, "~7", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(10, 4, 4, "~8", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(14, 4, 4, "~9", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(18, 4, 4, "~/", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction);
	*this
		+ new UIW_BUTTON(2, 5, 4, "M~R", BTF_NO_TOGGLE, WOF_NO_FLAGS, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(6, 5, 4, "~4", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(10, 5, 4, "~5", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(14, 5, 4, "~6", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(18, 5, 4, "~-", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(2, 6, 4, "M~S", BTF_NO_TOGGLE, WOF_NO_FLAGS, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(6, 6, 4, "~1", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(10, 6, 4, "~2", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(14, 6, 4, "~3", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(18, 6, 4, "~*", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ (mpButton = new UIW_BUTTON(2, 7, 4, "M+", BTF_NO_TOGGLE, WOF_NO_FLAGS, CALCULATOR::ButtonFunction))
		+ new UIW_BUTTON(6, 7, 4, "~.", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(10, 7, 4, "~0", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(14, 7, 4, "~=", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_BUTTON(18, 7, 4, "~+", BTF_NO_TOGGLE, WOF_JUSTIFY_CENTER, CALCULATOR::ButtonFunction)
		+ new UIW_ICON (0, 0, "calc", "Calculator", ICF_MINIMIZE_OBJECT, WOF_NO_FLAGS);
		 

	// Add hot key to M+ button.
	mpButton->HotKey('p');
}

EVENT_TYPE CALCULATOR::ButtonFunction(UI_WINDOW_OBJECT *button, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return 0;
	
	CALCULATOR *calculator = (CALCULATOR *)button->parent;
	calculator->Display((UIW_BUTTON *)button);
	return S_CONTINUE;
}

void CALCULATOR::Display(UIW_BUTTON *button)
{
	// Switch on the button value.
	switch (button->HotKey())
	{

	// Clear the calculator.
	case 'c' :
		decimal =
			operand1 =
			operand2 = 0.0;
		operation = '=';	   
		break;

	case 'e' :
		operand2 = 0.0;
		break;

	// Memory operations.
	case 'm' :
		memory = 0.0;
		memField->DataSet("");
		break;
		
	case 'r' :
		decimal = 0.0;
		operatorLast = FALSE;
		operand1 = operand2;
		operand2 = memory;
		break;

   	case 's' :
		memory = operand2;
		memField->DataSet("M");
		break;

	case 'p' :
		memory += operand2;
		memField->DataSet("M");
		break;

	// Operations.
	case '/' :
	case '*' :
	case '-' :
	case '+' :
	case '=' :
		if (operation != '=' && !operatorLast)
		{
			switch (operation)
			{
			case '/' :
				if (operand2 != 0.0)
					operand2 = operand1 / operand2;
				else
					operand2 = 0.0;
				break;

			case '*' :
				operand2 *= operand1;
				break;

			case '-' :
				operand2 = operand1 - operand2;
				break;

			case '+' :
				operand2 += operand1;
				break;

			}
			operand1 = 0.0;
		}
		decimal = 0.0;
		operatorLast = TRUE;
		operation = button->HotKey();
		break;

	// Decimal placement.
	case '.' :
		if (operatorLast)
		{
			operatorLast = FALSE;
			operand1 = operand2;
			operand2 = 0.0;
		}
		if (decimal == 0.0)
			decimal = .1;
		break;

	// Digit pressed.
	default:
		if (operatorLast)
		{
			operatorLast = FALSE;
			operand1 = operand2;
			operand2 = 0.0;
		}
		if (decimal == 0.0 ) 
			operand2 = operand2 * 10 + button->HotKey() - '0';
		else if (decimal > 0.0000001)
		{
			operand2 = operand2 + (button->HotKey() - '0') * decimal;
			decimal /= 10.0;
		}
		break;
	}

	// Update the displayed number.
	numberField->DataSet(&operand2);
}



#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		display = new UI_TEXT_DISPLAY;
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Create a window in the screen center.
	int left = display->columns / display->cellWidth / 2 - 13;
	int top = display->lines / display->cellHeight / 2 - 5;
	
	*windowManager + new CALCULATOR(left, top, "Calculator");

	// Wait for user response.
	EVENT_TYPE ccode;
	do
	{
		// Get input from the user.
		UI_EVENT event;
		eventManager->Get(event);

		// Send event information to the window manager.
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
