//	DISPLAY.CPP (DISPLAY) - Example program using UI_DISPLAY.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#define USE_RAW_KEYS
#include <ui_win.hpp>

int fill = FALSE;
#define EDGE 70

UI_PALETTE rectanglePalette =
	{ ' ', attrib(BLACK, LIGHTGRAY), attrib(MONO_NORMAL, MONO_BLACK),
	PTN_INTERLEAVE_FILL, BLACK, BLUE, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };
UI_PALETTE ellipsePalette =
	{ ' ', attrib(BLACK, LIGHTGRAY), attrib(MONO_NORMAL, MONO_BLACK),
	PTN_INTERLEAVE_FILL, RED, YELLOW, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };
UI_PALETTE polygonPalette =
	{ ' ', attrib(BLACK, LIGHTGRAY), attrib(MONO_NORMAL, MONO_BLACK),
	PTN_SOLID_FILL, MAGENTA, BROWN, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };
UI_PALETTE textPalette =
	{ ' ', attrib(BLACK, LIGHTGRAY), attrib(MONO_NORMAL, MONO_BLACK),
	PTN_SOLID_FILL, GREEN, LIGHTRED, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };
UI_PALETTE linePalette =
	{ ' ', attrib(BLACK, LIGHTGRAY), attrib(MONO_NORMAL, MONO_BLACK),
	PTN_INTERLEAVE_FILL, LIGHTGRAY, CYAN, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };

void DrawRectangle(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion, int fill)
{
	// Top, left.
	display->Rectangle(screenID, clipRegion.left - EDGE,
		clipRegion.top - EDGE, clipRegion.left + EDGE,
		clipRegion.top + EDGE, &rectanglePalette, 1, fill, FALSE, &clipRegion);

	// Bottom, left.
	display->Rectangle(screenID, clipRegion.left - EDGE,
		clipRegion.bottom - EDGE, clipRegion.left + EDGE,
		clipRegion.bottom + EDGE, &rectanglePalette, 1, fill, FALSE, &clipRegion);

	// Bottom, right.
	display->Rectangle(screenID, clipRegion.right - EDGE,
		clipRegion.bottom - EDGE, clipRegion.right + EDGE,
		clipRegion.bottom + EDGE, &rectanglePalette, 1, fill, FALSE, &clipRegion);

	// Top, right.
	display->Rectangle(screenID, clipRegion.right - EDGE,
		clipRegion.top - EDGE, clipRegion.right + EDGE,
		clipRegion.top + EDGE, &rectanglePalette, 1, fill, FALSE, &clipRegion);

	// In the middle.
	display->Rectangle(screenID, clipRegion.left + EDGE,
		clipRegion.top + EDGE, clipRegion.right - EDGE,
		clipRegion.bottom - EDGE, &rectanglePalette, 5, fill, FALSE, &clipRegion);
}

void DrawEllipse(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion, int fill)
{
	// Top, left.
	display->Ellipse(screenID, clipRegion.left, clipRegion.top,
		0, 360, EDGE, EDGE, &ellipsePalette, fill, FALSE, &clipRegion);

	// Bottom, left.
	display->Ellipse(screenID, clipRegion.left, clipRegion.bottom,
		0, 360, EDGE, EDGE, &ellipsePalette, fill, FALSE, &clipRegion);

	// Bottom, right.
	display->Ellipse(screenID, clipRegion.right, clipRegion.bottom,
		0, 360, EDGE, EDGE, &ellipsePalette, fill, FALSE, &clipRegion);

	// Top, right.
	display->Ellipse(screenID, clipRegion.right, clipRegion.top,
		0, 360, EDGE, EDGE, &ellipsePalette, fill, FALSE, &clipRegion);
}

void DrawPolygon(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion, int fill)
{
	int pointsTL[] =
	{
		clipRegion.left + EDGE, clipRegion.top + EDGE,
		clipRegion.left - EDGE, clipRegion.top,
		clipRegion.left, clipRegion.top - EDGE,
		clipRegion.left + EDGE, clipRegion.top + EDGE
	};

	int pointsBL[] =
	{
		clipRegion.left + EDGE, clipRegion.bottom - EDGE,
		clipRegion.left - EDGE, clipRegion.bottom,
		clipRegion.left, clipRegion.bottom + EDGE,
		clipRegion.left + EDGE, clipRegion.bottom - EDGE
	};

	int pointsBR[] =
	{
		clipRegion.right - EDGE, clipRegion.bottom - EDGE,
		clipRegion.right + EDGE, clipRegion.bottom,
		clipRegion.right, clipRegion.bottom + EDGE,
		clipRegion.right - EDGE, clipRegion.bottom - EDGE
	};

	int pointsTR[] =
	{
		clipRegion.right - EDGE, clipRegion.top + EDGE,
		clipRegion.right + EDGE, clipRegion.top,
		clipRegion.right, clipRegion.top - EDGE,
		clipRegion.right - EDGE, clipRegion.top + EDGE
	};

	// Top, left.
	display->Polygon(screenID, 4, pointsTL, &polygonPalette, fill, FALSE, &clipRegion);

	// Bottom, left.
	display->Polygon(screenID, 4, pointsBL, &polygonPalette, fill, FALSE, &clipRegion);

	// Bottom, right.
	display->Polygon(screenID, 4, pointsBR, &polygonPalette, fill, FALSE, &clipRegion);

	// Top, right.
	display->Polygon(screenID, 4, pointsTR, &polygonPalette, fill, FALSE, &clipRegion);
}

void DrawBitmap(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion)
{
	UCHAR bitmapArray[] =
	{
#define _ BLACK,
#define O BACKGROUND,
#define I YELLOW,
		O O O O O O O O O O O O I I I I I I I I O O O O O O O O O O O O
		O O O O O O O O O I I I I I I I I I I I I I I O O O O O O O O O
		O O O O O O O I I I I I I I I I I I I I I I I I I O O O O O O O
		O O O O O O I I I I I I I I I I I I I I I I I I I I O O O O O O
		O O O O O I I I I I I I I I I I I I I I I I I I I I I O O O O O
		O O O O I I I I I I I I I I I I I I I I I I I I I I I I O O O O
		O O O I I I I I I I I I I I I I I I I I I I I I I I I I I O O O
		O O I I I I I I I I I I I I I I I I I I I I I I I I I I I I O O
		O O I I I I I I I _ _ _ I I I I I I I I _ _ _ I I I I I I I O O
		O I I I I I I I _ _ _ _ _ I I I I I I _ _ _ _ _ I I I I I I I O
		O I I I I I I I _ _ _ _ _ I I I I I I _ _ _ _ _ I I I I I I I O
		O I I I I I I I _ _ _ _ _ I I I I I I _ _ _ _ _ I I I I I I I O
		I I I I I I I I I _ _ _ I I I I I I I I _ _ _ I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I I
		I I I I I _ _ I I I I I I I I I I I I I I I I I I _ _ I I I I I
		O I I I I _ _ I I I I I I I I I I I I I I I I I I _ _ I I I I O
		O I I I I _ _ _ I I I I I I I I I I I I I I I I _ _ _ I I I I O
		O I I I I I _ _ I I I I I I I I I I I I I I I I _ _ I I I I I O
		O O I I I I _ _ _ I I I I I I I I I I I I I I _ _ _ I I I I O O
		O O I I I I I _ _ _ _ I I I I I I I I I I _ _ _ _ I I I I I O O
		O O O I I I I I _ _ _ _ _ _ I I I I I _ _ _ _ _ I I I I I O O O
		O O O O I I I I I I _ _ _ _ _ _ _ _ _ _ _ _ I I I I I I O O O O
		O O O O O I I I I I I I I _ _ _ _ _ _ I I I I I I I I O O O O O
		O O O O O O I I I I I I I I I I I I I I I I I I I I O O O O O O
		O O O O O O O I I I I I I I I I I I I I I I I I I O O O O O O O
		O O O O O O O O O I I I I I I I I I I I I I I O O O O O O O O O
		O O O O O O O O O O O O I I I I I I I I O O O O O O O O O O O O
#undef _
#undef O
#undef I
	};

	// Top, left.
	display->Bitmap(screenID, clipRegion.left - 16,
		clipRegion.top - 16, 32, 32, bitmapArray, NULL, &clipRegion);

	// Bottom, left.
	display->Bitmap(screenID, clipRegion.left - 16,
		clipRegion.bottom - 16, 32, 32, bitmapArray, NULL, &clipRegion);

	// Bottom, right.
	display->Bitmap(screenID, clipRegion.right - 16,
		clipRegion.bottom - 16, 32, 32, bitmapArray, NULL, &clipRegion);

	// Top, right.
	display->Bitmap(screenID, clipRegion.right - 16,
		clipRegion.top - 16, 32, 32, bitmapArray, NULL, &clipRegion);

	// Center.
	display->Bitmap(screenID, clipRegion.left + (clipRegion.right - clipRegion.left) / 2 - 16,
		clipRegion.top + (clipRegion.bottom - clipRegion.top) / 2 - 16, 32, 32, bitmapArray);
}

void DrawText(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion)
{
	char *string = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	// Top.
	display->Text(screenID, clipRegion.left +
		(clipRegion.right - clipRegion.left - display->TextWidth(string, screenID)) / 2,
		clipRegion.top - (display->TextHeight(string, screenID) / 2), string,
		&textPalette, -1, fill, FALSE, &clipRegion);

	// Left.
	display->Text(screenID, clipRegion.left - (display->TextWidth(string, screenID) / 2),
		clipRegion.top + (clipRegion.bottom - clipRegion.top) / 2 -
		display->TextHeight(string, screenID) / 2, string,
		&textPalette, -1, fill, FALSE, &clipRegion);

	// Right.
	display->Text(screenID, clipRegion.right - (display->TextWidth(string, screenID) / 2),
		clipRegion.top + (clipRegion.bottom - clipRegion.top) / 2 -
		display->TextHeight(string, screenID) / 2, string,
		&textPalette, -1, fill, FALSE, &clipRegion);

	// Bottom.
	display->Text(screenID, clipRegion.left +
		(clipRegion.right - clipRegion.left - display->TextWidth(string, screenID)) / 2,
		clipRegion.bottom - (display->TextHeight(string, screenID) / 2), string,
		&textPalette, -1, fill, FALSE, &clipRegion);
}

void DrawLines(UI_DISPLAY *display, SCREENID screenID, UI_REGION &clipRegion)
{
	// Top, left to bottom, right.
	display->Line(screenID, clipRegion.left - EDGE,
		clipRegion.top, clipRegion.right, clipRegion.bottom + EDGE,
		&linePalette, 1, FALSE, &clipRegion);

	// Bottom, right to top, left.
	display->Line(screenID, clipRegion.right + EDGE,
		clipRegion.bottom, clipRegion.left, clipRegion.top - EDGE,
		&linePalette, 1, FALSE, &clipRegion);

	// Bottom, left to top, right.
	display->Line(screenID, clipRegion.left - EDGE,
		clipRegion.bottom, clipRegion.right, clipRegion.top - EDGE,
		&linePalette, 1, FALSE, &clipRegion);

	// Top, right to bottom, left.
	display->Line(screenID, clipRegion.right + EDGE,
		clipRegion.top, clipRegion.left, clipRegion.bottom + EDGE,
		&linePalette, 1, FALSE, &clipRegion);
}

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		return (1);
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Create a window with a NON_FIELD_REGION text field.
	UIW_WINDOW *mainWindow = UIW_WINDOW::Generic(5, 1, 70, 12, "DISPLAY");
	UI_WINDOW_OBJECT *clipWindow = new UIW_WINDOW(10, 2, 48, 8, WOF_BORDER);
	*mainWindow
		+ new UIW_PROMPT(1, 0, 66,
			"Press <ENTER> to draw...          Press <ALT-F4> to exit", WOF_JUSTIFY_CENTER)
		+ clipWindow;

	// Put main window onto display.
	*windowManager
		+ mainWindow;

	// Wait for user response.
	EVENT_TYPE ccode;
	UI_EVENT event;
	do
	{
		eventManager->Get(event, Q_NORMAL);
#ifdef _WINDOWS
		if (event.type == E_MSWINDOWS && event.message.message == WM_CHAR &&
			event.message.wParam == ' ')
#else
		if (event.type == E_KEY && event.rawCode == SPACE)
#endif
		{
			UI_REGION moveRegion = clipWindow->true;

			display->RegionMove(moveRegion, moveRegion.left,
				moveRegion.top - display->cellHeight, mainWindow->screenID, mainWindow->screenID);
		}
		else
#ifdef _WINDOWS
		if (event.type == E_MSWINDOWS && event.message.message == WM_CHAR &&
			event.message.wParam == ENTER)
#else
		if (event.type == E_KEY && event.rawCode == ENTER)
#endif
		{
			windowManager->Event(UI_EVENT(S_REDISPLAY));

			UI_REGION clipRegion = clipWindow->true;
			fill = fill == FALSE;

			// Draw a clipped rectangle.
			DrawRectangle(display, mainWindow->screenID, clipRegion, fill);

			// Draw a clipped ellipse.
			DrawEllipse(display, mainWindow->screenID, clipRegion, fill);

			// Draw a clipped polygon.
			DrawPolygon(display, mainWindow->screenID, clipRegion, fill);

			// Draw clipped bitmaps.
			DrawBitmap(display, mainWindow->screenID, clipRegion);

			// Draw clipped text.
			DrawText(display, mainWindow->screenID, clipRegion);
	
			// Draw clipped lines.
			DrawLines(display, mainWindow->screenID, clipRegion);
		}
		else
			ccode = windowManager->Event(event);
	} while(ccode != L_EXIT && ccode != S_NO_OBJECT);

	//	Clean up now
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
