//	GRAPH.CPP (GRAPH) - Extended graphics example.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdlib.h>
#include <ui_win.hpp>

// Palettes used for drawing graphics.
UI_PALETTE graphPalette[] =
{
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, RED, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260',attrib( BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, YELLOW, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, GREEN, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, BLUE, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_INTERLEAVE_FILL, BLACK, LIGHTRED, BW_BLACK, BW_WHITE,	GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, BLACK, LIGHTGREEN, BW_BLACK, BW_WHITE, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, BLACK, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, RED, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY },
	{ '\260', attrib(BLACK, BLACK), attrib(MONO_DIM, MONO_BLACK),
		PTN_SOLID_FILL, GREEN, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY }
};

// Abstract base class for all graphic window objects.
class WO_GRAPHIC : public UI_WINDOW_OBJECT
{
public:
	UI_PALETTE *palette;

	WO_GRAPHIC(UI_PALETTE *_palette)
		: UI_WINDOW_OBJECT(0, 0, 0, 0, WOF_NO_FLAGS, WOAF_NO_FLAGS),
		  palette(_palette)
	{ woStatus = WOS_GRAPHICS; }

	virtual void Draw() = 0;
	virtual EVENT_TYPE Event(const UI_EVENT &event);
};


#ifdef _WINDOWS
static int _graphicOffset = -1;
static FARPROC _graphicCallback = (FARPROC)DefWindowProc;
static FARPROC _graphicJumpInstance = NULL;

long FAR PASCAL _export GraphicJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	WO_GRAPHIC *object = (WO_GRAPHIC *)GetWindowLong(hWnd, _graphicOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE WO_GRAPHIC::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_graphicJumpInstance)
			_graphicJumpInstance = (FARPROC)GraphicJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		break;

	case S_CREATE:
		UI_WINDOW_OBJECT::Event(event);
		RegisterObject("WO_GRAPHIC", "STATIC", &_graphicOffset,
			&_graphicJumpInstance, &_graphicCallback, "graphic");
		break;

	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_DRAWITEM)
			Draw();
		else if (message == WM_PAINT)
		{
			PAINTSTRUCT ps;
			BeginPaint(screenID, &ps);
			Draw();
			EndPaint(screenID, &ps);
		}

		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

#else

EVENT_TYPE WO_GRAPHIC::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_OBJECT);
	switch (ccode)
	{
	// Display object.
	case S_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		if (true.Overlap(event.region))
			Draw();
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
	}

	// Return the control code.
	return (ccode);
}

#endif

// Window object line class definition.
// Lines must be drawn from left to right.
class WO_LINE : public WO_GRAPHIC
{
public:
	char topLeftToBottomRight;

	WO_LINE(int x1, int y1, int x2, int y2, UI_PALETTE *palette);
  
	virtual void Draw();
};

WO_LINE::WO_LINE(int x1, int y1, int x2, int y2, UI_PALETTE *_palette)
	: WO_GRAPHIC(_palette)

{
	topLeftToBottomRight = ((x1 < x2 && y1 < y2) || (x2 < x1 && y2 < y1));

	// Set Window Object region relative to parent window.
	relative.left = Min(x1, x2);
	relative.right = Max(x1, x2);
	relative.top = Min(y1, y2);
	relative.bottom = Max(y1, y2);
}

void WO_LINE::Draw()
{
#ifdef _WINDOWS
	UI_REGION region = {0, 0, true.right - true.left, true.bottom - true.top};
#else
	UI_REGION region = true;
#endif

	if (topLeftToBottomRight)
		display->Line(screenID, region.left, region.top, region.right, region.bottom, palette, 1, FALSE);
	else
		display->Line(screenID, region.left, region.bottom, region.right, region.top, palette, 1, FALSE);
}

// Window object sector class definition.
class WO_SECTOR : public WO_GRAPHIC
{
public:
	int stAngle, endAngle, xRadius, yRadius;

	WO_SECTOR(int x, int y, int stAngle, int endAngle, int xRadius,
		int yRadius, UI_PALETTE *palette);

	virtual void Draw();
};

WO_SECTOR::WO_SECTOR(int x, int y, int _stAngle, int _endAngle, int _xRadius,
	int _yRadius, UI_PALETTE *_palette)
	: WO_GRAPHIC(_palette), xRadius(_xRadius), yRadius(_yRadius),
	  stAngle(_stAngle), endAngle(_endAngle)
{
	// Set Window Object region relative to parent window.
	relative.left = x - _xRadius;
	relative.right = x + _xRadius;
	relative.top = y - _yRadius;
	relative.bottom = y + _yRadius;
}

void WO_SECTOR::Draw()
{
#ifdef _WINDOWS
	UI_REGION region = {0, 0, true.right - true.left, true.bottom - true.top};
#else
	UI_REGION region = true;
#endif

	display->Ellipse(screenID, region.left + xRadius, region.top + yRadius, stAngle, endAngle,
		xRadius, yRadius, palette, TRUE, FALSE);
}


// Window object sector class definition.
class WO_PIE_GRAPH : public WO_GRAPHIC
{
public:
	int numSectors, *angles;

	WO_PIE_GRAPH(int _numSectors, int *_angles, UI_PALETTE *_palette);

	virtual void Draw();
};

WO_PIE_GRAPH::WO_PIE_GRAPH(int _numSectors, int *_angles, UI_PALETTE *_palette)
	: WO_GRAPHIC(_palette), numSectors(_numSectors), angles(_angles)
{
	woFlags |= WOF_NON_FIELD_REGION;
}

void WO_PIE_GRAPH::Draw()
{
#ifdef _WINDOWS
	UI_REGION region = {0, 0, true.right - true.left, true.bottom - true.top};
#else
	UI_REGION region = true;
#endif

	UI_PALETTE *windowPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, PM_ANY, ID_WINDOW);
	display->Rectangle(screenID, true, windowPalette, 1, TRUE);

	int xRadius = (region.right - region.left) / 2 - 20;
	int yRadius = (region.bottom - region.top) / 2 - 20;
	int x = region.left + xRadius + 20;
	int y = region.top + yRadius + 20;
//	UI_PALETTE *backPalette = UI_WINDOW_OBJECT::LogicalPalette(S_DISPLAY_INACTIVE);
//	display->Rectangle(screenID, region.left, region.top, region.right, region.bottom, backPalette,
//		1, TRUE, FALSE);
	for (int index = 0; index < numSectors; index++)
	{
		display->Ellipse(screenID, x, y, angles[index], angles[index + 1],
			xRadius, yRadius, &palette[index], TRUE, FALSE);
	}
}

// Window object bar class definition.
class WO_BAR : public WO_GRAPHIC
{
public:
	WO_BAR(int left, int top, int width, int height, UI_PALETTE *palette);

	virtual void Draw();
};

WO_BAR::WO_BAR(int left, int top, int width, int height, UI_PALETTE *_palette)
	: WO_GRAPHIC(_palette)
{
	// Set Window Object region relative to parent window.
	relative.left = left;
	relative.right = left + width;
	relative.top = top;
	relative.bottom = top + height;
}

void WO_BAR::Draw()
{
#ifdef _WINDOWS
	UI_REGION region = {0, 0, true.right - true.left, true.bottom - true.top};
#else
	UI_REGION region = true;
#endif

	display->Rectangle(screenID, region.left, region.top, region.right, region.bottom, palette, 1, TRUE, FALSE);
}

int _sectors[] =
{
	0, 56, 112, 133, 210, 240, 360
};

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		exit (1);
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Create four windows.
	UIW_WINDOW *barWindow = new UIW_WINDOW(40, 1, 35, 9, WOF_NO_FLAGS, WOAF_NO_FLAGS);
	UIW_WINDOW *pieWindow = new UIW_WINDOW(4, 10, 60, 5, WOF_NO_FLAGS, WOAF_NO_FLAGS);
	UIW_WINDOW *sectorWindow = new UIW_WINDOW(14, 5, 52, 8, WOF_NO_FLAGS, WOAF_NO_FLAGS);
	UIW_WINDOW *lineWindow = new UIW_WINDOW(7, 2, 32, 6, WOF_NO_FLAGS, WOAF_NO_FLAGS);

	// Add the bars and other objects to window2.
	*barWindow
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ new UIW_SYSTEM_BUTTON(SYF_GENERIC)
		+ new UIW_TITLE("WO_BAR", WOF_JUSTIFY_CENTER);

	*barWindow
		+ new WO_BAR(16, 56, 20, 84, &graphPalette[0])
		+ new WO_BAR(16 + display->TextWidth("1985 "),
			98, 20, 42, &graphPalette[1])
		+ new WO_BAR(16 + display->TextWidth("1985 1986 "),
			14, 20, 126, &graphPalette[2])
		+ new WO_BAR(16 + display->TextWidth("1985 1986 1987 "),
			28, 20, 112, &graphPalette[3])
		+ new WO_BAR(16 + display->TextWidth("1985 1986 1987 1988 "),
			70, 20, 70, &graphPalette[4])
		+ new WO_BAR(16 + display->TextWidth("1985 1986 1987 1988 1989 "),
			126, 20, 14, &graphPalette[5])
		+ new UIW_PROMPT(2, 6, "1985 1986 1987 1988 1989 1990", WOF_NO_FLAGS);
		// Add the pie elements and other objects to pieWindow.
	*pieWindow
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ new UIW_SYSTEM_BUTTON(SYF_GENERIC)
		+ new UIW_TITLE("WO_PIE_GRAPH (Auto Sizing)", WOF_JUSTIFY_CENTER)
		+ new WO_PIE_GRAPH(6, _sectors, graphPalette);

	*sectorWindow
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ new UIW_SYSTEM_BUTTON(SYF_GENERIC)
		+ new UIW_TITLE("WO_SECTOR_GRAPH", WOF_JUSTIFY_CENTER);

	*sectorWindow
		+ new WO_SECTOR(205, 80, 0, 360, 75, 30, &graphPalette[6])
		+ new WO_SECTOR(205, 70, 0, 56, 75, 30, &graphPalette[0])
		+ new WO_SECTOR(205, 70, 56, 112, 75, 30, &graphPalette[1])
		+ new WO_SECTOR(205, 70, 112, 133, 75, 30, &graphPalette[2])
		+ new WO_SECTOR(205, 70, 133, 210, 75, 30, &graphPalette[3])
		+ new WO_SECTOR(205, 70, 210, 240, 75, 30, &graphPalette[4])
		+ new WO_SECTOR(205, 70, 240, 360, 75, 30, &graphPalette[5]);

	*sectorWindow
		+ new WO_LINE(130, 70, 130, 80, &graphPalette[6])
		+ new WO_LINE(280, 70, 280, 80, &graphPalette[6])
		+ new UIW_PROMPT(21, 1, "Rent", WOF_NO_FLAGS)
		+ new UIW_PROMPT(15, 2, "Food", WOF_NO_FLAGS)
		+ new UIW_PROMPT(14, 4, "Clothing", WOF_NO_FLAGS)
		+ new UIW_PROMPT(32, 1, "Auto", WOF_NO_FLAGS)
		+ new UIW_PROMPT(41, 2, "Other", WOF_NO_FLAGS)
		+ new UIW_PROMPT(35, 5, "Entertainment", WOF_NO_FLAGS);

	// Initialize data for line graph.
	int redLine[] = { 75, 80, 63, 52, 14, 21, 7 };
	int greenLine[] = { 80, 70, 65, 42, 50, 35, 20 };

	// Add the border and other objects to lineWindow.
	*lineWindow
		+ new UIW_BORDER
		+ new UIW_MAXIMIZE_BUTTON
		+ new UIW_MINIMIZE_BUTTON
		+ new UIW_SYSTEM_BUTTON(SYF_GENERIC)
		+ new UIW_TITLE("WO_LINE", WOF_JUSTIFY_CENTER);

	*lineWindow
		+ new WO_LINE(8, 5, 8, 82, &graphPalette[6])
		+ new WO_LINE(8, 5, 200, 5, &graphPalette[6])
		+ new WO_LINE(200, 5, 200, 82, &graphPalette[6])
		+ new WO_LINE(8, 82, 200, 82, &graphPalette[6])
		+ new UIW_PROMPT(1, 4, "Jan Feb Mar Apr May Jun", WOF_NO_FLAGS);

	// Create the graph lines for lineWindow.
	int x;
	for (x = 0; x < 6; x++)
		*lineWindow + new WO_LINE(8 + x * 32, redLine[x], 8 + (x + 1) * 32,
			redLine[x+1], &graphPalette[7]);

	for (x = 0; x < 6; x++)
		*lineWindow + new WO_LINE(8 + x * 32, greenLine[x], 8 + (x + 1) * 32,
			greenLine[x+1], &graphPalette[8]);

	// Add the windows to the window manager.
	*windowManager
		+ barWindow
		+ pieWindow
		+ sectorWindow
		+ lineWindow;

	// Wait for user response.
	EVENT_TYPE ccode;
	UI_EVENT event;
	do
	{
		eventManager->Get(event, Q_NORMAL);
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
