//	PIANO.CPP (PIANO) - Sample electronic piano program.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#define USE_RAW_KEYS

#include <ui_win.hpp>
#include <stdio.h>
#include "sound.hpp"

// Color palette definitions for piano keys.
static UI_PALETTE ebonyPalette = { ' ', attrib(BLACK, BLACK), attrib(MONO_NORMAL, MONO_BLACK),
		PTN_SOLID_FILL, BLACK, BLACK, BW_BLACK, BW_BLACK, GS_BLACK, GS_BLACK };
static UI_PALETTE ivoryPalette = { ' ', attrib(BLACK, WHITE), attrib(MONO_BLACK, MONO_NORMAL),
		PTN_SOLID_FILL, BLACK, WHITE, BW_BLACK, BW_WHITE, GS_BLACK, GS_WHITE };
static UI_PALETTE whitePalette = { ' ', attrib(WHITE, WHITE), attrib(MONO_HIGH, MONO_HIGH),
		PTN_SOLID_FILL, WHITE, WHITE, BW_WHITE, BW_WHITE, GS_WHITE, GS_WHITE };
static UI_PALETTE lightPalette = { ' ', attrib(LIGHTGRAY, LIGHTGRAY), attrib(MONO_NORMAL, MONO_NORMAL),
		PTN_SOLID_FILL, LIGHTGRAY, LIGHTGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY };
static UI_PALETTE darkPalette = { ' ', attrib(DARKGRAY, DARKGRAY), attrib(MONO_NORMAL, MONO_NORMAL),
		PTN_SOLID_FILL, DARKGRAY, DARKGRAY, BW_BLACK, BW_BLACK, GS_GRAY, GS_GRAY };

const USHORT KYF_NO_FLAGS	= 0x0000;
const USHORT KYF_SQUARE		= 0x0001;
const USHORT KYF_LEFT 		= 0x0002;
const USHORT KYF_MIDDLE		= 0x0004;
const USHORT KYF_RIGHT		= 0x0008;
const USHORT KYF_EBONY		= 0x0010;
const USHORT KYF_IVORY		= 0x0020;

const USHORT KYS_NO_STATUS	= 0x0000;
const USHORT KYS_DOWN		= 0x0001;

class PIANO_KEY : public UI_WINDOW_OBJECT, public UI_SOUND
{
public:
	PIANO_KEY(int left, char _hotKey, int freq,
		USHORT _kyFlags = KYF_SQUARE | KYF_EBONY);

	USHORT kyFlags;
	USHORT kyStatus;

	virtual EVENT_TYPE Event(const UI_EVENT &event);
	void HighLight();
};

class PIANO : public UIW_WINDOW
{
public:
	PIANO(int left, int top);

	virtual EVENT_TYPE Event(const UI_EVENT &event);
};


PIANO_KEY::PIANO_KEY(int left, char _hotKey, int freq, USHORT _kyFlags) :
	UI_WINDOW_OBJECT(left, 1, FlagSet(_kyFlags, KYF_EBONY) ? 2 : 3,
	 	FlagSet(_kyFlags, KYF_EBONY) ? 2 : 4, WOF_BORDER, WOAF_NON_CURRENT),
	UI_SOUND(freq, 100), kyFlags(_kyFlags), kyStatus(0)
{
	hotKey = _hotKey;
}

EVENT_TYPE PIANO_KEY::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = UI_WINDOW_OBJECT::LogicalEvent(event, ID_BUTTON);
	UI_PALETTE *palette;
	switch (ccode)
	{
	case S_NON_CURRENT:
		if (FlagSet(kyStatus, KYS_DOWN))
		{
			SoundOff();
			kyStatus &= ~KYS_DOWN;
			HighLight();
		}
	case S_CURRENT:
	case S_DISPLAY_INACTIVE:
	case S_DISPLAY_ACTIVE:
		palette = FlagSet(kyFlags, KYF_EBONY) ? &ebonyPalette : &ivoryPalette;

		if (display && (ccode == S_NON_CURRENT || ccode == S_CURRENT ||
			UI_WINDOW_OBJECT::NeedsUpdate(event, ccode)))
			{
				int halfHeight = 2 * display->cellHeight -
					(display->postSpace + display->preSpace);
				UI_REGION bottomRegion;
				UI_REGION topRegion;
				topRegion.top = true.top;
				topRegion.bottom = true.top + halfHeight - 1;
				bottomRegion.left = true.left;
				bottomRegion.top = topRegion.bottom + 1;
				bottomRegion.right = true.right;
				bottomRegion.bottom = true.bottom;
				if (FlagSet(kyFlags, KYF_LEFT))
				{
					topRegion.left = true.left;
					topRegion.right = true.right - display->cellWidth;
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &topRegion);
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &bottomRegion);
					HighLight();
				}
				else if (FlagSet(kyFlags, KYF_MIDDLE))
				{
					topRegion.left = true.left + display->cellWidth;
					topRegion.right = true.right - display->cellWidth;
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &topRegion);
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &bottomRegion);
					HighLight();
				}
				else if (FlagSet(kyFlags, KYF_RIGHT))
				{
					topRegion.left = true.left + display->cellWidth;
					topRegion.right = true.right;
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &topRegion);
					display->Rectangle(screenID, true, palette, 1, TRUE,
						FALSE, &bottomRegion);
					HighLight();
				}
				else // FlagSet(kyFlags, KYF_SQUARE));
				{
					display->Rectangle(screenID, true, palette, 1, TRUE);
					HighLight();
				}
			}
		break;

	case L_BEGIN_SELECT:
		if (!FlagSet(kyStatus, KYS_DOWN))
		{
			kyStatus |= KYS_DOWN;
			HighLight();
			SoundOn();
		}
		break;

	case L_CONTINUE_SELECT:
		if (!FlagSet(kyStatus, KYS_DOWN) && true.Overlap(event.position))
		{
			kyStatus |= KYS_DOWN;
			HighLight();
			SoundOn();
		}
		else if (FlagSet(kyStatus, KYS_DOWN) && !true.Overlap(event.position))
		{
			SoundOff();
			kyStatus &= ~KYS_DOWN;
			HighLight();
		}
		break;

	case L_END_SELECT:
		if (FlagSet(kyStatus, KYS_DOWN))
		{
			SoundOff();
			kyStatus &= ~KYS_DOWN;
			HighLight();
		}
		break;

	case L_SELECT:
		kyStatus |= KYS_DOWN;
		HighLight();
		Beep();
		kyStatus &= ~KYS_DOWN;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}
	return (ccode);
}

void PIANO_KEY::HighLight()
{
	if (display->isText)
		return;
	// Get the palette for the left side's shadows.
	UI_PALETTE *shadowPalette = FlagSet(kyStatus, KYS_DOWN) ? &lightPalette : &whitePalette;

	// Draw the highlight on the left side.
	if (FlagSet(kyFlags, KYF_LEFT | KYF_SQUARE))
		display->Line(screenID, true.left + 1, true.top + 1, true.left + 1,
			true.bottom - 2, shadowPalette);
	else
		display->Line(screenID, true.left + 1,
			true.top + 2 * display->cellHeight - (display->postSpace + display->preSpace),
			true.left + 1, true.bottom - 2, shadowPalette);

	// Get the palette for the bottom and right side's shadows.
	shadowPalette = FlagSet(kyStatus, KYS_DOWN) ? &darkPalette : &lightPalette;

	// Draw the highlight across the bottom.
	display->Line(screenID, true.left + 1, true.bottom - 1, true.right - 1,
		true.bottom - 1, shadowPalette);

	// Draw the highlight on the right side.
	if (FlagSet(kyFlags, KYF_RIGHT | KYF_SQUARE))
		display->Line(screenID, true.right - 1, true.top + 1, true.right - 1,
			true.bottom - 2, shadowPalette);
	else
		display->Line(screenID, true.right - 1,
			true.top + 2 * display->cellHeight - (display->postSpace + display->preSpace),
			true.right - 1, true.bottom - 2, shadowPalette);
}

PIANO::PIANO(int left, int top) : UIW_WINDOW(left, top, 47, 6)
{
	*this
		+ new PIANO_KEY(1, 'A', 131, KYF_LEFT | KYF_IVORY)
		+ new PIANO_KEY(4, 'Q', 147, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(7, 'W', 165, KYF_RIGHT | KYF_IVORY)
		+ new PIANO_KEY(10, 'E', 175, KYF_LEFT | KYF_IVORY)
		+ new PIANO_KEY(13, 'R', 196, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(16, 'T', 220, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(19, 'Y', 247, KYF_RIGHT | KYF_IVORY)
		+ new PIANO_KEY(22, 'U', 262, KYF_LEFT | KYF_IVORY);
	*this
		+ new PIANO_KEY(25, 'I', 294, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(28, 'O', 329, KYF_RIGHT | KYF_IVORY)
		+ new PIANO_KEY(31, 'P', 349, KYF_LEFT | KYF_IVORY)
		+ new PIANO_KEY(34, '[', 392, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(37, ']', 440, KYF_MIDDLE | KYF_IVORY)
		+ new PIANO_KEY(40, ';', 494, KYF_RIGHT | KYF_IVORY)
		+ new PIANO_KEY(43, '\'', 523, KYF_SQUARE | KYF_IVORY);
	*this
		+ new PIANO_KEY(3, '1', 139, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(6, '2', 156, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(12, '4', 185, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(15, '5', 208, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(18, '6', 233, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(24, '8', 277, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(27, '9', 311, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(33, '-', 370, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(36, '=', 415, KYF_SQUARE | KYF_EBONY)
		+ new PIANO_KEY(39, BACKSPACE, 466, KYF_SQUARE | KYF_EBONY);
}

#ifdef _WINDOWS
EVENT_TYPE PIANO::Event(const UI_EVENT &event)
{
	if (event.type == S_CREATE)
	{
		EVENT_TYPE answer = UIW_WINDOW::Event(event);
		for (UI_WINDOW_OBJECT *nObject = Last(); nObject;
			nObject = nObject->Previous())
				nObject->screenID = screenID;
		return (answer);
	}

	WORD message = event.message.message;

	if (event.type == E_MSWINDOWS)
		switch (message)
		{
		case WM_PAINT:
		{
			EVENT_TYPE answer = UIW_WINDOW::Event(event);
			for (UI_WINDOW_OBJECT *nObject = Last(); nObject;
				nObject = nObject->Previous())
					nObject->Event(UI_EVENT(S_DISPLAY_ACTIVE, 0, true));
			return (answer);
		}
		case WM_KEYDOWN:
		{
			// Treat key like a hotKey.
			char key = LOBYTE(event.message.wParam);
			unsigned tHotKey = (key >= 'a' && key <= 'z') ? key + ('A' - 'a') : key;

			// See if one of the objects has this as a hot key.
			for (UI_WINDOW_OBJECT *nObject = First();
				nObject && nObject->HotKey() != tHotKey;
				nObject = nObject->Next())
				;

			// If it is not for the current object then change who is current.
			if (nObject && nObject != Current())
			{
				if (Current())
					Current()->Event(UI_EVENT(S_NON_CURRENT, 0, true));
				nObject->Event(UI_EVENT(S_CURRENT, 0, true));
				current = nObject;
			}
			if (nObject)
				return (nObject->Event(UI_EVENT(L_SELECT)));
		}

		case WM_LBUTTONDOWN:
		case WM_MOUSEMOVE:
		case WM_LBUTTONUP:
		{
			for (UI_WINDOW_OBJECT *nObject = Last();
				nObject && !nObject->true.Overlap(event.position);
				nObject = nObject->Previous())
				;

			// If no object overlaps then pass it on.
			if (!nObject)
				return (UIW_WINDOW::Event(event));

			// If it is not for the current object then change who is current.
			if (event.message.wParam && nObject != Current())
			{
				if (Current())
					Current()->Event(UI_EVENT(S_NON_CURRENT, 0, true));
				nObject->Event(UI_EVENT(S_CURRENT, 0, true));
				current = nObject;
			}

			// Pass the message down.
			if (message == WM_LBUTTONDOWN)
				return (nObject->Event(UI_EVENT(L_SELECT)));

			if (message == WM_LBUTTONUP && !event.message.wParam)
				return (S_CONTINUE);
				

		}
	}
	return (UIW_WINDOW::Event(event));
}

#else
EVENT_TYPE PIANO::Event(const UI_EVENT &event)
{
	if (event.type == E_MOUSE)
	{
		// Find out which object the mouse event is for.
		for (UI_WINDOW_OBJECT *nObject = Last();
			nObject && !nObject->true.Overlap(event.position);
			nObject = nObject->Previous())
			;

		// If no object overlaps then pass it on.
		if (!nObject)
			return (UIW_WINDOW::Event(event));

		// If it is not for the current object then change who is current.
		if (event.rawCode && nObject != Current())
		{
			if (Current())
				Current()->Event(UI_EVENT(S_NON_CURRENT, 0, true));
			nObject->Event(UI_EVENT(S_CURRENT, 0, true));
			current = nObject;
		}

		// Pass the message down.
		return (nObject->Event(event));
	}
	else if (event.type == E_KEY)
	{
		// Treat key like a hotKey.
		unsigned tHotKey =
			(event.key.value >= 'a' && event.key.value <= 'z') ?
				event.key.value + ('A' - 'a') : event.key.value;

		// See if one of the objects has this as a hot key.
		for (UI_WINDOW_OBJECT *nObject = First();
			nObject && nObject->HotKey() != tHotKey;
			nObject = nObject->Next())
			;

		// If a match is found send it an L_SELECT.
		if (nObject)
			return (nObject->Event(UI_EVENT(L_SELECT)));
	}
	return (UIW_WINDOW::Event(event));
}
#endif

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		display = new UI_TEXT_DISPLAY;
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Initialize the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Create a window in the screen center.
	int centerX = display->columns / display->cellWidth / 2;
	int centerY = display->lines / display->cellHeight / 2;
	UIW_WINDOW *window = new UIW_WINDOW(centerX - 25, centerY - 4, 52, 10, WOF_NO_FLAGS, WOAF_NO_FLAGS);

	// Add border, etc. to window.
	*window
		+ new UIW_BORDER
		+ new UIW_MINIMIZE_BUTTON
		+ &(*new UIW_SYSTEM_BUTTON
			+ new UIW_POP_UP_ITEM("~Move", MNIF_MOVE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM("Mi~nimize", MNIF_MINIMIZE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0)
			+ new UIW_POP_UP_ITEM
			+ new UIW_POP_UP_ITEM("~Close", MNIF_CLOSE, BTF_NO_TOGGLE, WOF_NO_FLAGS, 0))
		+ new UIW_TITLE("Beethoven", WOF_JUSTIFY_CENTER)
		+ new PIANO(1, 1);

	// Add the keys.

	// Add the window to the window manager.
	*windowManager + window;

	// Wait for user response.
	EVENT_TYPE ccode;
	do
	{
		// Get input from the user.
		UI_EVENT event;
		eventManager->Get(event);

		// Send event information to the window manager.
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
