//	Zinc Interface Library - BITMAP.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#include <mem.h>
#endif
#include <string.h>
#include "ui_dsp.hpp"
#pragma hdrstop

#ifdef _WINDOWS
// extern void ZincToWindowsBitmap(UI_DISPLAY *display, int bitmapWidth,
// 	int bitmapHeight, UCHAR *bitmapArray, UI_PALETTE *paletteArray,
// 	HBITMAP *hColorBitmap, HBITMAP *hMonoBitmap);
// 
// extern void WindowsToZincBitmap(UI_DISPLAY *display, HBITMAP hColorBitmap,
// 	HBITMAP hMonoBitmap, int *bitmapWidth, int *bitmapHeight,
// 	UCHAR **bitmapArray);
// 
// extern void WindowsIconToBitmap(UI_DISPLAY *display, HICON icon,
// 	HBITMAP *hColorBitmap, HBITMAP *hMonoBitmap);
// 
// extern void WindowsBitmapToIcon(UI_DISPLAY *display, HBITMAP hColorBitmap,
// 	HBITMAP hMonoBitmap, HICON *icon);
// 
// extern void ZincToWindowsIcon(UI_DISPLAY *display, int iconWidth,
// 	int iconHeight, UCHAR *iconArray, UI_PALETTE *paletteArray,
// 	HICON *icon);
// 
// extern void WindowsToZincIcon(UI_DISPLAY *display, HICON icon, int *iconWidth,
// 	int *iconHeight, UCHAR **iconArray);
// 
// 
// The Zinc bitmap format consists of a width, height, and array of color
// indices in the range 0x00 to 0xFE (not actual colors), and a possible
// palette array.  If a palette array is not specified a default palette
// array (not palette map) of 16 predefined colors is used.  The predefined
// colors are in order:
//   0-BLACK, 1-BLUE, 2-GREEN, 3-CYAN, 4-RED, 5-MAGENTA, 6-BROWN,
//   7-LIGHTGRAY, 8-DARKGRAY, 9-LIGHTBLUE, 10-LIGHTGREEN, 11-LIGHTCYAN,
//   12-LIGHTRED, 13-LIGHTMAGENTA, 14-YELLOW, 15-WHITE
// The index of BACKGROUND (0xFF) is reserved for no color or transparency.
//
// The Zinc icon format consists of the same as above except that no
// palette can currently be defined.  Only the predefined colors and BACKGROUND
// can be used.

// All functions below allocate memory for the destination bitmaps if NULL.

// This function converts a Zinc format bitmap into a Windows bitmap.
// The hMonoBitmap is taken from the 0xFF transparent indices present in
// the Zinc bitmap.  NULL can be used for hMonoBitmap for no transparency.

void ZincToWindowsBitmap(UI_DISPLAY *display, int bitmapWidth,
	int bitmapHeight, UCHAR *bitmapArray, UI_PALETTE *paletteArray,
	HBITMAP *hColorBitmap, HBITMAP *hMonoBitmap)
{
	HBITMAP hBitmap;
	HBITMAP hOldBitmap;
	HDC hDC = GetDC(0);
	HDC hMemDC = CreateCompatibleDC(hDC);
	UCHAR *pixel;
	int column, row;

	if (hColorBitmap)
	{
		// Create a bitmap compatable with the display.
		hBitmap = CreateCompatibleBitmap(hDC, bitmapWidth, bitmapHeight);

		// Select the bitmap into the memory DC.
		hOldBitmap = SelectObject(hMemDC, hBitmap);

		if (bitmapArray)
		{
			// Draw the Zinc bitmap onto the memory DC.
			pixel = bitmapArray;
			for (row = 0; row < bitmapHeight; row++)
			{
				for (column = 0; column < bitmapWidth; column++)
				{
					if (paletteArray)
						SetPixel(hMemDC, column, row, display->MapColor(&paletteArray[*pixel], TRUE));
					else if (*pixel <= MAX_COLORMAP_INDEX)
						SetPixel(hMemDC, column, row, display->MapColor(&display->colorMap[*pixel], TRUE));
					else
						SetPixel(hMemDC, column, row, RGB_BLACK);
					pixel++;
				}
			}
		}
		else
			// Create a white icon if no bitmap array was passed.
			BitBlt(hMemDC, 0, 0, bitmapWidth, bitmapHeight, NULL, 0, 0, WHITENESS);

		// Get the bitmap.
		hBitmap = SelectObject(hMemDC, hOldBitmap);
		*hColorBitmap = hBitmap;
	}

	if (hMonoBitmap)
	{
		hBitmap = CreateBitmap(bitmapWidth, bitmapHeight, 1, 1, NULL);
		hOldBitmap = SelectObject(hMemDC, hBitmap);
		BitBlt(hMemDC, 0, 0, bitmapWidth, bitmapHeight, NULL, 0, 0, BLACKNESS);

		if (bitmapArray)
		{
			// Draw the Zinc bitmap onto the memory DC.
			pixel = bitmapArray;
			for (row = 0; row < bitmapHeight; row++)
			{
				for (column = 0; column < bitmapWidth; column++)
				{
					if (*pixel == BACKGROUND)
						SetPixel(hMemDC, column, row, RGB_WHITE);
					pixel++;
				}
			}
		}
		else
			// Create a white icon if no bitmap array was passed.
			BitBlt(hMemDC, 0, 0, bitmapWidth, bitmapHeight, NULL, 0, 0, WHITENESS);

		// Get the bitmap.
		hBitmap = SelectObject(hMemDC, hOldBitmap);
		*hMonoBitmap = hBitmap;

		
		
		
		
		
		BITMAP bitmap;
		GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bitmap);
	}
	
	// Clean up.
	DeleteDC(hMemDC);
	ReleaseDC(0, hDC);
}

void WindowsToZincBitmap(UI_DISPLAY *display, HBITMAP hColorBitmap,
	HBITMAP hMonoBitmap, int *bitmapWidth, int *bitmapHeight,
	UCHAR **bitmapArray)
{
	BITMAP bitmap;
	HBITMAP hBitmap;
	HBITMAP hOldBitmap;
	HDC hDC = GetDC(0);
	HDC hMemDC = CreateCompatibleDC(hDC);
	UCHAR *pixel;
	int column, row;

	if (hColorBitmap)
	{
		hBitmap = hColorBitmap;
		hOldBitmap = SelectObject(hMemDC, hBitmap);
		GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bitmap);
		*bitmapWidth = bitmap.bmWidth;
		*bitmapHeight = bitmap.bmHeight;
		if (!(*bitmapArray))
			*bitmapArray = new UCHAR[bitmap.bmWidth * bitmap.bmHeight];

		pixel = *bitmapArray;
		for (row = 0; row < *bitmapHeight; row++)
			for (column = 0; column < *bitmapWidth; column++)
			{
				COLOR color = GetPixel(hMemDC, column, row);
				*pixel = 0;

				for (UCHAR i = 0; i <= MAX_COLORMAP_INDEX; i++)
					if (color == display->MapColor(&display->colorMap[i], TRUE))
					{
						*pixel = i;
						break;
					}
				pixel++;
			}

		hBitmap = SelectObject(hMemDC, hOldBitmap);
	}

	if (hMonoBitmap)
	{
		hBitmap = hMonoBitmap;
		hOldBitmap = SelectObject(hMemDC, hBitmap);
		if (!(*bitmapArray))
			*bitmapArray = new UCHAR[bitmap.bmWidth * bitmap.bmHeight];

		pixel = *bitmapArray;
		for (row = 0; row < *bitmapHeight; row++)
			for (column = 0; column < *bitmapWidth; column++)
			{
				if (GetPixel(hMemDC, column, row) != RGB_BLACK)
					*pixel = BACKGROUND;
				pixel++;
			}

		hBitmap = SelectObject(hMemDC, hOldBitmap);
	}
	
	// Clean up.
	ReleaseDC(0, hDC);
	DeleteDC(hMemDC);
}

void WindowsIconToBitmap(UI_DISPLAY *display, HICON icon,
	HBITMAP *hColorBitmap, HBITMAP *hMonoBitmap)
{
	HBITMAP hBitmap;
	HBITMAP hOldBitmap;
	HBITMAP hANDBitmap;
	HBITMAP hOldANDBitmap;
	HDC hDC = GetDC(0);
	HDC hMemDC = CreateCompatibleDC(hDC);
	HDC hANDMemDC = CreateCompatibleDC(hDC);
	int iconWidth = GetSystemMetrics(SM_CXICON);
	int iconHeight = GetSystemMetrics(SM_CYICON);

	if (hColorBitmap)
	{
		// Select a bitmap of icon size into the memory DC.
		hBitmap = CreateCompatibleBitmap(hDC, iconWidth, iconHeight);
		hOldBitmap = SelectObject(hMemDC, hBitmap);

		// Draw the icon on to the bitmap.
		DrawIcon(hMemDC, 0, 0, icon);

		// Get the bitmap.
		hBitmap = SelectObject(hMemDC, hOldBitmap);
		*hColorBitmap = hBitmap;
	}

	if (hMonoBitmap)
	{
		// Select a bitmap of icon size into the memory DC.
		hBitmap = CreateBitmap(iconWidth, iconHeight, 1, 1, NULL);
		hANDBitmap = CreateBitmap(iconWidth, iconHeight, 1, 1, NULL);
		hOldBitmap = SelectObject(hMemDC, hBitmap);
		hOldANDBitmap = SelectObject(hANDMemDC, hANDBitmap);

		// Draw the icon on to the bitmap.
		BitBlt(hMemDC, 0, 0, iconWidth, iconHeight, NULL, 0, 0, WHITENESS);
		DrawIcon(hMemDC, 0, 0, icon);
		BitBlt(hANDMemDC, 0, 0, iconWidth, iconHeight, NULL, 0, 0, BLACKNESS);
		DrawIcon(hANDMemDC, 0, 0, icon);
		BitBlt(hMemDC, 0, 0, iconWidth, iconHeight, hANDMemDC, iconWidth, iconHeight, SRCERASE);

		// Get the bitmap.
		hBitmap = SelectObject(hMemDC, hOldBitmap);
		hANDBitmap = SelectObject(hMemDC, hOldANDBitmap);
		*hMonoBitmap = hBitmap;


		BITMAP bitmap;
		GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&bitmap);
		BitBlt(hDC, 0, 0, iconWidth, iconHeight, hMemDC, iconWidth, iconHeight, SRCCOPY);
	}

	// Clean up.
	DeleteDC(hMemDC);
	DeleteDC(hANDMemDC);
	ReleaseDC(0, hDC);
}

void WindowsBitmapToIcon(UI_DISPLAY *display, HBITMAP hColorBitmap,
	HBITMAP hMonoBitmap, HICON *icon)
{
	if (!hColorBitmap)
		return;

	BITMAP colorInfo;
	BITMAP monoInfo;

	// Find size of the icon.
	GetObject(hColorBitmap, sizeof(BITMAP), (LPSTR)&colorInfo);
	DWORD size = colorInfo.bmWidth * colorInfo.bmHeight * colorInfo.bmPlanes * colorInfo.bmBitsPixel;
	UCHAR *colorBits = new UCHAR[(int)size + 1];
	UCHAR *monoBits = new UCHAR[(int)size + 1];

	// Get the color bits.
	size = GetBitmapBits(hColorBitmap, size, (LPSTR)colorBits);

	// If monochrome bitmap then use bits, else set bits to black.
	if (hMonoBitmap)
		GetObject(hMonoBitmap, sizeof(BITMAP), (LPSTR)&monoInfo);
	if (!hMonoBitmap || monoInfo.bmPlanes != 1 || monoInfo.bmBitsPixel != 1)
		memset(monoBits, 0, (size_t)size);
	else
		size = GetBitmapBits(hMonoBitmap, size, (LPSTR)monoBits);

	// Create the icon.
	*icon = CreateIcon(display->hInstance, colorInfo.bmWidth, colorInfo.bmHeight, colorInfo.bmPlanes, colorInfo.bmBitsPixel, (LPSTR)monoBits, (LPSTR)colorBits);

	// Clean up.
	delete colorBits;
	delete monoBits;
}

void ZincToWindowsIcon(UI_DISPLAY *display, int iconWidth,
	int iconHeight, UCHAR *iconArray, UI_PALETTE *paletteArray,
	HICON *icon)
{
	HBITMAP hColorBitmap;
	HBITMAP hMonoBitmap;

	ZincToWindowsBitmap(display, iconWidth, iconHeight, iconArray,
		paletteArray, &hColorBitmap, &hMonoBitmap);
	WindowsBitmapToIcon(display, hColorBitmap, hMonoBitmap, icon);
}

void WindowsToZincIcon(UI_DISPLAY *display, HICON icon, int *iconWidth,
	int *iconHeight, UCHAR **iconArray)
{
	HBITMAP hColorBitmap;
	HBITMAP hMonoBitmap;

	WindowsIconToBitmap(display, icon, &hColorBitmap, &hMonoBitmap);
	WindowsToZincBitmap(display, hColorBitmap, hMonoBitmap, iconWidth,
		iconHeight,iconArray);
}

#endif
