//	Zinc Interface Library - COMBO.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
const SYSTEM_EVENT S_RESET_TEXT		= -500;
#define CELL_RATIO	3 / 4

// ----- UIW_COMBO_BOX ------------------------------------------------------

UIW_COMBO_BOX::UIW_COMBO_BOX(int left, int top, int width, int height,
	int (*_compareFunction)(void *element1, void *element2),
	WNF_FLAGS _wnFlags, WOF_FLAGS _woFlags, WOAF_FLAGS _woAdvancedFlags) :
	UIW_WINDOW(left, top, width, height, _woFlags, _woAdvancedFlags),
	list(0, 0, width, height - 1, _compareFunction, _wnFlags | WNF_AUTO_SELECT,
	WOF_BORDER, WOAF_TEMPORARY | WOAF_NO_DESTROY)
{
	// Initialize the combo-box information.
	wnFlags = _wnFlags | WNF_AUTO_SELECT;
	UIW_COMBO_BOX::Information(INITIALIZE_CLASS, NULL);
}

UIW_COMBO_BOX::UIW_COMBO_BOX(int left, int top, int width, int height,
	int (*_compareFunction)(void *element1, void *element2), 
	WOF_FLAGS flagSetting, UI_ITEM *item) :
	UIW_WINDOW(left, top, width, height, WOF_NO_FLAGS, WOAF_NO_FLAGS),
	list(0, 0, width, height - 1, _compareFunction, WNF_NO_WRAP | WNF_AUTO_SELECT,
	WOF_BORDER, WOAF_TEMPORARY | WOAF_NO_DESTROY)
{
	// Initialize the combo-box information.
	wnFlags |= WNF_NO_WRAP | WNF_AUTO_SELECT;
	UIW_COMBO_BOX::Information(INITIALIZE_CLASS, NULL);

	// Add the string items.
	for (int i = 0; item[i].text; i++)
	{
		UIW_STRING *string = new UIW_STRING(0, 0, width, item[i].text, -1,
			item[i].flags, WOF_BORDER | WOF_AUTO_CLEAR,
			(USER_FUNCTION)item[i].data);
		if (FlagSet(flagSetting, item[i].value))
			string->woStatus |= WOS_SELECTED;
		*this + string;
	}
}

UIW_COMBO_BOX::~UIW_COMBO_BOX(void)
{
}

void UIW_COMBO_BOX::Destroy(void)
{
	// Destroy the combo box elements.
#ifdef _WINDOWS
	SendMessage(screenID, CB_RESETCONTENT, 0, 0);
#endif
	UI_LIST::Destroy();
}

#ifdef _WINDOWS
static int _comboboxOffset = -1;
static FARPROC _comboboxCallback = (FARPROC)DefWindowProc;
static FARPROC _comboboxJumpInstance = NULL;

long FAR PASCAL _export ComboboxJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UIW_COMBO_BOX *object = (UIW_COMBO_BOX *)GetWindowLong(hWnd, _comboboxOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_COMBO_BOX::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_COMBO_BOX);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_comboboxJumpInstance)
			_comboboxJumpInstance = (FARPROC)ComboboxJumpProcedure;
		UIW_WINDOW::Event(event);
		list.Event(event);
		break;

	case S_CREATE:
		{
		UI_WINDOW_OBJECT::Event(event);
		dwStyle |= (list.dwStyle & (WS_VSCROLL | WS_HSCROLL));
		RegisterObject("UIW_COMBO_BOX", "COMBOBOX", &_comboboxOffset,
			&_comboboxJumpInstance, &_comboboxCallback, NULL);
		list.screenID = screenID;
		for (object = list.First(); object; object = object->Next())
			object->Event(event);
		}
		break;

	case S_CURRENT:
	case S_REDISPLAY:
		ccode = UI_WINDOW_OBJECT::Event(event);
		if (current && SendMessage(screenID, CB_GETCURSEL, 0, 0) != list.Index(current))
			SendMessage(screenID, CB_SETCURSEL, list.Index(current), 0);
		break;

	case S_NON_CURRENT:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;

	case S_ADD_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (list.Index(object) == -1)
			list.Event(event);
		else if (object != list.current)
		{
			if (list.current)
				list.Current()->woStatus &= ~WOS_CURRENT;
			list.current = object;
			list.Current()->woStatus |= WOS_CURRENT;
			SendMessage(screenID, CB_SETCURSEL, list.Index(current), 0);
		}
		break;		// Stop at the combo-box level.

	case S_SUBTRACT_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (list.Index(object) != -1)
		{
			if (screenID)
				SendMessage(screenID, CB_DELETESTRING, UI_LIST::Index(object), 0);
			list.Subtract(object);
		}
		break;

	case S_DEINITIALIZE:
		list.Event(event);
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	default:
		{
		WORD message = event.message.message;
		MEASUREITEMSTRUCT *measure = (MEASUREITEMSTRUCT *)event.message.lParam;
		if ((ccode == L_SELECT && !FlagSet(wnFlags, WNF_BITMAP_CHILDREN) &&
			 !FlagSet(woFlags, WOF_VIEW_ONLY)) ||
			(message == WM_COMMAND && HIWORD(event.message.lParam) == 1))
		{
			ccode = UI_WINDOW_OBJECT::Event(event);
			long selection = SendMessage(screenID, CB_GETCURSEL, 0, 0);
			object = (selection >= 0) ? (UI_WINDOW_OBJECT *)list.UI_LIST::Get(LOWORD(selection)) : NULL;
			if (object)
				object->Event(UI_EVENT(L_SELECT));
		}
		else if (event.type != E_MSWINDOWS)
			ccode = UIW_WINDOW::Event(event);
		else if (message == WM_COMPAREITEM)
			ccode = list.Event(event);
		else if (message == WM_MEASUREITEM && measure->itemID == 0xFFFF)
		{
			measure->itemWidth = relative.right - relative.left - 2;
			measure->itemHeight = display->cellHeight - display->preSpace - display->postSpace;
			for (object = list.First(); object; object = object->Next())
				if (measure->itemHeight < object->relative.bottom - object->relative.top - 2)
					measure->itemHeight = object->relative.bottom - object->relative.top - 2;
		}
		else if (message == WM_MEASUREITEM)
		{
			object = (UI_WINDOW_OBJECT *)measure->itemData;
			object->Event(event);
		}
		else if (message == WM_COMMAND && !FlagSet(woStatus, WOS_CURRENT) &&
			HIWORD(event.message.lParam) == WM_KEYDOWN)
		{
			woStatus |= WOS_WINDOWS_ACTION;
			UI_WINDOW_OBJECT::Event(UI_EVENT(S_ADD_OBJECT));
			woStatus &= ~WOS_WINDOWS_ACTION;
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		}
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
static char *_tDownArrow = "[]";
static UCHAR _gDownArrow[65] =
{
	7, 9,
	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
	BACKGROUND,	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BACKGROUND,
	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
	BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
	BACKGROUND,	BACKGROUND,	BACKGROUND,	BACKGROUND,	BACKGROUND,	BACKGROUND,	BACKGROUND,
	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK
};

class UIW_COMBO_BUTTON : public UIW_BUTTON
{
public:
	UIW_COMBO_BUTTON(void) : UIW_BUTTON(0, 0, 0, NULL, BTF_NO_TOGGLE,
		WOF_JUSTIFY_CENTER | WOF_BORDER)
		{ woAdvancedFlags |= WOAF_NON_CURRENT; depth = 1; }
	~UIW_COMBO_BUTTON(void) { text = NULL; bitmapArray = NULL; }
	virtual EVENT_TYPE Event(const UI_EVENT &event);
};

class UIW_COMBO_STRING : public UIW_STRING
{
public:
	UIW_COMBO_STRING(void) : UIW_STRING(0, 0, 0, NULL, 128, STF_NO_FLAGS, WOF_BORDER) { }
	virtual EVENT_TYPE Event(const UI_EVENT &event);
};

EVENT_TYPE UIW_COMBO_BUTTON::Event(const UI_EVENT &event)
{
 	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (display->isText)
			text = _tDownArrow;
		else
		{
			UCHAR *bitmap = _gDownArrow;
			bitmapWidth = bitmap[0];
			bitmapHeight = bitmap[1];
			bitmapArray = &bitmap[2];
		}
		ccode = UIW_BUTTON::Event(event);
		break;

	case L_BEGIN_SELECT:
		ccode = UI_WINDOW_OBJECT::Event(event);
		if (ccode != S_UNKNOWN)
			eventManager->Put(UI_EVENT(L_SELECT));
		break;

	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		userFlags = FALSE;
		// Continue to default.
	default:
		ccode = UIW_BUTTON::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

EVENT_TYPE UIW_COMBO_STRING::Event(const UI_EVENT &event)
{
 	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event);
	switch (ccode)
	{
	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		if (userObject)
		{
			UI_WINDOW_OBJECT::Event(event);
			break;
		}
		// Continue to default.
	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

EVENT_TYPE UIW_COMBO_BOX::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;
	UI_WINDOW_OBJECT *oldCurrent = list.Current();

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_COMBO_BOX);
	switch (ccode)
	{
	case S_INITIALIZE:
		UIW_WINDOW::Event(event);
		list.Event(event);
		if (!display->isText && FlagSet(wnFlags, WNF_BITMAP_CHILDREN))
		{
			int maxHeight = display->cellHeight - display->preSpace - display->postSpace;
			for (UI_WINDOW_OBJECT *object = list.First(); object; object = object->Next())
				if (maxHeight <= object->relative.bottom - object->relative.top)
					maxHeight = object->relative.bottom - object->relative.top + 1;
			relative.bottom = relative.top + maxHeight;
		}
		else if (!display->isText)
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace;
		else
			relative.bottom = relative.top;
		break;

	case S_SIZE:
	case S_CREATE:
		UIW_WINDOW::Event(event);
		button->Event(event);
		button->true = true;
		button->true.left = display->isText ? true.right - 2 :
			true.right - display->cellHeight * CELL_RATIO;
		if (string)
		{
			string->Event(event);
			string->true = true;
			string->true.right = (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || FlagSet(woFlags, WOF_VIEW_ONLY)) ?
				button->true.left : button->true.left - display->cellWidth;
			if (display->isText && (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || FlagSet(woFlags, WOF_VIEW_ONLY)))
				string->true.right--;
		}
		list.Event(event);
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	case L_SELECT:
		if ((!FlagSet(wnFlags, WNF_BITMAP_CHILDREN) && !FlagSet(woFlags, WOF_VIEW_ONLY)) &&
			event.type == E_KEY && event.key.value == ' ')
			return (string->Event(event));
		{
		list.RegionConvert(list.relative, parent ? FALSE : TRUE);
		list.wnFlags = wnFlags | WNF_AUTO_SELECT;
		int height = list.relative.bottom - list.relative.top;
		int width = relative.right - relative.left;
		if (!FlagSet(wnFlags, WNF_BITMAP_CHILDREN) && !FlagSet(woFlags, WOF_VIEW_ONLY))
			width -= display->cellWidth;
		list.relative.left = true.right - width;
		list.relative.top = true.bottom;
		if (display->isText)
			list.relative.top++;
		list.relative.right = true.right;
		list.relative.bottom = list.relative.top + height;
		*windowManager + &list;
		}
		break;

	case L_FIRST:
	case L_LAST:
		if (!FlagSet(wnFlags, WNF_BITMAP_CHILDREN) && !FlagSet(woFlags, WOF_VIEW_ONLY))
			return (string->Event(event));
		// Continue to L_UP.
	case L_UP:
	case L_DOWN:
	case L_PGUP:
	case L_PGDN:
		list.woStatus |= WOS_INTERNAL_ACTION;
		list.Event(event);
		list.woStatus &= ~WOS_INTERNAL_ACTION;
		// Make sure the user function (if any) is called for the items.
		if (oldCurrent != list.Current())
		{
			oldCurrent->UI_WINDOW_OBJECT::Event(UI_EVENT(S_NON_CURRENT));
			list.Current()->UI_WINDOW_OBJECT::Event(UI_EVENT(S_CURRENT));
		}
		break;

	case S_ADD_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (list.Index(object) == -1)
			list.Event(event);
		break;		// Stop at the combo-box level.

	case S_SUBTRACT_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (list.Index(object) != -1)
			list.Event(event);
		break;		// Stop at the combo-box level.

	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		current = string;
		UIW_WINDOW::Event(event);
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || FlagSet(woFlags, WOF_VIEW_ONLY))
			oldCurrent = NULL;
		break;

	case L_VIEW:
	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		// Continue to default.
	default:
		ccode = UIW_WINDOW::Event(event);
		break;
	}

	// Redraw the current combo-item.
	if (oldCurrent != list.Current())
		UIW_COMBO_BOX::Information(RESET_SELECTION, list.Current());

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_COMBO_BOX::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	static char text[128];

	// Switch on the request.
	if (!objectID) objectID = ID_COMBO_BOX;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = list.windowID[0] = ID_COMBO_BOX;
		windowID[1] = list.windowID[1] = ID_VT_LIST;
		windowID[2] = list.windowID[2] = ID_LIST;
		windowID[3] = list.windowID[3] = ID_WINDOW;
		list.parent = this;
		list.wnFlags |= WNF_AUTO_SELECT;
#ifndef _WINDOWS
		// Add the combo fields.
		button = new UIW_COMBO_BUTTON;
		UIW_WINDOW::Add(button);
		string = new UIW_COMBO_STRING;
		UIW_WINDOW::Add(string);
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || FlagSet(woFlags, WOF_VIEW_ONLY))
			string->woFlags |= WOF_VIEW_ONLY;
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_WINDOW::Information(CHANGED_FLAGS, data, ID_COMBO_BOX);
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN))
		{
			dwStyle |= CBS_DROPDOWNLIST | CBS_OWNERDRAWVARIABLE;
			woStatus |= WOS_OWNERDRAW;
			list.woStatus |= WOS_OWNERDRAW;
		}
		else if (FlagSet(woFlags, WOF_VIEW_ONLY))
			dwStyle |= CBS_DROPDOWNLIST;
		else
			dwStyle |= CBS_DROPDOWN | CBS_AUTOHSCROLL;
		if (compareFunction)
		{
			dwStyle |= CBS_SORT;
			list.compareFunction = compareFunction;
		}
		if (screenID && request == CHANGED_FLAGS && objectID == ID_COMBO_BOX)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#else
		if (compareFunction)
			list.compareFunction = compareFunction;
		if (screenID && request == CHANGED_FLAGS && objectID == ID_COMBO_BOX)
		{
			UI_EVENT event(S_INITIALIZE);
			Event(event);
			event.type = S_CREATE;
			Event(event);
		}
#endif
		break;

	case GET_TEXT:
	case GET_BITMAP_ARRAY:
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || request != GET_TEXT)
			return (list.Information(request, data, objectID));
#ifdef _WINDOWS
		if (screenID)
			SendMessage(screenID, WM_GETTEXT, 128, (LONG)text);
		if (!data)
			return (text);
		*(char **)data = text;
		break;
#else
		return (string->Information(request, data, objectID));
#endif

	case SET_TEXT:
	case SET_BITMAP_ARRAY:
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || request != GET_TEXT)
			return (list.Information(request, data, objectID));
		strcpy(text, data ? (char *)data : "");
#ifdef _WINDOWS
		if (screenID)
		{
			InvalidateRect(screenID, NULL, TRUE);
			SendMessage(screenID, WM_SETTEXT, 0, (LONG)text);
		}
#endif
		break;

#ifndef _WINDOWS
	case RESET_SELECTION:
		if (!list.Current())
			string->DataSet("");
		else if (!FlagSet(wnFlags, WNF_BITMAP_CHILDREN) || display->isText)
			string->DataSet((char *)list.Current()->Information(GET_TEXT, NULL));
		else
		{
			UI_WINDOW_OBJECT *object = list.Current();
			string->userObject = object;
			UI_REGION _clip = object->clip;
			UI_REGION _true = object->true;
			SCREENID _screenID = object->screenID;
			WOF_FLAGS _woFlags = object->woFlags;
			WOS_STATUS _woStatus = object->woStatus;
			UI_WINDOW_OBJECT *_parent = object->parent;
			object->clip = string->clip;
			object->true = string->true;
			object->screenID = string->screenID;
			object->woFlags = (string->woFlags & ~(WOF_JUSTIFY_CENTER | WOF_JUSTIFY_RIGHT)) |
				(_woFlags & (WOF_JUSTIFY_CENTER | WOF_JUSTIFY_RIGHT));
			object->woStatus = string->woStatus;
			object->parent = FlagSet(woStatus, WOS_CURRENT) ? this : object->parent;
			object->Event(UI_EVENT(S_REDISPLAY));
			object->clip = _clip;
			object->true = _true;
			object->screenID = _screenID;
			object->woFlags = _woFlags;
			object->woStatus = _woStatus;
			object->parent = _parent;
		}
		break;
#endif

	case GET_NUMBERID_OBJECT:
	case GET_STRINGID_OBJECT:
		{
		void *match = UI_WINDOW_OBJECT::Information(request, data, objectID);
		if (!match)
			match = list.Information(request, data, objectID);
		data = match;
		}
		break;

	case PRINT_INFORMATION:
	case PRINT_USER_FUNCTION:
		UI_WINDOW_OBJECT::Information(request, data, objectID);
		list.Information(request, data, objectID);
		break;

	default:
		data = UIW_WINDOW::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_COMBO_BOX::UIW_COMBO_BOX(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_WINDOW(0, 0, 20, 5, WOF_NO_FLAGS),
	list(0, 0, 20, 5, NULL, WNF_NO_WRAP, WOF_BORDER, WOAF_TEMPORARY | WOAF_NO_DESTROY)
{
	// Initialize the text information.
	Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	UIW_COMBO_BOX::Information(INITIALIZE_CLASS, NULL);
}

void UIW_COMBO_BOX::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the combo-box information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	list.Load(NULL, directory, file);
	wnFlags |= list.wnFlags;
}

void UIW_COMBO_BOX::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the combo-box information.
#ifdef _WINDOWS
	UI_WINDOW_OBJECT::Store(name, directory, file);
#else
	UI_REGION region = relative;
	if (!display->isText && FlagSet(woStatus, WOS_GRAPHICS))
		relative.bottom = relative.top +
			(display->cellHeight - display->preSpace - display->postSpace) +
			(list.relative.bottom - list.relative.top);
	else
		relative.bottom = relative.top + (list.relative.bottom - list.relative.top);
	UI_WINDOW_OBJECT::Store(name, directory, file);
	relative = region;
#endif
	list.Store(NULL, directory, file);
}
#endif

