//	Zinc Interface Library - DATE.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "ui_gen.hpp"
#pragma hdrstop

static int _daysInMonth[] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

static int LeapYear(int year)
{
	return ((year % 400 == 0) || ((year % 4 == 0) && (year % 100 != 0)));
}

int UI_DATE::DayOfWeek(void)
{
	extern void z_date_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_date_dummy();

	return ((int)(((value + 1) % 7) + 1));
}

int UI_DATE::DaysInMonth(void)
{
	int year, month, day;
	Export(&year, &month, &day);
	if (month == 2 && LeapYear(year))
		return (29);
	return (_daysInMonth[month - 1]);
}

int UI_DATE::DaysInYear(void)
{
	int year, month, day;
	Export(&year, &month, &day);
	return (LeapYear(year) ? 366 : 365);
}

void UI_DATE::Export(int *pYear, int *pMonth, int *pDay, int *pDayOfWeek)
{
	// Compute the separate date values using algorithm 199 of CACM.
	long tValue = value - 1721119L;
	long year = (tValue * 4 - 1) / 146097L;
	tValue = tValue * 4 - 1 - 146097L * year;
	long day = tValue / 4;
	tValue = (day * 4 + 3) / 1461L;
	day = day * 4 + 3 - 1461L * tValue;
	day = (day + 4) / 4;
	long month = (5 * day - 3) / 153;
	day = 5 * day - 3 - 153 * month;
	day = (day + 5) / 5;
	year = 100 * year + tValue;
	if (month < 10)
		month += 3;
	else
	{
		month -= 9;
		year++;
	}

	// Set the integer variables.
	if (pYear)
		*pYear = (int)year;
	if (pMonth)
		*pMonth = (int)month;
	if (pDay)
		*pDay = (int)day;
	if (pDayOfWeek)
		*pDayOfWeek = DayOfWeek();
}

void UI_DATE::Export(char *string, DTF_FLAGS dtFlags)
{
	// Make sure the country information is initialized.
	if (!UI_INTERNATIONAL::initialized)
		UI_INTERNATIONAL::Initialize();
	char *tString = string;

	// Determine the seperator character.
	char *yearSeparator, *monthSeparator, *daySeparator;
	if (FlagSet(dtFlags, DTF_MILITARY_FORMAT))
	{
		dtFlags &= (DTF_MILITARY_FORMAT | DTF_UPPER_CASE);
		dtFlags |= DTF_SHORT_MONTH | DTF_ZERO_FILL;
		yearSeparator = monthSeparator = daySeparator = " ";
	}
	else if (FlagSet(dtFlags, DTF_DASH))
		yearSeparator = monthSeparator = daySeparator = "-";
	else if (FlagSet(dtFlags, DTF_SLASH))
		yearSeparator = monthSeparator = daySeparator = "/";
	else
		yearSeparator = monthSeparator = daySeparator = dateSeparator;

	// Get the integer date values.
	char year[32], month[32], day[32];
	int yearValue, monthValue, dayValue, dayOfWeekValue;
	Export(&yearValue, &monthValue, &dayValue, &dayOfWeekValue);

	// Determine the year value.
	if (FlagSet(dtFlags, DTF_SHORT_YEAR) || (FlagSet(dtFlags, DTF_ZERO_FILL) && yearValue < 10))
		sprintf(year, "%02d", yearValue % 100);
	else
		sprintf(year, "%d", yearValue);

	// Determine the month value.
	if (FlagSet(dtFlags, DTF_ALPHA_MONTH | DTF_SHORT_MONTH))
	{
		int value = FlagSet(dtFlags, DTF_SHORT_MONTH) ? 0 : 1;
		strcpy(month, monthTable[--monthValue * 2 + value]);
		if (FlagSet(dtFlags, DTF_MILITARY_FORMAT))
			month[3] = '\0';
		else
		{
			monthSeparator = " ";
			daySeparator = ", ";
		}
	}
	else if (FlagSet(dtFlags, DTF_ZERO_FILL) && monthValue < 10)
		sprintf(month, "%02d", monthValue);
	else
		sprintf(month, "%d", monthValue);

	// Determine the day value.
	if (FlagSet(dtFlags, DTF_ZERO_FILL) && dayValue < 10)
		sprintf(day, "%02d", dayValue);
	else
		sprintf(day, "%d", dayValue);

	// Add the day of week value to the date.
	if (FlagSet(dtFlags, DTF_DAY_OF_WEEK | DTF_SHORT_DAY))
	{
		int value = FlagSet(dtFlags, DTF_SHORT_DAY) ? 0 : 1;
		tString += sprintf(tString, "%s ", dayTable[(dayOfWeekValue - 1) * 2 + value]);
	}

	// Format the date according to the country information.
	if (FlagSet(dtFlags, DTF_JAPANESE_FORMAT))
		sprintf(tString, "%s%s%s%s%s", year, yearSeparator, month, monthSeparator, day);
	else if (FlagSet(dtFlags, DTF_EUROPEAN_FORMAT | DTF_MILITARY_FORMAT))
		sprintf(tString, "%s%s%s%s%s", day, daySeparator, month, monthSeparator, year);
	else if (FlagSet(dtFlags, DTF_US_FORMAT))
		sprintf(tString, "%s%s%s%s%s", month, monthSeparator, day, daySeparator, year);
	else if (dateFormat == 2)
		sprintf(tString, "%s%s%s%s%s", year, yearSeparator, month, monthSeparator, day);
	else if (dateFormat == 1)
		sprintf(tString, "%s%s%s%s%s", day, daySeparator, month, monthSeparator, year);
	else
		sprintf(tString, "%s%s%s%s%s", month, monthSeparator, day, daySeparator, year);
	if (FlagSet(dtFlags, DTF_UPPER_CASE))
		strupr(string);
}

void UI_DATE::Export(int *packedDate)
{
	int year, month, day;

	Export(&year, &month, &day);
	*packedDate = ((year - 1980) << 9) | (month << 5) | day;
}

DTI_RESULT UI_DATE::Import(void)
{
	// Set the date values according to the system date.
#ifdef __ZTC__
	struct dos_date_t info;
	dos_getdate(&info);
	Import(info.year, info.month, info.day);
#endif
#ifdef _MSC_VER
	struct _dosdate_t info;
	_dos_getdate(&info);
	Import(info.year, info.month, info.day);
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	struct date info;
	getdate(&info);
	Import(info.da_year, info.da_mon, info.da_day);
#endif

	return (DTI_OK);
}

DTI_RESULT UI_DATE::Import(const UI_DATE &date)
{
	// Set the date value according to the passed date.
	value = date.value;

	return (DTI_OK);
}

DTI_RESULT UI_DATE::Import(int year, int month, int day)
{
	// Make sure year, month and day are valid.
	if (!year || !month || !day)
	{
		UI_DATE date;
		date.Export(year ? NULL : &year, month ? NULL : &month, day ? NULL : &day);
	}
	if (year < 100)
		year += 1900;

	// Compute the julian date from three integer values using algorithm 199 of CACM.
	if (year < 0 || month < 0 || month > 12 || day < 0 ||
		(month == 2 && day == 29 && !LeapYear(year)) ||
		((month != 2 || day != 29) && day > _daysInMonth[month-1]))
		return (DTI_INVALID);
	if (month > 2)	      
		month -= 3;
	else
	{
		month += 9;
		year--;
	}
	value = ((146097L * (year / 100)) / 4) + ((1461L * (year % 100)) / 4) + (153L * month + 2) / 5 + day + 1721119L;

	return (DTI_OK);
}

DTI_RESULT UI_DATE::Import(const char *string, DTF_FLAGS dtFlags)
{
	// Make sure the country information is initialized.
	if (!UI_INTERNATIONAL::initialized)
		UI_INTERNATIONAL::Initialize();

	// Check for a system date.
	if ((!string || !string[0]) && FlagSet(dtFlags, DTF_SYSTEM))
		return (UI_DATE::Import());

	// Set up the date variables and determine their proper order.
	int date[3] = { 0, 0, 0 };
	int year, month, day;
	if (FlagSet(dtFlags, DTF_JAPANESE_FORMAT))
		year = 0, month = 1, day = 2;
	else if (FlagSet(dtFlags, DTF_EUROPEAN_FORMAT | DTF_MILITARY_FORMAT))
		day = 0, month = 1, year = 2;
	else if (FlagSet(dtFlags, DTF_US_FORMAT))
		month = 0, day = 1, year = 2;
	else if (dateFormat == 2)
		year = 0, month = 1, day = 2;
	else if (dateFormat == 1)
		day = 0, month = 1, year = 2;
	else
		month = 0, day = 1, year = 2;

	// Parse out the string date.
	for (int i = 0; i < 3 && *string; i++)
	{
		while (*string && !isdigit(*string) && !isalpha(*string))
			string++;
		if (isdigit(*string))
		{
			for (int value = 0; *string && isdigit(*string); *string++)
				value = value * 10 + *string - '0';
			date[i] = value;
		}
		else if (isalpha(*string))
		{
			int found = FALSE;
			const char *tString = string;
			while (*string && isalpha(*string))
				string++;
			for (int j = 0; j < 24 && !found; j++)
				if (!strnicmp(tString, monthTable[j], (int)(string - tString)))
				{
					date[i] = (j / 2) + 1;
					if (year == i)
						year = month;
					else if (day == i)
						day = month;
					month = i;
					found = TRUE;
				}
			if (!found)
				i--;
		}
	}

	// Set the date according to the parsed values.
	return (Import(date[year], date[month], date[day]));
}

DTI_RESULT UI_DATE::Import(int date)
{
	return(Import(1980 + (((int)(date & 0xFE00)) >> 9), ((int)(date & 0x01E0)) >> 5, date & 0x001F));
}
