//	Zinc Interface Library - FMTSTR.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop

// ----- UIW_FORMATTED_STRING -----------------------------------------------

UIW_FORMATTED_STRING::UIW_FORMATTED_STRING(int left, int top, int width,
	char *_compressedText, char *_editMask, char *_deleteText,
	WOF_FLAGS _woFlags, USER_FUNCTION _userFunction) :
	UIW_STRING(left, top, width, NULL, 64, STF_NO_FLAGS, _woFlags, _userFunction)
{
	// Initialize the formatted string information.
	compressedText = new char[64];
	editMask = new char[64];
	deleteText = new char[64];
	strcpy(compressedText, _compressedText ? _compressedText : "");
	strcpy(editMask, _editMask ? _editMask : "");
	strcpy(deleteText, _deleteText ? _deleteText : "");
	UIW_FORMATTED_STRING::Information(INITIALIZE_CLASS, NULL);
}

UIW_FORMATTED_STRING::~UIW_FORMATTED_STRING(void)
{
	// Restore the formatted string information.
	delete compressedText;
	delete editMask;
	delete deleteText;
}

char *UIW_FORMATTED_STRING::DataGet(int _compressedText)
{
	// Return the formatted string.
	UIW_STRING::DataGet();
	return (_compressedText ? compressedText : text);
}

void UIW_FORMATTED_STRING::DataSet(char *_text)
{
	// Reset the formatted string.
	if (text == _text || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = _text;
	else
		strcpy(text, _text ? _text : "");
	UIW_STRING::DataSet(text);
}

EVENT_TYPE UIW_FORMATTED_STRING::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_FORMATTED_STRING);
	switch (ccode)
	{
	case L_SELECT:
	case S_NON_CURRENT:
		if (!FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			UIW_STRING::DataGet();
			Import(text);
			Export(text, TRUE);
			UIW_STRING::DataSet(text);
		}
		ccode = UIW_STRING::Event(event);
		break;

	case S_CREATE:
		Export(text, TRUE);
		// Continue to default.
	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void UIW_FORMATTED_STRING::Export(char *destination, int expanded)
{
	// Export the formatted string value.
	if (expanded)
	{
		char *source = compressedText;
		for (int i = 0; editMask[i]; i++)
			if (editMask[i] != 'L' && *source)
				*destination++ = *source++;
			else
				*destination++ = deleteText[i];
		*destination = '\0';
	}
	else
		strcpy(destination, compressedText);
}

FMI_RESULT UIW_FORMATTED_STRING::Import(char *source)
{
	// Import the formatted string value.
	char *destination = compressedText;
	for (char *mask = editMask; *mask && *source; mask++)
		if (*mask != 'L')
		{
			for (int processed = FALSE; !processed && *source; source++)
			{
				int editValue = *mask;
				int textValue = *source;
				int isAlpha = (textValue == ' ' || isalpha(textValue));
				int isDigit = isdigit(textValue);
				if (isupper(editValue))
					textValue = toupper(textValue);
				editValue = toupper(editValue);
				if ((editValue == 'X' && textValue >= ' ' && textValue <= '~') ||
					(editValue == 'N' && isDigit) ||
					(editValue == 'A' && isAlpha) ||
					(editValue == 'C' && (isDigit || isAlpha)))
				{
					*destination++ = *source = textValue;
					processed = TRUE;
				}
			}
		}
	*destination = '\0';

	// Return success.
	return (FMI_OK);
}

void *UIW_FORMATTED_STRING::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_FORMATTED_STRING;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_FORMATTED_STRING;
		windowID[1] = ID_STRING;
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_STRING::Information(CHANGED_FLAGS, data, ID_FORMATTED_STRING);
		if (screenID && request == CHANGED_FLAGS && objectID == ID_FORMATTED_STRING)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#endif
		break;

	default:
		data = UIW_STRING::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_FORMATTED_STRING::UIW_FORMATTED_STRING(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_STRING(0, 0, 10, NULL, 64, STF_NO_FLAGS, WOF_NO_FLAGS)
{
	// Initialize the formatted string information.
	compressedText = new char[64];
	editMask = new char[64];
	deleteText = new char[64];
	UIW_FORMATTED_STRING::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_FORMATTED_STRING::Information(INITIALIZE_CLASS, NULL);
}

void UIW_FORMATTED_STRING::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the formatted string information.
	UIW_STRING::Load(name, directory, file);
	file->Load(compressedText, 64);
	file->Load(editMask, 64);
	file->Load(deleteText, 64);
}

void UIW_FORMATTED_STRING::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the formatted string information.
	UIW_STRING::Store(name, directory, file);
	file->Store(compressedText);
	file->Store(editMask);
	file->Store(deleteText);
}
#endif

