//	Zinc Interface Library - ICON.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop

// ----- UIW_ICON -----------------------------------------------------------

UIW_ICON::UIW_ICON(int left, int top, char *_iconName, char *_title,
	ICF_FLAGS _icFlags, WOF_FLAGS _woFlags, USER_FUNCTION _userFunction) :
	UI_WINDOW_OBJECT(left, top, 0, 0, _woFlags, WOAF_NO_FLAGS),
	icFlags(_icFlags), iconWidth(32), iconHeight(32), iconArray(NULL)
{
	// Initialize the icon information.
	title = ui_strdup(_title);
	if (title)
	{
		ui_strrepc(title, '~', '&');
		char *tHotKey = strchr(title, '&');
		if (tHotKey)
			hotKey = tolower(tHotKey[1]);
	}
	iconName = ui_strdup(_iconName);
	userFunction = _userFunction;
	UIW_ICON::Information(INITIALIZE_CLASS, NULL);
}

UIW_ICON::~UIW_ICON(void)
{
	if (title)
		delete title;
	if (iconName)
		delete iconName;
	if (iconArray)
		delete iconArray;
#ifdef _WINDOWS
	if (icon && FlagSet(woStatus, WOS_DESTROY_ICON))
		DestroyIcon(icon);
#endif
}

char *UIW_ICON::DataGet(void)
{
 	return (title);
}

void UIW_ICON::DataSet(char *_title)
{
	// Reset the title.
	if (title && title != _title && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete title;
	if (title == _title || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		title = _title;
	else
		title = ui_strdup(_title);

	// Check for a hotkey.
	if (title)
	{
		ui_strrepc(title, '~', '&');
		char *tHotKey = strchr(title, '&');
		if (tHotKey)
			hotKey = tolower(tHotKey[1]);
	}

	// Redisplay the string.
#ifdef _WINDOWS
	if (screenID)
		SendMessage(screenID, WM_SETTEXT, 0, (LONG)title);
#else
	if (screenID && !FlagSet(woStatus, WOS_INTERNAL_ACTION))
		Event(UI_EVENT(S_REDISPLAY));
#endif
}

#ifdef _WINDOWS
extern void ZincToWindowsIcon(UI_DISPLAY *display, int iconWidth,
	int iconHeight, UCHAR *iconArray, UI_PALETTE *paletteArray,
	HICON *icon);
extern void WindowsToZincIcon(UI_DISPLAY *display, HICON icon,
	int *iconWidth, int *iconHeight, UCHAR **iconArray);

static int _iconOffset = -1;
static FARPROC _iconCallback = (FARPROC)DefWindowProc;
static FARPROC _iconJumpInstance = NULL;

long FAR PASCAL _export IconJumpProcedure(HWND hWnd, WORD wMsg,
	WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _iconOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_ICON::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	const int ICON_OFFSET = 4;

	// Initialize the variables.
	DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
	UI_REGION region = { draw->rcItem.left, draw->rcItem.top,
		draw->rcItem.right, draw->rcItem.bottom };
	if (draw->CtlType != ODT_BUTTON)
		region.left += relative.left;
	UI_MSWINDOWS_DISPLAY::hDC = draw->hDC;

	// Check the icon status.
	if (FlagSet(draw->itemState, ODS_SELECTED))
		woStatus |= WOS_SELECTED;
	else
		woStatus &= ~WOS_SELECTED;
	if (FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS))
		woStatus |= WOS_CURRENT;
	else if (FlagSet(draw->itemAction, ODA_FOCUS))
		woStatus &= ~WOS_CURRENT;

	// Draw the icon.
	if (FlagSet(draw->itemAction, ODA_DRAWENTIRE | ODA_SELECT))
	{
		if (draw->CtlType == ODT_BUTTON)
			lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ? PM_SELECTED : PM_ACTIVE, ID_ICON);
		else
		{
			lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ? PM_SELECTED : PM_ACTIVE, ID_WINDOW_OBJECT);
			HBRUSH fillBrush = CreateSolidBrush(display->MapColor(lastPalette, FALSE));
			FillRect(draw->hDC, &draw->rcItem, fillBrush);
			DeleteObject(fillBrush);
		}
		if (title && FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		{
			DrawText(ID_DIRECT, region, title, lastPalette, FALSE, ccode);
			region.left = region.right - iconWidth - ICON_OFFSET;
		}
		else if (title && FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		{
			region.top += iconHeight;
			DrawText(ID_DIRECT, region, title, lastPalette, FALSE, ccode);
			region.top -= iconHeight;
			region.bottom = region.top + iconHeight;
			region.right = (region.left + region.right + iconWidth + 1) / 2;
			region.left = region.right - iconWidth;
		}
		else if (draw->CtlType != ODT_BUTTON)
			region.left += ICON_OFFSET;
		if (icon)
		{
			DrawIcon(draw->hDC, region.left, region.top, icon);
			region.left += iconWidth + ICON_OFFSET;
		}
		if (title && !FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER))
			DrawText(ID_DIRECT, region, title, lastPalette, FALSE, ccode);
	}
	if (FlagSet(draw->itemAction, ODA_FOCUS))
		DrawFocusRect(draw->hDC, &draw->rcItem);

	return (TRUE);
}

EVENT_TYPE UIW_ICON::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_ICON);
	switch (ccode)
	{
	case S_INITIALIZE:
		// Read in the icon if so specified.
		if (!icon && iconArray)
		{
			ZincToWindowsIcon(display, iconWidth, iconHeight, iconArray, NULL, &icon);
			woStatus |= WOS_DESTROY_ICON;
		}
		// Initialize the icon information.
		if (!_iconJumpInstance)
			_iconJumpInstance = (FARPROC)IconJumpProcedure;
		if (FlagSet(woFlags, WOF_SUPPORT_OBJECT))
			icFlags |= ICF_MINIMIZE_OBJECT;
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			parent->Information(SET_ICON, &icon);
		// Continue to S_SIZE.
	case S_SIZE:
		// Compute the icon region.
		if (ccode == S_SIZE)
			UI_WINDOW_OBJECT::Event(event);
		relative.right = relative.left + iconWidth + 2;
		relative.bottom = relative.top + iconHeight + 2;
		if (title && FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		{
			relative.bottom += display->TextHeight(title, screenID, font);
			if (relative.right - relative.left < display->TextWidth(title, screenID, font))
				relative.right = relative.left + display->TextWidth(title, screenID, font);
		}
		true = relative;
		break;

	case S_CREATE:
		if (FlagSet(woFlags, WOF_SUPPORT_OBJECT))
			break;
		else if (parent->Inherited(ID_COMBO_BOX))
		{
			DWORD index = SendMessage(parent->screenID, CB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)title);
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, CB_SETCURSEL, LOWORD(index), 0);
		}
		else if (parent->Inherited(ID_LIST))
		{
			DWORD index = SendMessage(parent->screenID, LB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)title);
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, LB_SETSEL, TRUE, index);
		}
		else
			RegisterObject("UIW_ICON", "STATIC", &_iconOffset,
				&_iconJumpInstance, &_iconCallback, title);
		break;

	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_PAINT)
		{
			PAINTSTRUCT ps;
			UI_MSWINDOWS_DISPLAY::hDC = BeginPaint(screenID, &ps);
			DrawIcon(UI_MSWINDOWS_DISPLAY::hDC, 0, 0, icon);
			if (title)
			{
				RECT rect;
				GetClientRect(screenID, &rect);
				UI_REGION region;
				region.left = rect.left;
				region.top = rect.bottom - display->TextHeight(title, screenID, font);
				region.right = rect.right;
				region.bottom = rect.bottom;
				SelectObject(UI_MSWINDOWS_DISPLAY::hDC, UI_MSWINDOWS_DISPLAY::fontTable[font]);
				DrawText(ID_DIRECT, region, title, lastPalette, TRUE, ccode);
			}
			EndPaint(screenID, &ps);
		}
		else if (message == WM_DRAWITEM)
			ccode = DrawItem(event, ccode);
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_ICON::DrawItem(const UI_EVENT &, EVENT_TYPE ccode)
{
	// Check for text mode operation.
	if (display->isText)
	{
		UI_REGION region = true;
		DrawText(screenID, region, title, lastPalette, FALSE, ccode);
		return (ccode);
	}

	// Draw the icon.
	UI_REGION region = iconRegion;
	if (FlagSet(woFlags, WOF_BORDER))
		DrawBorder(screenID, region, FALSE, ccode);
	if (iconArray)
	{
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT) && parent && parent->parent)
			display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
		else if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			display->Rectangle(screenID, region, display->backgroundPalette, 0, TRUE, FALSE, &clip);
		display->Bitmap(screenID, region.left, region.top, iconWidth, iconHeight, iconArray, NULL, &clip);
	}
	else
		display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
	// Draw the title.
	if (title)
	{
		region = titleRegion;
		DrawText(screenID, region, title, lastPalette, TRUE, ccode);
	}

	// Return the control code.
	return (ccode);
}

EVENT_TYPE UIW_ICON::Event(const UI_EVENT &event)
{
	static initializedTime = FALSE;
	static UI_TIME lastTime;
	if (!initializedTime)
	{
		lastTime.Import();
		initializedTime = TRUE;
	}

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_ICON);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (display->isText)
			iconWidth = iconHeight = 1;
		else
			iconWidth = iconHeight = 32;
		if (FlagSet(woFlags, WOF_SUPPORT_OBJECT))
			icFlags |= ICF_MINIMIZE_OBJECT;
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
		{
			RegionConvert(relative, TRUE);
			relative.left = parent->relative.left;
			relative.top = parent->relative.top;
		}
		else
			UI_WINDOW_OBJECT::Event(event);
		relative.right = relative.left + iconWidth - 1;
		relative.bottom = relative.top + iconHeight - 1;
		if (!display->isText && title)
		{
			relative.bottom += display->cellHeight / 4;
			relative.bottom += display->TextHeight(title, screenID, font);
			int width = display->TextWidth(title, screenID, font);
			if (width > iconWidth)
				relative.right = relative.left + width + display->cellWidth;
		}
		else if (title)
		{
			relative.right = relative.left + display->TextWidth(title) - 1;
			relative.bottom = relative.top;
		}
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
		{
			true = relative;
			parent->Information(SET_ICON, this);
		}
		break;

	case S_SIZE:
	case S_CREATE:
		if (!FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			ccode = UI_WINDOW_OBJECT::Event(event);
		else
			screenID = parent->screenID;
		if (!display->isText && title)
		{
			iconRegion.left = (true.right - true.left > iconWidth) ?
				(true.left + true.right - iconWidth) / 2 : true.left;
			iconRegion.top = true.top;
			iconRegion.right = iconRegion.left + iconWidth - 1;
			iconRegion.bottom = iconRegion.top + iconHeight - 1;
			titleRegion.bottom = true.bottom;
			titleRegion.top = titleRegion.bottom - display->TextHeight(title, screenID, font);
			titleRegion.left = true.left;
			titleRegion.right = true.right;
		}
		else
			iconRegion = true;
		break;

	case S_REGION_DEFINE:
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT) && display->isText)
		{
			UI_REGION region;
			region.left = Max(true.left, clip.left);
			region.top = Max(true.top, clip.top);
			region.right = Min(true.right, clip.right);
			region.bottom = Min(true.bottom, clip.bottom);
			display->RegionDefine(screenID, region);
		}
		else if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
		{
			UI_REGION region;
			region.left = Max(iconRegion.left, clip.left);
			region.top = Max(iconRegion.top, clip.top);
			region.right = Min(iconRegion.right, clip.right);
			region.bottom = Min(iconRegion.bottom, clip.bottom);
			display->RegionDefine(screenID, region);
			region.left = Max(titleRegion.left, clip.left);
			region.top = Max(titleRegion.top, clip.top);
			region.right = Min(titleRegion.right, clip.right);
			region.bottom = Min(titleRegion.bottom, clip.bottom);
			display->RegionDefine(screenID, region);
		}
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
		{
			if (FlagSet(parent->woStatus, WOS_MINIMIZED))
				woFlags &= ~WOF_NON_SELECTABLE;
			else
				woFlags |= WOF_NON_SELECTABLE;
		}
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT) && !FlagSet(parent->woStatus, WOS_MINIMIZED))
			woStatus &= ~WOS_REDISPLAY;
		else if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			DrawItem(event, ccode);
			woStatus &= ~WOS_REDISPLAY;
		}
		break;

	case L_BEGIN_SELECT:
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT) &&
			(iconRegion.Overlap(event.position) || titleRegion.Overlap(event.position)))
		{
			UI_TIME time;
			if (time - lastTime < doubleClickRate)
				parent->Event(UI_EVENT(L_RESTORE, 0x0001));
			else
				parent->Event(UI_EVENT(L_MOVE, 0x0001, event.position));
			lastTime = time;
		}
		else if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			return (S_UNKNOWN);
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;

	case L_END_SELECT:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (userFunction &&
			(iconRegion.Overlap(event.position) || titleRegion.Overlap(event.position)))
			ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_ICON::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
#ifdef _WINDOWS
	// Initialize the standard windows icons.
	struct ICON_ITEM
	{
		char *name;
		LPSTR identification;
	} _iconTable[] = {
		{ "APPLICATION", 	(LPSTR)IDI_APPLICATION },
		{ "ASTERISK", 		(LPSTR)IDI_ASTERISK },
		{ "EXCLAMATION", 	(LPSTR)IDI_EXCLAMATION },
		{ "HAND", 			(LPSTR)IDI_HAND },
		{ "QUESTION", 		(LPSTR)IDI_QUESTION },
		{ 0, 0 } };
#endif

	// Switch on the request.
	if (!objectID) objectID = ID_ICON;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_ICON;
		if (FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			woFlags |= WOF_SUPPORT_OBJECT;
		font = FNT_SMALL_FONT;
		woFlags |= WOF_JUSTIFY_CENTER;
#ifdef _WINDOWS
		icon = 0;
		// Read in the icon if so specified.
		if (iconName && display)
		{
			for (int i = 0; _iconTable[i].name && !icon; i++)
				if (!stricmp(iconName, _iconTable[i].name))
					icon = LoadIcon(0, _iconTable[i].identification);
			if (!icon)
				icon = LoadIcon(display->hInstance, iconName);
			if (icon && iconArray)
			{
				delete iconArray;
				iconArray = NULL;
			}
		}
#endif
#ifdef ZIL_PERSISTENCE
#ifdef _WINDOWS
		if (!icon && iconName && !iconArray && defaultStorage && !defaultStorage->storageError)
#else
		if (iconName && !iconArray && defaultStorage && !defaultStorage->storageError)
#endif
		{
			defaultStorage->ChDir("~UI_ICON");
			UI_STORAGE_OBJECT iFile(*defaultStorage, iconName, ID_ICON_IMAGE, UIS_READ);
			if (!iFile.objectError)
			{
				short _value; iFile.Load(&_value); iconWidth = _value;
				iFile.Load(&_value); iconHeight = _value;
				iconArray = new UCHAR[iconWidth * iconHeight];
				iFile.Load(iconArray, sizeof(UCHAR), iconWidth * iconHeight);
			}
		}
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (parent && FlagSet(icFlags, ICF_MINIMIZE_OBJECT))
			parent->Information(SET_ICON, &icon);
		else
		{
			if (request == CHANGED_FLAGS)
				UI_WINDOW_OBJECT::Information(CHANGED_FLAGS, data, ID_ICON);
			if (screenID && request == CHANGED_FLAGS && objectID == ID_ICON)
			{
				DestroyWindow(screenID);
				Event(UI_EVENT(S_CREATE));
				SetWindowPos(screenID, previous ? Previous()->screenID : 0,
					0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
			}
		}
#endif
		break;

	case GET_TEXT:
		if (!data)
			return (title);
		*(char **)data = title;
		break;

	case SET_TEXT:
		if (title)
			delete title;
		title = ui_strdup((char *)data);
#ifdef _WINDOWS
		if (screenID)
		{
			InvalidateRect(screenID, NULL, TRUE);
			SendMessage(screenID, WM_SETTEXT, 0, (LONG)title);
		}
#endif
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_ICON)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &icFlags;
		else if (request == GET_FLAGS)
			*(ICF_FLAGS *)data = icFlags;
		else if (request == SET_FLAGS)
			icFlags |= *(ICF_FLAGS *)data;
		else
			icFlags &= ~(*(ICF_FLAGS *)data);
		break;

	case GET_ICON_ARRAY:
#ifdef _WINDOWS
		if (!iconArray)
			WindowsToZincIcon(display, icon, &iconWidth, &iconHeight, &iconArray);
#endif
		if (!data)
			return (iconArray);
		*(UCHAR **)data = iconArray;
		break;

	case SET_ICON_ARRAY:
		if (iconArray)
			delete iconArray;
		if (data)
		{
			iconArray = new UCHAR[iconWidth * iconHeight];
			memcpy(iconArray, (UCHAR *)data, iconWidth * iconHeight);
		}
		else
			iconArray = NULL;
#ifdef _WINDOWS
		if (icon && FlagSet(woStatus, WOS_DESTROY_ICON))
			DestroyIcon(icon);
		if (iconArray)
			ZincToWindowsIcon(display, iconWidth, iconHeight, iconArray, NULL, &icon);
		else
			icon = 0;
#endif
		woStatus |= WOS_DESTROY_ICON;
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_ICON::UIW_ICON(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UI_WINDOW_OBJECT(0, 0, 15, 1, WOF_NO_FLAGS, WOAF_NO_FLAGS),
	iconWidth(32), iconHeight(32), iconArray(NULL)
{
	// Initialize the icon information.
	UIW_ICON::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_ICON::Information(INITIALIZE_CLASS, NULL);
}

void UIW_ICON::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the icon information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	file->Load(&icFlags);
	file->Load(&title);
	file->Load(&iconName);

	// Read the bitmap information.
	if (iconName)
	{
		directory->ChDir("~UI_ICON");
		UI_STORAGE_OBJECT iFile(*directory, iconName, ID_ICON_IMAGE, UIS_READ);
		if (!iFile.objectError)
		{
			short _value; iFile.Load(&_value); iconWidth = _value;
			iFile.Load(&_value); iconHeight = _value;
			iconArray = new UCHAR[iconWidth * iconHeight];
			iFile.Load(iconArray, sizeof(UCHAR), iconWidth * iconHeight);
		}
	}
}

void UIW_ICON::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the icon information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	file->Store(icFlags);
	file->Store(title);
	file->Store(iconName);

	// Store the bitmap information.
#ifdef _WINDOWS
	if (iconName && icon && !iconArray)
		WindowsToZincIcon(display, icon, &iconWidth, &iconHeight, &iconArray);
#endif
	if (iconName && iconArray)
	{
		if (directory->ChDir("~UI_ICON"))
		{
			directory->ChDir("~");
			directory->MkDir("UI_ICON");
			directory->ChDir("~UI_ICON");
		}
		UI_STORAGE_OBJECT iFile(*directory, iconName, ID_ICON_IMAGE, UIS_CREATE | UIS_READWRITE);
		if (!iFile.objectError)
		{
			short _value = iconWidth; iFile.Store(_value);
			_value = iconHeight; iFile.Store(_value);
			iFile.Store(iconArray, sizeof(UCHAR), iconWidth * iconHeight);
		}
	}
}
#endif

