//	Zinc Interface Library - INT.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
#define MAX_LENGTH	64

// ----- UIW_INTEGER --------------------------------------------------------

UIW_INTEGER::UIW_INTEGER(int left, int top, int width, int *_number,
	const char *_range, NMF_FLAGS _nmFlags, WOF_FLAGS _woFlags,
	USER_FUNCTION _userFunction) :
	UIW_STRING(left, top, width, NULL, MAX_LENGTH, STF_NO_FLAGS, _woFlags, _userFunction),
	number(NULL), nmFlags(_nmFlags)
{
	// Initialize the integer information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		number = new int;
	else
	{
		text = new char[MAX_LENGTH+1];
		text[0] = '\0';
	}
	range = ui_strdup(_range);
	UIW_INTEGER::Information(INITIALIZE_CLASS, NULL);
	UIW_INTEGER::DataSet(_number);
}

UIW_INTEGER::~UIW_INTEGER(void)
{
	// Restore the integer information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete number;
	else
		delete text;
	if (range)
		delete range;
}

int UIW_INTEGER::DataGet(void)
{
	// Return the integer value.
	UIW_STRING::DataGet();
	*number = atoi(text);
	return (*number);
}

void UIW_INTEGER::DataSet(int *_number)
{
	// Reset the integer value.
	if (number == _number || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		number = _number;
	else if (_number)
		*number = *_number;
	else
		*number = 0;
	sprintf(text, "%d", *number);
	UIW_STRING::DataSet(text);
}

EVENT_TYPE UIW_INTEGER::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_INTEGER);
	switch (ccode)
	{
	case L_SELECT:
	case S_NON_CURRENT:
		if (!FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			int tNumber = *number;				// Keep a temporary value in case of error.
			UIW_STRING::DataGet();
			*number = atoi(text);
			ccode = UIW_STRING::Event(event);	// Call the user or valitime function.
			if (ccode == -1)
				*number = tNumber;				// An error occurred, restore the old value.
			else
				woStatus &= ~WOS_UNANSWERED;
			sprintf(text, "%d", *number);
			UIW_STRING::DataSet(text);
		}
		else
			ccode = UIW_STRING::Event(event);
		break;

	case S_CREATE:
		sprintf(text, "%d", *number);
		// Continue to default.
	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void *UIW_INTEGER::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_INTEGER;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_INTEGER;
		windowID[1] = ID_NUMBER;
		windowID[2] = ID_STRING;
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_STRING::Information(CHANGED_FLAGS, data, ID_INTEGER);
		if (screenID && request == CHANGED_FLAGS && objectID == ID_INTEGER)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#endif
		break;

	default:
		data = UIW_STRING::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

int UIW_INTEGER::Validate(int processError)
{
	// Set up the number error messages.
	static struct
	{
		NMI_RESULT errorCode;
		char *message;
	} errorTable[] =
	{
		{ NMI_OUT_OF_RANGE, 	"The number %s must be in the range %s."	},
		{ NMI_OK,				0											}
	};
#ifdef ZIL_PERSISTENCE
	// Check for internationalization.
#endif

	// Check for an absolute number error.
	char *stringNumber = (char *)UIW_STRING::Information(GET_TEXT, NULL);
	int currentNumber = atoi(stringNumber);
	NMI_RESULT errorCode = NMI_OK;

	// Check for a range error.
	if (range && errorCode == NMI_OK)
		errorCode = NMI_OUT_OF_RANGE;
	for (char *tRange = range; tRange && errorCode == NMI_OUT_OF_RANGE; )
	{
		char minNumber[MAX_LENGTH], maxNumber[MAX_LENGTH];
		tRange = ParseRange(tRange, minNumber, maxNumber);
		if (currentNumber >= atoi(minNumber) &&
			currentNumber <= atoi(maxNumber))
			errorCode = NMI_OK;
	}

	// Process the error code.
	woStatus &= ~WOS_INVALID;
	if (errorCode == NMI_OK)			// Set up the new number.
	{
		sprintf(stringNumber, "%d", currentNumber);
		UIW_STRING::Information(SET_TEXT, stringNumber);
		return (errorCode);
	}
	else if (!errorSystem)				// Restore the original number.
	{
		sprintf(stringNumber, "%d", *number);
		UIW_STRING::Information(SET_TEXT, stringNumber);
		return (errorCode);
	}
	else if (!processError)
		return (errorCode);

	// Generate the error message and wait for a response.
	for (int i = 0; errorTable[i].message; i++)
		if (errorTable[i].errorCode == errorCode)
		{
			woStatus |= WOS_INTERNAL_ACTION;
			UIS_STATUS errorStatus = errorSystem->ReportError(windowManager,
				WOS_INVALID, errorTable[i].message, stringNumber, range);
			woStatus &= ~WOS_INTERNAL_ACTION;
			if (errorStatus == WOS_INVALID)
				woStatus |= WOS_INVALID;
			else
				return (-1);		// This will cause the number to be restored.
			break;
		}
	return (errorCode);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_INTEGER::UIW_INTEGER(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_STRING(0, 0, 10, NULL, MAX_LENGTH, STF_NO_FLAGS, WOF_NO_FLAGS)
{
	// Initialize the number information.
	UIW_INTEGER::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_INTEGER::Information(INITIALIZE_CLASS, NULL);
}

void UIW_INTEGER::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the integer information.
	UIW_STRING::Load(name, directory, file);
	file->Load(&nmFlags);
	file->Load(&range);

	number = new int;
	*number = atoi(text);
}

void UIW_INTEGER::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the integer information.
	UIW_STRING::Store(name, directory, file);
	file->Store(nmFlags);
	file->Store(range);
}
#endif
