//	Zinc Interface Library - PROMPT.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop

// ----- UIW_PROMPT ---------------------------------------------------------

UIW_PROMPT::UIW_PROMPT(int left, int top, char *_text, WOF_FLAGS _woFlags) :
	UI_WINDOW_OBJECT(left, top, 0, 1, _woFlags | WOF_NON_SELECTABLE, WOAF_NO_FLAGS),
	text(NULL)
{
	// Initialize the prompt information.
	UIW_PROMPT::Information(INITIALIZE_CLASS, NULL);
	UIW_PROMPT::DataSet(_text);
}

UIW_PROMPT::UIW_PROMPT(int left, int top, int width, char *_text, WOF_FLAGS _woFlags) :
	UI_WINDOW_OBJECT(left, top, width, 1, _woFlags | WOF_NON_SELECTABLE, WOAF_NO_FLAGS),
	text(NULL)
{
	// Initialize the prompt information.
	UIW_PROMPT::Information(INITIALIZE_CLASS, NULL);
	UIW_PROMPT::DataSet(_text);
}

UIW_PROMPT::~UIW_PROMPT(void)
{
	if (text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
}

char *UIW_PROMPT::DataGet(void)
{
	return (text);
}

void UIW_PROMPT::DataSet(char *_text)
{
	// Reset the prompt.
	if (text && text != _text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
	if (text == _text || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = _text;
	else
		text = ui_strdup(_text);

	// Check for a hotkey.
	if (text)
	{
		ui_strrepc(text, '~', '&');
		char *tHotKey = strchr(text, '&');
		if (tHotKey)
			hotKey = tolower(tHotKey[1]);
	}

	// Redisplay the prompt.
	if (screenID)
	{
#ifdef _WINDOWS
		InvalidateRect(screenID, NULL, TRUE);
		SendMessage(screenID, WM_SETTEXT, 0, (LONG)text);
#endif
	}
}

#ifdef _WINDOWS
static int _promptOffset = -1;
static FARPROC _promptCallback = (FARPROC)DefWindowProc;
static FARPROC _promptJumpInstance = NULL;

long FAR PASCAL _export PromptJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UIW_PROMPT *object = (UIW_PROMPT *)GetWindowLong(hWnd, _promptOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_PROMPT::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	// Initialize the variables.
	DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
	UI_REGION region = { draw->rcItem.left + 1, draw->rcItem.top + 1,
		draw->rcItem.right - 1, draw->rcItem.bottom - 1 };
	lastPalette = parent->Inherited(ID_TOOL_BAR) ?
		parent->LogicalPalette(ccode, ID_TOOL_BAR) : NULL;
	DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
	return (TRUE);
}

EVENT_TYPE UIW_PROMPT::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_promptJumpInstance)
			_promptJumpInstance = (FARPROC)PromptJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		if (hotKey && Next())
			Next()->HotKey(hotKey);
		break;

	case S_CREATE:
		if (!FlagSet(woFlags, WOF_JUSTIFY_CENTER | WOF_JUSTIFY_RIGHT))
		{
			relative.right = (next && Next()->relative.top == relative.top) ?
				Next()->relative.left - 2 : 
				relative.left + display->TextWidth(text, screenID, font);
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace - 1;
		}
		UI_WINDOW_OBJECT::Event(event);
		RegisterObject("UIW_PROMPT", "STATIC", &_promptOffset,
			&_promptJumpInstance, &_promptCallback, text);
		break;

	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_DRAWITEM)
			ccode = DrawItem(event, ccode);
		else if (message == WM_PAINT)
		{
			lastPalette = parent->LogicalPalette(ccode, parent->SearchID());
			UI_REGION region = true;
			if (FlagSet(woFlags, WOF_BORDER))
				DrawBorder(screenID, region, FALSE, ccode);
			ccode = DrawText(screenID, region, text, lastPalette, FALSE, ccode);
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_PROMPT::DrawItem(const UI_EVENT &, EVENT_TYPE ccode)
{
	// Draw the prompt.
	UI_REGION region = true;
	if (FlagSet(woFlags, WOF_BORDER))
		DrawBorder(screenID, region, FALSE, ccode);
	if (parent->Inherited(ID_TOOL_BAR))
		lastPalette = parent->LogicalPalette(ccode, ID_TOOL_BAR);
	DrawText(screenID, region, text, lastPalette, FALSE, ccode);
	return (ccode);
}

EVENT_TYPE UIW_PROMPT::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_PROMPT);
	switch (ccode)
	{
	case S_INITIALIZE:
		UI_WINDOW_OBJECT::Event(event);
		if (hotKey && Next())
			Next()->HotKey(hotKey);
		break;

	case S_SIZE:
	case S_CREATE:
		if (!FlagSet(woFlags, WOF_JUSTIFY_CENTER | WOF_JUSTIFY_RIGHT))
		{
			relative.right = (next && Next()->relative.top == relative.top) ?
				Next()->relative.left - 1 : 
				relative.left + display->TextWidth(text, screenID, font);
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace - 1;
		}
		UI_WINDOW_OBJECT::Event(event);
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			DrawItem(event, ccode);
			woStatus &= ~WOS_REDISPLAY;
		}
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_PROMPT::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_PROMPT;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_PROMPT;
		if (text)
		{
			ui_strrepc(text, '~', '&');
			char *tHotKey = strchr(text, '&');
			if (tHotKey)
				hotKey = tolower(tHotKey[1]);
		}
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UI_WINDOW_OBJECT::Information(CHANGED_FLAGS, data, ID_PROMPT);
		if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
			dwStyle |= SS_RIGHT;
		else
			dwStyle |= SS_LEFT;
		if (screenID && request == CHANGED_FLAGS && objectID == ID_PROMPT)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_PROMPT)
			Event(UI_EVENT(S_SIZE));	// Recompute the prompt size.
#endif
		break;

	case GET_TEXT:
		if (!data)
			return (DataGet());
		*(char **)data = DataGet();
		break;

	case SET_TEXT:
		DataSet((char *)data);
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_PROMPT::UIW_PROMPT(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UI_WINDOW_OBJECT(0, 0, 0, 1, WOF_NON_SELECTABLE, WOAF_NO_FLAGS)
{
	// Initialize the prompt information.
	UIW_PROMPT::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_PROMPT::Information(INITIALIZE_CLASS, NULL);
	UIW_PROMPT::DataSet(text);
}

void UIW_PROMPT::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Read the prompt information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	file->Load(&text);
}

void UIW_PROMPT::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Write the prompt information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	file->Store(text);
}
#endif

