//	Zinc Interface Library - REAL1.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "ui_win.hpp"
#pragma hdrstop
#define MAX_LENGTH	64

// ----- UIW_REAL -----------------------------------------------------------

UIW_REAL::UIW_REAL(int left, int top, int width, double *_number,
	const char *_range, NMF_FLAGS _nmFlags, WOF_FLAGS _woFlags,
	USER_FUNCTION _userFunction) :
	UIW_STRING(left, top, width, NULL, MAX_LENGTH, STF_NO_FLAGS, _woFlags, _userFunction),
	number(NULL), nmFlags(_nmFlags)
{
	// Initialize the real information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		number = new double;
	else
	{
		text = new char[MAX_LENGTH+1];
		text[0] = '\0';
	}
	range = ui_strdup(_range);
	UIW_REAL::Information(INITIALIZE_CLASS, NULL);
	UIW_REAL::DataSet(_number);
}

UIW_REAL::~UIW_REAL(void)
{
	// Restore the real information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete number;
	else
		delete text;
	if (range)
		delete range;
}

double UIW_REAL::DataGet(void)
{
	// Return the double value.
	UIW_STRING::DataGet();
	*number = atof(text);
	return (*number);
}

static void dosprintf(char *text, double number, int flags)
{
	char tmp[20];

	if (FlagSet(flags, NMF_SCIENTIFIC))
		strcpy(tmp, "%le");
	else if (NMF_DIGITS(flags) >= 0)
		sprintf(tmp, "%%.%dlg", NMF_DIGITS(flags)+1);
	else
		strcpy(tmp, "%lg");
	sprintf(text, tmp, number);
}

void UIW_REAL::DataSet(double *_number)
{
	// Reset the double value.
	if (number == _number || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		number = _number;
	else if (_number)
		*number = *_number;
	else
		*number = 0.0;
	dosprintf(text, *number, nmFlags);
	// sprintf(text, "%lg", *number);
	UIW_STRING::DataSet(text);
}

EVENT_TYPE UIW_REAL::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_REAL);
	switch (ccode)
	{
	case L_SELECT:
	case S_NON_CURRENT:
		if (!FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			double tNumber = *number;			// Keep a temporary value in case of error.
			UIW_STRING::DataGet();
			*number = atof(text);
			ccode = UIW_STRING::Event(event);	// Call the user or valitime function.
			if (ccode == -1)
				*number = tNumber;				// An error occurred, restore the old value.
			else
				woStatus &= ~WOS_UNANSWERED;
			dosprintf(text, *number, nmFlags);
			// sprintf(text, "%lg", *number);
			UIW_STRING::DataSet(text);
		}
		else
			ccode = UIW_STRING::Event(event);
		break;

	case S_CREATE:
		dosprintf(text, *number, nmFlags);
		// sprintf(text, "%lg", *number);
		// Continue to default.
	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void *UIW_REAL::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_REAL;
		windowID[1] = ID_NUMBER;
		windowID[2] = ID_STRING;
		break;

	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_STRING::Information(CHANGED_FLAGS, data, ID_INTEGER);
		if (screenID && request == CHANGED_FLAGS && objectID == ID_INTEGER)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#endif
		break;

	default:
		data = UIW_STRING::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

int UIW_REAL::Validate(int processError)
{
	// Set up the number error messages.
	static struct
	{
		NMI_RESULT errorCode;
		char *message;
	} errorTable[] =
	{
		{ NMI_OUT_OF_RANGE, 	"The number %s must be in the range %s."	},
		{ NMI_OK,				0											}
	};
#ifdef ZIL_PERSISTENCE
	// Check for internationalization.
#endif

	// Check for an absolute number error.
	char *stringNumber = (char *)UIW_STRING::Information(GET_TEXT, NULL);
	double currentNumber = atof(stringNumber);
	NMI_RESULT errorCode = NMI_OK;

	// Check for a range error.
	if (range && errorCode == NMI_OK)
		errorCode = NMI_OUT_OF_RANGE;
	for (char *tRange = range; tRange && errorCode == NMI_OUT_OF_RANGE; )
	{
		char minNumber[MAX_LENGTH], maxNumber[MAX_LENGTH];
		tRange = ParseRange(tRange, minNumber, maxNumber);
		if (currentNumber >= atof(minNumber) &&
			currentNumber <= atof(maxNumber))
			errorCode = NMI_OK;
	}

	// Process the error code.
	woStatus &= ~WOS_INVALID;
	if (errorCode == NMI_OK)			// Set up the new number.
	{
		dosprintf(stringNumber, currentNumber, nmFlags);
		// sprintf(stringNumber, "%lg", currentNumber);
		UIW_STRING::Information(SET_TEXT, stringNumber);
		return (errorCode);
	}
	else if (!errorSystem)				// Restore the original number.
	{
		dosprintf(stringNumber, *number, nmFlags);
		// sprintf(stringNumber, "%lg", *number);
		UIW_STRING::Information(SET_TEXT, stringNumber);
		return (errorCode);
	}
	else if (!processError)
		return (errorCode);

	// Generate the error message and wait for a response.
	for (int i = 0; errorTable[i].message; i++)
		if (errorTable[i].errorCode == errorCode)
		{
			woStatus |= WOS_INTERNAL_ACTION;
			UIS_STATUS errorStatus = errorSystem->ReportError(windowManager,
				WOS_INVALID, errorTable[i].message, stringNumber, range);
			woStatus &= ~WOS_INTERNAL_ACTION;
			if (errorStatus == WOS_INVALID)
				woStatus |= WOS_INVALID;
			else
				return (-1);		// This will cause the number to be restored.
			break;
		}
	return (errorCode);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_REAL::UIW_REAL(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_STRING(0, 0, 10, NULL, MAX_LENGTH, STF_NO_FLAGS, WOF_NO_FLAGS)
{
	// Initialize the number information.
	UIW_REAL::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_REAL::Information(INITIALIZE_CLASS, NULL);
}

void UIW_REAL::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the real information.
	UIW_STRING::Load(name, directory, file);
	file->Load(&nmFlags);
	file->Load(&range);

	number = new double;
	*number = atof(text);
}

void UIW_REAL::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the real information.
	UIW_STRING::Store(name, directory, file);
	file->Store(nmFlags);
	file->Store(range);
}
#endif

