//	Zinc Interface Library - STORE.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#include <mem.h>
#endif
#include <io.h>
#include <fcntl.h>
#include <errno.h>
#if defined(__ZTC__) | defined(_MSC_VER)
#include <direct.h>
#else
#include <dir.h>
#endif
#include <dos.h>
#include <sys/stat.h>
#include "ui_gen.hpp"
#pragma hdrstop

#ifdef TESTING
#define ui_strlen	strlen
#define CheckStorageError(x)	if (storageError != 0) fileabort(-1);
#define CheckObjectError(x)	if (objectError != 0) fileabort(-1);
#define CheckObjectErrorNull()	if (objectError != 0) fileabort(-1);
#else
#define CheckStorageError(x)	if (storageError != 0) return x;
#define CheckObjectError(x)	if (objectError != 0) return x;
#define CheckObjectErrorNull()	if (objectError != 0) return;
#endif	/* TESTING */

#define DOT		"."
#define DOTDOT		".."
#define DIRECTORY_SEPARATOR	'~'
#define ROOT		"~"
#define IREAD		0
#define IWRITE		1

#define MAGIC_NUMBER	0x05AF
#define MAJOR_VERSION	3
#define MINOR_VERSION	0
#define DIRECTORY_TAG	0x8000

#define SHORT_PATTERN	0x0201
#define LONG_PATTERN	0x04030201L

#define BYTES_PER_DATA_BLOCK		256
#define DISK_ADDR_PER_DATA_BLOCK	(BYTES_PER_DATA_BLOCK / sizeof(disk_addr))
#define BYTES_PER_INODE_BLOCK		1024
#define DISK_ADDR_PER_INODE_BLOCK	(BYTES_PER_INODE_BLOCK / sizeof(disk_addr))
#define INODES_PER_INODE_BLOCK		(BYTES_PER_INODE_BLOCK / sizeof(struct inode))
#define BYTES_PER_BLOCK			(BYTES_PER_DATA_BLOCK)
#define COPY_BYTES			4096
#define SIZE_OF_EMPTY_DIR		24

#define LENGTHOF(x)	(sizeof(x)/sizeof((x)[0]))
#define MIN(x, y)	(x < y ? x : y)

struct ZINC_SIGNATURE {
	char copyrightNotice[64];
  	UCHAR majorVersion;
	UCHAR minorVersion;
	USHORT magicNumber;
};

static struct ZINC_SIGNATURE _signature =
	{ "Zinc Data File Version 3.0\032", MAJOR_VERSION, MINOR_VERSION, MAGIC_NUMBER };

struct super_block {
	struct ZINC_SIGNATURE signature;
	time_t createtime;
	time_t modifytime;
	char long_pattern[sizeof(long)];
	char short_pattern[sizeof(short)];
	USHORT revision;
	disk_addr free_inode_list;
	disk_addr free_data_list;
	disk_addr inode_direct[82];
	disk_addr inode_s_indirect;
	disk_addr inode_d_indirect;
};

struct free_list_block {
	disk_addr free_blocks[DISK_ADDR_PER_DATA_BLOCK -1];
	disk_addr next;
};

typedef disk_addr indirect_inode_block[DISK_ADDR_PER_INODE_BLOCK];

struct openfile {
	int open_count;		/* Number of times the file is open */
	int modified;		/* if the modifytime needs updating */
	USHORT inum;		/* Inode number */
	struct inode inode;	/* Inode of this file */
	USHORT revision;
	USHORT country;
};

struct cachedata {
	int pos;			/* position of data buffer */
	disk_addr blknum;		/* absolute block number on disk */
	UCHAR dirty;			/* 1 if block needs to be written */
	UCHAR used;			/* 1 if this block is being used */
};


/* --- support ------------------------------------------------------------- */

#ifdef TESTING
static void fileabort(int i)
{
	printf("error abort: %d\n", i);
	perror("");
	abort();
}
#endif

static int writeat(int fd, long pos, void *buf, int len)
{
	if (lseek(fd, pos, SEEK_SET) != pos) {
#ifdef TESTING
		fileabort(errno);
#endif
		return errno;
	}
	if (write(fd, buf, len) != len) {
#ifdef TESTING
		fileabort(errno);
#endif
		return errno;
	}
	return 0;
}

static int readat(int fd, long pos, void *buf, int len)
{
	if (lseek(fd, pos, SEEK_SET) != pos) {
#ifdef TESTING
		fileabort(errno);
#endif
		return errno;
	}
	if (read(fd, buf, len) != len) {
#ifdef TESTING
		fileabort(errno);
#endif
		return errno;
	}
	return 0;
}


/* --- UI_STORAGE statics -------------------------------------------------- */

void UI_STORAGE::AppendFullPath(char *fullPath, const char *pathName,
	const char *fileName, const char *extension)
{
	if (pathName != fullPath)
		strcpy(fullPath, pathName);
	if (fullPath[0] != '\0' && fullPath[ui_strlen(fullPath)-1] != '\\' &&
	    fullPath[ui_strlen(fullPath)-1] != ':')
		strcat(fullPath, "\\");
	strcat(fullPath, fileName);
	if (extension)
		ChangeExtension(fullPath, extension);
	strupr(fullPath);
}

void UI_STORAGE::ChangeExtension(char *pathName, const char *newExtension)
{
	char *oldExtension = strrchr(pathName, '.');
	if (oldExtension)
		*oldExtension = '\0';
	if (newExtension)
		strcat(pathName, newExtension);
}

int UI_STORAGE::ValidName(const char *name, int createStorage)
{
	char path[128];
	if (createStorage)
	{
		UI_STORAGE::StripFullPath(name, path);
		strcat(path, "\\*.*");
	}
	else
		strcpy(path, name);
#ifdef __ZTC__
	if (findfirst(path, FA_DIREC) != NULL)
		return (TRUE);
	else
		return (FALSE);
#endif
#ifdef _MSC_VER
	struct _find_t fileBlock;
	return (!_dos_findfirst(path, _A_SUBDIR, &fileBlock));
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	struct ffblk fileBlock;
	return (!findfirst(path, &fileBlock, FA_DIREC));
#endif
}


void UI_STORAGE::StripFullPath(const char *fullPath, char *pathName,
	char *fileName, char *objectName, char *objectPathName)
{
	int i;

	// Determine the path/file split area.
	const char *endPath = strrchr(fullPath, '\\');
	if (!endPath)
		endPath = strrchr(fullPath, ':');
	if (!endPath)
		endPath = fullPath;
	if (*endPath == ':')
		endPath++;
	if (*endPath == '\\' && endPath-fullPath == 2 && endPath[-1] == ':')
		endPath++;

	// Determine the file/object split area.
	const char *endFile = strchr(endPath, DIRECTORY_SEPARATOR);
	if (!endFile)
		endFile = &endPath[ui_strlen(endPath)];

	// find the end of the object path name
	const char *endOPath = strrchr(endFile, DIRECTORY_SEPARATOR);
	if (!endOPath)
		endOPath = &endFile[ui_strlen(endFile)];

	// Construct the path name.
	if (pathName) {
		i = (int)(endPath - fullPath);
		strncpy(pathName, fullPath, i);
		pathName[i] = '\0';
		strupr(pathName);
	}
	if (*endPath == '\\') endPath++;

	// Construct the file name.
	if (fileName) {
		i = (int)(endFile - endPath);
		strncpy(fileName, endPath, i);
		fileName[i] = '\0';
		strupr(fileName);
	}

	// Construct the object path name.
	if (objectPathName) {
		i = (int)(endOPath - endFile);
		strncpy(objectPathName, endFile, i);
		objectPathName[i] = '\0';
	}
	if (*endOPath == DIRECTORY_SEPARATOR) endOPath++;

	// Construct the object name.
	if (objectName)
		strcpy(objectName, endOPath);
}


/* --- UI_STORAGE privates ------------------------------------------------- */

int UI_STORAGE::checkOpen(USHORT inum)
{
	int i;

	for (i=0; i < openlen; i++)
		if (openfiles[i].open_count && openfiles[i].inum == inum)
			return i;
	storageError = EACCES;
	return -1;
}

UI_STORAGE_OBJECT *UI_STORAGE::walkPath(const char *name, int stripLast)
{
	char *strt, *stop, *tmpname;
	UI_STORAGE_OBJECT *startdir;

	if (!name || ! *name) {
//		storageError = EINVAL;
		return NULL;
	}
	startdir = current_directory;
	tmpname = new char[ui_strlen(name)+2];
	strt = stop = strcpy(tmpname, name);
	tmpname[ui_strlen(name)+1] = '\0';
	// if (stripLast) then don't try to get into the last part of the path
	if (stripLast) {
		int i = ui_strlen(tmpname);
		while (i && tmpname[i] != DIRECTORY_SEPARATOR) i--;
		tmpname[i] = '\0';
		tmpname[i+1] = '\0';
	}
	if (!*strt) {
#if __BORLANDC__ >= 0x0300
		delete []tmpname;
#else
		delete [ui_strlen(name)+1]tmpname;
#endif
		return NULL;
	}
	for (;;) {
		/* strip out a name */
		while (*stop && *stop != DIRECTORY_SEPARATOR) stop++;
		*stop = '\0';
		if (stop == strt) {
			/* null name, start from root */
			if (openfiles[startdir->inode_index].inum != 0) {
				struct directory_entry dentry = { 0, 0, 0, 0, "" };
				struct inode ientry;

				if (rw_inode(0, &ientry, IREAD) < 0)
					break;
//	No one does this right yet...
//	startdir->~UI_STORAGE_OBJECT();
				if (startdir != current_directory)
					startdir->partialDestruct();
				else
					startdir = new UI_STORAGE_OBJECT();
				startdir->flags = current_directory->flags;
				startdir->openTheFile(*this, &dentry, &ientry,
						      FALSE);
			}
			stop++;
			if (*stop == '\0') {
#if __BORLANDC__ >= 0x0300
				delete []tmpname;
#else
				delete [ui_strlen(name)+1]tmpname;
#endif
				return startdir;
			}
			strt = stop;
		} else if (! *strt) {
//			storageError = EINVAL;
			break;
		} else if (!startdir->find_name(strt)) {
			/* it doesn't exist */
//			storageError = ENOENT;
			break;
		} else {
			UI_STORAGE_OBJECT *dir = new UI_STORAGE_OBJECT();
			struct directory_entry dentry;
			struct inode ientry;
			int i;

			startdir->Load(&dentry);
			dir->objectID = dentry.objectID;
			(void) strcpy(dir->stringID, dentry.stringID);
			dir->flags = startdir->flags;
			if ((i = checkOpen(dentry.inum)) < 0) {
				storageError = 0;	// not an error yet
				if (rw_inode(dentry.inum, &ientry, IREAD) < 0) {
					delete dir;
					break;
				}
			} else
				ientry = openfiles[i].inode;
			dir->openTheFile(*this, &dentry, &ientry, FALSE);
			if (!(openfiles[dir->inode_index].inode.use_count &
			      DIRECTORY_TAG)) {
				delete dir;
//				storageError = ENOENT;
				break;
			}
			if (current_directory != startdir)
				delete startdir;
			startdir = dir;
			stop++;
			if (*stop == '\0') {
#if __BORLANDC__ >= 0x0300
				delete []tmpname;
#else
				delete [ui_strlen(name)+1]tmpname;
#endif
				return startdir;
			}
			strt = stop;
		}
	}
	if (current_directory != startdir)
		delete startdir;
#if __BORLANDC__ >= 0x0300
	delete []tmpname;
#else
	delete [ui_strlen(name)+1]tmpname;
#endif
	return NULL;
}

void UI_STORAGE::walkPartialPath(const char *pathname,
				 UI_STORAGE_OBJECT **parentdir,
				 const char **filename)
{
	*parentdir = walkPath(pathname, TRUE);
	*filename = pathname + ui_strlen(pathname);
	while (*filename != pathname && **filename != DIRECTORY_SEPARATOR)
		(*filename)--;
	if (**filename == DIRECTORY_SEPARATOR)
		(*filename)++;
	if (*parentdir == NULL)
		*parentdir = current_directory;
}

void UI_STORAGE::free_data(unsigned int blknum)
{
	int i;
	struct free_list_block *flblk;

	if (blknum == 0) return;
	if (sb->free_data_list == 0) {
		flblk = (free_list_block *)read_data(blknum);
		if (flblk == NULL) return;
		for (i=0; i < DISK_ADDR_PER_DATA_BLOCK-1; i++)
			flblk->free_blocks[i] = 0;
		flblk->next = 0;
		write_data(flblk);
		sb->free_data_list = blknum;
	} else {
		flblk = (free_list_block *)read_data(sb->free_data_list);
		if (flblk == NULL) return;
		for (i=0; i < DISK_ADDR_PER_DATA_BLOCK-1; i++)
			if (flblk->free_blocks[i] == 0) break;
		if (i >= DISK_ADDR_PER_DATA_BLOCK-1) {
			for (i=0; i < DISK_ADDR_PER_DATA_BLOCK-1; i++)
				flblk->free_blocks[i] = 0;
			flblk->next = sb->free_data_list;
			sb->free_data_list = blknum;
		} else {
			flblk->free_blocks[i] = blknum;
		}
		write_data(flblk);
	}
}

disk_addr UI_STORAGE::alloc_data(void)
{
	disk_addr blkptr;

	if (sb->free_data_list == 0) {
		int i;
		long pos;
		char *buff;

		/* extend the file */
		pos = lseek(fd, 0L, SEEK_END);
		if (pos < 0L) {
			storageError = errno;
#ifdef TESTING
			fileabort(errno);
#endif
			return 0;
		}
		blkptr = pos / BYTES_PER_BLOCK;
		if (pos % BYTES_PER_BLOCK != 0) blkptr++;
		buff = new char[BYTES_PER_BLOCK];
		memset(buff, 0, BYTES_PER_BLOCK);
		i = write(fd, buff, BYTES_PER_BLOCK);
#if	__BORLANDC__ >= 0x0300
		delete []buff;
#else
		delete [BYTES_PER_BLOCK]buff;
#endif
		if (i != BYTES_PER_BLOCK) {
			storageError = errno;
#ifdef TESTING
			fileabort(errno);
#endif
			return 0;
		}
	} else {
		int i;
		struct free_list_block *flblk;

		flblk = (free_list_block *)read_data(sb->free_data_list);
		if (flblk == NULL) return 0;
		for (i=0; i < DISK_ADDR_PER_DATA_BLOCK-1; i++)
			if (flblk->free_blocks[i] != 0) break;
		if (i >= DISK_ADDR_PER_DATA_BLOCK-1) {
			blkptr = sb->free_data_list;
			sb->free_data_list = flblk->next;
		} else {
			blkptr = flblk->free_blocks[i];
			flblk->free_blocks[i] = 0;
		}
		write_data(flblk);
	}
	return blkptr;
}

disk_addr UI_STORAGE::append_inode(USHORT inum)
{
	int i, j;
	struct inode *ipage, *tmp;
	long pos;

	tmp = ipage = new struct inode[INODES_PER_INODE_BLOCK];
	/* append another block to file and make them free */
	for (i=0; i < INODES_PER_INODE_BLOCK; i++, tmp++) {
		tmp->size = 0;
		tmp->createtime = 0;
		tmp->modifytime = 0;
		for (j=0; j < LENGTHOF(tmp->direct); j++)
			tmp->direct[j] = 0;
		tmp->s_indirect = 0;
		tmp->d_indirect = 0;
		tmp->fragment_block = 0;
		tmp->use_count = 0;
		tmp->fragment_index = inum + i + 1;
	}
	pos = lseek(fd, 0L, SEEK_END);
	if (pos < 0L) return 0;
	if (pos % BYTES_PER_BLOCK != 0) {
		pos = lseek(fd, BYTES_PER_BLOCK - (pos % BYTES_PER_BLOCK), SEEK_CUR);
		if (pos < 0L) return 0;
	}
	i = write(fd, (char *)ipage, BYTES_PER_INODE_BLOCK);
#if	__BORLANDC__ >= 0x0300
	delete []ipage;
#else
	delete [INODES_PER_INODE_BLOCK]ipage;
#endif
	if (i != BYTES_PER_INODE_BLOCK) return -1;
	if (inum) sb->free_inode_list = inum;
	return (USHORT)(pos / BYTES_PER_BLOCK);
}

USHORT UI_STORAGE::alloc_inode(void)
// returns 0 on failure or end of inodes.  This is safe because
// inode 0 is always hard assigned to the root directory.
{
	int i;
	struct inode ibuf;

	i = sb->free_inode_list;
	if (rw_inode(i, &ibuf, IREAD) < 0) return 0;
	sb->free_inode_list = ibuf.fragment_index;
	return i;
}

#define INODE_INDIRECT_SIZE	(DISK_ADDR_PER_INODE_BLOCK*sizeof(*iblk))

int UI_STORAGE::rw_inode(USHORT inum, inode *ientry, int direction)
{
	disk_addr inumblk, inumbyt, blkptr;
	long pos;

	inumblk = inum / INODES_PER_INODE_BLOCK;
	inumbyt = inum % INODES_PER_INODE_BLOCK;
	/* direct blocks */
	if (inumblk < LENGTHOF(sb->inode_direct)) {
		if (sb->inode_direct[inumblk] == 0)
			sb->inode_direct[inumblk] = append_inode(inum);
		inumblk = sb->inode_direct[inumblk];
		goto END;
	}
	inumblk -= LENGTHOF(sb->inode_direct);
	/* single indirect blocks */
	if (inumblk < DISK_ADDR_PER_INODE_BLOCK) {
		if (sb->inode_s_indirect == 0)
			sb->inode_s_indirect = append_inode(0);
		pos = sb->inode_s_indirect * BYTES_PER_BLOCK;
		disk_addr *iblk = new disk_addr[DISK_ADDR_PER_INODE_BLOCK];
		if ((storageError = readat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
			delete []iblk;
#else
			delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
			return -1;
		}
		if (iblk[inumblk] == 0) {
			iblk[inumblk] = append_inode(inum);
			if ((storageError = writeat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
				delete []iblk;
#else
				delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
				return -1;
			}
		}
		inumblk = iblk[inumblk];
#if	__BORLANDC__ >= 0x0300
		delete []iblk;
#else
		delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
	} else {
		inumblk -= DISK_ADDR_PER_INODE_BLOCK;
		/* double indirect blocks */
		blkptr = inumblk % DISK_ADDR_PER_INODE_BLOCK;
		inumblk /= DISK_ADDR_PER_INODE_BLOCK;
		if (inumblk >= DISK_ADDR_PER_INODE_BLOCK) {
//			storageError = EINVDAT;
			storageError = EINVAL;
#ifdef TESTING
			fileabort(storageError);
#endif
			return -1;
		}
		if (sb->inode_d_indirect == 0)
			sb->inode_d_indirect = append_inode(0);
		pos = sb->inode_d_indirect * BYTES_PER_BLOCK;
		disk_addr *iblk = new disk_addr[DISK_ADDR_PER_INODE_BLOCK];
		if ((storageError = readat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
			delete []iblk;
#else
			delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
			return -1;
		}
		if (iblk[inumblk] == 0) {
			iblk[inumblk] = append_inode(0);
			if ((storageError = writeat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
				delete []iblk;
#else
				delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
				return -1;
			}
		}
		pos = iblk[inumblk] * BYTES_PER_BLOCK;
		if ((storageError = readat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
			delete []iblk;
#else
			delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
			return -1;
		}
		if (iblk[blkptr] == 0) {
			iblk[blkptr] = append_inode(inum); 
			if ((storageError = writeat(fd, pos, iblk, INODE_INDIRECT_SIZE)) != 0) {
#if	__BORLANDC__ >= 0x0300
				delete []iblk;
#else
				delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
				return -1;
			}
		}
		inumblk = iblk[blkptr];
#if	__BORLANDC__ >= 0x0300
		delete []iblk;
#else
		delete [DISK_ADDR_PER_INODE_BLOCK]iblk;
#endif
	}
END:
	pos = (long)inumblk*BYTES_PER_BLOCK + inumbyt*sizeof(inode);
	switch (direction) {
	case IREAD:
		if ((storageError = readat(fd, pos, ientry, sizeof(*ientry))) != 0)
			return -1;
		return 0;
	case IWRITE:
		if ((storageError = writeat(fd, pos, ientry, sizeof(*ientry))) != 0)
			return -1;
		return 0;
	}
	storageError = EINVAL;
#ifdef TESTING
	fileabort(storageError);
#endif
	return -1;
}

int UI_STORAGE::find_slot(void)
{
	int i;

	for (i=0; i < openlen; i++)
		if (openfiles[i].open_count <= 0) break;
	if (i >= openlen) {
		openfile *nopen;

		nopen = new openfile[openlen+5];
		if (nopen == NULL) {
			storageError = ENOMEM;
#ifdef TESTING
			fileabort(ENOMEM);
#endif
			return -1;
		}
		for (i=0; i < openlen; i++)
			nopen[i] = openfiles[i];
		for (i=openlen; i < openlen+5; i++)
			nopen[i].open_count = 0;
#if	__BORLANDC__ >= 0x0300
		delete []openfiles;
#else
		delete [openlen]openfiles;
#endif
		openfiles = nopen;
		i = openlen;
		openlen += 5;
	}
	return i;
}


void *UI_STORAGE::read_data(disk_addr blknum)
{
	int i, j, cpos;
	long dpos;
	struct cachedata tmpcd;

	if (blknum == 0) return NULL;
	for (i=0; i < cachelen; i++) {
		if (cd[i].blknum == blknum) {
			tmpcd = cd[i];
			tmpcd.used++;
			for (j=i; j > 0; j--) cd[j] = cd[j-1];
			cd[0] = tmpcd;
			return &cache[tmpcd.pos*BYTES_PER_BLOCK];
		}
	}
	tmpcd = cd[cachelen-1];
	for (j=cachelen-1; j > 0; j--) cd[j] = cd[j-1];
	cpos = tmpcd.pos * BYTES_PER_BLOCK;
	if (tmpcd.dirty) {
		dpos = (long)tmpcd.blknum * BYTES_PER_BLOCK;
		if ((storageError = writeat(fd, dpos, &cache[cpos], BYTES_PER_BLOCK)) != 0)
			return NULL;
		tmpcd.dirty = 0;
	}
	tmpcd.blknum = blknum;
	tmpcd.used = 1;
	cd[0] = tmpcd;
	dpos = (long)blknum * BYTES_PER_BLOCK;
	if ((storageError = readat(fd, dpos, &cache[cpos], BYTES_PER_BLOCK)) != 0)
		return NULL;
	return &cache[cpos];
}

void UI_STORAGE::write_data(void *data)
{
	int i, j;

	if ((char *)data < cache || (char *)data >= &cache[cachelen*BYTES_PER_BLOCK]) {
		storageError = ERANGE;
		abort();
	}
	i = (short)(((char *)data - cache) / BYTES_PER_BLOCK);
	// This is probably always 0 (or 1)
	for (j=0; j < cachelen; j++)
		if (cd[j].pos == i) {
			cd[j].dirty = FlagSet(flags, UIS_READWRITE);
			cd[j].used = 0;
			return;
		}
	// storageError = ERANGE;
#ifndef _WINDOWS
	printf("Fatal internal UI_STORAGE error.\n");
#endif
	abort();
}

/* --- UI_STORAGE publics -------------------------------------------------- */

UI_STORAGE::UI_STORAGE(const char *name, UIS_FLAGS pflags)
{
	extern void z_store_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_store_dummy();

	int i;

	storageError = 0;
	openlen = 0;
	openfiles = NULL;
	current_directory = NULL;
	first_time = 1;
	fd = -1;
	sb = NULL;
	cachelen = 0;
	cd = NULL;
	cache = NULL;

	flags = pflags;
	StripFullPath(name, pname, fname);
	sb = new super_block;
	if (!FlagSet(flags, UIS_CREATE)) {
		struct stat statb;
		char tmp[128];
		const char *path = searchPath ? searchPath->FirstPathName() : "";

		while (path != NULL) {
			AppendFullPath(tmp, path, pname, NULL);
			AppendFullPath(tmp, tmp, fname, NULL);
		if (stat(tmp, &statb) >= 0) break;
			path = searchPath ? searchPath->NextPathName() : NULL;
		}
		if (path != NULL) {
			AppendFullPath(tmp, path, pname, NULL);
			(void) strcpy(pname, tmp);
		} else if (FlagSet(flags, UIS_OPENCREATE))
			flags |= UIS_CREATE | UIS_READWRITE;
	}
	/* create the file */
	if (FlagSet(flags, UIS_READWRITE)) {
		AppendFullPath(tname, pname, tmpnam(NULL), NULL);
		fd = open(tname, O_BINARY|O_CREAT|O_TRUNC|O_RDWR, S_IREAD|S_IWRITE);
	} else {
		char tmp[128];
		tname[0] = '\0';
		AppendFullPath(tmp, pname, fname, NULL);
		fd = open(tmp, O_BINARY|O_RDONLY);
	}
	if (fd < 0) {
		storageError = errno;
#ifdef TESTING_ALL
		fileabort(errno);
#endif
		return;
	}
	if (FlagSet(flags, UIS_CREATE)) {
		long lpat;
		short spat;

		/* write the super block */
		memset(sb, 0, sizeof(*sb));
		sb->signature = _signature;
		(void) time(&sb->createtime);
		sb->revision = 0;
		lpat = LONG_PATTERN;
		memcpy((char *)&sb->long_pattern, (char *)&lpat, sizeof(lpat));
		spat = SHORT_PATTERN;
		memcpy((char *)&sb->short_pattern, (char *)&spat, sizeof(spat));
		if (Flush() < 0) return;
	} else if (FlagSet(flags, UIS_READWRITE)) {
		/* copy the file across */
		int fd1, i;
		char *buff;
		char tmp[128];

		AppendFullPath(tmp, pname, fname, NULL);
		fd1 = open(tmp, O_BINARY|O_RDONLY);
		buff = new char[COPY_BYTES];
		lseek(fd, 0L, SEEK_SET);
		while ((i = read(fd1, buff, COPY_BYTES)) > 0) {
			write(fd, buff, i);
		}
#if	__BORLANDC__ >= 0x0300
		delete []buff;
#else
		delete [COPY_BYTES]buff;
#endif
		close(fd1);
	}
	if ((storageError = readat(fd, 0L, sb, sizeof(*sb))) != 0 ||
	    sb->signature.magicNumber != MAGIC_NUMBER ||
	    sb->signature.majorVersion < MAJOR_VERSION) {
		storageError = EINVAL;
		if (tname[0])
			unlink(tname);
		close(fd);
		fd = -1;
		delete sb;
		return;
	}
	sb->revision++;
	cachelen = cacheSize;
	cache = new char[cachelen * BYTES_PER_DATA_BLOCK];
	cd = new cachedata[cachelen];
	if (cache == NULL || cd == NULL) {
#if	__BORLANDC__ >= 0x0300
		delete []cache; cache = NULL;
		delete []cd; cd = NULL;
#else
		delete [cachelen * BYTES_PER_DATA_BLOCK]cache; cache = NULL;
		delete [cachelen]cd; cd = NULL;
#endif
		cachelen = 0;
		storageError = ENOMEM;
		close(fd);
		fd = -1;
		return;
	}
	for (i=0; i < cachelen; i++) {
		cd[i].pos = i;
		cd[i].blknum = 0;
		cd[i].dirty = 0;
		cd[i].used = 0;
	}
	/* initialize the open file table to all unused */
	openlen = 5;
	openfiles = new openfile[openlen];
	for (i=0; i < openlen; i++)
		openfiles[i].open_count = 0;
	if (FlagSet(flags, UIS_CREATE)) {
		if (MkDir(ROOT) < 0) return;
	} else {
		/* this is start up code */
		current_directory = new UI_STORAGE_OBJECT();
		current_directory->file = this;
		current_directory->inode_index = 0;
		current_directory->position = 0;
		current_directory->flags = flags;
		openfiles[0].inum = 0;
		openfiles[0].open_count = 1;
		openfiles[0].modified = 0;
		if (rw_inode(0, &openfiles[0].inode, IREAD) < 0)
			return;
	}
	storageError = 0;
}

UI_STORAGE::~UI_STORAGE(void)
{
	if (fd < 0) return;
	Flush();
	delete current_directory;
	close(fd);
	if (tname[0])
		unlink(tname);
	fd = -1;
#if	__BORLANDC__ >= 0x0300
	delete []cache;
	delete []cd;
	delete []openfiles;
#else
	delete [cachelen * BYTES_PER_DATA_BLOCK]cache;
	delete [cachelen]cd;
	delete [openlen]openfiles;
#endif
	delete sb;
}

int UI_STORAGE::Save(int revisions)
{
	char tmp[128];

	AppendFullPath(tmp, pname, fname, NULL);
	CheckStorageError(-1);
	if (!FlagSet(flags, UIS_READWRITE)) {
		storageError = EACCES;
		return -1;
	}
	if (modified) (void) time(&sb->modifytime);
	if (Flush() < 0) return -1;
	if (first_time) {
		char tmp1[128], tmp2[128];

		AppendFullPath(tmp1, pname, fname, ".BK?");
		(void) strcpy(tmp2, tmp1);
		tmp1[ui_strlen(tmp1)-1] = revisions + '0';
		unlink(tmp1);
		for (int j=revisions-1; j > 0; j--) {
			tmp2[ui_strlen(tmp2)-1] = j + '0';
			rename(tmp2, tmp1);
			tmp1[ui_strlen(tmp1)-1] = j + '0';
		}
		if (revisions) rename(tmp, tmp1);
	}
	int fd1, i;
	char *buff;

	fd1 = open(tmp, O_BINARY|O_CREAT|O_TRUNC|O_RDWR, S_IREAD|S_IWRITE);
	lseek(fd, 0L, SEEK_SET);
	buff = new char[COPY_BYTES];
	while ((i = read(fd, buff, COPY_BYTES)) > 0)
		if (write(fd1, buff, i) != i) return -1;
	if (i < 0) return -1;
#if	__BORLANDC__ >= 0x0300
	delete []buff;
#else
	delete [COPY_BYTES]buff;
#endif
	close(fd1);
	first_time = 0;
	return 0;
}

int UI_STORAGE::SaveAs(const char *new_name, int revisions)
{
	CheckStorageError(-1);
	if (!FlagSet(flags, UIS_READWRITE)) {
		storageError = EACCES;
		return -1;
	}
	if (modified) (void) time(&sb->modifytime);
	if (Flush() < 0) return -1;
	StripFullPath(new_name, pname, fname);
	first_time = 1;
	return Save(revisions);
}

int UI_STORAGE::Version(void)
{
	return 100*sb->signature.majorVersion + sb->signature.minorVersion;
}

int UI_STORAGE::Flush(void)
{
	int i;
	long pos;

	CheckStorageError(-1);
	for (i=0; i < cachelen; i++)
		if (cd[i].dirty) {
			pos = (long)cd[i].blknum * BYTES_PER_BLOCK;
			if ((storageError = writeat(fd, pos, &cache[cd[i].pos * BYTES_PER_BLOCK], BYTES_PER_BLOCK)) != 0)
				return -1;
			cd[i].dirty = 0;
		}
	if ((storageError = writeat(fd, 0L, sb, sizeof(*sb))) != 0)
		return -1;
	for (i=0; i < openlen; i++)
		if (openfiles[i].open_count > 0)
			if (rw_inode(openfiles[i].inum, &openfiles[i].inode, IWRITE) < 0)
				return -1;
#if 0
	extern unsigned int _version;
	if (_version >= 0x0330)
		bdos(0x68, 
#endif
	return 0;
}

int UI_STORAGE::RenameObject(const char *old_object, const char *new_name)
{
	long delentry;
	struct directory_entry dentry, oldentry;

	CheckStorageError(-1);
	if (current_directory->find_name(new_name)) {
		/* if it does exist */
		storageError = EACCES;
		return -1;
	}
	if (!current_directory->find_name(old_object)) {
		/* if it doesn't exist */
		storageError = ENOENT;
		return -1;
	}
	delentry = current_directory->position;
	current_directory->Load(&oldentry);
	/* if it is already open, abort */
	if (checkOpen(oldentry.inum) >= 0)
		return -1;
	storageError = 0;
	struct openfile *o = &openfiles[current_directory->inode_index];
	long pos = current_directory->position;

	while (pos < o->inode.size) {
		current_directory->position = pos;
		current_directory->Load(&dentry);
		pos = current_directory->position;
 
		current_directory->position = delentry;
		current_directory->Store(dentry);
		delentry = current_directory->position;
	}
	(void) strcpy(oldentry.stringID, new_name);
	current_directory->Store(oldentry);
	o->inode.size = current_directory->position;
	return 0;
}

int UI_STORAGE::DestroyObject(const char *name)
{
	int j, i;
	long delentry;
	disk_addr *iblk1, *iblk2;
	struct directory_entry dentry;
	struct inode ientry;
	UI_STORAGE_OBJECT *parentdir;
	const char *fname;

	CheckStorageError(-1);

	walkPartialPath(name, &parentdir, &fname);
	if (!parentdir->find_name(fname)) {
		/* if it doesn't exist */
		storageError = ENOENT;
		if (parentdir != current_directory)
			delete parentdir;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	delentry = parentdir->position;
	parentdir->Load(&dentry);
	/* if it is already open, abort */
	if (checkOpen(dentry.inum) >= 0) {
		if (parentdir != current_directory)
			delete parentdir;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	if (rw_inode(dentry.inum, &ientry, IREAD) < 0) {
		if (parentdir != current_directory)
			delete parentdir;
		return -1;
	}
	/* if it is a directory, abort */
	if (ientry.use_count & DIRECTORY_TAG) {
		storageError = EACCES;
		if (parentdir != current_directory)
			delete parentdir;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	/* if this isn't really a file return error */
	if (ientry.use_count <= 0) {
		storageError = EINVAL;
		if (parentdir != current_directory)
			delete parentdir;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	ientry.use_count--;
	if (ientry.use_count > 0) {
		if (parentdir != current_directory)
			delete parentdir;
		return rw_inode(dentry.inum, &ientry, IWRITE);
	}
	ientry.use_count = 0;
	for (i=0; i < LENGTHOF(ientry.direct); i++) {
		free_data(ientry.direct[i]);
		ientry.direct[i] = 0;
	}
	if (ientry.s_indirect != 0) {
		iblk1 = (disk_addr *)read_data(ientry.s_indirect);
		if (iblk1 == NULL) {
			if (parentdir != current_directory)
				delete parentdir;
			return -1;
		}
		for (i=0; i < DISK_ADDR_PER_DATA_BLOCK; i++)
			free_data(iblk1[i]);
		free_data(ientry.s_indirect);
		ientry.s_indirect = 0;
	}
	if (ientry.d_indirect != 0) {
		iblk1 = (disk_addr *)read_data(ientry.d_indirect);
		if (iblk1 == NULL) {
			if (parentdir != current_directory)
				delete parentdir;
			return -1;
		}
		for (i=0; i < DISK_ADDR_PER_DATA_BLOCK; i++) {
			if (iblk1[i] == 0) continue;
			iblk2 = (disk_addr *)read_data(iblk1[i]);
			if (iblk2 == NULL) {
				if (parentdir != current_directory)
					delete parentdir;
				return -1;
			}
			for (j=0; j < DISK_ADDR_PER_DATA_BLOCK; j++)
				free_data(iblk2[j]);
			free_data(iblk1[i]);
		}
		free_data(ientry.d_indirect);
		ientry.d_indirect = 0;
	}
	ientry.fragment_index = sb->free_inode_list;
	sb->free_inode_list = dentry.inum;
	if (rw_inode(dentry.inum, &ientry, IWRITE) < 0) {
		if (parentdir != current_directory)
			delete parentdir;
		return -1;
	}

	struct openfile *o = &openfiles[parentdir->inode_index];
	long pos = parentdir->position;

	while (pos < o->inode.size) {
		parentdir->position = pos;
		parentdir->Load(&dentry);
		pos = parentdir->position;
 
		parentdir->position = delentry;
		parentdir->Store(dentry);
		delentry = parentdir->position;
	}
	o->inode.size = parentdir->position = delentry;
	if (parentdir != current_directory)
		delete parentdir;
	return 0;
}

int UI_STORAGE::MkDir(const char *name)
{
	int i;
	struct directory_entry dentry;
	struct inode ientry;
	UI_STORAGE_OBJECT *dir;
	UI_STORAGE_OBJECT *parentdir;
	const char *fname;

	CheckStorageError(-1);
#ifdef TESTING_ALL
	memset(&dentry, 0, sizeof(dentry));
#endif
	if (!FlagSet(flags, UIS_READWRITE)) return -1;
	/* this is ROOT */
	if (strcmp(name, ROOT) == 0) {
		dentry.inum = alloc_inode();	// this will be a 0!!
		fname = name;
		parentdir = NULL;
	} else {
		walkPartialPath(name, &parentdir, &fname);
		/* if it does exist */
		if (parentdir->find_name(fname)) {
//			storageError = EEXIST;
			if (parentdir != current_directory)
				delete parentdir;
#ifdef TESTING_ALL
			fileabort(storageError);
#endif
			return -1;
		}
		/* allocate an inode */
		if ((dentry.inum = alloc_inode()) == 0) {
			if (parentdir != current_directory)
				delete parentdir;
#ifdef TESTING
			fileabort(storageError);
#endif
			return -1;
		}
	}
	dentry.objectID = 0;
	dentry.revision = 0;
	dentry.country = 0;

	ientry.use_count = 1;
	ientry.size = 0;
	for (i=0; i < LENGTHOF(ientry.direct); i++)
		ientry.direct[i] = 0;
	ientry.s_indirect = 0;
	ientry.d_indirect = 0;
	ientry.fragment_block = 0;
	ientry.fragment_index = 0;
	(void) time((time_t *)&ientry.createtime);
	modified = 0;
	sb->modifytime = ientry.modifytime = ientry.createtime;

	if ((i = find_slot()) < 0) {
		if (parentdir != current_directory)
			delete parentdir;
		return -1;
	}
	struct openfile *o = &openfiles[i];

	o->inum = dentry.inum;
	o->open_count = 1;
	o->inode = ientry;
	o->modified = 0;

	dir = new UI_STORAGE_OBJECT();
	dir->inode_index = i;
	dir->file = this;
	dir->position = 0;
	dir->flags = UIS_READWRITE;

	if (current_directory == NULL)
		current_directory = dir;

	/* Write the "." (myself) entry */
	(void) strcpy(dentry.stringID, DOT);
	dir->Store(dentry);
	if (dentry.inum != 0) {
		struct openfile *p = &openfiles[parentdir->inode_index];
		/* Write my name in my parent */
		(void) strcpy(dentry.stringID, fname);
		/* write name to parent directory */
		parentdir->position = p->inode.size;
		parentdir->Store(dentry);
		/* mark a pointer to .. */
		dentry.inum = p->inum;
		p->inode.use_count++;
	}
	o->inode.use_count++;
	/* Point me at my parent */
	(void) strcpy(dentry.stringID, DOTDOT);
	dir->Store(dentry);
	o->inode.use_count |= DIRECTORY_TAG;
	if (o->inum != 0) delete dir;
	if (parentdir != current_directory)
		delete parentdir;
	return 0;
}

int UI_STORAGE::RmDir(const char *name)
{
	struct inode ientry;
	struct directory_entry dentry;
	UI_STORAGE_OBJECT *parentdir;
	const char *fname;

	CheckStorageError(-1);

	walkPartialPath(name, &parentdir, &fname);
	if (!parentdir->find_name(fname)) {
		/* if it doesn't exist */
//		storageError = ENOENT;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	parentdir->Load(&dentry);
	if (checkOpen(dentry.inum) >= 0)
		return -1;
	if (rw_inode(dentry.inum, &ientry, IREAD) < 0)
		return -1;
	/* make sure it's a dir and empty (except for "." and "..") */
	if (ientry.size >= SIZE_OF_EMPTY_DIR) {
//		storageError = EACCES;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	/* Make sure this is really a directory and it is only used 2 times */
	if (ientry.use_count != (DIRECTORY_TAG|2)) {
//		storageError = EACCES;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	/* and not current_directory */
	if (dentry.inum == openfiles[current_directory->inode_index].inum) {
//		storageError = EACCES;
#ifdef TESTING_ALL
		fileabort(storageError);
#endif
		return -1;
	}
	/* deallocate it */
	ientry.use_count &= ~DIRECTORY_TAG;
	ientry.use_count--;
// ????? what if use_count doesn't go to 0
	if (rw_inode(dentry.inum, &ientry, IWRITE) < 0)
		return -1;
	openfiles[parentdir->inode_index].inode.use_count--;
	if (parentdir != current_directory)
		delete parentdir;
	return DestroyObject(name);
}

int UI_STORAGE::ChDir(const char *name)
{
	UI_STORAGE_OBJECT *startdir;

	CheckStorageError(-1);
	startdir = walkPath(name, FALSE);
	if (startdir == NULL) return -1;
	if (startdir != current_directory) delete current_directory;
	current_directory = startdir;
	return 0;
}

UI_STATS_INFO *UI_STORAGE::Stats(void)
{
	static UI_STATS_INFO stats;

	stats.createTime = sb->createtime;
	stats.modifyTime = sb->modifytime;
	stats.size = lseek(fd, 0L, SEEK_END);
	stats.useCount = 1;		// Unix will change this
	stats.revision = sb->revision;
	stats.countryID = 0;
	stats.inum = 0;			// Unix will change this
	return &stats;
}

/* --- FindFirst ----------------------------------------------------------- */

static char search_info[MAX_NAME_LENGTH];

static const char *search_pat;
static long pat_position;
static OBJECTID search_ID;
static long ID_position;

static int matchString(const char *pattern, const char *item)
{
	while (*pattern) {
		if (*pattern == *item || *pattern == '?') {
			pattern++;
			item++;
			continue;
		}
		if (*pattern != '*') return FALSE;
		if (!pattern[1]) return TRUE;
		for (int i = 0; item[i]; i++)
			if (matchString(pattern+1, item+i)) return TRUE;
		return FALSE;
	}
	return !*item;
}

char *UI_STORAGE::FindFirstObject(const char *pattern)
{
	CheckStorageError(NULL);
	pat_position = 0;
	search_pat = pattern;
	return FindNextObject();
}

char *UI_STORAGE::FindNextObject(void)
{
	struct directory_entry dentry;

	CheckStorageError(NULL);
	current_directory->position = pat_position;
	do {
		if (current_directory->position >= openfiles[current_directory->inode_index].inode.size)
			return NULL;
		current_directory->Load(&dentry);
	} while (!matchString(search_pat, dentry.stringID));
	pat_position = current_directory->position;
	(void) strcpy(search_info, dentry.stringID);
	return search_info;
}

char *UI_STORAGE::FindFirstID(OBJECTID nobjectID)
{
	CheckStorageError(NULL);
	ID_position = 0;
	search_ID = nobjectID;
	return FindNextID();
}

char *UI_STORAGE::FindNextID(void)
{
	struct directory_entry dentry;

	CheckStorageError(NULL);
	current_directory->position = ID_position;
	do {
		if (current_directory->position >= openfiles[current_directory->inode_index].inode.size)
			return NULL;
		current_directory->Load(&dentry);
	} while (dentry.objectID != search_ID);
	ID_position = current_directory->position;
	(void) strcpy(search_info, dentry.stringID);
	return search_info;
}

/* --- UI_STORAGE_OBJECT privates ------------------------------------------ */

int UI_STORAGE_OBJECT::find_name(const char *name)
{
	long i;
	struct directory_entry dentry;

	// Assume this is a directory
	position = 0;
	while (position < file->openfiles[inode_index].inode.size) {
		i = position;
		Load(&dentry);
		if (strcmp(name, dentry.stringID) == 0) {
			position = i;
			return TRUE;
		}
	}
	return FALSE;
}


openfile *UI_STORAGE_OBJECT::checkobject(void)
{
	openfile *o;

	if (file == NULL) {
		objectError = ENOENT;
#ifdef TESTING
		fileabort(objectError);
#endif
		return NULL;
	}
	if (inode_index < 0 || inode_index >= file->openlen) {
		objectError = EBADF;
#ifdef TESTING
		fileabort(objectError);
#endif
		return NULL;
	}
	o = &file->openfiles[inode_index];
	if (o->open_count < 0) {
		objectError = EBADF;
#ifdef TESTING
		fileabort(objectError);
#endif
		return NULL;
	}
	return o;
}

disk_addr UI_STORAGE_OBJECT::getblockptr(disk_addr blknum)
{
	unsigned int blkptr;
	disk_addr *iblk;
	struct inode *inode;

	/* direct blocks */
	inode = &file->openfiles[inode_index].inode;
	if (blknum < LENGTHOF(inode->direct)) {
		if (inode->direct[blknum] == 0)
			inode->direct[blknum] = file->alloc_data();
		return inode->direct[blknum];
	}
	blknum -= LENGTHOF(inode->direct);
	/* single indirect blocks */
	if (blknum < DISK_ADDR_PER_DATA_BLOCK) {
		if (inode->s_indirect == 0)
			inode->s_indirect = file->alloc_data();
		iblk = (disk_addr *)file->read_data(inode->s_indirect);
		if (iblk == NULL) return 0;
		if (iblk[blknum] == 0)
			iblk[blknum] = file->alloc_data();
		file->write_data(iblk);
		return iblk[blknum];
	}
	blknum -= DISK_ADDR_PER_DATA_BLOCK;
	/* double indirect blocks */
	blkptr = blknum % DISK_ADDR_PER_DATA_BLOCK;
	blknum /= DISK_ADDR_PER_DATA_BLOCK;
	if (blknum < DISK_ADDR_PER_DATA_BLOCK) {
		if (inode->d_indirect == 0)
			inode->d_indirect = file->alloc_data();
		iblk = (disk_addr *)file->read_data(inode->d_indirect);
		if (iblk == NULL) return 0;
		if (iblk[blknum] == 0)
			iblk[blknum] = file->alloc_data();
		file->write_data(iblk);
		iblk = (disk_addr *)file->read_data(iblk[blknum]);
		if (iblk == NULL) return 0;
		if (iblk[blkptr] == 0)
			iblk[blkptr] = file->alloc_data();
		file->write_data(iblk);
		return iblk[blkptr];
	}
	return 0;		/* error */
}

void UI_STORAGE_OBJECT::openTheFile(UI_STORAGE &pfile,
				    struct directory_entry *dentry,
				    struct inode *ientry, int truncate)
{
	int i;

	/* if it is already open, dup the file */
	if ((i = pfile.checkOpen(dentry->inum)) >= 0) {
		pfile.openfiles[i].open_count++;
	} else {
		pfile.storageError = 0;		// not an error yet
		/* find an empty slot */
		if ((i = pfile.find_slot()) < 0) {
			objectError = ENOMEM;
			return;
		}
		pfile.openfiles[i].inum = dentry->inum;
		pfile.openfiles[i].open_count = 1;
		pfile.openfiles[i].inode = *ientry;
		pfile.openfiles[i].modified = 0;
		pfile.openfiles[i].revision = dentry->revision;
		pfile.openfiles[i].country = dentry->country;
	}
	// This next line will be changed in 3.1
	if (truncate) pfile.openfiles[i].inode.size = 0;
	inode_index = i;
	position = 0;
	file = &pfile;
}

int UI_STORAGE_OBJECT::rw_data(void *_buffer, unsigned N, int direction)
{
	int i, cnt, cpy, pos;
	char *bblk;
	openfile *o;

	char *buffer = (char *)_buffer;

	CheckObjectError(-1);
	if (direction == IWRITE && !FlagSet(flags, UIS_READWRITE)) return -1;
	if ((o = checkobject()) == NULL) return -1;
	if (N <= 0) return 0;
	/* Write the first partial block */
	pos = (short)(position % BYTES_PER_DATA_BLOCK);
	cpy = BYTES_PER_DATA_BLOCK - pos;
	i = (short)(position / BYTES_PER_DATA_BLOCK);
	cnt = 0;

	if (direction == IREAD && o->inode.size < position + N)
		N = (short)(o->inode.size - position);
	if (cpy > N) cpy = N;
	/* write blocks until done */
	while (cnt < N) {
		bblk = (char *)file->read_data(getblockptr(i));
		if (bblk == NULL) return -1;
		if (direction == IWRITE) memcpy(&bblk[pos], buffer, cpy);
		else memcpy(buffer, &bblk[pos], cpy);
		file->write_data(bblk);
		buffer += cpy;
		i++;
		cnt += cpy;
		pos = 0;
		cpy = MIN(BYTES_PER_DATA_BLOCK, N-cnt);
	}
	position += cnt;
	if (direction == IWRITE) {
		if (o->inode.size < position) o->inode.size = position;
		o->modified = 1;
		file->modified = 1;
	}
	return cnt;
}

/* --- UI_STORAGE_OBJECT publics ------------------------------------------- */

UI_STORAGE_OBJECT::UI_STORAGE_OBJECT(UI_STORAGE &pfile, const char *name,
				     OBJECTID nobjectID, UIS_FLAGS pflags)
{
	int i;
	struct inode ientry;
	struct directory_entry dentry;
	UI_STORAGE_OBJECT *parentdir;
	const char *fname;

	objectError = 0;
	file = NULL;
	inode_index = -1;
	position = -1;

	if (name == NULL) {
		objectError = EINVAL;
#ifdef TESTING
		fileabort(objectError);
#endif
		return;
	}

	flags = pflags;
	if (!FlagSet(pfile.flags, UIS_READWRITE)) {
		flags |= UIS_READ;
		flags &= ~UIS_READWRITE;
	}

	pfile.walkPartialPath(name, &parentdir, &fname);
	if (!parentdir->find_name(fname)) {
		/* if it doesn't exist */
		if (!FlagSet(flags, UIS_CREATE) &&
		    !FlagSet(flags, UIS_OPENCREATE)) {
			objectError = ENOENT;
			if (parentdir != pfile.current_directory)
				delete parentdir;
#ifdef TESTING_ALL
			fileabort(objectError);
#endif
			return;
		}
		// We ARE going to be able to write this object!!!
		flags |= UIS_READWRITE;
#ifdef TESTING_ALL
		memset(&dentry, 0, sizeof(dentry));
#endif
		if ((dentry.inum = pfile.alloc_inode()) == 0) {
			objectError = pfile.storageError;
			pfile.storageError = 0;
			if (parentdir != pfile.current_directory)
				delete parentdir;
#ifdef TESTING
			fileabort(objectError);
#endif
			return;
		}
		objectID = dentry.objectID = nobjectID;
		(void) strcpy(dentry.stringID, fname);
		(void) strcpy(stringID, fname);
		dentry.revision = 0;
		dentry.country = 0;
		parentdir->Store(dentry);

		ientry.use_count = 1;
		ientry.size = 0;
		for (i=0; i < LENGTHOF(ientry.direct); i++)
			ientry.direct[i] = 0;
		ientry.s_indirect = 0;
		ientry.d_indirect = 0;
		ientry.fragment_block = 0;
		ientry.fragment_index = 0;
	} else {
		parentdir->Load(&dentry);
		objectID = dentry.objectID;
		(void) strcpy(stringID, dentry.stringID);
		if ((i = pfile.checkOpen(dentry.inum)) < 0) {
			pfile.storageError = 0;		// not an error yet
			if (pfile.rw_inode(dentry.inum, &ientry, IREAD) < 0) {
				objectError = pfile.storageError;
				pfile.storageError = 0;
				if (parentdir != pfile.current_directory)
					delete parentdir;
				return;
			}
		} else
			ientry = pfile.openfiles[i].inode;
		if (!ientry.use_count) {
			objectError = ENOENT;
#ifdef TESTING
			fileabort(objectError);
#endif
			if (parentdir != pfile.current_directory)
				delete parentdir;
			return;
		}
		if (FlagSet(flags, UIS_CREATE)) {
			flags |= UIS_READWRITE;
			(void) time((time_t *)&ientry.createtime);
			pfile.modified = 0;
			pfile.sb->modifytime = ientry.modifytime =
				ientry.createtime;
		}
	}
	openTheFile(pfile, &dentry, &ientry, FlagSet(flags, UIS_CREATE));
	if (parentdir != pfile.current_directory)
		delete parentdir;
}

void UI_STORAGE_OBJECT::partialDestruct(void)
// This is really the complete Destructor, but since we can't call
// it directly (some compilers won't allow it) and I need to call it,
// here it is with this name.
{
	openfile *o;

	if ((o = checkobject()) == NULL) return;
	o->open_count--;
	if (o->open_count <= 0 && FlagSet(flags, UIS_READWRITE)) {
		if (o->modified) (void) time((time_t *)&o->inode.modifytime);
//		if (FlagSet(flags, UIS_TEMPORARY)) ????;
		if (file->rw_inode(o->inum, &o->inode, IWRITE) < 0) {
			objectError = file->storageError;
			file->storageError = 0;
		}
	}
	inode_index = -1;
}

void UI_STORAGE_OBJECT::Touch(void)
{
	openfile *o;

	CheckObjectErrorNull();
	if ((o = checkobject()) == NULL) return;
	(void) time((time_t *)&o->inode.modifytime);
	file->sb->modifytime = o->inode.modifytime;
	o->modified = file->modified = 0;
}

UI_STATS_INFO *UI_STORAGE_OBJECT::Stats(void)
{
	openfile *o;
	static UI_STATS_INFO stats;

	if ((o = checkobject()) == NULL) return NULL;
	stats.createTime = o->inode.createtime;
	stats.modifyTime = o->inode.modifytime;
	stats.size = o->inode.size;
	stats.useCount = o->inode.use_count;
	stats.revision = o->revision;
	stats.countryID = o->country;
	stats.inum = o->inum;
	return &stats;
}


/* --- Loads/Stores -------------------------------------------------------- */

#ifdef UNIX

int UI_STORAGE_OBJECT::Load(char *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(UCHAR *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(short *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(USHORT *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(long *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(ULONG *value)
{
	return rw_data(value, sizeof(*value), IREAD);
}

int UI_STORAGE_OBJECT::Load(void *buff, int size, int len)
{
	return rw_data(buff, size*len, IREAD);
}

int UI_STORAGE_OBJECT::Store(char value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(UCHAR value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(short value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(USHORT value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(long value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(ULONG value)
{
	return rw_data(&value, sizeof(value), IWRITE);
}

int UI_STORAGE_OBJECT::Store(void *buff, int size, int len)
{
	return rw_data(buff, size*len, IWRITE);
}
#endif // UNIX

int UI_STORAGE_OBJECT::Load(directory_entry *dirent)
{
	int i, j;

	CheckObjectError(-1);
	if ((i = Load(&dirent->inum)) < 0) return -1;
	if ((j = Load(&dirent->objectID)) < 0) return -1;
	i += j;
	if ((j = Load(&dirent->revision)) < 0) return -1;
	i += j;
	if ((j = Load(&dirent->country)) < 0) return -1;
	i += j;
	if ((j = Load(dirent->stringID, sizeof(dirent->stringID))) < 0)
		return -1;
	return i + j;
}

int UI_STORAGE_OBJECT::Store(const directory_entry dirent)
{
	int i, j;

	CheckObjectError(-1);
	if ((i = Store(dirent.inum)) < 0) return -1;
	if ((j = Store(dirent.objectID)) < 0) return -1;
	i += j;
	if ((j = Store(dirent.revision)) < 0) return -1;
	i += j;
	if ((j = Store(dirent.country)) < 0) return -1;
	i += j;
	if ((j = Store(dirent.stringID)) < 0) return -1;
	return i + j;
}

int UI_STORAGE_OBJECT::Load(char *string, int len)
{
	USHORT size;
	int cnt1, cnt2;

	CheckObjectError(-1);
	*string = '\0';
	cnt1 = rw_data(&size, sizeof(size), IREAD);
	if (cnt1 != sizeof(size)) return cnt1;
	if (size > len) {
		position -= cnt1;
		return -1;
	}
	cnt2 = 0;
	if (size)
		cnt2 = rw_data(string, size, IREAD);
	string[size] = '\0';
	return (cnt2 != size ? cnt2 : cnt1 + cnt2);
}

int UI_STORAGE_OBJECT::Load(char **string)
{
	USHORT size;
	int cnt1, cnt2;

	CheckObjectError(-1);
	cnt1 = rw_data(&size, sizeof(size), IREAD);
	cnt2 = 0;
	if (cnt1 == sizeof(size) && size) {
		*string = new char[size + 1];
		cnt2 = rw_data(*string, size, IREAD);
		(*string)[size] = '\0';
	} else
		*string = NULL;
	return (cnt2 != size ? cnt2 : cnt1 + cnt2);
}

int UI_STORAGE_OBJECT::Store(const char *string)
{
	USHORT size;
	int cnt1, cnt2;

	CheckObjectError(-1);
	size = (string == NULL ? 0 : ui_strlen(string));
	cnt1 = rw_data(&size, sizeof(size), IWRITE);
	if (cnt1 != sizeof(size)) return cnt1;
	cnt2 = (size == 0 ? 0 : rw_data((void *)string, size, IWRITE));
	return (cnt2 != size ? cnt2 : cnt1 + cnt2);
}
