//	Zinc Interface Library - STRING.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
const int STRING_OFFSET = 4;

// ----- UIW_STRING ---------------------------------------------------------

UIW_STRING::UIW_STRING(int left, int top, int width, char *_text,
	int _maxLength, STF_FLAGS _stFlags, WOF_FLAGS _woFlags,
	USER_FUNCTION _userFunction) :
	UI_WINDOW_OBJECT(left, top, width, 1, _woFlags, WOAF_NO_FLAGS),
	text(NULL), maxLength(_maxLength), stFlags(_stFlags)
{
	// Initialize the string information.
	if (maxLength == -1)
		maxLength = _text ? ui_strlen(_text) + 1 : width;
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = new char[maxLength+1];
	userFunction = _userFunction;
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::DataSet(_text);
}

UIW_STRING::~UIW_STRING(void)
{
	// Restore the string information.
	if (text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA) && !FlagSet(stFlags, STF_SUBSTRING))
		delete text;
}

char *UIW_STRING::DataGet(void)
{
#ifdef _WINDOWS
	// Get the text information from Windows.
	if (screenID && !FlagSet(woFlags, WOF_VIEW_ONLY))
		SendMessage(screenID, WM_GETTEXT, maxLength, (LONG)text);
#endif

	// Return the text information.
	return (text);
}

void UIW_STRING::DataSet(char *_text, int _maxLength)
{
	// Reset the text length.
	if (_maxLength != -1)
	{
		if (_maxLength > maxLength && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		{
			char *tText = new char[_maxLength+1];
			strncpy(tText, text ? text : "", maxLength);
			tText[maxLength] = '\0';
			if (_text == text)
				_text = tText;
			if (text)
				delete text;
			text = tText;
		}
		maxLength = _maxLength;
	}

	// Reset the text and compute the length.
	if (text == _text || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = _text;
	else
	{
		strncpy(text, _text ? _text : "", maxLength);
		text[maxLength] = '\0';
	}
	if (FlagSet(woStatus, WOS_UNANSWERED))
		strcpy(text, "");
#ifndef _WINDOWS
	if (!FlagSet(stFlags, STF_SUBSTRING))
		length = ui_strlen(text);
	cursor = (FlagSet(woFlags, WOF_AUTO_CLEAR) && !FlagSet(woFlags, WOF_VIEW_ONLY)) ? length : 0;
	beginMark = endMark = -1;
	leftClip = 0;
#endif

	// Redisplay the string.
#ifdef _WINDOWS
	if (screenID)
	{
		if (_maxLength != -1)
			SendMessage(screenID, EM_LIMITTEXT, maxLength - 1, FALSE);
		SendMessage(screenID, WM_SETTEXT, 0, (LONG)text);
		if (FlagSet(woFlags, WOF_AUTO_CLEAR))
			SendMessage(screenID, EM_SETSEL, 0, 0x7FFF0000L);
	}
#else
	if (screenID && !FlagSet(woStatus, WOS_INTERNAL_ACTION))
		Event(UI_EVENT(S_REDISPLAY));
#endif
}

#ifdef _WINDOWS
static int _stringOffset = -1;
static FARPROC _stringCallback = (FARPROC)DefWindowProc;
static FARPROC _stringJumpInstance = NULL;

long FAR PASCAL _export StringJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	// Get the object pointer and call the C++ class object.
	UIW_STRING *object = (UIW_STRING *)GetWindowLong(hWnd, _stringOffset);
	if (FlagSet(object->stFlags, STF_VARIABLE_NAME) && wMsg == WM_CHAR && wParam == ' ')
		wParam = '_';
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_STRING::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	// Initialize the variables.
	DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
	UI_REGION region = { draw->rcItem.left, draw->rcItem.top,
		draw->rcItem.right, draw->rcItem.bottom };
	region.left += relative.left;
	UI_MSWINDOWS_DISPLAY::hDC = draw->hDC;

	// Check the string status.
	if (FlagSet(draw->itemState, ODS_SELECTED))
		woStatus |= WOS_SELECTED;
	else
		woStatus &= ~WOS_SELECTED;
	if (FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS) &&
		!FlagSet(woStatus, WOS_CURRENT))
	{
		UI_EVENT event(S_ADD_OBJECT);
		event.data = this;
		parent->woStatus |= WOS_WINDOWS_ACTION;
		parent->Event(event);
		parent->woStatus &= ~WOS_WINDOWS_ACTION;
	}

	// Draw the string information.
	if (FlagSet(draw->itemAction, ODA_DRAWENTIRE | ODA_SELECT))
	{
		lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ? PM_SELECTED : PM_ACTIVE, ID_STRING);
		HBRUSH fillBrush = CreateSolidBrush(display->MapColor(lastPalette, FALSE));
		FillRect(draw->hDC, &draw->rcItem, fillBrush);
		DeleteObject(fillBrush);
		region.left += STRING_OFFSET;
		DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
		if (!FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS))
			DrawFocusRect(draw->hDC, &draw->rcItem);
	}
	if (FlagSet(draw->itemAction, ODA_FOCUS))
		DrawFocusRect(draw->hDC, &draw->rcItem);

	// Return success.
	return (TRUE);
}

EVENT_TYPE UIW_STRING::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = FlagSet(woFlags, WOF_VIEW_ONLY) ? LogicalEvent(event, ID_STRING) : event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_stringJumpInstance)
			_stringJumpInstance = (FARPROC)StringJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		if (!display->isText)
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace;
		break;

	case S_CREATE:
		if (FlagSet(woStatus, WOS_UNANSWERED))
			strcpy(text, "");
		if (parent->Inherited(ID_COMBO_BOX))
		{
			DWORD index = SendMessage(parent->screenID, CB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)text);
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, CB_SETCURSEL, LOWORD(index), 0);
		}
		else if (parent->Inherited(ID_LIST))
		{
			DWORD index = SendMessage(parent->screenID, LB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)text);
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, LB_SETSEL, TRUE, index);
		}
		else
		{
			UI_WINDOW_OBJECT::Event(event);
			if (FlagsSet(woFlags, WOF_NON_FIELD_REGION))
			{
				true.bottom++;
				true.top = true.bottom - display->cellHeight + 1;
			}
			RegisterObject("UIW_STRING", "EDIT", &_stringOffset,
				&_stringJumpInstance, &_stringCallback, text);
			SendMessage(screenID, EM_LIMITTEXT, maxLength - 1, FALSE);
		}
		break;

	case S_SIZE:
		if (!screenID || FlagSet(woStatus, WOS_INTERNAL_ACTION))
			break;
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
		{
			true.bottom++;
			true.top = true.bottom - display->cellHeight + 1;
		}
		break;

	case L_BACKSPACE:
	case L_DELETE:
	case L_DELETE_WORD:
	case L_DELETE_EOL:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		// Continue to default.
	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_DRAWITEM)
			ccode = DrawItem(event, ccode);
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) || message != WM_CHAR)
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
extern char _dialogSize[];

char *UIW_STRING::pasteBuffer = NULL;
int UIW_STRING::pasteLength = 0;

const int SCROLL_COUNT	= 5;

char UIW_STRING::CharacterConvert(char key)
{
	if (FlagSet(stFlags, STF_LOWER_CASE))
		key = tolower(key);
	if (FlagSet(stFlags, STF_UPPER_CASE))
		key = toupper(key);
	if (FlagSet(stFlags, STF_VARIABLE_NAME) && key == ' ')
		key = '_';
	return (key);
}

int UIW_STRING::CursorUpdate(int newOffset, int scrollWidth, int showCursor)
{
	int oldLeftClip = leftClip;

	// Check for valid region.
	if (active.right <= active.left)
	{
		leftClip = cursor = newOffset;
		return (scrollWidth);
	}

	// Check for cursor moving to the left of the field.
	while (leftClip > 0 && newOffset < leftClip)
		leftClip = (leftClip > SCROLL_COUNT) ? leftClip - SCROLL_COUNT : 0;
	char saveChar = text[newOffset];
	text[newOffset] = '\0';
	int count = text[leftClip] ?
		display->TextWidth(&text[leftClip], screenID, font | IGNORE_UNDERSCORE) : 0;

	// Check for cursor moving to the right of the field.
	while (active.left + count > active.right)
	{
		int width = newOffset - leftClip;
		leftClip = (width > SCROLL_COUNT) ? leftClip + SCROLL_COUNT : newOffset;
		count = display->TextWidth(&text[leftClip], screenID, font | IGNORE_UNDERSCORE);
	}
	if (showCursor && FlagSet(parent->woStatus, WOS_CURRENT))
	{
		if (active.left + count >= clip.left && active.top >= clip.top &&
			active.left + count <= clip.right && active.top <= clip.bottom)
		{
			eventManager->DeviceState(E_CURSOR, insertMode ? DC_INSERT : DC_OVERSTRIKE);
			eventManager->DevicePosition(E_CURSOR, active.left + count, active.top);
		}
		else
			eventManager->DeviceState(E_CURSOR, D_OFF);
	}
	text[newOffset] = saveChar;

	// Reset the string flag and cursor position.
	cursor = newOffset;

	// Determine and return the actual scroll offset.
	if (oldLeftClip < leftClip)
	{
		saveChar = text[leftClip];
		text[leftClip] = '\0';
		scrollWidth = -display->TextWidth(&text[oldLeftClip], screenID, font | IGNORE_UNDERSCORE);
		text[leftClip] = saveChar;
	}
	else if (oldLeftClip > leftClip)
	{
		saveChar = text[oldLeftClip];
		text[oldLeftClip] = '\0';
		scrollWidth = display->TextWidth(&text[leftClip], screenID, font | IGNORE_UNDERSCORE);
		text[oldLeftClip] = saveChar;
	}

	// Return the scroll width.
	return (scrollWidth);
}

EVENT_TYPE UIW_STRING::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	// Draw the string item.
	UI_REGION region = true;
	if (FlagSet(woFlags, WOF_BORDER))
		DrawBorder(screenID, region, FALSE, ccode);
	display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
	if (FlagSet(stFlags, STF_PASSWORD))	// Check for a password.
	{
		woStatus &= ~WOS_REDISPLAY;
		return (ccode);
	}
	else if (!display->isText)
	{
		region.left += STRING_OFFSET;
		region.right -= STRING_OFFSET;
	}
	DrawText(screenID, region, text, lastPalette, FALSE, ccode);
	woStatus &= ~WOS_REDISPLAY;
	return (ccode);
}

EVENT_TYPE UIW_STRING::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode;
	int insertCount = 0;
	int deleteCount = 0;
	int scrollWidth = 0;
	int tCursor = cursor;
	int tBeginMark = beginMark;
	int tEndMark = endMark;

	// Switch on the event type.
	ccode = LogicalEvent(event, ID_STRING);
	if (ccode == L_BACKSPACE && (FlagSet(woFlags, WOF_VIEW_ONLY) || tCursor == 0))
		return (S_UNKNOWN);
	else if (ccode == L_BACKSPACE)
	{
		tCursor--;
		ccode = L_DELETE;
	}
	else if (ccode == L_CUT_PASTE)
		ccode = (beginMark != -1) ? L_CUT : L_PASTE;
	switch (ccode)
	{
	case S_INITIALIZE:
		ccode = UI_WINDOW_OBJECT::Event(event);
		if (!display->isText)
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace;
		break;

	case S_SIZE:
	case S_CREATE:
		ccode = UI_WINDOW_OBJECT::Event(event);
		active = true;
		if (ccode == S_CREATE && FlagSet(woStatus, WOS_UNANSWERED))
			strcpy(text, "");
		break;

	case S_REDISPLAY:
		return (UI_WINDOW_OBJECT::Event(event));

	case S_NON_CURRENT:
	case S_CURRENT:
		if (FlagSet(stFlags, STF_SUBSTRING))
			;
		else if (ccode != S_CURRENT)
		{
			woStatus &= ~WOS_AUTO_MARK;
			beginMark = endMark = -1;
		}
		else if (FlagSet(woFlags, WOF_AUTO_CLEAR) && !FlagSet(woFlags, WOF_VIEW_ONLY | WOF_NON_SELECTABLE))
		{
			woStatus |= WOS_AUTO_MARK;
			tCursor = length;
			beginMark = 0, endMark = length;
		}
		// Continue to S_DISPLAY_ACTIVE.
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		if (!screenID)
			break;
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_REDISPLAY) && ccode != S_CURRENT &&
			FlagSet(woFlags, WOF_JUSTIFY_CENTER | WOF_JUSTIFY_RIGHT | WOF_NON_SELECTABLE))
		{
			DrawItem(event, ccode);
			return (ccode);
		}
		else if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			// Compute the active region.
			if (!FlagSet(stFlags, STF_SUBSTRING))
				length = ui_strlen(text);
			active = true;
			if (FlagSet(woFlags, WOF_BORDER))
				DrawBorder(screenID, active, FALSE, ccode);
			display->Rectangle(screenID, active, lastPalette, 0, TRUE, FALSE, &clip);
			if (!display->isText)	// Center the text vertically.
			{
				active.left += STRING_OFFSET;
				active.right -= STRING_OFFSET;
				int height = display->TextHeight(text, screenID, font | IGNORE_UNDERSCORE);
				active.top += (active.bottom - active.top + 1 - height) / 2;
				active.bottom = active.top + height;
			}

			// Update the string information.
			scrollWidth = CursorUpdate(tCursor, scrollWidth, (ccode == S_CURRENT) ? TRUE : FALSE);
			StringUpdate(insertCount, deleteCount, scrollWidth);
			return (ccode);
		}
		else if (ccode == S_CURRENT && !FlagSet(woFlags, WOF_VIEW_ONLY))
			CursorUpdate(cursor, 0, TRUE);
		return (ccode);

	case L_UP:
	case L_LEFT:
		if (tCursor > 0)
			tCursor--;
		else
			ccode = S_UNKNOWN;
		break;

	case L_DOWN:
	case L_RIGHT:
		if (tCursor < length)
			tCursor++;
		else
			ccode = S_UNKNOWN;
		if (FlagSet(woStatus, WOS_AUTO_MARK))
		{
			beginMark = endMark = -1;
			woStatus &= ~WOS_AUTO_MARK;
		}
		break;

	case L_BOL:
		tCursor = 0;
		break;

	case L_EOL:
		tCursor = length;
		if (FlagSet(woStatus, WOS_AUTO_MARK))
		{
			beginMark = endMark = -1;
			woStatus &= ~WOS_AUTO_MARK;
		}
		break;

	case L_WORD_LEFT:
		if (tCursor > 0)
			tCursor--;
		else
			ccode = S_UNKNOWN;
		while (tCursor > 0 && text[tCursor] == ' ')
			tCursor--;
		while (tCursor > 0 && text[tCursor-1] != ' ')
			tCursor--;
		break;

	case L_WORD_RIGHT:
		while (tCursor < length && text[tCursor] != ' ')
			tCursor++;
		while (tCursor < length && text[tCursor] == ' ')
			tCursor++;
		if (tCursor == cursor)
			ccode = S_UNKNOWN;
		else if (FlagSet(woStatus, WOS_AUTO_MARK))
		{
			beginMark = endMark = -1;
			woStatus &= ~WOS_AUTO_MARK;
		}
		break;

	case L_INSERT_TOGGLE:
		if (!FlagSet(woFlags, WOF_VIEW_ONLY))
		{
			insertMode = !insertMode;
			eventManager->DeviceState(E_CURSOR, insertMode ? DC_INSERT : DC_OVERSTRIKE);
		}
		break;

	case L_MARK:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		else if (beginMark == -1)
			beginMark = endMark = cursor;
		else
			beginMark = endMark = -1;
		break;

	case S_HSCROLL:
		scrollWidth = event.scroll.delta;
		leftClip += scrollWidth;
		if (leftClip < 0)
			leftClip = 0;
		cursor = tCursor = leftClip;
		break;

	case L_VIEW:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) && true.Overlap(event.position))
			eventManager->DeviceState(E_MOUSE, DM_EDIT);
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		return (ccode);

	case L_BEGIN_MARK:
	case L_CONTINUE_MARK:
	case L_END_MARK:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		if (ccode == L_END_MARK && true.Overlap(event.position) &&
			userFunction && parent->Inherited(ID_LIST))
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, L_SELECT);
		}
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) && true.Overlap(event.position))
		{
			tCursor = leftClip - 1;
			int left = active.left;
			do
			{
				left += _dialogSize[text[++tCursor]];
			} while (tCursor < length && left <= event.position.column);
			if (FlagSet(woStatus, WOS_AUTO_MARK))
			{
				woStatus &= ~WOS_AUTO_MARK;
				beginMark = endMark = -1;
			}
			if (ccode == L_BEGIN_MARK || (ccode == L_CONTINUE_MARK && beginMark == -1))
			{
				beginMark = endMark = tCursor;
				tBeginMark = tEndMark = -1;
			}
			else if (ccode == L_END_MARK)
			{
				if (beginMark == endMark)
					beginMark = endMark = -1;
				woStatus |= WOS_AUTO_MARK;
			}
		}
		else
			ccode = S_UNKNOWN;
		break;

	case L_CUT:
	case L_DELETE:
	case L_DELETE_EOL:
	case L_DELETE_WORD:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		else if (beginMark != -1)
		{
			deleteCount = endMark - beginMark;
			if (ccode == L_CUT)
			{
				if (pasteBuffer)
					delete pasteBuffer;
				pasteLength = deleteCount;
				pasteBuffer = new char[pasteLength];
				memcpy(pasteBuffer, &text[beginMark], pasteLength);
			}
			tCursor = beginMark;
			beginMark = endMark = -1;
			length -= deleteCount;
			memmove(&text[tCursor], &text[tCursor+deleteCount], length - tCursor + 1);
			break;
		}
		else if (tCursor == length && FlagSet(stFlags, STF_SUBSTRING))
			return (S_UNKNOWN);
		{
		int emptyQueue;
		UI_EVENT tEvent;
		tEvent = event;
		do
		{
			if (ccode == L_DELETE)
				deleteCount = (tCursor < length) ? 1 : 0;
			else if (ccode == L_DELETE_EOL)
				deleteCount = length - tCursor;
			else if (ccode == L_DELETE_WORD)
			{
				while (tCursor > 0 && text[tCursor-1] != ' ')
					tCursor--;
				deleteCount = tCursor;
				while (deleteCount < length && text[deleteCount] != ' ')
					deleteCount++;
				while (deleteCount < length && text[deleteCount] == ' ')
					deleteCount++;
				deleteCount -= tCursor;
			}
			if (deleteCount > 0)
			{
				length -= deleteCount;
				memmove(&text[tCursor], &text[tCursor+deleteCount], length - tCursor + 1);
			}
			emptyQueue = eventManager->Get(tEvent, Q_NO_BLOCK | Q_NO_DESTROY);
			if (!emptyQueue && tEvent.type == event.type && tEvent.rawCode == event.rawCode)
				eventManager->Get(tEvent);
			else
				emptyQueue = TRUE;
		} while (!emptyQueue);
		}
		break;

	case L_PASTE:
		if (!FlagSet(woFlags, WOF_VIEW_ONLY) && length + pasteLength < maxLength)
		{
			length += pasteLength;
			memmove(&text[tCursor+pasteLength], &text[tCursor], length - tCursor);
			for (int i = 0; i < pasteLength; i++)
			{
				char key = CharacterConvert(pasteBuffer[i]);
				text[tCursor++] = key;
			}
			insertCount = pasteLength;
		}
		break;

	case L_SELECT:
		if (cursor > length)
			cursor = length;
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;

	case E_KEY:
		if (beginMark != -1)
		{
			UIW_STRING::Event(UI_EVENT(L_CUT));
			tCursor = cursor;
			tBeginMark = beginMark, tEndMark = endMark;
		}
		if (FlagSet(woFlags, WOF_VIEW_ONLY) || !isprint(event.key.value))
			break;
		{
		insertCount = 0;
		int emptyQueue;
		UI_EVENT tEvent;
		tEvent = event;
		do
		{
			char key = CharacterConvert(tEvent.key.value);
			if (length < maxLength - 1 && (insertMode || tCursor == length ||
				text[tCursor] == '\r' || text[tCursor] == '\n'))
			{
				length++;
				memmove(&text[tCursor+1], &text[tCursor], length - tCursor);
				text[tCursor++] = key;
				insertCount++;
			}
			else if (!insertMode && tCursor < length)
			{
				text[tCursor++] = key;
				insertCount++;
			}
			emptyQueue = eventManager->Get(tEvent, Q_NO_BLOCK | Q_NO_DESTROY);
			if (!emptyQueue && tEvent.type == event.type && isprint(tEvent.key.value) &&
				tEvent.key.value != '\n')
				eventManager->Get(tEvent);
			else
				emptyQueue = TRUE;
		} while (!emptyQueue);
		}
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Check the mark information.
	if (beginMark != -1)
	{
		if (cursor == beginMark)
			beginMark = tCursor;
		else
			endMark = tCursor;
		if (beginMark > endMark)	// Make sure the mark order is correct.
		{
			int temp = beginMark;
			beginMark = endMark;
			endMark = temp;
		}
	}
	if (cursor != tCursor && beginMark != -1 && ccode != S_CURRENT)
	{
		// Reset the mark region if the mark status is set.
		if (FlagSet(woStatus, WOS_AUTO_MARK))
		{
			beginMark = endMark = -1;
			woStatus &= ~WOS_AUTO_MARK;
		}
		else
			tBeginMark = tEndMark = -1;
	}

	// Recompute the cursor position and redraw the string.
	if (!screenID)
	{
		cursor = tCursor;
		return (ccode);
	}
	if (cursor != tCursor)
		scrollWidth = CursorUpdate(tCursor, scrollWidth, TRUE);
	if (insertCount || deleteCount || scrollWidth ||
		beginMark != tBeginMark || endMark != tEndMark)
		StringUpdate(insertCount, deleteCount, scrollWidth);

	// Return the control code.
	return (ccode);
}

void UIW_STRING::StringUpdate(int insertCount, int deleteCount, int scrollWidth)
{
	// Check for passwords.
	if (FlagSet(stFlags, STF_PASSWORD) || active.right <= active.left)
	{
		if (insertCount || deleteCount)
			woStatus |= WOS_CHANGED;
		woStatus &= ~WOS_REDISPLAY;
		return;
	}

	// Compute the actual clip region and save the display.
	UI_REGION clipRegion = clip;
	if (clipRegion.right > active.right)
		clipRegion.right = active.right;
	if (clipRegion.bottom > active.bottom)
		clipRegion.bottom = active.bottom;

	// Draw the string information.
	int left = active.left;
	int scrollLeft = leftClip;
	if (!scrollWidth && (insertCount || deleteCount))
	{
		scrollLeft = cursor - insertCount;
		char saveChar = text[scrollLeft];
		text[scrollLeft] = '\0';
		left += display->TextWidth(&text[leftClip], screenID, font | IGNORE_UNDERSCORE);
		text[scrollLeft] = saveChar;
	}
	int top = active.top;
	int beginOffset;
	int endOffset = 0;
	for (int i = 0; endOffset < length; i++)
	{
		beginOffset = (endOffset > scrollLeft) ? endOffset : scrollLeft;
		if (i == 0)			// Begin mark region.
			endOffset = (beginMark != -1) ? beginMark : length;
		else if (i == 1)	// End mark region.
			endOffset = endMark;
		else				// End of string.
			endOffset = length;
		if (beginOffset != endOffset && endOffset > scrollLeft && left < clipRegion.right)
		{
			char saveChar = text[endOffset];
			text[endOffset] = '\0';
			display->Text(screenID, left, top, &text[beginOffset],
				(i == 1) ? display->xorPalette : lastPalette, 
				endOffset - beginOffset, TRUE, FALSE, &clipRegion,
				font | IGNORE_UNDERSCORE);
			left += display->TextWidth(&text[beginOffset], screenID, font | IGNORE_UNDERSCORE);
			text[endOffset] = saveChar;
		}
	}
	// Update the end of the field.
	if (left <= clipRegion.right)
		display->Rectangle(screenID, left, active.top, active.right,
			active.bottom, lastPalette, 0, TRUE, FALSE, &clip);
	lineWidth = left - scrollWidth - active.left + 1;

	// Restore the display.
	if (insertCount || deleteCount)
		woStatus |= WOS_CHANGED;
	woStatus &= ~WOS_REDISPLAY;
}
#endif

void *UIW_STRING::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_STRING;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_STRING;
#ifndef _WINDOWS
		leftClip = cursor = lineWidth = 0;
		beginMark = endMark = -1;
		insertMode = TRUE;
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UI_WINDOW_OBJECT::Information(CHANGED_FLAGS, data, ID_STRING);
		dwStyle |= ES_AUTOHSCROLL | ES_OEMCONVERT;
		if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
			dwStyle |= ES_CENTER;
		else if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
			dwStyle |= ES_RIGHT;
		if (FlagSet(stFlags, STF_LOWER_CASE))
			dwStyle |= ES_LOWERCASE;
		else if (FlagSet(stFlags, STF_UPPER_CASE))
			dwStyle |= ES_UPPERCASE;
		else if (FlagSet(stFlags, STF_PASSWORD))
			dwStyle |= ES_PASSWORD;
		if (screenID && request == CHANGED_FLAGS && objectID == ID_STRING)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#endif
		break;

	case GET_TEXT:
		if (!data)
			return (UIW_STRING::DataGet());
		*(char **)data = UIW_STRING::DataGet();
		break;

	case SET_TEXT:
		UIW_STRING::DataSet((char *)data);
		break;

	case GET_TEXT_LENGTH:
		if (!data)
			return (&maxLength);
		*(int *)data = maxLength;
		break;

	case SET_TEXT_LENGTH:
		UIW_STRING::DataSet(text, *(int *)data);
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_STRING)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &stFlags;
		else if (request == GET_FLAGS)
			*(STF_FLAGS *)data = stFlags;
		else if (request == SET_FLAGS)
			stFlags |= *(STF_FLAGS *)data;
		else
			stFlags &= ~(*(STF_FLAGS *)data);
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

char *UIW_STRING::ParseRange(char *buffer, char *minValue, char *maxValue)
{
	// Parse the minimum value.
	int position = 0, offset = 0;
	while (buffer[offset] != '\0' &&
		(buffer[offset] != '.' || buffer[offset+1] != '.') &&
		buffer[offset] != '/')
		minValue[position++] = buffer[offset++];
	minValue[position] = '\0';

	// See if it is a standalone value.
	if (buffer[offset] == '/' || buffer[offset] == '\0')
	{
		strcpy(maxValue, minValue);
		return (buffer[offset] ? &buffer[++offset] : NULL);
	}

	// Increment the offset.
	while (buffer[offset] == '.')
		offset++;

	// Parse the maximum value.
	position = 0;
	while (buffer[offset] != '\0' && buffer[offset] != '/')
		maxValue[position++] = buffer[offset++];
	maxValue[position] = '\0';

	// Return the offset position.
	return (buffer[offset] ? &buffer[++offset] : NULL);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_STRING::UIW_STRING(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UI_WINDOW_OBJECT(0, 0, 15, 1, WOF_NO_FLAGS, WOAF_NO_FLAGS)
{
	// Initialize the string information.
	UIW_STRING::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::DataSet(text);
}

void UIW_STRING::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the string information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	file->Load(&stFlags);
	short _value; file->Load(&_value); maxLength = _value;
	text = new char[maxLength+1]; file->Load(text, maxLength);
}

void UIW_STRING::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the string information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	file->Store(stFlags);
	short _value = maxLength; file->Store(_value);
	file->Store(text);
}
#endif

