//	Zinc Interface Library - TIME.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include "ui_gen.hpp"
#pragma hdrstop

// ----- UI_TIME ------------------------------------------------------------

void UI_TIME::Export(int *hour, int *minute, int *second, int *hundredth)
{
	// Set the integer variables.
	long value1 = value;
	long value2 = value1 / 360000L;
	if (hour)
		*hour = (int)value2;
	value1 -= value2 * 360000L;
	value2 = value1 / 6000L;
	if (minute)
		*minute = (int)value2;
	value1 -= value2 * 6000L;
	value2 = value1 / 100L;
	if (second)
		*second = (int)value2;
	value1 -= value2 * 100L;
	if (hundredth)
		*hundredth = (int)value1;
}

void UI_TIME::Export(char *string, TMF_FLAGS tmFlags)
{
	extern void z_time_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_time_dummy();

	// Make sure the country information is initialized.
	if (!UI_INTERNATIONAL::initialized)
		UI_INTERNATIONAL::Initialize();

	// Determine the seperator character.
	char *separator = timeSeparator;
	if (FlagSet(tmFlags, TMF_COLON_SEPARATOR))
		separator = ":";
	else if (FlagSet(tmFlags, TMF_NO_SEPARATOR))
		separator = "";

	// Determine the time format.
	long hourValue = -1;
	char *format = FlagSet(tmFlags, TMF_ZERO_FILL | TMF_NO_SEPARATOR) ? "%02ld" : "%ld";

	// Format the string.
	strcpy(string, "");
	long value1 = value;
	long value2;
	long ratio = 360000L;
	for (int i = 0; i < 4; i++)
	{
		// Strip off the relavant value.
		value2 = value1 / ratio;
		value1 -= value2 * ratio;
		if (ratio == 360000L)
			ratio = 6000L;
		else if (ratio == 6000L)
			ratio = 100L;
		else
			ratio = 1L;

		// See if the string should contain the value.
		if ((i == 0 && !FlagSet(tmFlags, TMF_NO_HOURS)) ||
			(i == 1 && !FlagSet(tmFlags, TMF_NO_MINUTES)) ||
			(i == 2 && FlagSet(tmFlags, TMF_SECONDS)) ||
			(i == 3 && FlagSet(tmFlags, TMF_HUNDREDTHS)))
		{
			// Append the number separator.
			if (*string)
				strcat(string, (i == 3) ? decimalSeparator : separator);
			// Check for the 12 hour timing.
			if (i == 0 && (FlagSet(tmFlags, TMF_TWELVE_HOUR) ||
				(timeFormat == 0 && !FlagSet(tmFlags, TMF_TWENTY_FOUR_HOUR))))
			{
				hourValue = value2;
				value2 = (value2 == 0 || value2 == 12) ? 12 : value2 % 12;
			}
			// Append the time value.
			char sValue[8];
			sprintf(sValue, *string ? "%02ld" : format, value2);
			strcat(string, sValue);
		}
	}

	// Set the am/pm information.
	if (hourValue >= 0)
	{
		// Append the am/pm string.
		strcat(string, " ");
		strcat(string, (hourValue >= 12) ? pmPtr : amPtr);
		// Convert to upper or lower case.
		if (FlagSet(tmFlags, TMF_UPPER_CASE))
			strupr(string);
		else if (FlagSet(tmFlags, TMF_LOWER_CASE))
			strlwr(string);
	}
}

void UI_TIME::Export(int *packedTime)
{
	int hour, minute, second;
	
	Export(&hour, &minute, &second);
	*packedTime = (hour << 11) | (minute << 5) | second / 2;
}

TMI_RESULT UI_TIME::Import(void)
{
	// Set the time values according to the system time.
#ifdef __ZTC__
	struct dos_time_t info;
	dos_gettime(&info);
	Import(info.hour, info.minute, info.second, info.hsecond);
#endif
#ifdef _MSC_VER
	struct dostime_t info;
	_dos_gettime(&info);
	Import(info.hour, info.minute, info.second, info.hsecond);
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	struct time info;
	gettime(&info);
	Import(info.ti_hour, info.ti_min, info.ti_sec, info.ti_hund);
#endif

	return (TMI_OK);
}

TMI_RESULT UI_TIME::Import(const UI_TIME &time)
{
	// Set the time value according to the passed time.
	value = time.value;

	return (TMI_OK);
}

TMI_RESULT UI_TIME::Import(int hour, int minute, int second, int hundredth)
{
	// Set the time values according to the specified integers.
	if (hour < 0 || hour > 23 || minute < 0 || minute > 59 ||
		second < 0 || second > 59 || hundredth < 0 || hundredth > 99)
		return (TMI_INVALID);
	value = (3600L * hour + 60L * minute + second) * 100 + hundredth;

	return (TMI_OK);
}

TMI_RESULT UI_TIME::Import(const char *string, TMF_FLAGS tmFlags)
{
	// Check for a system time.
	if ((!string || !string[0]) && FlagSet(tmFlags, TMF_SYSTEM))
		return (UI_TIME::Import());

	// Parse out the string time.
	const char *saveString = string;
	int time[4] = { 0, 0, 0, 0 };
	int maxTime[4] = { 23, 59, 59, 99 };
	for (int i = 0; i < 4 && *string; i++)
		if ((i == 0 && !FlagSet(tmFlags, TMF_NO_HOURS)) ||
			(i == 1 && !FlagSet(tmFlags, TMF_NO_MINUTES)) ||
			(i == 2 && FlagSet(tmFlags, TMF_SECONDS)) ||
			(i == 3 && FlagSet(tmFlags, TMF_HUNDREDTHS)))
		{
			while (*string && !isdigit(*string))
				string++;
			while (*string && isdigit(*string))
			{
				int value = time[i] * 10 + *string - '0';
				if (value > maxTime[i])
					return (TMI_INVALID);
				time[i] = value;
				string++;
			}
		}

	// Find the am/pm string (if any).
	string = saveString;
	while (*string && !isalpha(*string))
		string++;
	if (isalpha(*string) && !strnicmp(string, pmPtr, 1) && time[0] < 12)
		time[0] += 12;
	else if (isalpha(*string) && !strnicmp(string, amPtr, 1) && time[0] == 12)
		time[0] -= 12;

	// Set the time according to the parsed values.
	Import(time[0], time[1], time[2], time[3]);

	return (TMI_OK);
}

TMI_RESULT UI_TIME::Import(int time)
{
	return(Import((time & 0xF800) >> 11, (time & 0x07E0) >> 5, 2 * (time & 0x001F)));
}
