//	Zinc Interface Library - TXTDSP.CPP
//	COPYRIGHT (C) 1990, 1991.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#include <dos.h>
#include <conio.h>
#include <string.h>
#include <stdlib.h>
#include "ui_dsp.hpp"
#pragma hdrstop

static USHORT *_screen;
static USHORT *_moveBuffer;
static int _originalMode = -1;
static int _originalCursor;
static int _virtualCount = 0;
static UI_REGION _virtualRegion;
static char _stopDevice = FALSE;

#ifdef __ZTC__
#include <disp.h>

#define OutPortByte outp

void GetText(int left, int top, int right, int bottom, void *buffer)
{
	disp_peekbox(buffer, top - 1, left - 1, bottom - 1, right - 1);
}
void PutText(int left, int top, int right, int bottom, void *buffer)
{
	disp_pokebox(buffer, top - 1, left - 1, bottom - 1, right - 1);
}
void MoveText(int left, int top, int right, int bottom, int destX, int destY)
{
	if (right < left || bottom < top)
		return;
	USHORT *buffer = _moveBuffer;
	disp_peekbox(buffer, top - 1, left - 1, bottom - 1, right - 1);
	disp_pokebox(buffer, destY - 1, destX - 1, destY + bottom - top - 1, destX + right - left - 1);
}
#endif
#ifdef _MSC_VER
#include <graph.h>

#define OutPortByte _outp
#define MKFP(s,o)	    (((long)s << 16) | o)
#define MDA_SEGMENT		0xb000
#define COLOR_SEGMENT	0xb800

unsigned _videoSegment;
int _videoWidth;

void GetText(int left, int top, int right, int bottom, void *_buffer)
{
	USHORT far *buffer = (USHORT far *)_buffer;
	int y, x;
	int width = (right - left + 1) * 2;
	unsigned videoOffset = (top - 1) * _videoWidth + (left - 1) * 2;
		
	for (y = top; y <= bottom; y++)
	{

		for (x = 0; x < width; x += 2)
			*buffer++ = *((USHORT far*)MKFP(_videoSegment, videoOffset + x));
		
		videoOffset += _videoWidth;
	}
}
void PutText(int left, int top, int right, int bottom, void *_buffer)
{
	USHORT far *buffer = (USHORT far *)_buffer;
	int y, x;
	int width = (right - left + 1) * 2;
	unsigned videoOffset = (top - 1) * _videoWidth + (left - 1) * 2;
		
	for (y = top; y <= bottom; y++)
	{

		for (x = 0; x < width; x += 2)
			*((USHORT far*)MKFP(_videoSegment, videoOffset + x)) = *buffer++;
		
		videoOffset += _videoWidth;
	}
}
void MoveText(int left, int top, int right, int bottom, int destX, int destY)
{
	if (right < left || bottom < top)
		return;
	USHORT *buffer = _moveBuffer;
	GetText(left, top, right, bottom, buffer);
	PutText(destX, destY, destX + right - left, destY + bottom - top, buffer);
}
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#define OutPortByte outportb
#define GetText gettext
#define PutText puttext
#define MoveText movetext
#endif

#define CellXOR(cell) ((cell & 0x00FF) | (~(cell & 0xFF00) & 0xFF00))

// ----- UI_TEXT_DISPLAY ----------------------------------------------------

UI_TEXT_DISPLAY::UI_TEXT_DISPLAY(int _mode) :
	UI_DISPLAY(TRUE)
{
	union REGS inregs, outregs;

#ifdef __ZTC__
	// Save original text mode.
	if (_originalMode == -1)
		_originalMode = disp_getmode();
	if (_mode == TDM_43x80 &&
		_originalMode != 0 && _originalMode != 2 && _originalMode != 7)
	{
		disp_open();
		disp_close();
		disp_set43();
	}
	else if (_mode != TDM_AUTO)
		disp_setmode(_mode);
	disp_open();
	disp_hidecursor();

	// Check for special text mode monitors.
	mode = disp_getmode();
	isMono = (mode == 0 || mode == 2 || mode == 5 || mode == 6 || mode == 7) ?
		TRUE : FALSE;

	lines = disp_numrows;
	columns = disp_numcols;
#endif
#ifdef _MSC_VER
	// Save original text mode.
	struct _videoconfig textInfo;
	if (_originalMode == -1)
	{
		_getvideoconfig(&textInfo);
		_originalMode = textInfo.mode;
	}
	if (_mode == TDM_43x80 &&
		_originalMode != 0 && _originalMode != 2 && _originalMode != 7)
	{
		_setvideomoderows(_DEFAULTMODE, 43);
	}
	else if (_mode != TDM_AUTO)
		_setvideomode(_mode);

	// Check for special text mode monitors.
	_getvideoconfig(&textInfo);
	mode = textInfo.mode;
	isMono = (mode == 0 || mode == 2 || mode == 5 || mode == 6 || mode == 7) ?
		TRUE : FALSE;

	_videoSegment = (mode == 7) ? MDA_SEGMENT : COLOR_SEGMENT;
	_videoWidth = textInfo.numtextcols * 2;

	lines = textInfo.numtextrows;
	columns = textInfo.numtextcols;

	// Hide the text cursor;
	_originalCursor = _gettextcursor();
	_settextcursor(0x2000);
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	// Save original text mode.
	struct text_info textInfo;
	if (_originalMode == -1)
	{
		gettextinfo(&textInfo);
		_originalMode = textInfo.currmode;
	}

	// Initialize the text display.
	textmode(_mode);
	_setcursortype(_NOCURSOR);

	// Check for special text mode monitors.
	gettextinfo(&textInfo);
	mode = textInfo.currmode;
	isMono = (mode == 0 || mode == 2 || mode == 5 || mode == 6 || mode == 7) ?
		TRUE : FALSE;

	lines = textInfo.screenheight;
	columns = textInfo.screenwidth;
#endif

	// Turn the blink attribute off.
	inregs.x.ax = 0x1003;
	inregs.h.bl = 0x0000;
	int86(0x10, &inregs, &outregs);

//#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UCHAR newSetting = *((UCHAR far *)0x465L);
	newSetting = newSetting & ~0x20;
	OutPortByte(*((USHORT far *)0x463L) + 4, newSetting);
	*((UCHAR far *)0x465L) = newSetting;		// Update BIOS data area
//#endif

	cellWidth = cellHeight = 1;	
	preSpace = postSpace = 0;

	// Fill the screen according to the specified palette.
	USHORT color = isMono ? backgroundPalette->monoAttribute : backgroundPalette->colorAttribute;
	color = (color << 8);
	int screenSize = columns * lines;
	_screen = new USHORT[screenSize];
	_moveBuffer = new USHORT[screenSize];
	for (int i = 0; i < screenSize; i++)
		_screen[i] = color;
	int offset = 0;
	for (i = 1; i <= lines; i++)
	{
		PutText(1, i, columns, i, &_screen[offset]);
		offset += columns;
	}

	extern char _dialogSize[];
	memset(_dialogSize, 1, 256);

	// Define the screen display region.
	Add(NULL, new UI_REGION_ELEMENT(ID_SCREEN, 0, 0, columns - 1, lines - 1));
	installed = TRUE;
}

UI_TEXT_DISPLAY::~UI_TEXT_DISPLAY(void)
{
	// Restore the display.
	if (installed)
	{
		delete _screen;
#ifdef __ZTC__
		disp_setcursortype(DISP_CURSORBLOCK);
		disp_move(0, 0);
		disp_eeop();
		disp_showcursor();
		disp_close();
		if (lines > 25)
			disp_reset43();
#endif
#ifdef _MSC_VER
		_settextcursor(_originalCursor);
		_setvideomode(_originalMode);
		_clearscreen(_GCLEARSCREEN);
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
		_setcursortype(_NORMALCURSOR);
		textmode(_originalMode);
		clrscr();
#endif
	}
}

void UI_TEXT_DISPLAY::Bitmap(SCREENID, int, int, int, int, const UCHAR *,
	const UI_PALETTE *, const UI_REGION *)
{
}

void UI_TEXT_DISPLAY::Ellipse(SCREENID, int, int, int, int, int, int,
	const UI_PALETTE *, int, int, const UI_REGION *)
{
}

void UI_TEXT_DISPLAY::DeviceMove(IMAGE_TYPE imageType, int newColumn, int newLine)
{
	if (imageType != MOUSE_IMAGE || _stopDevice)
		return;
	_stopDevice = TRUE;

	UI_DISPLAY_IMAGE *view = &displayImage[MOUSE_IMAGE];
	int column = view->region.left;
	int line = view->region.top;
	view->region.left = view->region.right = newColumn;
	view->region.top = view->region.bottom = newLine;

	USHORT screen;
	if (column >= 0 && column < columns && line >= 0 && line < lines)
	{
		screen = _screen[line * columns + column];
		PutText(column + 1, line + 1, column + 1, line + 1, &screen);
	}
	if (newColumn >= 0 && newColumn < columns && newLine >= 0 && newLine < lines)
	{
		screen = CellXOR(_screen[newLine * columns + newColumn]);
		PutText(newColumn + 1, newLine + 1, newColumn + 1, newLine + 1, &screen);
	}

	_stopDevice = FALSE;
}

void UI_TEXT_DISPLAY::DeviceSet(IMAGE_TYPE imageType, int column, int line,
	int, int, UCHAR *)
{
	DeviceMove(imageType, column, line);
}

void UI_TEXT_DISPLAY::Line(SCREENID screenID, int column1, int line1,
	int column2, int line2, const UI_PALETTE *palette, int width, int xor,
	const UI_REGION *clipRegion)
{
	// Ensure a left-top to right-bottom line.
	if (column2 < column1)
	{
		int column = column1;
		column1 = column2;
		column2 = column;
	}
	if (line2 < line1)
	{
		int line = line1;
		line1 = line2;
		line2 = line;
	}

	// Set up the line region.
	UI_REGION region, tRegion;
	if (!RegionInitialize(region, clipRegion, column1, line1, column2, line2))
		return;

	// Draw the line on the display.
	static USHORT horizontal[] = { 0xC4, 0xCD };
	static USHORT vertical[] = { 0xB3, 0xBA };
	int changedScreen = FALSE;
	USHORT color = isMono ? palette->monoAttribute : palette->colorAttribute;
	if (line1 == line2 && width >= 1)
		color = (color << 8) | horizontal[width - 1];
	else if (column1 == column2 && width >= 1)
		color = (color << 8) | vertical[width - 1];
	else
		color = (color << 8) | palette->fillCharacter;
	for (UI_REGION_ELEMENT *dRegion = First(); dRegion; dRegion = dRegion->Next())
		if (screenID == ID_DIRECT ||
			(screenID == dRegion->screenID && dRegion->region.Overlap(region, tRegion)))
		{
			if (xor || screenID == ID_DIRECT)
				tRegion = region;
			if (!changedScreen)
				changedScreen = VirtualGet(screenID, region.left, region.top, region.right, region.bottom);
			for (int row = tRegion.top; row <= tRegion.bottom; row++)
			{
				int offset = columns * row + tRegion.left;
				for (int column = tRegion.left; column <= tRegion.right; column++, offset++)
					_screen[offset] = xor ? CellXOR(_screen[offset]) : color;
			}
			if (xor || screenID == ID_DIRECT)
				break;
		}

	// Update the screen.
	if (changedScreen)
		VirtualPut(screenID);
}

COLOR UI_TEXT_DISPLAY::MapColor(const UI_PALETTE *palette, int)
{
	// Match the color request based on the type of display.
	if (isMono)
		return (palette->monoAttribute);
	return (palette->colorAttribute);
}

void UI_TEXT_DISPLAY::Polygon(SCREENID, int, const int *, const UI_PALETTE *,
	int, int, const UI_REGION *)
{
}

void UI_TEXT_DISPLAY::Rectangle(SCREENID screenID, int left, int top, int right,
	int bottom, const UI_PALETTE *palette, int width, int fill, int xor,
	const UI_REGION *clipRegion)
{
	// Assign the rectangle to the region structure.
	UI_REGION region, tRegion;
	if (!RegionInitialize(region, clipRegion, left, top, right, bottom))
		return;

	// Check for the text shadow ``feature''.
	if (!palette)
	{
		VirtualGet(screenID, region.left, region.top, region.right, region.bottom);
		UI_REGION clip;
		if (clipRegion)				// Un-optimized for Zortech bug.
			clip = *clipRegion;
		else
			clip = region;
		int offset = columns * region.top + region.right;
		if (top == region.top && right == region.right)
		{
			_screen[offset] = (_screen[offset] & 0xF000) | 0x00DC;
			region.top++;
			offset += columns;
		}
		if (right == region.right)
		{
			for (int line = region.top; line < region.bottom; line++, offset += columns)
				if (line >= clip.top && line <= clip.bottom)
					_screen[offset] = (_screen[offset] & 0xF000) | 0x00DB;
			if (bottom != region.bottom && region.bottom >= clip.top && region.bottom <= clip.bottom)
				_screen[offset] = (_screen[offset] & 0xF000) | 0x00DB;
		}
		if (left == region.left)
			region.left++;
		offset = columns * region.bottom + region.left;
		if (bottom == region.bottom)
		{
			for (int column = region.left; column <= region.right; column++, offset++)
				if (column >= clip.left && column <= clip.right)
					_screen[offset] = (_screen[offset] & 0xF000)  | 0x00DF;
		}
		VirtualPut(screenID);
		return;
	}

	// Check for a no fill-pattern.
	if (!fill && top != bottom)
	{
		static char cornerUL[] = { 0xDA, 0xC9 };
		static char cornerUR[] = { 0xBF, 0xBB };
		static char cornerLL[] = { 0xC0, 0xC8 };
		static char cornerLR[] = { 0xD9, 0xBC };
		VirtualGet(screenID, region.left, region.top, region.right, region.bottom);
		if (left + 1 < right - 1)
		{
			Line(screenID, left + 1, top, right - 1, top, palette, width, xor, clipRegion);
			Line(screenID, left + 1, bottom, right - 1, bottom, palette, width, xor, clipRegion);
		}
		if (top + 1 < bottom - 1)
		{
			Line(screenID, left, top + 1, left, bottom - 1, palette, width, xor, clipRegion);
			Line(screenID, right, top + 1, right, bottom - 1, palette, width, xor, clipRegion);
		}
		Text(screenID, left, top, &cornerUL[width - 1], palette, 1, FALSE, xor, clipRegion);
		Text(screenID, right, top, &cornerUR[width - 1], palette, 1, FALSE, xor, clipRegion);
		Text(screenID, left, bottom, &cornerLL[width - 1], palette, 1, FALSE, xor, clipRegion);
		Text(screenID, right, bottom, &cornerLR[width - 1], palette, 1, FALSE, xor, clipRegion);
		VirtualPut(screenID);
		return;
	}

	// Draw the rectangle on the display.
	int changedScreen = FALSE;
	USHORT color = isMono ? palette->monoAttribute : palette->colorAttribute;
	color = (color << 8) | palette->fillCharacter;
	for (UI_REGION_ELEMENT *dRegion = First(); dRegion; dRegion = dRegion->Next())
		if (screenID == ID_DIRECT ||
			(screenID == dRegion->screenID && dRegion->region.Overlap(region, tRegion)))
		{
			if (xor || screenID == ID_DIRECT)
				tRegion = region;
			if (!changedScreen)
				changedScreen = VirtualGet(screenID, region.left, region.top, region.right, region.bottom);
			for (int row = tRegion.top; row <= tRegion.bottom; row++)
			{
				int offset = columns * row + tRegion.left;
				for (int column = tRegion.left; column <= tRegion.right; column++, offset++)
					_screen[offset] = xor ? CellXOR(_screen[offset]) : color;
			}
			if (xor || screenID == ID_DIRECT)
				break;
		}

	// Update the screen.
	if (changedScreen)
		VirtualPut(screenID);
}

void UI_TEXT_DISPLAY::RectangleXORDiff(const UI_REGION &oldRegion,
	const UI_REGION &newRegion)
{
	// Set up the hide region.
	if (oldRegion.left == newRegion.left && oldRegion.top == newRegion.top &&
		oldRegion.right == newRegion.right && oldRegion.bottom == newRegion.bottom)
		return;
	UI_REGION region;
	region.left = Max(Min(oldRegion.left, newRegion.left), 0);
	region.top = Max(Min(oldRegion.top, newRegion.top), 0);
	region.right = Min(Max(oldRegion.right, newRegion.right), columns - 1);
	region.bottom = Min(Max(oldRegion.bottom, newRegion.bottom), lines - 1);

	// XOR the screen region.
	int row, column, offset;
	VirtualGet(ID_SCREEN, region.left, region.top, region.right, region.bottom);
	for (row = Max(oldRegion.top, region.top);
		row <= Min(oldRegion.bottom, region.bottom); row++)
	{
		offset = columns * row + Max(oldRegion.left, region.left);
		for (column = Max(oldRegion.left, region.left);
			column <= Min(oldRegion.right, region.right); column++, offset++)
			if (row == oldRegion.top || row == oldRegion.bottom ||
				column == oldRegion.left || column == oldRegion.right)
				_screen[offset] = CellXOR(_screen[offset]);
	}
	for (row = Max(newRegion.top, region.top);
		row <= Min(newRegion.bottom, region.bottom); row++)
	{
		offset = columns * row + Max(newRegion.left, region.left);
		for (column = Max(newRegion.left, region.left);
			column <= Min(newRegion.right, region.right); column++, offset++)
			if (row == newRegion.top || row == newRegion.bottom ||
				column == newRegion.left || column == newRegion.right)
			_screen[offset] = (_screen[offset] & 0x00FF) | (~(_screen[offset] & 0xFF00) & 0xFF00);
	}

	// Update the screen.
	VirtualPut(ID_SCREEN);
}

void UI_TEXT_DISPLAY::RegionDefine(SCREENID screenID, int left, int top,
	int right, int bottom)
{
	// See if it is a full screen definition.
	UI_REGION region = { left, top, right, bottom };
	if (region.left <= 0 && region.top <= 0 && region.right >= columns - 1 && region.bottom >= lines - 1)
	{
		UI_REGION_LIST::Destroy();
		Add(0, new UI_REGION_ELEMENT(screenID, 0, 0, columns - 1, lines - 1));
		return;
	}

	// Throw away regions with a negative width or height.
	if (region.right < region.left || region.bottom < region.top)
		return;
	
	// Throw away regions that are completely off the screen.
	if (region.left >= columns || region.right < 0 ||
		region.top >= lines || region.bottom < 0)
		return;

	// Clip regions partially off the screen to fit on the screen.
	if (region.left < 0)
		region.left = 0;
	if (region.right >= columns)
		region.right = columns - 1;
	if (region.top < 0)
		region.top = 0;
	if (region.bottom >= lines)
		region.bottom = lines - 1;

	// Split any overlapping regions.
	Split(screenID, region, FALSE);

	// Define the new display region.
	UI_REGION_LIST::Add(0, new UI_REGION_ELEMENT(screenID, region));
}

void UI_TEXT_DISPLAY::RegionMove(const UI_REGION &oldRegion,
	int newColumn, int newLine, SCREENID oldScreenID, SCREENID)
{
	UI_REGION region;
	if (!RegionInitialize(region, NULL, Min(oldRegion.left, newColumn),
		Min(oldRegion.top, newLine),
		Max(oldRegion.right, newColumn + (oldRegion.right - oldRegion.left)),
		Max(oldRegion.bottom, newLine + (oldRegion.bottom - oldRegion.top))))
		return;
	VirtualGet(oldScreenID, region.left, region.top, region.right, region.bottom);
	for (int row = region.top; row <= region.bottom; row++)
	{
		int offset = row * columns + region.left;
		PutText(region.left + 1, row + 1, region.right + 1, row + 1, &_screen[offset]);
	}
	MoveText(oldRegion.left + 1, oldRegion.top + 1, oldRegion.right + 1,
		oldRegion.bottom + 1, newColumn + 1, newLine + 1);
	for (row = region.top; row <= region.bottom; row++)
	{
		int offset = row * columns + region.left;
		GetText(region.left + 1, row + 1, region.right + 1, row + 1, &_screen[offset]);
	}
	VirtualPut(oldScreenID);
}

void UI_TEXT_DISPLAY::Text(SCREENID screenID, int left, int top,
	const char *text, const UI_PALETTE *palette, int length, int,
	int xor, const UI_REGION *clipRegion, LOGICAL_FONT logicalFont)
{
	// Make sure we have a valid string.
	if (!text || !text[0])
		return;

	// Set up the fill line.
	static char fillLine[256];
	if (length < 0)
		length = ui_strlen(text);
	else if (length > 255)
		length = 255;
	strncpy(fillLine, text, length);
	fillLine[length] = '\0';

	// Check for special characters.
	char *hotKey = FlagSet(logicalFont, IGNORE_UNDERSCORE) ? NULL : strchr(fillLine, '&');
	if (hotKey)
		strcpy(hotKey, hotKey + 1);
	for (char *string = strchr(fillLine, '\t'); string; string = strchr(string, '\t'))
		*string = ' ';
	for (string = strchr(fillLine, '\r'); string; string = strchr(string, '\r'))
		*string = ' ';
	for (string = strchr(fillLine, '\n'); string; string = strchr(string, '\n'))
		*string = ' ';

	// Convert the cell coordinates.
	UI_REGION region, tRegion;
	if (!RegionInitialize(region, clipRegion, left, top,
		left + TextWidth(fillLine, screenID, logicalFont) - 1,
		top + TextHeight(fillLine, screenID, logicalFont) - 1))
		return;

	// Draw the rectangle on the display.
	int changedScreen = FALSE;
	USHORT color = isMono ? palette->monoAttribute : palette->colorAttribute;
	color = (color << 8);
	for (UI_REGION_ELEMENT *dRegion = First(); dRegion; dRegion = dRegion->Next())
		if (xor || screenID == ID_DIRECT ||
			(screenID == dRegion->screenID && dRegion->region.Overlap(region, tRegion)))
		{
			if (xor || screenID == ID_DIRECT)
				tRegion = region;
			if (!changedScreen)
				changedScreen = VirtualGet(screenID, region.left, region.top, region.right, region.bottom);
			int sOffset = columns * tRegion.top + tRegion.left;
			int tOffset = tRegion.left - left;
			for (int column = tRegion.left; column <= tRegion.right; column++, sOffset++, tOffset++)
				_screen[sOffset] = xor ? CellXOR(_screen[sOffset]) :
					color | (unsigned char)fillLine[tOffset];
			if (xor || screenID == ID_DIRECT)
				break;
		}

	// Update the screen.
	if (changedScreen)
		VirtualPut(screenID);
}

int UI_TEXT_DISPLAY::TextHeight(const char *, SCREENID, LOGICAL_FONT)
{
	return (1);
}

int UI_TEXT_DISPLAY::TextWidth(const char *string, SCREENID, LOGICAL_FONT logicalFont)
{
	if (!string || !string[0])
		return (0);
	if (!FlagSet(logicalFont, IGNORE_UNDERSCORE) && strchr(string, '&'))
		return (ui_strlen(string) - 1);
	return (ui_strlen(string));
}

int UI_TEXT_DISPLAY::VirtualGet(SCREENID, int left, int top, int right, int bottom)
{
#ifndef ZINC_DEBUG
	if (--_virtualCount == -1)
	{
#endif
 		_stopDevice = TRUE;
		_virtualRegion.left = Max(0, left);
		_virtualRegion.top = Max(0, top);
		_virtualRegion.right = Min(columns - 1, right);
		_virtualRegion.bottom = Min(lines - 1, bottom);
		if (_virtualRegion.Overlap(displayImage[MOUSE_IMAGE].region))
		{
			UI_DISPLAY_IMAGE *view = &displayImage[MOUSE_IMAGE];
			int column = view->region.left;
			int line = view->region.top;
			USHORT screen = _screen[line * columns + column];
			PutText(column + 1, line + 1, column + 1, line + 1, &screen);
		}
#ifndef ZINC_DEBUG
	}
	else
	{
		_virtualRegion.left = Max(0, Min(_virtualRegion.left, left));
		_virtualRegion.top = Max(0, Min(_virtualRegion.top, top));
		_virtualRegion.right = Min(columns - 1, Max(_virtualRegion.right, right));
		_virtualRegion.bottom = Min(lines - 1, Max(_virtualRegion.bottom, bottom));
	}
#endif
	return (TRUE);
}

int UI_TEXT_DISPLAY::VirtualPut(SCREENID)
{
#ifndef ZINC_DEBUG
	if (++_virtualCount == 0)
	{
#endif
		for (int row = _virtualRegion.top; row <= _virtualRegion.bottom; row++)
		{
			int offset = row * columns + _virtualRegion.left;
			PutText(_virtualRegion.left + 1, row + 1,
				_virtualRegion.right + 1, row + 1, &_screen[offset]);
		}
		if (_virtualRegion.Overlap(displayImage[MOUSE_IMAGE].region))
		{
			UI_DISPLAY_IMAGE *view = &displayImage[MOUSE_IMAGE];
			int column = view->region.left;
			int line = view->region.top;
			USHORT screen = CellXOR(_screen[line * columns + column]);
			PutText(column + 1, line + 1, column + 1, line + 1, &screen);
		}

		_stopDevice = FALSE;
#ifndef ZINC_DEBUG
	}
#endif
	return (TRUE);
}
