//	Zinc Interface Library - WIN2.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#include <mem.h>
#endif
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop

// ----- Definition for the Windows callback functions -----
#ifdef _WINDOWS
#ifdef __ZTC__
typedef LONG (FAR PASCAL *DEFWINPROC)(HWND, unsigned, WORD, LONG);
//typedef long (FAR PASCAL *DEFWINPROC)();	// For early versions of Zortech.
#endif
#ifdef _MSC_VER
typedef LONG (FAR PASCAL *DEFWINPROC)(HWND, UINT, UINT, LONG);
#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
typedef LONG (FAR PASCAL *DEFWINPROC)(HWND, WORD, WORD, LONG);
#endif

static int _offset = -1;
static WORD _controlID = 0;
static UI_WINDOW_OBJECT *_object = NULL;
static FARPROC _jumpInstance = NULL;

long FAR PASCAL _export ObjectInitProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	_object->screenID = hWnd;
	SetWindowLong(hWnd, _offset, (LONG)_object);
	SetWindowLong(hWnd, GWL_WNDPROC, (LONG)_jumpInstance);
	if (_object->parent)
		SetWindowWord(hWnd, GWW_ID, _controlID);
	_object = NULL;
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _offset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}
#endif

// ----- UI_WINDOW_OBJECT ---------------------------------------------------

UI_WINDOW_OBJECT::UI_WINDOW_OBJECT(int left, int top, int width, int height,
	WOF_FLAGS _woFlags, WOAF_FLAGS _woAdvancedFlags) : woFlags(_woFlags),
	woAdvancedFlags(_woAdvancedFlags)
{
	numberID = 0;
	stringID[0] = '\0';
	userFlags = UIF_NO_FLAGS;
	userStatus = UIS_NO_FLAGS;
	userObject = NULL;
	userFunction = NULL;
	userObjectName = NULL;
	userFunctionName = NULL;
	helpContext = NO_HELP_CONTEXT;
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);

	// Intialize the screen object information.
	true.left = relative.left = left;
	true.top = relative.top = top;

	true.right = relative.right = (left >= 0 && width > 0) ?
		left + width - 1 : width;
	true.bottom = relative.bottom = (top >= 0 && height > 0) ?
		(height == 1) ? top : top + height - 1 : height;
}

UI_WINDOW_OBJECT::~UI_WINDOW_OBJECT(void)
{
	if (userObjectName)
		delete userObjectName;
	if (userFunctionName)
		delete userFunctionName;
}

#ifdef _WINDOWS
EVENT_TYPE UI_WINDOW_OBJECT::DrawBorder(SCREENID screenID, UI_REGION &tRegion,
	int fillRegion, EVENT_TYPE ccode)
{
	// Draw the outer rectangle.
	HDC hDC;
	PAINTSTRUCT ps;
	RECT rect = { tRegion.left, tRegion.top, tRegion.right, tRegion.bottom };
	if (screenID != ID_DIRECT)
	{
		hDC = BeginPaint(screenID, &ps);
		GetClientRect(screenID, &rect);
	}
	else
		hDC = UI_MSWINDOWS_DISPLAY::hDC;
	HBRUSH brush = CreateSolidBrush(GetSysColor(COLOR_WINDOWFRAME));
	FrameRect(hDC, &rect, brush);
	DeleteObject(brush);
	--tRegion;

	// Fill the rest of the region.
	if (fillRegion)
	{
		rect.left += 1, rect.top +=1, rect.right -=1, rect.bottom -=1;
		HBRUSH brush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
		FillRect(hDC, &rect, brush);
		DeleteObject(brush);
	}

	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, hDC);
	return (TRUE);
}

EVENT_TYPE UI_WINDOW_OBJECT::DrawShadow(SCREENID screenID, UI_REGION &tRegion,
	int depth, int fillRegion, EVENT_TYPE)
{
	// Draw the outer rectangle.
	HDC hDC = (screenID != ID_DIRECT) ? GetDC(screenID) : UI_MSWINDOWS_DISPLAY::hDC;
	HPEN border = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_WINDOWFRAME));
	HPEN oldForeground = SelectObject(hDC, border);
	RECT rect = { tRegion.left, tRegion.top, tRegion.right, tRegion.bottom };
	HBRUSH brush = CreateSolidBrush(GetSysColor(COLOR_WINDOWFRAME));
	FrameRect(hDC, &rect, brush);
	--tRegion;

	// Draw the shadow in normal or depressed state.
	HPEN lightShadow = GetStockObject(WHITE_PEN);
	HPEN darkShadow = CreatePen(PS_SOLID, 1, GetSysColor(COLOR_BTNSHADOW));
	if (depth > 0)
	{
		// Draw the light shadow.
		SelectObject(hDC, lightShadow);
		for (int i = 0; i < Abs(depth); i++)
		{
			MoveTo(hDC, tRegion.left + i, tRegion.bottom - i - 1);
			LineTo(hDC, tRegion.left + i, tRegion.top + i);
			LineTo(hDC, tRegion.right - i - 1, tRegion.top + i);
		}

		// Draw the dark shadow.
		SelectObject(hDC, darkShadow);
		for (i = 0; i < 2; i++)
		{
			MoveTo(hDC, tRegion.right - 1, tRegion.top);
			LineTo(hDC, tRegion.right - 1, tRegion.bottom - 1);
			LineTo(hDC, tRegion.left - 1, tRegion.bottom - 1);
			--tRegion;
		}
		if (Abs(depth) < 2)
		{
			tRegion.left--;
			tRegion.top--;
		}
	}
	else
	{
		// Draw the dark shadow.
		SelectObject(hDC, darkShadow);
		MoveTo(hDC, tRegion.left, tRegion.bottom - 1);
		LineTo(hDC, tRegion.left, tRegion.top);
		LineTo(hDC, tRegion.right - 1, tRegion.top);
		tRegion.left++;
		tRegion.top++;
	}
	SelectObject(hDC, oldForeground);
	DeleteObject(lightShadow);
	DeleteObject(darkShadow);
	DeleteObject(border);
	DeleteObject(brush);

	// Fill the rest of the region.
	if (fillRegion)
	{
		COLOR background = GetSysColor(COLOR_BTNFACE);
		HBRUSH fillBrush = CreateSolidBrush(background);
		RECT region = { tRegion.left, tRegion.top, tRegion.right, tRegion.bottom };
		FillRect(hDC, &region, fillBrush);
		DeleteObject(fillBrush);
	}
	if (depth < 0)
	{
		tRegion.left -= depth;
		tRegion.top -= depth;
	}

	if (screenID != ID_DIRECT)
		ReleaseDC(screenID, hDC);
	return (TRUE);
}

EVENT_TYPE UI_WINDOW_OBJECT::DrawText(SCREENID screenID, UI_REGION &tRegion,
	const char *text, UI_PALETTE *palette, int fillRegion, EVENT_TYPE)
{
	HDC hDC;
	PAINTSTRUCT ps;
	RECT region = { tRegion.left, tRegion.top, tRegion.right, tRegion.bottom };
	COLOR foreground, background;
	if (screenID != ID_DIRECT)
	{
		hDC = BeginPaint(screenID, &ps);
		GetClientRect(screenID, &region);
		SelectObject(hDC, UI_MSWINDOWS_DISPLAY::fontTable[font]);
		foreground = palette ? display->MapColor(palette, TRUE) : GetSysColor(COLOR_WINDOWTEXT);
		background = palette ? display->MapColor(palette, FALSE) : GetSysColor(COLOR_WINDOW);
		HBRUSH fillBrush = CreateSolidBrush(background);
		FillRect(hDC, &region, fillBrush);
		DeleteObject(fillBrush);
	}
	else
	{
		hDC = UI_MSWINDOWS_DISPLAY::hDC;
		foreground = palette ? display->MapColor(palette, TRUE) : GetTextColor(hDC);
		background = palette ? display->MapColor(palette, FALSE) : GetBkColor(hDC);
	}

	int length = ui_strlen(text);
	WORD wFormat = DT_SINGLELINE | DT_VCENTER;
	if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		wFormat |= DT_CENTER;
	else if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		wFormat |= DT_RIGHT;
	else
		wFormat |= DT_LEFT;
	COLORREF oldForeground = SetTextColor(hDC, foreground);
	COLORREF oldBackground = SetBkColor(hDC, background);
	::DrawText(hDC, (LPSTR)text, length, &region, wFormat);
	SetTextColor(hDC, oldForeground);
	SetBkColor(hDC, oldBackground);
	if (screenID != ID_DIRECT)
		EndPaint(screenID, &ps);

	return (TRUE);
}

EVENT_TYPE UI_WINDOW_OBJECT::Event(const UI_EVENT &event)
{
	static initializedTime = FALSE;
	static UI_TIME lastTime;
	if (!initializedTime)
	{
		lastTime.Import();
		initializedTime = TRUE;
	}
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_OBJECT);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!numberID && parent)
		{
			for (object = parent; object->parent; object = object->parent)
				;
			numberID = object->NumberID();
			object->NumberID(numberID + 1);
			if (stringID[0] == '\0')
				sprintf(stringID, "FIELD_%d", numberID);
		}
		dwStyle |= parent ? WS_CHILD | WS_VISIBLE : WS_OVERLAPPED;
		RegionConvert(relative, (parent && !FlagSet(woFlags, WOF_NON_FIELD_REGION)) ? FALSE : TRUE);
		break;

	case S_SIZE:
		if (!screenID || FlagSet(woStatus, WOS_INTERNAL_ACTION))
			break;
		// Continue to S_CREATE.
	case S_CREATE:
		woStatus |= WOS_REDISPLAY;
		if (parent)
			parent->RegionMax(this);
		else
			RegionMax(this);
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
		// Check the object status.
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			lastPalette = LogicalPalette(ccode);
			break;
		}
		else if (ccode == S_CURRENT)
		{
			if (!FlagSet(woStatus, WOS_CURRENT))
				woStatus |= WOS_REDISPLAY;
			woStatus |= WOS_CURRENT;
		}
		else
		{
			if (FlagSet(woStatus, WOS_CURRENT))
				woStatus |= WOS_REDISPLAY;
			woStatus &= ~WOS_CURRENT;
		}
		// Continue to S_DISPLAY_ACTIVE.
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		lastPalette = LogicalPalette(ccode);
		if (!screenID || FlagSet(woStatus, WOS_WINDOWS_ACTION))
			;
		else if (ccode == S_CURRENT)
		{
			woStatus |= WOS_WINDOWS_ACTION;
			SendMessage(screenID, WM_ACTIVATE, 1, 0);
			if (FlagSet(woFlags, WOF_AUTO_CLEAR))
				SendMessage(screenID, EM_SETSEL, 0, 0x7FFF0000L);
			SetFocus(screenID);
			woStatus &= ~WOS_WINDOWS_ACTION;
		}
		else if (ccode == S_DISPLAY_ACTIVE || ccode == S_DISPLAY_INACTIVE)
		{
			SendMessage(screenID, WM_SETREDRAW, TRUE, 0);
			InvalidateRect(screenID, NULL, FALSE);
			ShowWindow(screenID, SW_SHOWNA);
			UpdateWindow(screenID);
		}

		// Call the associated user or validate function.
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION) ||
			(ccode != S_CURRENT && ccode != S_NON_CURRENT))
			break;

		woStatus |= WOS_INTERNAL_ACTION;	// Prevent the user function from 
		if (userFunction)					// being called more than once.
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, ccode);
		}
		else if (ccode == S_NON_CURRENT)
			ccode = (Validate(TRUE) == 0) ? 0 : -1;
		woStatus &= ~WOS_INTERNAL_ACTION;
		break;

	case S_ADD_OBJECT:
	case S_SUBTRACT_OBJECT:
		if (FlagSet(woStatus, WOS_WINDOWS_ACTION))
		{
			UI_EVENT event(ccode);
			event.data = this;
			object = parent ? parent : windowManager;
			object->woStatus |= WOS_WINDOWS_ACTION;
			object->Event(event);
			object->woStatus &= ~WOS_WINDOWS_ACTION;
		}
		else if (parent)
			parent->Event(event);
		break;

	case S_REDISPLAY:
		if (screenID)
		{
			woStatus |= WOS_REDISPLAY | WOS_INTERNAL_ACTION;	// Don't call user-function.
			SendMessage(screenID, WM_SETREDRAW, TRUE, 0);
			InvalidateRect(screenID, NULL, TRUE);
			ShowWindow(screenID, SW_SHOWNA);
			UpdateWindow(screenID);
			woStatus &= ~WOS_INTERNAL_ACTION;
		}
		else if (parent && parent->screenID)
			return (parent->Event(event));
		break;

	case L_HELP:
		if (event.type == E_MSWINDOWS)
			eventManager->Put(UI_EVENT(ccode));
		else if (parent && helpContext == NO_HELP_CONTEXT)
			ccode = S_UNKNOWN;
		else if (helpSystem)
			helpSystem->DisplayHelp(windowManager, helpContext);
		break;

	case L_SELECT:
		if (!FlagSet(woFlags, WOF_NON_SELECTABLE) && parent)
		{
			WNF_FLAGS wnFlags = WNF_NO_FLAGS;
			parent->Information(GET_FLAGS, &wnFlags, ID_WINDOW);
			if (FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
				woStatus ^= WOS_SELECTED;
			else
				woStatus |= WOS_SELECTED;
		}

		woStatus |= WOS_INTERNAL_ACTION;	// Prevent the user function from 
		if (userFunction)					// being called more than once.
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, L_SELECT);
		}
		else
			ccode = (Validate(TRUE) == 0) ? 0 : -1;
		woStatus &= ~WOS_INTERNAL_ACTION;
		break;

	case L_PREVIOUS:
	case L_NEXT:
		if (event.type == E_MSWINDOWS)
			eventManager->Put(UI_EVENT(ccode));
		else
			ccode = S_UNKNOWN;
		break;

	case E_KEY:
		return (CallWindowProc((FARPROC)defaultCallback, screenID, WM_CHAR, event.key.value, 0));

	default:
		ccode = S_UNKNOWN;
	}
	if (ccode != S_UNKNOWN || event.type != E_MSWINDOWS)
		return (ccode);
	else if (!windowManager)
		return (CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam));
		
	// Switch on the windows message.
	ccode = TRUE;
	WORD message = event.message.message;
	switch (message)
	{
	case WM_NCDESTROY:
		ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam);
		if (!parent)
		{
			woStatus |= WOS_WINDOWS_ACTION;
			*windowManager - this;
			woStatus &= ~WOS_WINDOWS_ACTION;
		}
		screenID = 0;
		break;

	case WM_MEASUREITEM:
		{
		MEASUREITEMSTRUCT *item = (MEASUREITEMSTRUCT *)event.message.lParam;
		item->itemWidth = relative.right - relative.left - 2;
		item->itemHeight = relative.bottom - relative.top - 2;
		}
		break;

	case WM_NCLBUTTONDBLCLK:
	case WM_NCLBUTTONDOWN:
	case WM_LBUTTONDBLCLK:
	case WM_LBUTTONDOWN:
		{
		if (!FlagSet(woStatus, WOS_EDIT_MODE))
		{
			ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
				event.message.message, event.message.wParam, event.message.lParam);
			break;
		}
		UI_TIME currentTime;
		UI_EVENT dEvent;
		UI_EVENT mEvent;
		mEvent = event;
		if (message == WM_NCLBUTTONDBLCLK || message == WM_NCLBUTTONDOWN)
		{
			RECT rect;
			GetWindowRect(screenID, &rect);
			mEvent.position.column -= rect.left;
			mEvent.position.line -= rect.top;
		}
		if (currentTime - lastTime < doubleClickRate)
		{
			dEvent.type = D_EDIT_OBJECT;
			dEvent.rawCode = searchID;
			dEvent.data = this;
			eventManager->Put(dEvent);
			break;
		}
		if (parent)
		{
			if (screenID && GetFocus() != screenID)
				SetFocus(screenID);
			Modify(mEvent);
		}
		else
			ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
				event.message.message, event.message.wParam, event.message.lParam);

		// Set the new current object.
		dEvent.type = D_SET_OBJECT;
		dEvent.rawCode = searchID;
		dEvent.data = this;
		eventManager->Put(dEvent);
		dEvent.type = D_SET_POSITION;
		dEvent.rawCode = M_LEFT | M_LEFT_CHANGE;
		dEvent.position = mEvent.position;
		eventManager->Put(dEvent);
		lastTime = currentTime;
		}
		break;

	case WM_NCRBUTTONDBLCLK:
	case WM_NCRBUTTONDOWN:
	case WM_RBUTTONDBLCLK:
	case WM_RBUTTONDOWN:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
		{
			UI_EVENT dEvent;
			dEvent.type = D_SET_POSITION;
			dEvent.rawCode = M_RIGHT | M_RIGHT_CHANGE;
			dEvent.position = event.position;
			eventManager->Put(dEvent);
			break;
		}
		else
			ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
				event.message.message, event.message.wParam, event.message.lParam);
		break;

	case WM_MOUSEACTIVATE:
		if (!FlagSet(woStatus, WOS_WINDOWS_ACTION | WOS_CURRENT))
		{
			woStatus |= WOS_WINDOWS_ACTION;
			UI_WINDOW_OBJECT::Event(UI_EVENT(S_ADD_OBJECT));
			woStatus &= ~WOS_WINDOWS_ACTION;
		}
		ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam);
		break;

	case WM_SETCURSOR:
		if ((FlagSet(woStatus, WOS_EDIT_MODE) && parent) ||
			(FlagSet(woAdvancedFlags, WOAF_NO_SIZE) && !parent))
		{
			SetCursor(LoadCursor(0, IDC_ARROW));
			ccode = TRUE;
			break;
		}
		ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam);
		break;

	default:
		ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam);
		break;
	}

	// Return the control code.
	return (ccode);
}

void UI_WINDOW_OBJECT::RegisterObject(char *name, char *baseName,
	int *offset, FARPROC *procInstance, FARPROC *defProcInstance,
	char *title, HMENU menu)
{
	// Set up the object init procedure.
	static FARPROC _initInstance = NULL;
	if (!_initInstance)
		_initInstance = MakeProcInstance((FARPROC)ObjectInitProcedure, display->hInstance);

	// Register the windows class object.
	extern int ZincClassAdd(char *name, FARPROC farproc, HANDLE hInstance);
	if (*offset == -1 && !baseName)
	{
		WNDCLASS wc;
		*offset = (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && parent) ?
			sizeof(LOCALHANDLE) : 0;
		if (*defProcInstance && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && parent)
			*defProcInstance = (FARPROC)DefMDIChildProc;
		else if (*defProcInstance)
			*defProcInstance = (FARPROC)DefWindowProc;
		else
			*defProcInstance = NULL;
		*procInstance = MakeProcInstance(*procInstance, display->hInstance);
		wc.style = 0;
		wc.lpfnWndProc = (DEFWINPROC)_initInstance;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = *offset + sizeof(UI_WINDOW_OBJECT *);
		wc.hInstance = display->hInstance;
		wc.hIcon = 0;
		wc.hCursor = LoadCursor(0, IDC_ARROW);
		if (Inherited(ID_TOOL_BAR))
			wc.hbrBackground = COLOR_BTNFACE + 1;
		else
			wc.hbrBackground = COLOR_WINDOW + 1;
		wc.lpszMenuName = NULL;
		wc.lpszClassName = name;
		RegisterClass(&wc);
		ZincClassAdd(name, *procInstance, wc.hInstance);
	}
	else if (*offset == -1)
	{
		WNDCLASS wc;
		GetClassInfo(0, baseName, &wc);
		*offset = wc.cbWndExtra;
		*defProcInstance = (FARPROC)wc.lpfnWndProc;
		*procInstance = MakeProcInstance(*procInstance, display->hInstance);
		wc.lpfnWndProc = (DEFWINPROC)_initInstance;
		wc.cbWndExtra += sizeof(UI_WINDOW_OBJECT *);
		wc.hInstance = display->hInstance;
		wc.lpszMenuName = NULL;
		wc.lpszClassName = name;
		RegisterClass(&wc);
		ZincClassAdd(name, *procInstance, wc.hInstance);
	}

	// Get the class registration information and seed the init procedure.
	if (*defProcInstance)
		defaultCallback = *defProcInstance;
	_object = this;

	_offset = *offset;
	_controlID = numberID;
	_jumpInstance = *procInstance;

	// Create the object.
	woStatus |= WOS_INTERNAL_ACTION;
	DWORD exFlags = WS_EX_NOPARENTNOTIFY;
	if (FlagSet(woAdvancedFlags, WOAF_DIALOG_OBJECT))
		exFlags |= WS_EX_DLGMODALFRAME;
	if (parent && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
	{
		SCREENID parentMDI;
		parent->Information(GET_MDIHANDLE, &parentMDI);
		MDICREATESTRUCT mdiCreate;
		mdiCreate.szClass = name;
		mdiCreate.szTitle = title;
		mdiCreate.hOwner = display->hInstance;
		mdiCreate.style = dwStyle;
		mdiCreate.x = true.left;
		mdiCreate.y = true.top;
		mdiCreate.cx = true.right - true.left + 1;
		mdiCreate.cy = true.bottom - true.top + 1;
		mdiCreate.lParam = 0;
		screenID = LOWORD(SendMessage(parentMDI, WM_MDICREATE, 0, (LONG)&mdiCreate));
	}
	else
		screenID = CreateWindowEx(exFlags, name, title, dwStyle,
			true.left, true.top, true.right - true.left + 1,
			true.bottom - true.top + 1, parent ? parent->screenID : 0,
			menu, display->hInstance, NULL);
	if (font != FNT_SYSTEM_FONT)
		SendMessage(screenID, WM_SETFONT, UI_MSWINDOWS_DISPLAY::fontTable[font], FALSE);
	woStatus &= ~WOS_INTERNAL_ACTION;
}
#else
EVENT_TYPE UI_WINDOW_OBJECT::DrawBorder(SCREENID screenID, UI_REGION &region,
	int fillRegion, EVENT_TYPE ccode)
{
	if (display->isText && (region.bottom == region.top || !FlagSet(woAdvancedFlags, WOAF_TEMPORARY)))
		return (ccode);

	// Draw the object border.
	UI_PALETTE *outline = LogicalPalette(ccode, ID_OUTLINE);
	display->Rectangle(screenID, region, outline, 1, FALSE, FALSE, &clip);
	--region;

	// Fill in the region.
	if (fillRegion)
		display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
	return (ccode);
}

EVENT_TYPE UI_WINDOW_OBJECT::DrawShadow(SCREENID screenID, UI_REGION &region,
	int depth, int fillRegion, EVENT_TYPE ccode)
{
	// Make sure there is depth to the object
	if (!depth || (display->isText && (Abs(depth) < 2 || region.bottom - region.top < 1)))
		return (ccode);

	// Draw the shadow.
 	if (!display->isText)
	{
		UI_PALETTE *lightShadow = LogicalPalette(ccode, (depth > 0) ? ID_WHITE_SHADOW : ID_DARK_SHADOW);
		UI_PALETTE *darkShadow = LogicalPalette(ccode, (depth > 0) ? ID_DARK_SHADOW : ID_LIGHT_SHADOW);
		depth = Abs(depth);
		for (int i = 0; i < depth; i++)
		{
			display->Line(screenID, region.right, region.top, region.right,
				region.bottom, darkShadow, 1, FALSE, &clip);
			display->Line(screenID, region.left, region.bottom, region.right - 1,
				region.bottom, darkShadow, 1, FALSE, &clip);
			display->Line(screenID, region.left, region.top, region.right - 1,
				region.top, lightShadow, 1, FALSE, &clip);
			display->Line(screenID, region.left, region.top + 1, region.left,
				region.bottom - 1, lightShadow, 1, FALSE, &clip);
			--region;
		}
	}
	else if (searchID == ID_BORDER)
	{
		display->Rectangle(screenID, region, NULL, 0, TRUE, FALSE,
			(parent->parent ? &clip : NULL));
		region.right--;
		region.bottom--;
	}
	else if (depth > 0)
	{
		UI_PALETTE *palette = parent->LogicalPalette(ccode);
		char shadow = '';
		display->Text(screenID, region.right, region.top, &shadow, palette, 1, FALSE, FALSE, &clip);
		shadow = '';
		for (int row = region.top + 1; row < region.bottom; row++)
			display->Text(screenID, region.right, row, &shadow, palette, 1, FALSE, FALSE, &clip);
		shadow = palette->fillCharacter;
		display->Text(screenID, region.left, region.bottom, &shadow, palette, 1, FALSE, FALSE, &clip);
		shadow = '';
		for (int column = region.left + 1; column <= region.right; column++)
			display->Text(screenID, column, region.bottom, &shadow, palette, 1, FALSE, FALSE, &clip);
		region.right--;
		region.bottom--;
	}
	else
	{
		UI_PALETTE *palette = parent->LogicalPalette(ccode);
		display->Rectangle(screenID, region, palette, 0, TRUE, FALSE, &clip);
		region.left++;
		region.bottom--;
	}

	// Fill in the region.
	if (fillRegion && region.bottom > region.top)
		display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
	return (ccode);
}

EVENT_TYPE UI_WINDOW_OBJECT::DrawText(SCREENID screenID, UI_REGION &region,
	const char *text, UI_PALETTE *palette, int fillRegion, EVENT_TYPE ccode)
{
	// Fill in the region.
	if (!palette)
		palette = lastPalette;
	display->VirtualGet(screenID, region);
	if (fillRegion)
		display->Rectangle(screenID, region, palette, 0, TRUE, FALSE, &clip);

	// Draw the text.
	if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		region.left = region.right - display->TextWidth(text, screenID, font);
	else if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		region.left = region.left + (region.right - region.left + 1 - display->TextWidth(text, screenID, font)) / 2;
	region.top += (region.bottom - region.top + 1 - display->TextHeight(text, screenID, font)) / 2;
	UI_REGION clipRegion = clip;
	if (clipRegion.right > region.right)
		clipRegion.right = region.right;
	if (clipRegion.bottom > region.bottom)
		clipRegion.bottom = region.bottom;
	display->Text(screenID, region.left, region.top, text, palette, -1, FALSE, FALSE, &clipRegion, font);
	char *hotKey = display->isText ? strchr(text, '&') : NULL;
	if (hotKey)
	{
		palette = LogicalPalette(ccode, ID_HOT_KEY);
		display->Text(screenID, region.left + (int)(hotKey - (char *)text),
			region.top, &hotKey[1], palette, 1, FALSE, FALSE, &clipRegion, font);
	}
	display->VirtualPut(screenID);
	return (ccode);
}

EVENT_TYPE UI_WINDOW_OBJECT::Event(const UI_EVENT &event)
{
	static initializedTime = FALSE;
	static UI_TIME lastTime;
	if (!initializedTime)
	{
		lastTime.Import();
		initializedTime = TRUE;
	}

	// Switch on the system event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_OBJECT);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!numberID && parent)
		{
			for (UI_WINDOW_OBJECT *object = parent; object->parent; object = object->parent)
				;
			numberID = object->NumberID();
			object->NumberID(numberID + 1);
			if (stringID[0] == '\0')
				sprintf(stringID, "FIELD_%d", numberID);
		}
		RegionConvert(relative, (parent && !FlagSet(woFlags, WOF_NON_FIELD_REGION)) ? FALSE : TRUE);
		break;

	case S_SIZE:
	case S_CREATE:
		woStatus |= WOS_REDISPLAY;
		if (!screenID)
		{
			if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT | WOAF_TEMPORARY))
				screenID = ++(windowManager->currentScreenID);
			else
				screenID = parent->screenID;
		}
		if (!parent || FlagSet(woAdvancedFlags, WOAF_TEMPORARY))
			RegionMax(this);
		else
			parent->RegionMax(this);
		break;

	case S_RESET_DISPLAY:
		{
		UI_REGION region;
		region = relative;
		if (!display->isText && FlagSet(woStatus, WOS_GRAPHICS) && FlagSet(woFlags, WOF_MINICELL))
		{
			long miniNX = display->miniNumeratorX, miniDX = display->miniDenominatorX;
			long miniNY = display->miniNumeratorY, miniDY = display->miniDenominatorY;
			region.left = (int)((miniDX * region.left) / (miniNX * display->cellWidth));
			region.top = (int)((miniDY * region.top) / (miniNY * display->cellHeight));
			region.right = (int)((miniDX * region.right) / (miniNX * display->cellWidth));
			region.bottom = (int)((miniDY * region.bottom) / (miniNY * display->cellHeight));
			woStatus &= ~WOS_GRAPHICS;
			woStatus |= WOS_REDISPLAY;
		}
		else if (!display->isText && FlagSet(woStatus, WOS_GRAPHICS))
		{
			int width = (region.right - region.left) / display->cellWidth;
			int height = (region.bottom - region.top) / display->cellHeight;
			region.left /= display->cellWidth;
			region.bottom /= display->cellHeight;
			region.right = region.left + width;
			region.top = region.bottom - height;
			woStatus &= ~WOS_GRAPHICS;
			woStatus |= WOS_REDISPLAY;
		}
		relative = region;
		true = relative;
		screenID = NULL;
		}
		break;

	case S_REGION_DEFINE:
		display->RegionDefine(screenID, true);
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
		// Check the object status.
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			lastPalette = LogicalPalette(ccode);
			break;
		}
		else if (ccode == S_CURRENT)
		{
			if (!FlagSet(woStatus, WOS_CURRENT))
				woStatus |= WOS_REDISPLAY;
			woStatus |= WOS_CURRENT;
		}
		else
		{
			if (FlagSet(woStatus, WOS_CURRENT))
				woStatus |= WOS_REDISPLAY;
			woStatus &= ~WOS_CURRENT;
		}
		// Continue to S_DISPLAY_ACTIVE.
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		{
		UI_PALETTE *palette = LogicalPalette(ccode);
		if (FlagSet(woStatus, WOS_REDISPLAY) || palette != lastPalette ||
			(ccode != S_NON_CURRENT && true.Overlap(event.region)))
		{
			woStatus |= WOS_REDISPLAY;
			lastPalette = palette;
		}
		// Call the associated user or validate function.
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION) ||
			(ccode != S_CURRENT && ccode != S_NON_CURRENT))
			break;

		woStatus |= WOS_INTERNAL_ACTION;	// Prevent the user function from 
		if (userFunction)					// being called more than once.
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, ccode);
		}
		else if (ccode == S_NON_CURRENT)
			ccode = (Validate(TRUE) == 0) ? 0 : -1;
		woStatus &= ~WOS_INTERNAL_ACTION;
		}
		break;

	case S_REDISPLAY:
		if (screenID)
		{
			// Find the root to determine activity.
			UI_EVENT event;
			event.type = FlagSet(woStatus, WOS_CURRENT) ? S_CURRENT : S_DISPLAY_ACTIVE;
			UI_WINDOW_OBJECT *root = this;
			while (root->parent)
			{
				root = root->parent;
				if (!FlagSet(root->woStatus, WOS_CURRENT))
					event.type = S_DISPLAY_ACTIVE;
			}
			if (!FlagSet(root->woStatus, WOS_CURRENT))
				event.type = S_DISPLAY_INACTIVE;
			event.region = true;
			woStatus |= WOS_REDISPLAY | WOS_INTERNAL_ACTION;	// Don't call user-function.
			Event(event);
			woStatus &= ~WOS_INTERNAL_ACTION;
		}
//		else if (parent && parent->screenID)
//			return (parent->Event(event));
		break;

	case S_ADD_OBJECT:
	case S_SUBTRACT_OBJECT:
		if (parent)
			parent->Event(event);
		break;

	case L_HELP:
		if (parent && helpContext == NO_HELP_CONTEXT)
			ccode = S_UNKNOWN;
		else if (helpSystem)
			helpSystem->DisplayHelp(windowManager, helpContext);
		break;

	case L_VIEW:
		if (true.Overlap(event.position))
			eventManager->DeviceState(E_MOUSE, DM_VIEW);
		break;

	case L_SELECT:
		if (!FlagSet(woFlags, WOF_NON_SELECTABLE) && parent && parent->Inherited(ID_LIST))
		{
			WNF_FLAGS wnFlags = WNF_NO_FLAGS;
			parent->Information(GET_FLAGS, &wnFlags, ID_WINDOW);
			if (FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
				woStatus ^= WOS_SELECTED;
			else
				woStatus |= WOS_SELECTED;
		}

		woStatus |= WOS_INTERNAL_ACTION;	// Prevent the user function from 
		if (userFunction)					// being called more than once.
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, L_SELECT);
		}
		else
			ccode = (Validate(TRUE) == 0) ? 0 : -1;
		woStatus &= ~WOS_INTERNAL_ACTION;
		break;

	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		{
		UI_EVENT dEvent;
		if (!true.Overlap(event.position))
			ccode = S_UNKNOWN;
		if (ccode == L_END_SELECT && !parent && FlagSet(woStatus, WOS_EDIT_MODE))
		{
			UI_TIME currentTime;
			if (currentTime - lastTime < doubleClickRate)
			{
				dEvent.type = D_EDIT_OBJECT;
				dEvent.rawCode = searchID;
				dEvent.data = this;
				eventManager->Put(dEvent);
			}
			else
				lastTime = currentTime;
			break;
		}
		else if (ccode != L_BEGIN_SELECT || !FlagSet(woStatus, WOS_EDIT_MODE))
			break;

		dEvent.position = event.position;
		int hOffset = 1, vOffset = 0;
		if (!display->isText)
			hOffset = vOffset = 5;
		else if (true.right == true.left)
			hOffset = 0;
		if (event.position.column >= true.left + hOffset && event.position.line >= true.top + vOffset &&
			event.position.column <= true.right - hOffset && event.position.line <= true.bottom - vOffset)
		{
			dEvent.type = L_MOVE;
			dEvent.rawCode = M_LEFT_CHANGE | M_TOP_CHANGE | M_RIGHT_CHANGE | M_BOTTOM_CHANGE;
		}
		else
		{
			dEvent.type = L_SIZE;
			dEvent.rawCode = 0;
			if (event.position.column <= true.left + hOffset)
				dEvent.rawCode |= M_LEFT_CHANGE;
			else if (event.position.column >= true.right - hOffset)
				dEvent.rawCode |= M_RIGHT_CHANGE;
			if (event.position.line <= true.top + vOffset)
				dEvent.rawCode |= M_TOP_CHANGE;
			else if (event.position.line >= true.bottom - vOffset)
				dEvent.rawCode |= M_BOTTOM_CHANGE;
		}
		if (parent)
			Modify(dEvent);

		// Set the current object.
		dEvent.type = D_SET_OBJECT;
		dEvent.rawCode = searchID;
		dEvent.data = this;
		eventManager->Put(dEvent);
		dEvent.type = D_SET_POSITION;
		dEvent.rawCode = M_LEFT | M_LEFT_CHANGE;
		dEvent.position = event.position;
		eventManager->Put(dEvent);
		if (!parent)
			break;
		UI_TIME currentTime;
		if (currentTime - lastTime < doubleClickRate)
		{
			dEvent.type = D_EDIT_OBJECT;
			dEvent.rawCode = searchID;
			dEvent.data = this;
			eventManager->Put(dEvent);
		}
		else
			lastTime = currentTime;
		}
		break;

	case L_BEGIN_ESCAPE:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
		{
			UI_EVENT dEvent;
			dEvent.type = D_SET_POSITION;
			dEvent.rawCode = event.rawCode;
			dEvent.position = event.position;
			eventManager->Put(dEvent);
			break;
		}
		// Continue to default.
	default:
		ccode = S_UNKNOWN;
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

unsigned UI_WINDOW_OBJECT::HotKey(unsigned _hotKey)
{
	if (_hotKey)
		hotKey = _hotKey;
	return (hotKey);
}

int UI_WINDOW_OBJECT::NeedsUpdate(const UI_EVENT &, EVENT_TYPE)
{
	return (TRUE);
}

void *UI_WINDOW_OBJECT::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_WINDOW_OBJECT;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = windowID[1] = windowID[2] = windowID[3] =
			windowID[4] = windowID[5] = ID_WINDOW_OBJECT;
		screenID = 0;
		parent = NULL;
		lastPalette = NULL;

		hotKey = 0;
		woStatus = defaultStatus;
		font = FNT_DIALOG_FONT;
		extern UI_EVENT_MAP *_eventMapTable;
		eventMapTable = _eventMapTable;
		extern UI_EVENT_MAP *_hotKeyMapTable;
		hotKeyMapTable = _hotKeyMapTable;
		extern UI_PALETTE_MAP *_normalPaletteMapTable;
		paletteMapTable = _normalPaletteMapTable;
		if (FlagSet(woFlags, WOF_INVALID))
			woStatus |= WOS_INVALID;
		if (FlagSet(woFlags, WOF_UNANSWERED))
			woStatus |= WOS_UNANSWERED;
#ifdef _WINDOWS
		dwStyle = 0;
		defaultCallback = NULL;
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (screenID)
			dwStyle = parent ? WS_CHILD | WS_VISIBLE : WS_OVERLAPPED;
		else
			dwStyle = 0;
		if (FlagSet(woFlags, WOF_BORDER))
			dwStyle |= WS_BORDER;
		if (FlagSet(woFlags, WOF_NON_SELECTABLE))
			dwStyle |= WS_DISABLED;
#endif
		break;

	case GET_NUMBERID_OBJECT:
		data = (numberID == *(USHORT *)data) ? this : NULL;
		break;

	case GET_STRINGID_OBJECT:
		data = !stricmp(stringID, (char *)data) ? this : NULL;
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_WINDOW_OBJECT)
			return (NULL);
		else if (request == GET_FLAGS && !data)
			data = &woFlags;
		else if (request == GET_FLAGS)
			*(WOF_FLAGS *)data = woFlags;
		else if (request == SET_FLAGS)
			woFlags |= *(WOF_FLAGS *)data;
		else
			woFlags &= ~(*(WOF_FLAGS *)data);
		break;

	case GET_STATUS:
	case SET_STATUS:
	case CLEAR_STATUS:
		if (objectID && objectID != ID_WINDOW_OBJECT)
			return (NULL);
		else if (request == GET_STATUS && !data)
			data = &woStatus;
		else if (request == GET_STATUS)
			*(WOS_STATUS *)data = woStatus;
		else if (request == SET_STATUS)
			woStatus |= *(WOS_STATUS *)data;
		else if (request == SET_STATUS)
			woStatus &= ~(*(WOS_STATUS *)data);
		break;

#ifdef _WINDOWS
	case GET_DWSTYLE:		*(DWORD *)data = dwStyle;			break;
	case SET_DWSTYLE:		dwStyle |= *(DWORD *)data;			break;
	case CLEAR_DWSTYLE:		dwStyle &= ~(*(DWORD *)data);		break;
#endif

#ifdef ZIL_PERSISTENCE
	case PRINT_INFORMATION:
		if (data && stringID[0] != '\0' && numberID < 0xFF00)
		{
			strupr(stringID);
			UI_STORAGE_OBJECT *hppEntry = (UI_STORAGE_OBJECT *)data;
			char buffer[128];
			sprintf(buffer, "const USHORT %-32s = 0x%04X;", stringID, numberID);
			hppEntry->Store(buffer);
		}
		break;

	case PRINT_USER_FUNCTION:
		if (userFunctionName)
		{
			UI_STORAGE_OBJECT *cppEntry = (UI_STORAGE_OBJECT *)data;
			cppEntry->Store(userFunctionName);
		}
		break;
#endif

	default:
		data = NULL;
		break;
	}

	// Return the information.
	return (data);
}

EVENT_TYPE UI_WINDOW_OBJECT::LogicalEvent(const UI_EVENT &event, OBJECTID currentID)
{
	// See if there is a matching identification.
	if (!currentID)
		currentID = windowID[0];
	OBJECTID id[5];
	for (int i = 0; i < 5; i++)
		if (currentID == windowID[i])
			break;
	id[0] = currentID;
	for (int j = 1; j < 5; j++)
		id[j] = (i < 4) ? windowID[++i] : ID_WINDOW_OBJECT;

	return (UI_EVENT_MAP::MapEvent(eventMapTable, event, id[0], id[1], id[2], id[3], id[4])); 
}

UI_PALETTE *UI_WINDOW_OBJECT::LogicalPalette(LOGICAL_EVENT logicalEvent, OBJECTID currentID)
{
	// Determine the logical palette.
	LOGICAL_PALETTE logicalPalette = PM_ANY;
	if (FlagSet(woFlags, WOF_NON_SELECTABLE))
		logicalPalette = PM_NON_SELECTABLE;
	else if (logicalEvent == S_CURRENT && FlagSet(woStatus, WOS_CURRENT))
		logicalPalette = PM_CURRENT;
	else if (logicalEvent == S_CURRENT || logicalEvent == S_DISPLAY_ACTIVE ||
		logicalEvent == S_NON_CURRENT ||
		(parent && FlagSet(parent->woStatus, WOS_CURRENT)))
		logicalPalette = PM_ACTIVE;
	if (FlagSet(woStatus, WOS_SELECTED))
		logicalPalette |= PM_SELECTED;
	if (currentID == ID_HOT_KEY)
	{
		logicalPalette |= PM_HOT_KEY;
		currentID = 0;
	}

	// See if there is a matching identification.
	if (!currentID)
		currentID = windowID[0];
	OBJECTID id[5];
	for (int i = 0; i < 5; i++)
		if (currentID == windowID[i])
			break;
	id[0] = currentID;
	for (int j = 1; j < 5; j++)
		id[j] = (i < 4) ? windowID[++i] : ID_WINDOW_OBJECT;

	return (UI_PALETTE_MAP::MapPalette(paletteMapTable, logicalPalette, id[0], id[1], id[2], id[3], id[4]));
}

#ifdef _WINDOWS
void UI_WINDOW_OBJECT::Modify(const UI_EVENT &event)
{
	// Make sure we can move the object.
	if ((parent && FlagSet(woFlags, WOF_NON_FIELD_REGION)) ||
		(event.type == L_SIZE && Inherited(ID_ICON)))
		return;

	RECT newRegion = { 0, 0, relative.right - relative.left, relative.bottom - relative.top };
	RECT oldRegion = { 5, 5, relative.right - relative.left - 5, relative.bottom - relative.top - 5 };
	RAW_CODE sizeFlags = 0;
	EVENT_TYPE operation = L_SIZE;
	if (event.type == L_MOVE || 
		(event.position.column >= oldRegion.left && event.position.line >= oldRegion.top &&
		event.position.column <= oldRegion.right && event.position.line <= oldRegion.bottom))
	{
		sizeFlags = M_LEFT_CHANGE | M_TOP_CHANGE | M_RIGHT_CHANGE | M_BOTTOM_CHANGE;
		operation = L_MOVE;
	}
	if (oldRegion.left > event.position.column)
		sizeFlags |= M_LEFT_CHANGE;
	else if (oldRegion.right < event.position.column)
		sizeFlags |= M_RIGHT_CHANGE;
	if (oldRegion.top > event.position.line)
		sizeFlags |= M_TOP_CHANGE;
	else if (oldRegion.bottom < event.position.line)
		sizeFlags |= M_BOTTOM_CHANGE;

	// Determine the minimum height and width of the object.
	int minHeight = display->cellHeight;
	if (parent)
		minHeight -= (display->preSpace + display->postSpace);
	else
		minHeight += 2 * UIW_BORDER::width;
	int minWidth = 2 * display->cellWidth;
	if (Inherited(ID_WINDOW) && !Inherited(ID_SCROLL_BAR))
		minWidth *= 5;

	// Determine the absolute region.
	UI_REGION absolute;
	if (parent)
	{
		RECT rect;
		GetClientRect(parent->screenID, &rect);
		absolute.left = -relative.left;
		absolute.top = -relative.top;
		absolute.right = (rect.right - rect.left + 1) + absolute.left;
		absolute.bottom = (rect.bottom - rect.top + 1) + absolute.top;
	}
	else
	{
		absolute.left = absolute.top = 0;
		absolute.right = display->columns - 1;
		absolute.bottom = display->lines - 1;
	}

	if (operation != L_MOVE && (!Inherited(ID_WINDOW) || Inherited(ID_COMBO_BOX)))
		sizeFlags &= ~(M_TOP_CHANGE | M_BOTTOM_CHANGE);
	else if (operation != L_MOVE && Inherited(ID_SCROLL_BAR))
	{
		SBF_FLAGS sbFlags = SBF_NO_FLAGS;
		Information(GET_FLAGS, &sbFlags, ID_SCROLL_BAR);
		if (!FlagSet(sbFlags, SBF_VERTICAL))
		{
			minHeight = relative.bottom - relative.top + 1;
			sizeFlags &= ~(M_TOP_CHANGE | M_BOTTOM_CHANGE);
		}
		if (!FlagSet(sbFlags, SBF_HORIZONTAL))
		{
			minWidth = relative.right - relative.left + 1;
			sizeFlags &= ~(M_LEFT_CHANGE | M_RIGHT_CHANGE);
		}
		if (!sizeFlags)
			return;
	}

	int xJump = display->cellWidth;
	int yJump = display->cellWidth;
	UI_POSITION origin = event.position;
	SetCapture(screenID);
	HDC hDC = GetDC(screenID);
	DrawFocusRect(hDC, &newRegion);
	WORD message;
	EVENT_TYPE ccode = 0;
	do
	{
		UI_EVENT event;
		eventManager->Get(event);
		message = event.message.message;
		int deltaX = 0, deltaY = 0;
		UI_POSITION oldOrigin = origin;
		if (message == WM_LBUTTONDOWN || message == WM_NCLBUTTONDOWN ||
			(message == WM_MOUSEMOVE && FlagSet(event.message.wParam, MK_LBUTTON)))
		{
			if (event.position.column < absolute.left)			// Check the absolute region.
				event.position.column = absolute.left;
			else if (event.position.column > absolute.right)
				event.position.column = absolute.right;
			if (event.position.line < absolute.top)
				event.position.line = absolute.top;
			else if (event.position.line > absolute.bottom)
				event.position.line = absolute.bottom;
			deltaX = event.position.column - origin.column;
			deltaY = event.position.line - origin.line;
			origin = event.position;
		}
		else if (message >= WM_KEYFIRST && message <= WM_KEYLAST)
		{
			ccode = LogicalEvent(event, ID_WINDOW_OBJECT);
			switch (ccode)
			{
			case L_UP:
			case L_DOWN:
				deltaY = (ccode == L_UP) ? deltaY - yJump : deltaY + yJump;
				if (newRegion.top + deltaY < absolute.top)
					deltaY = absolute.top - newRegion.top;
				else if (newRegion.bottom + deltaY > absolute.bottom)
					deltaY = absolute.bottom - newRegion.bottom;
				break;

			case L_LEFT:
			case L_RIGHT:
				deltaX = (ccode == L_LEFT) ? deltaX - xJump : deltaX + xJump;
				if (newRegion.left + deltaX < absolute.left)
					deltaX = absolute.left - newRegion.left;
				else if (newRegion.right + deltaX > absolute.right)
					deltaX = absolute.right - newRegion.right;
				break;

			default:
				if (event.key.value == 0x000D)		// ENTER
					ccode = L_SELECT;
				else if (event.key.value == 0x001B)	// ESCAPE
					ccode = L_CANCEL;
				break;
			}
		}

		// Update the new region.
		oldRegion = newRegion;
		if (FlagSet(sizeFlags, M_LEFT_CHANGE))
			newRegion.left += deltaX;
		if (FlagSet(sizeFlags, M_TOP_CHANGE))
			newRegion.top += deltaY;
		if (FlagSet(sizeFlags, M_RIGHT_CHANGE))
			newRegion.right += deltaX;
		if (FlagSet(sizeFlags, M_BOTTOM_CHANGE))
			newRegion.bottom += deltaY;
		if (operation == L_SIZE &&
			(newRegion.top + minHeight - 1 > newRegion.bottom ||
			 newRegion.left + minWidth - 1 > newRegion.right))
		{
			origin = oldOrigin;
			newRegion = oldRegion;
			deltaX = deltaY = 0;
		}

		// Update the new region.
		if (deltaX || deltaY)
		{
			DrawFocusRect(hDC, &oldRegion);
			DrawFocusRect(hDC, &newRegion);
		}
	} while (message != WM_LBUTTONUP && ccode != L_SELECT && ccode != L_CANCEL);
	DrawFocusRect(hDC, &newRegion);
	ReleaseDC(screenID, hDC);
	ReleaseCapture();

	if (ccode == L_CANCEL)
		return;				// Do not change the object region.
	if (newRegion.left || newRegion.top ||
		newRegion.right - newRegion.left != relative.right - relative.left ||
		newRegion.bottom - newRegion.top != relative.bottom - relative.top)
	{
		int width = (newRegion.right - newRegion.left + 1);
		int height = (newRegion.bottom - newRegion.top + 1);
		// Check for a cell boundary move or size.
		if (FlagSet(woStatus, WOS_EDIT_MODE) && FlagSet(woFlags, WOF_MINICELL))
		{
			long miniNX = display->miniNumeratorX, miniDX = display->miniDenominatorX;
			long miniNY = display->miniNumeratorY, miniDY = display->miniDenominatorY;

			long value = relative.left + newRegion.left;
			int ceil = (value >= 0) ? (int)(miniDX - 1) : int(1 - miniDX);
			value = (value * miniDX) / (miniNX * display->cellWidth);
			value = (value * miniNX * display->cellWidth + ceil) / miniDX;
			relative.left = (int)value;

			value = relative.top + newRegion.top;
			ceil = (value >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
			value = (value * miniDY) / (miniNY * display->cellHeight);
			value = (value * miniNY * display->cellHeight + ceil) / miniDY;
			relative.top = (int)value;

			if (operation != L_MOVE)
			{
				long value = relative.right + newRegion.right;
				int ceil = (value >= 0) ? (int)(miniDX - 1) : int(1 - miniDX);
				value = (value * miniDX) / (miniNX * display->cellWidth);
				value = (value * miniNX * display->cellWidth + ceil) / miniDX;
				relative.right = (int)value;

				value = relative.bottom + newRegion.bottom;
				ceil = (value >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
				value = (value * miniDY) / (miniNY * display->cellHeight);
				value = (value * miniNY * display->cellHeight + ceil) / miniDY;
				relative.bottom = (int)value;

				width = (relative.right - relative.left + 1);
				height = (relative.bottom - relative.top + 1);
			}
		}
		else if (FlagSet(woStatus, WOS_EDIT_MODE))
		{
			relative.left += newRegion.left / display->cellWidth * display->cellWidth;
			relative.top += newRegion.top / display->cellHeight * display->cellHeight;
			if (operation != L_MOVE)
				width = width / display->cellWidth * display->cellWidth;
			if (operation != L_MOVE && FlagSet(sizeFlags, M_TOP_CHANGE | M_BOTTOM_CHANGE))
			{
				height = (height + display->preSpace + display->postSpace) / display->cellHeight * display->cellHeight;
				height -= (display->preSpace + display->postSpace);
			}
		}
		else
		{
			relative.left += newRegion.left;
			relative.top += newRegion.top;
		}
		relative.right = relative.left + width - 1;
		if (relative.right < relative.left + minWidth)
			relative.right = relative.left + minWidth - 1;
		relative.bottom = relative.top + height - 1;
		if (relative.bottom < relative.top + minHeight)
			relative.bottom = relative.top + minHeight - 1;
		// Compensate for a border on rect.
		RECT rect = { true.left, true.top, true.right + 1, true.bottom + 1 };
		InvalidateRect(parent->screenID, &rect, TRUE);
		Event(UI_EVENT(S_SIZE));
		woStatus |= WOS_INTERNAL_ACTION;	// Don't call user-function.
		SetWindowPos(screenID, screenID, true.left, true.top,
			true.right - true.left + 1, true.bottom - true.top + 1,
			SWP_NOZORDER);
		rect.left = true.left, rect.top = true.top, rect.right = true.right + 1, rect.bottom = true.bottom + 1;
		InvalidateRect(parent->screenID, &rect, TRUE);
		UpdateWindow(parent->screenID);
		woStatus &= ~WOS_INTERNAL_ACTION;
	}
}
#else
void UI_WINDOW_OBJECT::Modify(const UI_EVENT &event)
{
	// Make sure we can move the object.
	if ((parent && FlagSet(woFlags, WOF_NON_FIELD_REGION)) ||
		(event.type == L_SIZE && Inherited(ID_ICON)))
		return;

	// Initialize the modify variables.
	UI_REGION newRegion = true;
	RAW_CODE sizeFlags = event.rawCode;
	UI_POSITION origin = event.position;
	EVENT_TYPE operation = L_SIZE;
	if (event.type == L_MOVE || Inherited(ID_ICON))
	{
		sizeFlags = M_LEFT_CHANGE | M_TOP_CHANGE | M_RIGHT_CHANGE | M_BOTTOM_CHANGE;
		operation = L_MOVE;
	}
	else if (!event.rawCode)
		sizeFlags = M_RIGHT_CHANGE | M_BOTTOM_CHANGE;

	// Determine the absolute region.
	UI_REGION absolute;
	if (parent)
		absolute = clip;
	else
	{
		absolute.left = absolute.top = 0;
		absolute.right = display->columns - 1;
		absolute.bottom = display->lines - 1;
	}

	// Determine the minimum height and width of the object.
	int minHeight = display->cellHeight;
	if (parent)
		minHeight -= (display->preSpace + display->postSpace);
	else if (!display->isText)
		minHeight += 2 * UIW_BORDER::width;
	int minWidth = 2 * display->cellWidth;
	if (Inherited(ID_WINDOW) && !Inherited(ID_SCROLL_BAR))
		minWidth *= 5;

	// Check for special object movement.
	if (operation != L_MOVE && (!Inherited(ID_WINDOW) || Inherited(ID_COMBO_BOX)))
		sizeFlags &= ~(M_TOP_CHANGE | M_BOTTOM_CHANGE);
	else if (operation != L_MOVE && Inherited(ID_SCROLL_BAR))
	{
		SBF_FLAGS sbFlags = SBF_NO_FLAGS;
		Information(GET_FLAGS, &sbFlags, ID_SCROLL_BAR);
		if (!FlagSet(sbFlags, SBF_VERTICAL))
		{
			minHeight = true.bottom - true.top + 1;
			sizeFlags &= ~(M_TOP_CHANGE | M_BOTTOM_CHANGE);
		}
		if (!FlagSet(sbFlags, SBF_HORIZONTAL))
		{
			minWidth = true.right - true.left + 1;
			sizeFlags &= ~(M_LEFT_CHANGE | M_RIGHT_CHANGE);
		}
		if (!sizeFlags)
			return;
	}

	// Reverse the window region.
	int xJump = display->cellWidth;
	int yJump = display->cellWidth;
	EVENT_TYPE ccode = S_UNKNOWN;
	display->Rectangle(ID_DIRECT, newRegion, display->xorPalette, 1, FALSE, TRUE);
	do
	{
		UI_EVENT event;
		eventManager->Get(event);
		ccode = UI_EVENT_MAP::MapEvent(eventMapTable, event);
		int deltaX = 0, deltaY = 0;
		UI_POSITION oldOrigin = origin;
		switch (ccode)
		{
		case L_BEGIN_SELECT:
			origin = event.position;
			// Continue to L_CONTINUE_SELECT.
		case L_CONTINUE_SELECT:
			if (event.position.column < absolute.left)			// Check the absolute region.
				event.position.column = absolute.left;
			else if (event.position.column > absolute.right)
				event.position.column = absolute.right;
			if (event.position.line < absolute.top)
				event.position.line = absolute.top;
			else if (event.position.line > absolute.bottom)
				event.position.line = absolute.bottom;
			deltaX = event.position.column - origin.column;		// Compute the change.
			deltaY = event.position.line - origin.line;
			origin = event.position;
			break;

		case L_UP:
		case L_DOWN:
			deltaY = (ccode == L_UP) ? deltaY - yJump : deltaY + yJump;
			if (newRegion.top + deltaY < absolute.top)
				deltaY = absolute.top - newRegion.top;
			else if (newRegion.bottom + deltaY > absolute.bottom)
				deltaY = absolute.bottom - newRegion.bottom;
			break;

		case L_LEFT:
		case L_RIGHT:
			deltaX = (ccode == L_LEFT) ? deltaX - xJump : deltaX + xJump;
			if (newRegion.left + deltaX < absolute.left)
				deltaX = absolute.left - newRegion.left;
			else if (newRegion.right + deltaX > absolute.right)
				deltaX = absolute.right - newRegion.right;
			break;
		}

		// Update the new region.
		UI_REGION oldRegion = newRegion;
		if (FlagSet(sizeFlags, M_LEFT_CHANGE))
			newRegion.left += deltaX;
		if (FlagSet(sizeFlags, M_TOP_CHANGE))
			newRegion.top += deltaY;
		if (FlagSet(sizeFlags, M_RIGHT_CHANGE))
			newRegion.right += deltaX;
		if (FlagSet(sizeFlags, M_BOTTOM_CHANGE))
			newRegion.bottom += deltaY;
		if (operation == L_SIZE &&
			(newRegion.top + minHeight - 1 > newRegion.bottom ||
			 newRegion.left + minWidth - 1 > newRegion.right))
		{
			origin = oldOrigin;
			newRegion = oldRegion;
		}
		else if (oldRegion != newRegion)
			display->RectangleXORDiff(oldRegion, newRegion);
	} while (ccode != L_END_SELECT && ccode != L_SELECT && ccode != L_CANCEL);

	// Restore the region.
	display->Rectangle(ID_DIRECT, newRegion, display->xorPalette, 1, FALSE, TRUE);
	if (ccode == L_CANCEL || newRegion == true)
		return;				// Do not change the object region.

	// Check for a cell boundary move or size.
	int width = (newRegion.right - newRegion.left + 1);
	int height = (newRegion.bottom - newRegion.top + 1);
	if (FlagSet(woStatus, WOS_EDIT_MODE) && FlagSet(woFlags, WOF_MINICELL))
	{
		long miniNX = display->miniNumeratorX, miniDX = display->miniDenominatorX;
		long miniNY = display->miniNumeratorY, miniDY = display->miniDenominatorY;

		long value = relative.left + newRegion.left - true.left;
		int ceil = (value >= 0) ? (int)(miniDX - 1) : int(1 - miniDX);
		value = (value * miniDX) / (miniNX * display->cellWidth);
		value = (value * miniNX * display->cellWidth + ceil) / miniDX;
		relative.left = (int)value;

		value = relative.top + newRegion.top - true.top;	// delta width.
		ceil = (value >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
		value = (value * miniDY) / (miniNY * display->cellHeight);
		value = (value * miniNY * display->cellHeight + ceil) / miniDY;
		relative.top = (int)value;

		if (operation != L_MOVE)
		{
			long value = relative.right + newRegion.right - true.right;
			int ceil = (value >= 0) ? (int)(miniDX - 1) : int(1 - miniDX);
			value = (value * miniDX) / (miniNX * display->cellWidth);
			value = (value * miniNX * display->cellWidth + ceil) / miniDX;
			relative.right = (int)value;

			value = relative.bottom + newRegion.bottom - true.bottom;	// delta width.
			ceil = (value >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
			value = (value * miniDY) / (miniNY * display->cellHeight);
			value = (value * miniNY * display->cellHeight + ceil) / miniDY;
			relative.bottom = (int)value;

			width = (relative.right - relative.left + 1);
			height = (relative.bottom - relative.top + 1);
		}
	}
	else if (FlagSet(woStatus, WOS_EDIT_MODE))
	{
		relative.left += (newRegion.left - true.left) / display->cellWidth * display->cellWidth;
		relative.top += (newRegion.top - true.top) / display->cellHeight * display->cellHeight;
		if (operation != L_MOVE)
			width = width / display->cellWidth * display->cellWidth;
		if (operation != L_MOVE && FlagSet(sizeFlags, M_TOP_CHANGE | M_BOTTOM_CHANGE))
		{
			height = (height + display->preSpace + display->postSpace) / display->cellHeight * display->cellHeight;
			height -= (display->preSpace + display->postSpace);
		}
	}
	else
	{
		relative.left += (newRegion.left - true.left);
		relative.top += (newRegion.top - true.top);
	}
	relative.right = relative.left + width - 1;
	if (relative.right < relative.left + minWidth)
		relative.right = relative.left + minWidth - 1;
	relative.bottom = relative.top + height - 1;
	if (relative.bottom < relative.top + minHeight)
		relative.bottom = relative.top + minHeight - 1;
	UI_REGION region = true;
	Event(UI_EVENT(S_CREATE));
	woStatus |= WOS_INTERNAL_ACTION;	// Don't call user-function.
	if (!parent)
	{
		if (display->isText)
			display->VirtualGet(ID_SCREEN, 0, 0, display->columns, display->lines);
		display->RegionDefine(ID_SCREEN, 0, 0, display->columns, display->lines);
		for (UI_WINDOW_OBJECT *object = windowManager->Last(); object; object = object->Previous())
			object->Event(UI_EVENT(S_REGION_DEFINE));
		display->Rectangle(ID_SCREEN, region, display->backgroundPalette, 0, TRUE);
		for (object = windowManager->Last(); object; object = object->Previous())
			if (object == this)
				object->Event(UI_EVENT(S_CURRENT, 0, true));
			else if (object->true.Overlap(region))
			{
				object->Event(UI_EVENT(S_DISPLAY_INACTIVE, 0, region));
				if (display->isText)	// Account for window shadowing.
				{
					region.left = Min(object->true.left, region.left);
					region.top = Min(object->true.top, region.top);
					region.right = Max(object->true.right + 1, region.right);
					region.bottom = Max(object->true.bottom + 1, region.bottom);
				}
			}
		if (display->isText)
			display->VirtualPut(ID_SCREEN);
	}
	else
	{
		if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
			parent->Event(UI_EVENT(S_REGION_DEFINE));
		if (!true.Overlap(region))
		{
			parent->Event(UI_EVENT(S_CURRENT, 0, region));
			woStatus |= WOS_REDISPLAY;
			parent->Event(UI_EVENT(S_CURRENT, 0, true));
		}
		else
		{
			region.left = Min(region.left, true.left);
			region.top = Min(region.top, true.top);
			region.right = Max(region.right, true.right);
			region.bottom = Max(region.bottom, true.bottom);
			woStatus |= WOS_REDISPLAY;
			parent->Event(UI_EVENT(S_CURRENT, 0, region));
		}
	}
	woStatus &= ~WOS_INTERNAL_ACTION;
}
#endif

USHORT UI_WINDOW_OBJECT::NumberID(USHORT _numberID)
{
	if (_numberID)
		numberID = _numberID;
	return (numberID);
}

void UI_WINDOW_OBJECT::RegionConvert(UI_REGION &region, int absolute)
{
	// Check for a convertable region.
	if (display->isText || FlagSet(woStatus, WOS_GRAPHICS))
	{
		if (relative.right < relative.left && relative.right < 0)
			region.right += display->columns;
		if (relative.bottom < relative.top && relative.bottom < 0)
			region.bottom += display->lines;
		return;
	}

	// Convert the coordinates.
	int cellWidth = display->cellWidth, cellHeight = display->cellHeight;
	if (FlagSet(woFlags, WOF_MINICELL))
	{
		long miniNX = display->miniNumeratorX, miniDX = display->miniDenominatorX;
		long miniNY = display->miniNumeratorY, miniDY = display->miniDenominatorY;
		int ceil = (region.left >= 0) ? (int)(miniDX - 1) : (int)(1 - miniDX);
		region.left = (int)((miniNX * region.left * cellWidth + ceil) / miniDX);
		ceil = (region.top >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
		region.top = (int)((miniNY * region.top * cellHeight + ceil) / miniDY);
		ceil = (region.right >= 0) ? (int)(miniDX - 1) : (int)(1 - miniDX);
		region.right = (int)((miniNX * region.right * cellWidth + ceil) / miniDX);
		ceil = (region.bottom >= 0) ? (int)(miniDY - 1) : (int)(1 - miniDY);
		region.bottom = (int)((miniNY * region.bottom * cellHeight + ceil) / miniDY);
	}
	else
	{
		region.left = region.left * cellWidth;
		region.top = region.top * cellHeight;
		region.right = region.right * cellWidth + cellWidth - 1;
		region.bottom = region.bottom * cellHeight + cellHeight - 1;
	}

	// Check for absolute coordinates.
	if (!absolute && !FlagSet(woFlags, WOF_MINICELL))
	{
		region.top += display->preSpace;
		region.bottom -= display->postSpace;
	}

	// Check for negative values.
	if (region.left < 0)
		region.left += display->columns;
	if (region.top < 0)
		region.top += display->lines;
	if (region.right < 0)
		region.right += display->columns;
	if (region.bottom < 0)
		region.bottom += display->lines;

	// Update the object status.
	woStatus |= WOS_GRAPHICS;
}

void UI_WINDOW_OBJECT::RegionMax(UI_WINDOW_OBJECT *object)
{
	object->true = relative;
}

char *UI_WINDOW_OBJECT::StringID(const char *_stringID)
{
	if (_stringID)
		strcpy(stringID, _stringID);
	return (stringID);
}

int UI_WINDOW_OBJECT::Validate(int)
{
	return (0);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UI_WINDOW_OBJECT::UI_WINDOW_OBJECT(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Intialize the window object information.
	userFlags = UIF_NO_FLAGS;
	userStatus = UIS_NO_FLAGS;
	userObject = NULL;
	userFunction = NULL;
	helpContext = NO_HELP_CONTEXT;
	Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
}

void UI_WINDOW_OBJECT::Load(const char *, UI_STORAGE *, UI_STORAGE_OBJECT *file)
{
	short value;
	file->Load(&numberID);
	file->Load(stringID, 32);
	file->Load(&woFlags);
	file->Load(&woAdvancedFlags);
	file->Load(&value); relative.left = value;
	file->Load(&value); relative.top = value;
	file->Load(&value); relative.right = value;
	file->Load(&value); relative.bottom = value;
	true = relative;
	file->Load(&value); helpContext = value;
	file->Load(&userFlags);
	file->Load(&userStatus);
	file->Load(&userObjectName);
	if (userObjectName && userTable)
	{
		for (int i = 0; !userObject && userTable[i].data; i++)
			if (!stricmp(userTable[i].text, userObjectName))
				userObject = userTable[i].data;
	}
	file->Load(&userFunctionName);
	if (userFunctionName && userTable)
	{
		for (int i = 0; !userFunction && userTable[i].data; i++)
			if (!stricmp(userTable[i].text, userFunctionName))
				userFunction = (USER_FUNCTION)userTable[i].data;
	}
}

UI_WINDOW_OBJECT *UI_WINDOW_OBJECT::New(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Check for a valid directory and file.
	int tempDirectory = FALSE;
	if (name && !file)
	{
		char pathName[128], fileName[32], objectName[32], objectPathName[128];
		UI_STORAGE::StripFullPath(name, pathName, fileName, objectName, objectPathName);
		if (!directory)
		{
			UI_STORAGE::AppendFullPath(pathName, pathName, fileName);
			UI_STORAGE::ChangeExtension(pathName, ".DAT");
			directory = new UI_STORAGE(pathName, UIS_READ);
			if (directory->storageError)
			{
				delete directory;
				return (NULL);
			}
			tempDirectory = TRUE;
		}
		if (!file)
			return (new UIW_WINDOW(name, directory, NULL));
	}

	// Read the object based on its identification.
	UI_WINDOW_OBJECT *object = NULL;
	short searchID;
	file->Load(&searchID);
	for (int i = 0; objectTable[i].data; i++)
		if (objectTable[i].value == searchID)
		{
			NEW_FUNCTION newFunction = (NEW_FUNCTION)objectTable[i].data;
			object = newFunction(NULL, directory, file);
		}

	// Clean up the file and storage then return the object.
	if (tempDirectory)
		delete directory;
	if (object)
		return (object);

	// Error report for unknown object types.
	return (NULL);
}

void UI_WINDOW_OBJECT::Store(const char *, UI_STORAGE *, UI_STORAGE_OBJECT *file)
{
	// Mark the object as used.
	if (objectTable)
	{
		for (int i = 0; objectTable[i].data; i++)
			if (objectTable[i].value == searchID)
			{
				objectTable[i].flags = TRUE;
				break;
			}
	}

	// Store the object information.
	file->Store(numberID);
	file->Store(stringID);
	file->Store(woFlags);
	file->Store(woAdvancedFlags);
	UI_REGION region;
	if (parent)
		region = relative;
	else
		region = true;
	if (!display->isText && FlagSet(woStatus, WOS_GRAPHICS) && FlagSet(woFlags, WOF_MINICELL))
	{
		long miniNX = display->miniNumeratorX, miniDX = display->miniDenominatorX;
		long miniNY = display->miniNumeratorY, miniDY = display->miniDenominatorY;
		region.left = (int)((miniDX * region.left) / (miniNX * display->cellWidth));
		region.top = (int)((miniDY * region.top) / (miniNY * display->cellHeight));
		region.right = (int)((miniDX * region.right) / (miniNX * display->cellWidth));
		region.bottom = (int)((miniDY * region.bottom) / (miniNY * display->cellHeight));
	}
	else if (!display->isText && FlagSet(woStatus, WOS_GRAPHICS))
	{
		int width = (region.right - region.left) / display->cellWidth;
		int height = (region.bottom - region.top) / display->cellHeight;
		region.left /= display->cellWidth;
		region.bottom /= display->cellHeight;
		region.right = region.left + width;
		region.top = region.bottom - height;
	}
	short value = region.left; file->Store(value);
	value = region.top; file->Store(value);
	value = region.right; file->Store(value);
	value = region.bottom; file->Store(value);
	value = helpContext; file->Store(value);
	file->Store(userFlags);
	file->Store(userStatus);
	if (userObject && !userObjectName && userTable)
	{
		for (int i = 0; !userObjectName && userTable[i].data; i++)
			if (userTable[i].data == userObject)
				userObjectName = ui_strdup(userTable[i].text);
	}
	file->Store(userObjectName);
	if (userFunction && !userFunctionName && userTable)
	{
		for (int i = 0; !userFunctionName && userTable[i].data; i++)
			if (userTable[i].data == (void *)userFunction)
				userFunctionName = ui_strdup(userTable[i].text);
	}
	file->Store(userFunctionName);
}
#endif
