//	VLIST.HPP (VLIST) - This file contains source code for virtual list tutorial.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdio.h>
#define USE_RAW_KEYS
#include <ui_win.hpp>
#include "vlist.hpp"

// Event map table.
#ifdef _WINDOWS
static UI_EVENT_MAP vlistEventMapTable[] =
{
	{ ID_WINDOW_OBJECT, 	L_UP,				WM_KEYDOWN, 	GRAY_UP_ARROW },
	{ ID_WINDOW_OBJECT, 	L_UP,				WM_KEYDOWN, 	WHITE_UP_ARROW },
	{ ID_WINDOW_OBJECT, 	L_DOWN,				WM_KEYDOWN, 	GRAY_DOWN_ARROW },
	{ ID_WINDOW_OBJECT, 	L_DOWN,				WM_KEYDOWN, 	WHITE_DOWN_ARROW },
	{ ID_WINDOW_OBJECT, 	L_PGUP,				WM_KEYDOWN, 	GRAY_PGUP },
	{ ID_WINDOW_OBJECT, 	L_PGUP,				WM_KEYDOWN, 	WHITE_PGUP },
	{ ID_WINDOW_OBJECT, 	L_PGDN,				WM_KEYDOWN, 	GRAY_PGDN },
	{ ID_WINDOW_OBJECT, 	L_PGDN,				WM_KEYDOWN, 	WHITE_PGDN },
	{ ID_WINDOW_OBJECT, 	L_LEFT,				WM_KEYDOWN, 	GRAY_LEFT_ARROW },
	{ ID_WINDOW_OBJECT, 	L_LEFT,				WM_KEYDOWN, 	WHITE_LEFT_ARROW },
	{ ID_WINDOW_OBJECT, 	L_RIGHT,			WM_KEYDOWN, 	GRAY_RIGHT_ARROW },
	{ ID_WINDOW_OBJECT, 	L_RIGHT,			WM_KEYDOWN, 	WHITE_RIGHT_ARROW },
	{ ID_WINDOW_OBJECT, 	L_PREVIOUS,			WM_CHAR, 		BACKTAB },
	{ ID_WINDOW_OBJECT, 	L_NEXT,				WM_CHAR, 		TAB },
	{ ID_WINDOW_OBJECT, 	L_SELECT,			WM_CHAR, 		ENTER },

	// End of array.
	{ ID_END, 0, 0, 0 }
};
#endif

VIRTUAL_ELEMENT::VIRTUAL_ELEMENT(int left, int top, int width, int _height, int length) :
	UIW_STRING(left, top, width, "", length, STF_NO_FLAGS, WOF_NON_SELECTABLE),
	height(_height)
{
	// Set up event map table for Windows.
#ifdef _WINDOWS
	eventMapTable = vlistEventMapTable;
#endif
//	woAdvancedFlags = WOAF_NON_CURRENT;
}
EVENT_TYPE VIRTUAL_ELEMENT::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);

	switch (ccode)
	{
	case S_INITIALIZE:
		UIW_STRING::Event(event);
		if (display && !display->isText)
			relative.bottom = relative.top + height;
		break;

	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

VIRTUAL_LIST::VIRTUAL_LIST(const char *fileName, int _recordLength) :
	UIW_WINDOW(0, 0, 0, 0, WOF_NON_FIELD_REGION), recordLength(_recordLength),
	topRecordShowing(0), numberShowing(0)
{
	// Set up event map table for Windows.
#ifdef _WINDOWS
	eventMapTable = vlistEventMapTable;
#endif

	// Open the database, get the total number of records.
	file = fopen(fileName, "rb");
	fseek(file, 0L, SEEK_END);
	numberOfRecords = (int)(ftell(file) / recordLength);
}

VIRTUAL_LIST::~VIRTUAL_LIST(void)
{
	// Close the database.
	if (file)
		fclose(file);
}

EVENT_TYPE VIRTUAL_LIST::Event(const UI_EVENT &event)
{
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);

	switch (ccode)
	{
	case S_CREATE:
#ifndef _WINDOWS
	case S_SIZE:
#endif
		{
		// Call the default event.
		ccode = UIW_WINDOW::Event(event);

		// Calculate the number of elements that will fit in the list.
		int lineHeight = display->TextHeight("Mxq", screenID, font) + display->preSpace + display->postSpace;
		
		int newNumberShowing = (true.bottom - true.top + display->preSpace + display->postSpace);
		newNumberShowing /= lineHeight;
		newNumberShowing++;
		if (display->isText)
			newNumberShowing++;

		// Make sure the window is full of records.
		if (numberShowing != newNumberShowing)
		{
			numberShowing = newNumberShowing;
			Destroy();
			for (int line = 0; line < numberShowing - 1; line++)
			{
				VIRTUAL_ELEMENT *element = new VIRTUAL_ELEMENT(0,
					line * lineHeight, true.right - true.left + 1,
					lineHeight, recordLength + 10);
				element->woStatus |= WOS_GRAPHICS;
				LoadRecord(element, topRecordShowing + line);
				Add(element);
			}
			Event(UI_EVENT(S_REDISPLAY));
		}
		}
		break;

	case L_PREVIOUS:
	case L_UP:
		if (topRecordShowing != 0 && First())
		{
			for (VIRTUAL_ELEMENT *object = Last(); object && object->Previous(); object = object->Previous())
				object->DataSet(object->Previous());
			LoadRecord(First(), --topRecordShowing);
		}
		break;

	case L_NEXT:
	case L_DOWN:
		if (topRecordShowing < numberOfRecords - 1 && First())
		{
			for (VIRTUAL_ELEMENT *object = First(); object && object->Next(); object = object->Next())
				object->DataSet(object->Next());
			LoadRecord(Last(), ++topRecordShowing + numberShowing - 1);
		}
		break;

	case L_PGUP:
		if (topRecordShowing != 0)
		{
			topRecordShowing -= numberShowing;
			if (topRecordShowing < 0)
				topRecordShowing = 0;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	case L_PGDN:
		if (topRecordShowing != numberOfRecords - 1)
		{
			topRecordShowing += numberShowing;
			if (topRecordShowing > numberOfRecords - 1)
				topRecordShowing = numberOfRecords - 1;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	case L_TOP:
		if (topRecordShowing != 0)
		{
			topRecordShowing = 0;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	case L_BOTTOM:
		if (topRecordShowing != numberOfRecords - 1)
		{
			topRecordShowing = numberOfRecords - 1;
			int recordNumber = topRecordShowing;
			for (VIRTUAL_ELEMENT *object = First(); object; object = object->Next())
				LoadRecord(object, recordNumber++);
		}
		break;

	default:
		ccode = UIW_WINDOW::Event(event);
	}

	// Return the control code.
	return (ccode);
}

void VIRTUAL_LIST::LoadRecord(VIRTUAL_ELEMENT *element, int recordNumber)
{
	// Load the record from the file.
	if (recordNumber > numberOfRecords - 1)
		element->DataSet(-1, "");
	else
	{
		long offset = recordLength * recordNumber;
		fseek(file, offset, SEEK_SET);
		char *text = element->DataGet();
		fgets(text, recordLength - 1, file);
		text[recordLength - 1] = '\0';
		element->DataSet(recordNumber, text);
	}
}

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		display = new UI_TEXT_DISPLAY;
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Create a window containing the virtual list and attach it to the window manager.
	*windowManager
		+ &(*UIW_WINDOW::Generic(5, 1, 75, 12, "Dictionary")
			+ new VIRTUAL_LIST("VLIST", 80));

	// Wait for user response.
	EVENT_TYPE ccode;
	do
	{
		// Get input from the user.
		UI_EVENT event;
		eventManager->Get(event);

		// Send event information to the window manager.
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
