//	WORD3.CPP (WORD) - Dictionary tutorial, using the data file.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include <ui_win.hpp>

#define USE_WINDOW_DICTIONARY
#include "word_win.hpp"
#include "word3.hpp"

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		display = new UI_TEXT_DISPLAY;
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Initialize the error system.
	UI_WINDOW_OBJECT::errorSystem = new UI_ERROR_SYSTEM;

	// Create the dictionary window.
	DICTIONARY_WINDOW *dictionary = new DICTIONARY_WINDOW("WORD.DAT");

	if (!FlagSet(dictionary->woStatus, WOS_READ_ERROR))
		*windowManager + dictionary;
	else
	{
		delete dictionary;

		UI_WINDOW_OBJECT::errorSystem->ReportError(windowManager, WOS_NO_STATUS,
			"An error was encountered trying to open WORD_WIN.DAT");
	}

	// Wait for user response.
	EVENT_TYPE ccode;
	do
	{
		UI_EVENT event;
		eventManager->Get(event);
		ccode = windowManager->Event(event);
	} while (ccode != L_EXIT && ccode != S_NO_OBJECT);

	// Clean up.
	delete UI_WINDOW_OBJECT::errorSystem;
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}

DICTIONARY_WINDOW::DICTIONARY_WINDOW(char * dictionaryName) :
	UIW_WINDOW("WORD_WIN.DAT~WINDOW_DICTIONARY")
{
	// Set up the pointers to the window fields.
	inputField = (UIW_STRING *)Information(GET_STRINGID_OBJECT, "DCT_INPUT");
	definitionField = (UIW_TEXT *)Information(GET_STRINGID_OBJECT, "DCT_DEFINITION");
	antonymField = (UIW_STRING *)Information(GET_STRINGID_OBJECT, "DCT_ANTONYM");
	synonymField = (UIW_STRING *)Information(GET_STRINGID_OBJECT, "DCT_SYNONYM");

	// Set the user functions to the buttons.
	UIW_BUTTON *button;
	button = (UIW_BUTTON *)Get(DCT_LOOKUP_BUTTON);
	button->userFunction = DICTIONARY_WINDOW::ButtonFunction;

	button = (UIW_BUTTON *)Get(DCT_SAVE_BUTTON);
	button->userFunction = DICTIONARY_WINDOW::ButtonFunction;

	button = (UIW_BUTTON *)Get(DCT_DELETE_BUTTON);
	button->userFunction = DICTIONARY_WINDOW::ButtonFunction;

	// Set up the dictionary.
	dictionary = new DICTIONARY(dictionaryName);
	if (dictionary->storageError)
	{
		dictionary = NULL;
		errorSystem->ReportError(windowManager, WOS_NO_STATUS,
			"The dictionary file could not be opened with read write access.");
	}
	else
	{
		if (dictionary->ChDir("~DICTIONARY"))
		{
			dictionary->ChDir("~");
			dictionary->MkDir("DICTIONARY");
			dictionary->ChDir("~DICTIONARY");
		}
	}
}

EVENT_TYPE DICTIONARY_WINDOW::Event(const UI_EVENT &event)
{
	D_ENTRY *entry;
	EVENT_TYPE ccode = event.type;
	const char *stringID = NULL;

	switch(event.type)
	{
	case L_DCT_DELETE:
		stringID = dictionary->FindFirstObject(inputField->DataGet());
		if (stringID)
		{
			dictionary->DestroyObject(stringID);
		}
		else
			errorSystem->ReportError(windowManager, WOS_NO_STATUS,
				"That word was not found in the dictionary and cannot be deleted.");
		break;

	case L_DCT_LOOKUP:
		stringID = dictionary->FindFirstObject(inputField->DataGet());
		if (!stringID)
		{
			// The object was not found in the file.
			definitionField->DataSet("");
			antonymField->DataSet("");
			synonymField->DataSet("");

			// The word was not found.
			errorSystem->ReportError(windowManager, WOS_NO_STATUS,
				"That word was not found in the dictionary.");
		}
		else
		{
			entry = D_ENTRY::New(stringID, (UI_STORAGE *)dictionary, UIS_READ);

			// If the word was found then display the fields.
			if (entry->wasLoaded)
			{
				inputField->DataSet(entry->word);
				definitionField->DataSet(entry->definition);
				antonymField->DataSet(entry->antonym);
				synonymField->DataSet(entry->synonym);
			}
			delete entry;
		}
		break;

	case L_DCT_SAVE:
	{
		// If the entry exists, modify it.  Otherwise, create it.
		stringID = dictionary->FindFirstObject(inputField->DataGet());
		if (!stringID)
			entry = D_ENTRY::New(inputField->DataGet(), (UI_STORAGE *)dictionary,
				UIS_CREATE | UIS_READWRITE);
		else
			entry = D_ENTRY::New(inputField->DataGet(), (UI_STORAGE *)dictionary,
				UIS_READWRITE);

		// Don't save an empty field.
		const char *newWord = inputField->DataGet();
		if(strcmp(newWord, ""))
		{
			entry->word = strdup(newWord);
			entry->definition = strdup(definitionField->DataGet());
			entry->antonym = strdup(antonymField->DataGet());
			entry->synonym = strdup(synonymField->DataGet());
			entry->Save();
		}
		else
			errorSystem->ReportError(windowManager, WOS_NO_STATUS,
				"A blank word cannot be saved.");

		delete entry;
	}
		break;

	default:
		ccode = UIW_WINDOW::Event(event);
	};

	return ccode;
}

EVENT_TYPE DICTIONARY_WINDOW::ButtonFunction(UI_WINDOW_OBJECT *object, 
   UI_EVENT &event, EVENT_TYPE ccode)
{
	UI_EVENT tEvent;
	tEvent = event;

	if (ccode != L_SELECT)
		return ccode;

	OBJECTID buttonID = object->NumberID();
	switch (buttonID)
	{
	case DCT_DELETE_BUTTON:
		tEvent.type = L_DCT_DELETE;
	   break;

	case DCT_LOOKUP_BUTTON:
		tEvent.type = L_DCT_LOOKUP;
	   break;

	case DCT_SAVE_BUTTON:
		tEvent.type = L_DCT_SAVE;
	   break;

	default:
	   return event.type;
	}

	object->eventManager->Put(tEvent);
	return ccode;
}

D_ENTRY::D_ENTRY(const char *name, UI_STORAGE *file, UIS_FLAGS flags) :
	word(NULL), definition(NULL), antonym(NULL), synonym(NULL)
{
	object = new UI_STORAGE_OBJECT(*file, name, ID_DICTIONARY_ENTRY, flags);

	// Check to see if object was found in the file.
	if (object->objectError)
		wasLoaded = FALSE;
	else
	{
		wasLoaded = TRUE;

		// Load the word.
		USHORT stringLength;
		object->Load(&stringLength);
		if (stringLength)
		{
			word = new char[stringLength+1];
			object->Load(word, 1, stringLength);
			word[stringLength] = '\0';
		}

		// Load the definition.
		object->Load(&stringLength);
		if (stringLength)
		{
			definition = new char[stringLength+1];
			object->Load(definition, 1, stringLength);
			definition[stringLength] = '\0';
		}

		// Load the antonyms.
		object->Load(&stringLength);
		if (stringLength)
		{
			antonym = new char[stringLength+1];
			object->Load(antonym, 1, stringLength);
			antonym[stringLength] = '\0';
		}

		// Load the synonyms.
		object->Load(&stringLength);
		if (stringLength)
		{
			synonym = new char[stringLength+1];
			object->Load(synonym, 1, stringLength);
			synonym[stringLength] = '\0';
		}
	}
}

D_ENTRY::~D_ENTRY()
{
	// Free the memory if it was allocated.
	if (word)
		delete word;
	if (definition)
		delete definition;
	if (antonym)
		delete antonym;
	if (synonym)
		delete synonym;
	if (object)
		delete object;
}

D_ENTRY *D_ENTRY::New(const char *name, UI_STORAGE *directory, UIS_FLAGS flags)
{
	// Check for a valid directory and file.
	if (name)
		return (new D_ENTRY(name, directory, flags));
	else
       return NULL;
}

void D_ENTRY::Save()
{
	// Store the word.
	object->Store((USHORT)ui_strlen(word));
	object->Store(word, 1, ui_strlen(word));

	// Store the word definition.
	object->Store((USHORT)ui_strlen(definition));
	object->Store(definition, 1, ui_strlen(definition));
  
	// Store the antonyms.
	object->Store((USHORT)ui_strlen(antonym));
	object->Store(antonym, 1, ui_strlen(antonym));

	// Store the synonyms.
	object->Store((USHORT)ui_strlen(synonym));
	object->Store(synonym, 1, ui_strlen(synonym));
}
