//	PHONEBK.CPP (PHONEBK) - Phone book example program.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include <ui_win.hpp>

#define PHONE_BOOK_NAME "phone.dat"
const OBJECTID ID_PHONE_BOOK = 20000;

UI_STORAGE *_phoneBook;

EVENT_TYPE Find(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return (ccode);

	UIW_STRING *name = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "NAME");
	UIW_TEXT *address = (UIW_TEXT *)object->parent->Information(GET_STRINGID_OBJECT, "ADDRESS");
	UIW_STRING *workPhone = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "WORK_PHONE");
	UIW_STRING *homePhone = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "HOME_PHONE");

	_phoneBook->ChDir("~PHONE_BOOK");
	const char *stringID = _phoneBook->FindFirstObject(name->DataGet());
	if (!stringID)
	{
		// The object was not found in the file.
		address->DataSet("");
		workPhone->DataSet("");
		homePhone->DataSet("");

		// The word was not found.
		object->errorSystem->ReportError(object->windowManager, WOS_NO_STATUS,
			"That name was not found in the phone book.");
		object->Event(UI_EVENT(S_REDISPLAY));
	}
	else
	{
		UI_STORAGE_OBJECT *entry = new UI_STORAGE_OBJECT(*_phoneBook, stringID, ID_PHONE_BOOK, UIS_READ);

		// If the word was found then display the fields.
		if (!entry->objectError)
		{
			// Load the phone information.
			USHORT stringLength;
			char *string;

			// Load the address.
			entry->Load(&stringLength);
			if (stringLength)
			{
				string = new char[stringLength + 1];
				entry->Load(string, 1, stringLength);
				string[stringLength] = '\0';
				address->DataSet(string);
				delete string;
			}

			// Load the work phone number.
			entry->Load(&stringLength);
			if (stringLength)
			{
				string = new char[stringLength + 1];
				entry->Load(string, 1, stringLength);
				string[stringLength] = '\0';
				workPhone->DataSet(string);
				delete string;
			}

			//Load the home phone number.
			entry->Load(&stringLength);
			if (stringLength)
			{
				string = new char[stringLength + 1];
				entry->Load(string, 1, stringLength);
				string[stringLength] = '\0';
				homePhone->DataSet(string);
				delete string;
			}
		}
		delete entry;
	}

	return (ccode);
}

EVENT_TYPE Save(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return (ccode);

	UIW_STRING *name = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "NAME");
	if (!strcmp(name->DataGet(), ""))
	{
		object->errorSystem->ReportError(object->windowManager, WOS_NO_STATUS,
			"A blank phone entry cannot be saved.");
		object->Event(UI_EVENT(S_REDISPLAY));
		return (ccode);
	}
	UIW_TEXT *address = (UIW_TEXT *)object->parent->Information(GET_STRINGID_OBJECT, "ADDRESS");
	UIW_STRING *workPhone = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "WORK_PHONE");
	UIW_STRING *homePhone = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "HOME_PHONE");

	// Try to find the entry.
	_phoneBook->ChDir("~");
	_phoneBook->MkDir("PHONE_BOOK");
	_phoneBook->ChDir("~PHONE_BOOK");
	
	const char *stringID = _phoneBook->FindFirstObject(name->DataGet());
	UI_STORAGE_OBJECT *entry;
	if (!stringID)
		entry = new UI_STORAGE_OBJECT(*_phoneBook, name->DataGet(), ID_PHONE_BOOK, UIS_CREATE | UIS_READWRITE);
	else
		entry = new UI_STORAGE_OBJECT(*_phoneBook, stringID, ID_PHONE_BOOK, UIS_READWRITE);

	// If the word was found then display the fields.
	if (!entry->objectError)
	{
		// Store the phone information.
		char *string;

		// Store the address.
		string = address->DataGet();
		entry->Store((USHORT)ui_strlen(string));
		entry->Store(string, 1, ui_strlen(string));

		// Store the work phone number.
		string = workPhone->DataGet();
		entry->Store((USHORT)ui_strlen(string));
		entry->Store(string, 1, ui_strlen(string));

		// Store the home phone number.
		string = homePhone->DataGet();
		entry->Store((USHORT)ui_strlen(string));
		entry->Store(string, 1, ui_strlen(string));
	}
	delete entry;

	return (ccode);
}

EVENT_TYPE Delete(UI_WINDOW_OBJECT *object, UI_EVENT &, EVENT_TYPE ccode)
{
	if (ccode != L_SELECT)
		return (ccode);

	UIW_STRING *name = (UIW_STRING *)object->parent->Information(GET_STRINGID_OBJECT, "NAME");

	_phoneBook->ChDir("~PHONE_BOOK");
	const char *stringID = _phoneBook->FindFirstObject(name->DataGet());
	if (stringID)
		_phoneBook->DestroyObject(stringID);
	else
	{
		object->errorSystem->ReportError(object->windowManager, WOS_NO_STATUS,
			"That phone entry was not found in the phone book.");
		object->Event(UI_EVENT(S_REDISPLAY));
	}

	return (ccode);
}

#ifdef _WINDOWS

int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR, int nCmdShow)
{
	UI_DISPLAY *display = new UI_MSWINDOWS_DISPLAY(hInstance, hPrevInstance, nCmdShow);

#else

main()
{
	// Initialize the display (compiler dependent).
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
	UI_DISPLAY *display = new UI_BGI_DISPLAY;
#endif
#ifdef __ZTC__
	UI_DISPLAY *display = new UI_FG_DISPLAY;
#endif
#ifdef _MSC_VER
	UI_DISPLAY *display = new UI_MSC_DISPLAY;
#endif

	// Install a text display if no graphics capability.
	if (!display->installed)
	{
		delete display;
		display = new UI_TEXT_DISPLAY;
	}

#endif

	// Create the event manager and add devices.
	UI_EVENT_MANAGER *eventManager = new UI_EVENT_MANAGER(display);
	*eventManager
		+ new UID_KEYBOARD
		+ new UID_MOUSE
		+ new UID_CURSOR;

	// Create the window manager.
	UI_WINDOW_MANAGER *windowManager = new UI_WINDOW_MANAGER(display, eventManager);

	// Initialize the error system.
	UI_WINDOW_OBJECT::errorSystem = new UI_ERROR_SYSTEM;

	// Open the phone book.
	_phoneBook = new UI_STORAGE(PHONE_BOOK_NAME, UIS_OPENCREATE | UIS_READWRITE);
	if (_phoneBook->storageError)
		UI_WINDOW_OBJECT::errorSystem->ReportError(windowManager, WOS_NO_STATUS,
			"Error opening %s.", PHONE_BOOK_NAME);
	else
	{
		*windowManager + UI_WINDOW_OBJECT::New("phon_win.dat~PHONE_BOOK");

		// Wait for user response.
		EVENT_TYPE ccode;
		do
		{
			// Get input from the user.
			UI_EVENT event;
			eventManager->Get(event);

			// Send event information to the window manager.
			ccode = windowManager->Event(event);
		} while (ccode != L_EXIT && ccode != S_NO_OBJECT);
	}

	// Clean up.
	delete UI_WINDOW_OBJECT::errorSystem;
	delete windowManager;
	delete eventManager;
	delete display;

	return (0);
}
