//	SOUND.CPP (PIANO) - Sample electronic piano sound class.
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <dos.h>
#include <time.h>

#if defined __ZTC__	// Zortech compiler.
#define inportb inp
#define outportb outp
#elif defined _MSC_VER
#include <conio.h>
#define inportb _inp
#define outportb _outp
#endif

#include <ui_gen.hpp>
#include "sound.hpp"

void UI_SOUND::Beep(unsigned short hZ, unsigned short duration)
{
	// Turn sound on, wait for duration then turn it off.
	SoundOn(hZ);
	Pause(duration);
	SoundOff();
}
	
int UI_SOUND::Pause(unsigned duration)
{
	// Find what time it is to start with.
	clock_t startClock = clock();

	// If 0 is passed in use default duration.
	if (!duration)
		duration = defaultDuration;

	// If there is an error getting the time then exit immediately
	if (startClock == -1)
		return (CLOCK_ERROR);

	// Find what time to wait for (taking into account wrap around).
	clock_t waitFor = startClock + (((long)duration * CLK_TCK) / 1000L);
	clock_t wrappedFrom = MAX_CLOCK;
	if (waitFor < 0 || waitFor > MAX_CLOCK)
	{
		waitFor = MAX_CLOCK - startClock;
		wrappedFrom = startClock;
	}

	clock_t currentClock;
	do
	{
		currentClock = clock();
		if (currentClock == -1)
			return (CLOCK_ERROR);
	}while (currentClock <= waitFor || currentClock >= wrappedFrom);

	return (CLOCK_OKAY);
}

int UI_SOUND::SoundOn(unsigned short hZ)
{
	// If lower than 18Hz then forget it.
	int result = TOO_LOW;
	if(!hZ)
		hZ = defaultHZ;
	if (hZ < 18)
		return (result);

	// Otherwise, get wavelength in DOS click units.
	long waveLength = 1193181L / (long)hZ;

	// See if sound need to be turned on or if it already is.
	unsigned char mask = inportb(0x61);
	result = CHANGED_PITCH;
	if (!(mask & 0x01) || !(mask & 0x02))
	{
		mask |= 0x03;
		outportb(0x61, mask);
		result = SOUND_ON;
	}

	// Send out the pitch.
	outportb(0x43, 0xB6);
	outportb(0x42, LowByte(waveLength));
	outportb(0x42, HighByte(waveLength));
	return (result);
}

void UI_SOUND::SoundOff()
{
	// Mask off the sound.
	outportb(0x61, inportb(0x61 & ~0x03));
}
