//	Zinc Interface Library - BUTTON.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#define OEMRESOURCE
#include <ctype.h>
#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
const int BUTTON_OFFSET = 4;

// ----- UIW_BUTTON ---------------------------------------------------------

UIW_BUTTON::UIW_BUTTON(int left, int top, int width, char *_text,
	BTF_FLAGS _btFlags, WOF_FLAGS _woFlags, USER_FUNCTION _userFunction,
	EVENT_TYPE _value, char *_bitmapName) :
	UI_WINDOW_OBJECT(left, top, width, 1, _woFlags, WOAF_NO_FLAGS),
	text(NULL), btFlags(_btFlags), value(_value), depth(2),
	btStatus(BTS_NO_STATUS), bitmapWidth(0), bitmapHeight(0), bitmapArray(NULL)
{
	// Initialize the button information.
	bitmapName = ui_strdup(_bitmapName);
	userFunction = _userFunction;
	UIW_BUTTON::Information(INITIALIZE_CLASS, NULL);
	UIW_BUTTON::DataSet(_text);
}

UIW_BUTTON::~UIW_BUTTON(void)
{
	if (text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
	if (bitmapName)
		delete bitmapName;
	if (bitmapArray)
		delete bitmapArray;
#ifdef _WINDOWS
	if (colorBitmap)
		DeleteObject(colorBitmap);
	if (monoBitmap)
		DeleteObject(monoBitmap);
#endif
}

char *UIW_BUTTON::DataGet(int stripText)
{
	// Check for normal DataGet.
	if (!text || !stripText)
		return (text);

	// Strip the button text information.
	static char sText[128];
	int offset = 0;
	while (text[offset] == ' ')					// Check for beginning spaces.
		offset++;
	strcpy(sText, &text[offset]);
	offset = strlen(sText) - 1;
	while (offset >= 0 && sText[offset] == ' ')	// Check for trailing spaces.
		offset--;
	sText[offset + 1] = '\0';
	char *hotKey = strchr(sText, '&');			// Check for underline character.
	if (hotKey)
		memcpy(hotKey, hotKey + 1, ui_strlen(hotKey));
	return (sText);
}

void UIW_BUTTON::DataSet(char *_text)
{
	// Reset the string.
	if (text && text != _text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
	if (text == _text || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = _text;
	else
		text = ui_strdup(_text);

	// Check for a hotkey.
	if (text)
	{
		ui_strrepc(text, '~', '&');
		char *tHotKey = strchr(text, '&');
		if (tHotKey)
			hotKey = tolower(tHotKey[1]);
	}

	// Redisplay the string.
	if (screenID)
	{
#ifdef _WINDOWS
		InvalidateRect(screenID, NULL, TRUE);
		SendMessage(screenID, WM_SETTEXT, 0, (LONG)text);
#endif
	}
}

#ifdef _WINDOWS
extern void WindowsToZincBitmap(UI_DISPLAY *display, HBITMAP colorBitmap,
	HBITMAP monoBitmap, int *bitmapWidth, int *bitmapHeight,
	UCHAR **bitmapArray);
extern void ZincToWindowsBitmap(UI_DISPLAY *display, int bitmapWidth,
	int bitmapHeight, UCHAR *bitmapArray, UI_PALETTE *paletteArray,
	HBITMAP *colorBitmap, HBITMAP *monoBitmap);

static int _buttonOffset = -1;
static FARPROC _buttonCallback = (FARPROC)DefWindowProc;
static FARPROC _buttonJumpInstance = NULL;

long FAR PASCAL _export ButtonJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _buttonOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_BUTTON::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	// Initialize the variables.
	DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
	UI_REGION region = { draw->rcItem.left, draw->rcItem.top,
		draw->rcItem.right, draw->rcItem.bottom };
	if (draw->CtlType != ODT_BUTTON)
		region.left += relative.left;
	UI_MSWINDOWS_DISPLAY::hDC = draw->hDC;

	// Check the button status.
	if (FlagSet(draw->itemState, ODS_SELECTED))
		woStatus |= WOS_SELECTED;
	else
		woStatus &= ~WOS_SELECTED;
	if (FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS) &&
		!FlagSet(woStatus, WOS_CURRENT))
	{
		UI_EVENT event(S_ADD_OBJECT);
		event.data = this;
		parent->woStatus |= WOS_WINDOWS_ACTION;
		parent->Event(event);
		parent->woStatus &= ~WOS_WINDOWS_ACTION;
	}

	// Draw the button information.
	if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
		lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, PM_ACTIVE, ID_WINDOW_OBJECT);
	else if (!FlagSet(btFlags, BTF_NO_3D) && 
		(draw->CtlType == ODT_BUTTON || !FlagSet(woStatus, WOS_SELECTED)))
		lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ?
			PM_SELECTED : PM_ACTIVE, ID_BUTTON);
	else
		lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ?
			PM_SELECTED : PM_ACTIVE, ID_WINDOW_OBJECT);

	if (FlagSet(draw->itemAction, ODA_DRAWENTIRE | ODA_SELECT))
	{
		// Draw the button border.
		if (FlagSet(woFlags, WOF_BORDER))
			DrawBorder(screenID, region, FALSE, ccode);

		// Draw the button background.
		if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON) && !draw->CtlType == ODT_BUTTON)
		{
			HBRUSH fillBrush = CreateSolidBrush(display->MapColor(lastPalette, FALSE));
			FillRect(draw->hDC, &draw->rcItem, fillBrush);
			DeleteObject(fillBrush);
		}
		else if (!FlagSet(btFlags, BTF_NO_3D) && 
			(draw->CtlType == ODT_BUTTON || !FlagSet(woStatus, WOS_SELECTED)))
			DrawShadow(ID_DIRECT, region, FlagsSet(woStatus, WOS_SELECTED | WOS_CURRENT) ? -depth : depth, TRUE, ccode);
		else
		{
			HBRUSH fillBrush = CreateSolidBrush(display->MapColor(lastPalette, FALSE));
			FillRect(draw->hDC, &draw->rcItem, fillBrush);
			DeleteObject(fillBrush);
			if (!FlagSet(btFlags, BTF_NO_3D))
				region -= 3;
		}

		// Draw the text if it is centered or right justified.
		if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		{
			if (text)
			{
				woFlags &= ~WOF_JUSTIFY_RIGHT;
				DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
				woFlags |= WOF_JUSTIFY_RIGHT;
			}
			region.left = region.right - bitmapWidth - BUTTON_OFFSET;
		}
		else if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		{
			if (text)
			{
				int top = region.top;
				region.top = region.bottom - display->cellHeight + 4;
				DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
				region.bottom = region.top;
				region.top = top;
			}
			region.right = (region.left + region.right + bitmapWidth + 1) / 2;
			region.left = region.right - bitmapWidth;
		}
		else if (draw->CtlType != ODT_BUTTON)
			region.left += BUTTON_OFFSET;

		// Draw the bitmap image.
		int column = 0, line = 0;
		if (FlagSet(woStatus, WOS_SELECTED) && FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
			column = bitmapWidth;
		if (FlagSet(btFlags, BTF_RADIO_BUTTON))
			line = bitmapHeight;
		if (monoBitmap)
		{
			HDC hMemDC = CreateCompatibleDC(draw->hDC);
			COLORREF oldBackground = SetBkColor(draw->hDC, display->MapColor(lastPalette, FALSE));
			SelectObject(hMemDC, monoBitmap);
			BitBlt(draw->hDC, region.left,
				region.top + (region.bottom - region.top - bitmapHeight) / 2,
				bitmapWidth, bitmapHeight, hMemDC, column, line, SRCAND);
			SelectObject(hMemDC, colorBitmap);
			BitBlt(draw->hDC, region.left,
				region.top + (region.bottom - region.top - bitmapHeight) / 2,
				bitmapWidth, bitmapHeight, hMemDC, column, line, SRCINVERT);
			region.left += bitmapWidth + BUTTON_OFFSET;
			SetBkColor(draw->hDC, oldBackground);
			DeleteDC(hMemDC);
		}
		else if (colorBitmap)
		{
			HDC hMemDC = CreateCompatibleDC(draw->hDC);
			SelectObject(hMemDC, colorBitmap);
			BitBlt(draw->hDC, region.left,
				region.top + (region.bottom - region.top - bitmapHeight) / 2,
				bitmapWidth, bitmapHeight, hMemDC, column, line, SRCCOPY);
			region.left += bitmapWidth + BUTTON_OFFSET;
			DeleteDC(hMemDC);
		}

		// Draw the text if it is left justified.
		if (!FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER) && text)
			DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
		if (!FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS))
			DrawFocusRect(draw->hDC, &draw->rcItem);
	}
	if (FlagSet(draw->itemAction, ODA_FOCUS))
		DrawFocusRect(draw->hDC, &draw->rcItem);

	return (TRUE);
}

EVENT_TYPE UIW_BUTTON::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_BUTTON);
	switch (ccode)
	{
	case S_INITIALIZE:
		// Convert the button array if needed.
		if (!colorBitmap && bitmapArray)
			ZincToWindowsBitmap(display, bitmapWidth, bitmapHeight,
				bitmapArray, NULL, &colorBitmap, &monoBitmap);
		if (colorBitmap)
		{
			BITMAP bm;
			GetObject(colorBitmap, sizeof(BITMAP), (LPSTR)&bm);
			if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
			{
				bitmapHeight = bm.bmHeight / 3;
				bitmapWidth = bm.bmWidth / 4;
			}
			else
			{
				bitmapHeight = bm.bmHeight;
				bitmapWidth = bm.bmWidth;
			}
		}
		// Initialize the button information.
		if (!_buttonJumpInstance)
			_buttonJumpInstance = (FARPROC)ButtonJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		// Continue to S_CREATE.
	case S_CREATE:
		{
		if (!FlagSet(btFlags, BTF_AUTO_SIZE))
			relative.bottom = relative.top + display->cellHeight - display->preSpace - display->postSpace;
		int offset = 2;
		if (FlagSet(woFlags, WOF_BORDER))
			offset += 2;
		if (!FlagSet(btFlags, BTF_NO_3D))
			offset += 6;
		if (colorBitmap && FlagSet(btFlags, BTF_AUTO_SIZE) && FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		{
			relative.top = relative.bottom - bitmapHeight - offset;
			if (text)
				relative.top -= display->cellHeight;
			true.top = relative.top;
		}
		else if (FlagSet(btFlags, BTF_AUTO_SIZE))
			relative.top = relative.bottom - display->cellHeight * 10 / 9;
		if (colorBitmap && relative.bottom - relative.top < bitmapHeight + offset)
			relative.bottom = relative.top + bitmapHeight + offset;
		}
		if (ccode == S_INITIALIZE)
			break;
		if (parent->Inherited(ID_COMBO_BOX))
		{
			dwStyle = (dwStyle & 0xFFFFFF00L) + BS_OWNERDRAW;
			DWORD index = LOWORD(SendMessage(parent->screenID, CB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)text));
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, CB_SETCURSEL, LOWORD(index), 0);
		}
		else if (parent->Inherited(ID_LIST))
		{
			dwStyle = (dwStyle & 0xFFFFFF00L) + BS_OWNERDRAW;
			DWORD index = SendMessage(parent->screenID, LB_ADDSTRING, 0,
				FlagSet(parent->woStatus, WOS_OWNERDRAW) ? (LONG)this : (LONG)text);
			if (FlagSet(woStatus, WOS_SELECTED))
				SendMessage(parent->screenID, LB_SETSEL, TRUE, index);
		}
		else
		{
			UI_WINDOW_OBJECT::Event(event);
			if (colorBitmap && !FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
				dwStyle |= BS_OWNERDRAW;
			RegisterObject("UIW_BUTTON", "BUTTON", &_buttonOffset,
				&_buttonJumpInstance, &_buttonCallback, text);
			if (FlagSet(woStatus, WOS_SELECTED) &&
				FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
				SendMessage(screenID, BM_SETCHECK, 1, 0);
		}
		break;

	case L_LEFT:
	case L_UP:
		if (previous && Previous()->Inherited(ID_BUTTON))
			object = Previous();
		else
			for (object = this; object->Next() &&
				object->Next()->Inherited(ID_BUTTON); object = object->Next())
				;
		if (object && object != this)
		{
			UI_EVENT event(S_ADD_OBJECT);
			event.data = object;
			parent->Event(event);
		}
		break;

	case L_RIGHT:
	case L_DOWN:
		if (next && Next()->Inherited(ID_BUTTON))
			object = Next();
		else
			for (object = this; object->Previous() &&
				object->Previous()->Inherited(ID_BUTTON); object = object->Previous())
				;
		if (object && object != this)
		{
			UI_EVENT event(S_ADD_OBJECT);
			event.data = object;
			parent->Event(event);
		}
		break;

	case L_SELECT:
		ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
		if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON) && screenID)
			SendMessage(screenID, BM_SETCHECK, FlagSet(woStatus, WOS_SELECTED), 0);
		parent->Information(CHECK_SELECTION, NULL);
		break;

	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_DRAWITEM)
			ccode = DrawItem(event, ccode);
		else if (message == BM_SETSTATE)
		{
			if (event.message.wParam)
				btStatus |= BTS_DEPRESSED;
			else
				btStatus &= ~BTS_DEPRESSED;
			ccode = UI_WINDOW_OBJECT::Event(event);
		}
		else if (message == WM_MENUCHAR && event.message.wParam == hotKey)
			ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
		else if (message == WM_LBUTTONUP && !FlagSet(woStatus, WOS_EDIT_MODE))
		{
			if (FlagSet(btStatus, BTS_DEPRESSED))
			{
				ccode = UI_WINDOW_OBJECT::Event(event);
				Event(UI_EVENT(L_SELECT));
			}
			else
				ccode = UI_WINDOW_OBJECT::Event(event);
		}
		else if (message == WM_LBUTTONDBLCLK && !FlagSet(woStatus, WOS_EDIT_MODE) &&
			FlagSet(btFlags, BTF_DOUBLE_CLICK) && userFunction)
		{
			ccode = UI_WINDOW_OBJECT::Event(event);
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, L_DOUBLE_CLICK);
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_BUTTON::DrawItem(const UI_EVENT &, EVENT_TYPE ccode)
{
	UI_REGION region = true;

	// Handle text mode buttons.
	if (display->isText)
	{
		if (!text)
			return (ccode);
		if (FlagSet(woFlags, WOF_BORDER))
			DrawBorder(screenID, region, FALSE, ccode);
		DrawShadow(screenID, region, FlagSet(btStatus, BTS_DEPRESSED) ? -depth : depth, TRUE, ccode);
		if (FlagSet(btFlags, BTF_CHECK_BOX))
		{
			char activeCheck[5] = { '[', 'X', ']', ' ', 0x00 };
			char inactiveCheck[5] = { '[', ' ', ']', ' ', 0x00 };
			display->Text(screenID, region.left, region.top,
				FlagSet(woStatus, WOS_SELECTED) ? activeCheck : inactiveCheck,
				lastPalette, -1, FALSE, FALSE, &clip);
			region.left += 4;
		}
		else if (FlagSet(btFlags, BTF_RADIO_BUTTON))
		{
			char activeRadio[5] = { '(', 0x07, ')', ' ', 0x00 };
			char inactiveRadio[5] = { '(', ' ', ')', ' ', 0x00 };
			display->Text(screenID, region.left, region.top,
				FlagSet(woStatus, WOS_SELECTED) ? activeRadio : inactiveRadio,
				lastPalette, -1, FALSE, FALSE, &clip);
			region.left += 4;
		}
		DrawText(screenID, region, text, lastPalette, TRUE, ccode);
		return (ccode);
	}

	// Draw the border and shadow.
	if (FlagSet(woFlags, WOF_BORDER))
		DrawBorder(screenID, region, FALSE, ccode);
	UI_PALETTE *outline = LogicalPalette(ccode, ID_OUTLINE);
	if (depth > 1)
	{
		display->Line(screenID, region.left + 1, region.top,
			region.right - 1, region.top, outline, 1, FALSE, &clip);
		display->Line(screenID, region.left + 1, region.bottom,
			region.right - 1, region.bottom, outline, 1, FALSE, &clip);
		display->Line(screenID, region.left, region.top + 1,
			region.left, region.bottom - 1, outline, 1, FALSE, &clip);
		display->Line(screenID, region.right, region.top + 1,
			region.right, region.bottom - 1, outline, 1, FALSE, &clip);
		--region;
	}
	if (FlagSet(btFlags, BTF_NO_3D | BTF_CHECK_BOX | BTF_RADIO_BUTTON))
		lastPalette = LogicalPalette(ccode, ID_STRING);
	if (FlagSet(btFlags, BTF_NO_3D))
		display->Rectangle(screenID, region, lastPalette, 0, TRUE, FALSE, &clip);
	else
		DrawShadow(screenID, region, FlagSet(btStatus, BTS_DEPRESSED) ? -depth : depth, TRUE, ccode);

	// Draw the text.
	region.left += BUTTON_OFFSET;
	if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT | WOF_JUSTIFY_CENTER))
		region.right -= BUTTON_OFFSET;

	if (!bitmapArray && !FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
	{
		if (text)
			DrawText(screenID, region, text, lastPalette, FALSE, ccode);
		return (ccode);
	}

	// Draw the bitmap.
	int fieldWidth = region.right - region.left + 1;
	int fieldHeight = region.bottom - region.top + 1;
	if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
		bitmapWidth = bitmapHeight = display->TextHeight(NULL, screenID, font) + 2;
	int left = region.left;
	if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		left = region.right - bitmapWidth + 1;
	else if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
		left = region.left + (fieldWidth - bitmapWidth) / 2;
	int top = (FlagSet(woFlags, WOF_JUSTIFY_CENTER) && text) ?
		region.top + 2 : region.top + (fieldHeight - bitmapHeight) / 2;
	int right = left + bitmapWidth - 1;
	int bottom = top + bitmapHeight - 1;

	if (FlagSet(btFlags, BTF_CHECK_BOX))
	{
		display->Rectangle(screenID, left, top, right, bottom, lastPalette, 1, TRUE, FALSE, &clip);
		if (FlagSet(woStatus, WOS_SELECTED))
		{
			display->Line(screenID, left, top, right, bottom, lastPalette, 1, FALSE, &clip);
			display->Line(screenID, right, top, left, bottom, lastPalette, 1, FALSE, &clip);
		}
	}
	else if (FlagSet(btFlags, BTF_RADIO_BUTTON))
	{
		int radius = bitmapWidth / 2;
		display->Ellipse(screenID, left + radius, top + radius, 0, 360,
			radius, radius, lastPalette, TRUE, FALSE, &clip);
		if (FlagSet(woStatus, WOS_SELECTED))
			display->Ellipse(screenID, left + radius, top + radius, 0, 360,
				radius / 2, radius / 2, outline, TRUE, FALSE, &clip);
	}
	else if (bitmapArray)
		display->Bitmap(screenID, left, top, bitmapWidth, bitmapHeight, bitmapArray, NULL, &clip);

	// Draw the text.
	if (text)
	{
		WOF_FLAGS flags = woFlags;
		if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
			region.top = region.bottom - display->cellHeight + 4;
		else if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
		{
			region.right -= bitmapWidth - BUTTON_OFFSET;
			woFlags &= ~(WOF_JUSTIFY_RIGHT);
		}
		else
			region.left += bitmapWidth + BUTTON_OFFSET;
		DrawText(screenID, region, text, lastPalette, FALSE, ccode);
		woFlags = flags;
	}

	// Return the control code.
	return (ccode);
}

EVENT_TYPE UIW_BUTTON::Event(const UI_EVENT &event)
{
	static initializedTime = FALSE;
	static UI_TIME lastTime;
	if (!initializedTime)
	{
		lastTime.Import();
		initializedTime = TRUE;
	}
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_BUTTON);
	switch (ccode)
	{
	case S_INITIALIZE:
		// Compute the button region.
		UI_WINDOW_OBJECT::Event(event);
		if (!display->isText)
		{
			if (!FlagSet(btFlags, BTF_AUTO_SIZE))
				relative.top = relative.bottom - (display->cellHeight - display->preSpace - display->postSpace);
			int offset = 2;
			if (FlagSet(woFlags, WOF_BORDER))
				offset += 2;
			if (!FlagSet(btFlags, BTF_NO_3D))
				offset += 6;
			if (bitmapHeight && FlagSet(btFlags, BTF_AUTO_SIZE) && FlagSet(woFlags, WOF_JUSTIFY_CENTER))
			{
				relative.bottom = relative.top + bitmapHeight + offset;
				if (text)
					relative.bottom += display->cellHeight;
				true.bottom = relative.bottom;
			}
			else if (FlagSet(btFlags, BTF_AUTO_SIZE))
				relative.top = relative.bottom - display->cellHeight * 10 / 9;
			if (bitmapHeight && relative.bottom - relative.top < bitmapHeight + offset)
				relative.bottom = relative.top + bitmapHeight + offset;
		}
		else if (FlagSet(btFlags, BTF_AUTO_SIZE) && depth == 2)
			relative.top = relative.bottom - 1;
		else
			relative.bottom = relative.top;
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			DrawItem(event, ccode);
			woStatus &= ~WOS_REDISPLAY;
		}
		break;

	case L_LEFT:
	case L_UP:
		if (previous && Previous()->Inherited(ID_BUTTON))
			object = Previous();
		else
			for (object = this; object->Next() &&
				object->Next()->Inherited(ID_BUTTON); object = object->Next())
				;
		if (object && object != this)
		{
			UI_EVENT event(S_ADD_OBJECT);
			event.data = object;
			parent->Event(event);
		}
		break;

	case L_RIGHT:
	case L_DOWN:
		if (next && Next()->Inherited(ID_BUTTON))
			object = Next();
		else
			for (object = this; object->Previous() &&
				object->Previous()->Inherited(ID_BUTTON); object = object->Previous())
				;
		if (object && object != this)
		{
			UI_EVENT event(S_ADD_OBJECT);
			event.data = object;
			parent->Event(event);
		}
		break;

	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		{
		BTS_STATUS oldStatus = btStatus;
		ccode = UI_WINDOW_OBJECT::Event(event);
		if (ccode == L_END_SELECT && !FlagSet(btStatus, BTS_DEPRESSED))
			break;	// Case where selection was down-click from another object.
		else if (ccode == S_UNKNOWN || ccode == L_END_SELECT)
			btStatus &= ~BTS_DEPRESSED;
		else
			btStatus |= BTS_DEPRESSED;
		if (oldStatus != btStatus && !FlagSet(btFlags, BTF_NO_3D) && depth)
		{
			UI_REGION region = true; --region;
			if (display->isText)
				DrawItem(event, S_CURRENT);
			else if (FlagSet(btStatus, BTS_DEPRESSED))
				DrawShadow(screenID, region, -depth, FALSE, ccode);
			else
				DrawShadow(screenID, region, depth, FALSE, ccode);
		}
		}
		if ((FlagSet(btFlags, BTF_DOWN_CLICK) && ccode != L_BEGIN_SELECT) ||
			(!FlagSet(btFlags, BTF_DOWN_CLICK) && ccode != L_END_SELECT))
			break;
		// Continue to L_SELECT.
	case L_SELECT:
		{
		if (!FlagSet(btFlags, BTF_NO_TOGGLE) && !parent->Inherited(ID_LIST))
			woStatus ^= WOS_SELECTED;
		UI_TIME currentTime;
		if (ccode == L_END_SELECT && userFunction &&
			!parent->Inherited(ID_LIST) && currentTime - lastTime < doubleClickRate)
		{
			ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*userFunction)(this, uEvent, L_DOUBLE_CLICK);
		}
		else if (ccode == L_END_SELECT)
		{
			ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
			lastTime = currentTime;
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(UI_EVENT(L_SELECT));
		if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
		{
			lastPalette = LogicalPalette(S_CURRENT);
			DrawItem(event, S_CURRENT);
		}
		}
		break;

	default:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_BUTTON::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_BUTTON;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_BUTTON;
		if (text)
		{
			ui_strrepc(text, '~', '&');
			char *tHotKey = strchr(text, '&');
			if (tHotKey)
				hotKey = tolower(tHotKey[1]);
		}
#ifdef _WINDOWS
		// Read in the bitmap if so specified.
		colorBitmap = monoBitmap = 0;
		if (bitmapName && display)
		{
			char tempName[32];
			strcpy(tempName, "_");
			strcat(tempName, bitmapName);
			colorBitmap = LoadBitmap(display->hInstance, bitmapName);
			monoBitmap = LoadBitmap(display->hInstance, tempName);
			if (colorBitmap && bitmapArray)
			{
				delete bitmapArray;
				bitmapArray = NULL;
			}
		}
		else if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
			colorBitmap = LoadBitmap(0, (LPSTR)OBM_CHECKBOXES);
#endif
#ifdef ZIL_PERSISTENCE
#ifdef _WINDOWS
		if (!colorBitmap && bitmapName && !bitmapArray && defaultStorage && !defaultStorage->storageError)
#else
		if (bitmapName && !bitmapArray && defaultStorage && !defaultStorage->storageError)
#endif
		{
			defaultStorage->ChDir("~UI_BITMAP");
			UI_STORAGE_OBJECT bFile(*defaultStorage, bitmapName, ID_BITMAP_IMAGE, UIS_READ);
			if (!bFile.objectError)
			{
				short _value; bFile.Load(&_value); bitmapWidth = _value;
				bFile.Load(&_value); bitmapHeight = _value;
				bitmapArray = new UCHAR[bitmapWidth * bitmapHeight];
				bFile.Load(bitmapArray, sizeof(UCHAR), bitmapWidth * bitmapHeight);
			}
		}
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
		if (FlagSet(btFlags, BTF_SEND_MESSAGE) && !userFunction)
			userFunction = &UIW_BUTTON::Message;
		if (FlagSet(btFlags, BTF_CHECK_BOX | BTF_RADIO_BUTTON))
		{
			if (FlagSet(btFlags, BTF_RADIO_BUTTON))
				btFlags |= BTF_NO_TOGGLE;
			btFlags &= ~BTF_AUTO_SIZE;
			btFlags |= BTF_NO_3D;
		}
		if (FlagSet(btFlags, BTF_NO_3D))
			depth = 0;
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UI_WINDOW_OBJECT::Information(CHANGED_FLAGS, data, ID_BUTTON);
		if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
			dwStyle |= BS_LEFTTEXT;
		if (FlagSet(btFlags, BTF_CHECK_BOX))
			dwStyle |= BS_CHECKBOX;
//			dwStyle |= BS_AUTOCHECKBOX;
		else if (FlagSet(btFlags, BTF_RADIO_BUTTON))
//			dwStyle |= BS_RADIOBUTTON;
			dwStyle |= BS_AUTORADIOBUTTON;
		else if (colorBitmap)
			dwStyle |= BS_OWNERDRAW;
		if (screenID && request == CHANGED_FLAGS && objectID == ID_BUTTON)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_BUTTON &&
			FlagSet(btFlags, BTF_AUTO_SIZE))
		{
			Event(UI_EVENT(S_INITIALIZE));
			Event(UI_EVENT(S_SIZE));
		}
#endif
		break;

	case GET_TEXT:
		if (!data)
			return (DataGet());
		*(char **)data = DataGet();
		break;

	case SET_TEXT:
		DataSet((char *)data);
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_BUTTON)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &btFlags;
		else if (request == GET_FLAGS)
			*(BTF_FLAGS *)data = btFlags;
		else if (request == SET_FLAGS)
			btFlags |= *(BTF_FLAGS *)data;
		else
			btFlags &= ~(*(BTF_FLAGS *)data);
		break;

	case GET_STATUS:
	case SET_STATUS:
	case CLEAR_STATUS:
		if (objectID && objectID != ID_BUTTON)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_STATUS && !data)
			data = &btStatus;
		else if (request == GET_STATUS)
			*(BTS_STATUS *)data = btStatus;
		else if (request == SET_STATUS)
			btStatus |= *(BTS_STATUS *)data;
		else if (request == SET_STATUS)
			btStatus &= ~(*(BTS_STATUS *)data);
		break;

	case GET_VALUE:
		*(EVENT_TYPE *)data = value;
		break;

	case SET_VALUE:
		value = *(EVENT_TYPE *)data;
		break;

	case GET_BITMAP_WIDTH:
		*(int *)data = bitmapWidth;
		break;

	case SET_BITMAP_WIDTH:
		bitmapWidth = *(int *)data;
		break;

	case GET_BITMAP_HEIGHT:
		*(int *)data = bitmapHeight;
		break;

	case SET_BITMAP_HEIGHT:
		bitmapHeight = *(int *)data;
		break;

	case GET_BITMAP_ARRAY:
#ifdef _WINDOWS
		if (!bitmapArray)
			WindowsToZincBitmap(display, colorBitmap, monoBitmap,
				&bitmapWidth, &bitmapHeight, &bitmapArray);
#endif
		if (!data)
			return (bitmapArray);
		*(UCHAR **)data = bitmapArray;
		break;

	case SET_BITMAP_ARRAY:
		if (bitmapArray)
			delete bitmapArray;
		if (data)
		{
			bitmapArray = new UCHAR[bitmapWidth * bitmapHeight];
			memcpy(bitmapArray, (UCHAR *)data, bitmapWidth * bitmapHeight);
		}
		else
			bitmapArray = NULL;
#ifdef _WINDOWS
		if (colorBitmap)
			DeleteObject(colorBitmap);
		if (monoBitmap)
			DeleteObject(monoBitmap);
		if (bitmapArray)
		{
			ZincToWindowsBitmap(display, bitmapWidth, bitmapHeight,
				bitmapArray, NULL, &colorBitmap, &monoBitmap);
			dwStyle = (dwStyle & 0xFFFFFF00L) | BS_OWNERDRAW;
			if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
				dwStyle |= BS_LEFTTEXT;
		}
		else
		{
			colorBitmap = monoBitmap = 0;
			dwStyle = (dwStyle & 0xFFFFFF00L);
			if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
				dwStyle |= BS_LEFTTEXT;
		}
#endif
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

EVENT_TYPE UIW_BUTTON::Message(UI_WINDOW_OBJECT *object, UI_EVENT &event,
	EVENT_TYPE ccode)
{
	if (ccode == L_SELECT)
	{
		EVENT_TYPE command = ((UIW_BUTTON *)object)->value;
		event.type = command;
		event.rawCode = 0;
		event.data = object;
		if (command == L_RESTORE || command == L_MOVE || command == L_SIZE ||
			command == L_MINIMIZE || command == L_MAXIMIZE || command == S_CLOSE)
		{
			for (UI_WINDOW_OBJECT *tObject = object; tObject; tObject = tObject->parent)
				if (tObject->Inherited(ID_WINDOW) && !tObject->Inherited(ID_MENU))
				{
					event.data = tObject;
					break;
				}
		}
		object->eventManager->Put(event);
	}
	return (ccode);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_BUTTON::UIW_BUTTON(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UI_WINDOW_OBJECT(0, 0, 15, 1, WOF_NO_FLAGS, WOAF_NO_FLAGS),
	btStatus(BTS_NO_STATUS), bitmapWidth(0), bitmapHeight(0), bitmapArray(NULL)
{
	// Initialize the button information.
	UIW_BUTTON::Load(name, directory, file);
	UIW_BUTTON::Information(INITIALIZE_CLASS, NULL);
	UIW_BUTTON::DataSet(text);
}

void UIW_BUTTON::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the button information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	file->Load(&btFlags);
	short _value; file->Load(&_value); value = _value;
	file->Load(&_value); depth = _value;
	file->Load(&text);
	file->Load(&bitmapName);

	// Load the bitmap information.
	if (bitmapName)
	{
		directory->ChDir("~UI_BITMAP");
		UI_STORAGE_OBJECT bFile(*directory, bitmapName, ID_BITMAP_IMAGE, UIS_READ);
		if (!bFile.objectError)
		{
			short _value; bFile.Load(&_value); bitmapWidth = _value;
			bFile.Load(&_value); bitmapHeight = _value;
			bitmapArray = new UCHAR[bitmapWidth * bitmapHeight];
			bFile.Load(bitmapArray, sizeof(UCHAR), bitmapWidth * bitmapHeight);
		}
	}
}

void UIW_BUTTON::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the button information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	file->Store(btFlags);
	short _value = value; file->Store(_value);
	_value = depth; file->Store(_value);
	file->Store(text);
	file->Store(bitmapName);

	// Store the bitmap information.
#ifdef _WINDOWS
	if (bitmapName && colorBitmap && !bitmapArray)
		WindowsToZincBitmap(display, colorBitmap, monoBitmap,
			&bitmapWidth, &bitmapHeight, &bitmapArray);
#endif
	if (bitmapName && bitmapArray)
	{
		if (directory->ChDir("~UI_BITMAP"))
		{
			directory->ChDir("~");
			directory->MkDir("UI_BITMAP");
			directory->ChDir("~UI_BITMAP");
		}
		UI_STORAGE_OBJECT bFile(*directory, bitmapName, ID_BITMAP_IMAGE, UIS_CREATE | UIS_READWRITE);
		if (!bFile.objectError)
		{
			_value = bitmapWidth; bFile.Store(_value);
			_value = bitmapHeight; bFile.Store(_value);
			bFile.Store(bitmapArray, sizeof(UCHAR), bitmapWidth * bitmapHeight);
		}
	}
}
#endif

