//	Zinc Interface Library - DATE1.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
#define MAX_LENGTH	64

// ----- UIW_DATE -----------------------------------------------------------

UIW_DATE::UIW_DATE(int left, int top, int width, UI_DATE *_date,
	const char *_range, DTF_FLAGS _dtFlags, WOF_FLAGS _woFlags,
	USER_FUNCTION _userFunction) :
	UIW_STRING(left, top, width, NULL, MAX_LENGTH, STF_NO_FLAGS, _woFlags, _userFunction),
	date(NULL), dtFlags(_dtFlags)
{
	extern void z_date1_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_date1_dummy();

	// Initialize the date information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		date = new UI_DATE;
	else
	{
		text = new char[MAX_LENGTH+1];
		text[0] = '\0';
	}
	range = ui_strdup(_range);
	UIW_DATE::Information(INITIALIZE_CLASS, NULL);
	UIW_DATE::DataSet(_date);
}

UIW_DATE::~UIW_DATE(void)
{
	// Restore the date information.
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete date;
	else
		delete text;
	if (range)
		delete range;
}

UI_DATE *UIW_DATE::DataGet(void)
{
	// Return the date.
	UIW_STRING::DataGet();
	date->Import(text, dtFlags);
	return (date);
}

void UIW_DATE::DataSet(UI_DATE *_date)
{
	// Reset the date.
	if (date == _date || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		date = _date;
	else if (_date)
		date->Import(*_date);
	else
		date->Import();
	date->Export(text, dtFlags);
	UIW_STRING::DataSet(text);
}

EVENT_TYPE UIW_DATE::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_DATE);
	switch (ccode)
	{
	case L_SELECT:
	case S_NON_CURRENT:
		if (!FlagSet(woStatus, WOS_INTERNAL_ACTION))
		{
			UI_DATE tDate = *date;				// Keep a temporary date in case of error.
			UIW_STRING::DataGet();
			date->Import(text, dtFlags);
			ccode = UIW_STRING::Event(event);	// Call the user or validate function.
			if (ccode == -1)
				*date = tDate;					// An error occurred, restore the old date.
			else
				woStatus &= ~WOS_UNANSWERED;
			date->Export(text, dtFlags);
			UIW_STRING::DataSet(text);
		}
		else
			ccode = UIW_STRING::Event(event);
		break;

	case S_CREATE:
		date->Export(text, dtFlags);
		// Continue to default.
	default:
		ccode = UIW_STRING::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}

void *UIW_DATE::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_DATE;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_DATE;
		windowID[1] = ID_STRING;
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_STRING::Information(CHANGED_FLAGS, data, ID_DATE);
		if (screenID && request == CHANGED_FLAGS && objectID == ID_DATE)
		{
			DestroyWindow(screenID);
			Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#endif
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_DATE)
			data = UIW_STRING::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &dtFlags;
		else if (request == GET_FLAGS)
			*(DTF_FLAGS *)data = dtFlags;
		else if (request == SET_FLAGS)
			dtFlags |= *(DTF_FLAGS *)data;
		else
			dtFlags &= ~(*(DTF_FLAGS *)data);
		break;

	default:
		data = UIW_STRING::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

int UIW_DATE::Validate(int processError)
{
	// Set up the date error messages.
	static struct
	{
		DTI_RESULT errorCode;
		char *message;
	} errorTable[] =
	{
		{ DTI_INVALID,			"The date %s is in an invalid format." 		},
		{ DTI_AMBIGUOUS,		"The date %s has an ambiguous month name."	},
		{ DTI_INVALID_NAME,		"The date %s has an invalid month name."	},
		{ DTI_VALUE_MISSING,	"A date value must be entered."				},
		{ DTI_OUT_OF_RANGE, 	"The date %s must be in the range %s."		},
		{ DTI_OK,				0											}
	};
#ifdef ZIL_PERSISTENCE
	// Check for internationalization.
#endif

	// Check for an absolute date error.
	UI_DATE currentDate;
	char *stringDate = (char *)UIW_STRING::Information(GET_TEXT, NULL);
	DTI_RESULT errorCode = currentDate.Import(stringDate, dtFlags);

	// Check for a range error.
	if (range && errorCode == DTI_OK)
		errorCode = DTI_OUT_OF_RANGE;
	for (char *tRange = range; tRange && errorCode == DTI_OUT_OF_RANGE; )
	{
		char minDate[MAX_LENGTH], maxDate[MAX_LENGTH];
		tRange = ParseRange(tRange, minDate, maxDate);
		if (currentDate >= UI_DATE(minDate, rangeFlags) &&
			currentDate <= UI_DATE(maxDate, rangeFlags))
			errorCode = DTI_OK;
	}

	// Process the error code.
	woStatus &= ~WOS_INVALID;
	if (errorCode == DTI_OK)			// Set up the new date.
	{
		currentDate.Export(stringDate, dtFlags);
		UIW_STRING::Information(SET_TEXT, stringDate);
		return (errorCode);
	}
	else if (!errorSystem)				// Restore the original date.
	{
		date->Export(stringDate, dtFlags);
		UIW_STRING::Information(SET_TEXT, stringDate);
		return (errorCode);
	}
	else if (!processError)
		return (errorCode);

	// Generate the error message and wait for a response.
	for (int i = 0; errorTable[i].message; i++)
		if (errorTable[i].errorCode == errorCode)
		{
			woStatus |= WOS_INTERNAL_ACTION;
			UIS_STATUS errorStatus = errorSystem->ReportError(windowManager,
				WOS_INVALID, errorTable[i].message, stringDate, range);
			woStatus &= ~WOS_INTERNAL_ACTION;
			if (errorStatus == WOS_INVALID)
				woStatus |= WOS_INVALID;
			else
				return (-1);		// This will cause the date to be restored.
			break;
		}
	return (errorCode);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_DATE::UIW_DATE(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_STRING(0, 0, 15, NULL, MAX_LENGTH, STF_NO_FLAGS, WOF_NO_FLAGS)
{
	// Initialize the date information.
	UIW_DATE::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_STRING::Information(INITIALIZE_CLASS, NULL);
	UIW_DATE::Information(INITIALIZE_CLASS, NULL);
}

void UIW_DATE::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the date information.
	UIW_STRING::Load(name, directory, file);
	file->Load(&dtFlags);
	file->Load(&range);

	date = new UI_DATE;
	if (FlagSet(dtFlags, DTF_SYSTEM))
		date->Export(text, dtFlags);	// Set the text according to the system date.
	else
		date->Import(text, dtFlags);	// Set the date according to the loaded text.
}

void UIW_DATE::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the date information.
	UIW_STRING::Store(name, directory, file);
	file->Store(dtFlags);
	file->Store(range);
}
#endif

