//	Zinc Interface Library - HLIST.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include "ui_win.hpp"
#pragma hdrstop

// ----- UIW_HZ_LIST --------------------------------------------------------

UIW_HZ_LIST::UIW_HZ_LIST(int left, int top, int width, int height,
	int _cellWidth, int _cellHeight,
	int (*_compareFunction)(void *element1, void *element2),
	WNF_FLAGS _wnFlags, WOF_FLAGS _woFlags, WOAF_FLAGS _woAdvancedFlags) :
	UIW_WINDOW(left, top, width, height, _woFlags, _woAdvancedFlags),
	cellWidth(_cellWidth), cellHeight(_cellHeight)
{
	// Initialize the menu information.
	wnFlags = _wnFlags;
	compareFunction = _compareFunction;
	UIW_HZ_LIST::Information(INITIALIZE_CLASS, NULL);
}

UIW_HZ_LIST::UIW_HZ_LIST(int left, int top, int width, int height,
	int (*_compareFunction)(void *element1, void *element2), 
	WOF_FLAGS flagSetting, UI_ITEM *item) :
	UIW_WINDOW(left, top, width, height, WOF_BORDER, WOAF_NO_FLAGS)
{
	// Initialize the list information.
	wnFlags |= WNF_NO_WRAP;
	compareFunction = _compareFunction;
	UIW_HZ_LIST::Information(INITIALIZE_CLASS, NULL);

	// Add the string items.
	for (int i = 0; item[i].text; i++)
	{
		UIW_STRING *string = new UIW_STRING(0, 0, width, item[i].text, -1,
			item[i].flags, WOF_BORDER | WOF_AUTO_CLEAR, (USER_FUNCTION)item[i].data);
		if (FlagSet(flagSetting, item[i].value))
			string->woStatus |= WOS_SELECTED;
		*this + string;
	}
}

UIW_HZ_LIST::~UIW_HZ_LIST(void)
{
}

void UIW_HZ_LIST::Destroy(void)
{
	// Destroy the matrix elements.
#ifdef _WINDOWS
	if (screenID)
	{
		SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
		SendMessage(screenID, LB_RESETCONTENT, 0, 0);
	}
#endif
	UI_LIST::Destroy();
}

#ifdef _WINDOWS
static int _hlistOffset = -1;
static FARPROC _hlistCallback = (FARPROC)DefWindowProc;
static FARPROC _hlistJumpInstance = NULL;

long FAR PASCAL _export HListJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UIW_HZ_LIST *object = (UIW_HZ_LIST *)GetWindowLong(hWnd, _hlistOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_HZ_LIST::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_HZ_LIST);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_hlistJumpInstance)
			_hlistJumpInstance = (FARPROC)HListJumpProcedure;
		UIW_WINDOW::Event(event);
		break;

	case S_CREATE:
		UI_WINDOW_OBJECT::Event(event);
		RegisterObject("UIW_HZ_LIST", "LISTBOX", &_hlistOffset,
			&_hlistJumpInstance, &_hlistCallback, NULL);
		SendMessage(screenID, LB_SETCOLUMNWIDTH, cellWidth * display->cellWidth, 0);
		SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
		for (object = First(); object; object = object->Next())
			object->Event(event);
		SendMessage(screenID, WM_SETREDRAW, TRUE, 0);
		break;

	case S_CURRENT:
	case S_REDISPLAY:
		ccode = UI_WINDOW_OBJECT::Event(event);
		if (current && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE) &&
			SendMessage(screenID, LB_GETCURSEL, 0, 0) != Index(current))
			SendMessage(screenID, LB_SETCURSEL, Index(current), 0);
		break;

	case S_NON_CURRENT:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;

	case S_ADD_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (UI_LIST::Index(object) == -1)
			UIW_WINDOW::Event(event);
		else if (object != current)
		{
			if (current)
				Current()->woStatus &= ~WOS_CURRENT;
			current = object;
			Current()->woStatus |= WOS_CURRENT;
			if (!FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
				SendMessage(screenID, LB_SETCURSEL, Index(current), 0);
		}
		break;

	case S_SUBTRACT_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (UI_LIST::Index(object) != -1)
		{
			if (screenID)
				SendMessage(screenID, LB_DELETESTRING, UI_LIST::Index(object), 0);
			UI_LIST::Subtract(object);
		}
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	default:
		{
		WORD message = event.message.message;
		if (ccode == L_SELECT || (event.type == E_MSWINDOWS && message == WM_LBUTTONUP))
		{
			ccode = UI_WINDOW_OBJECT::Event(event);
			long selection = SendMessage(screenID, LB_GETCURSEL, 0, 0);
			object = (selection >= 0) ? (UI_WINDOW_OBJECT *)UI_LIST::Get(LOWORD(selection)) : NULL;
			if (object)
				object->Event(UI_EVENT(L_SELECT));
		}
		else if (event.type != E_MSWINDOWS)
			ccode = UIW_WINDOW::Event(event);
		else if (message == WM_COMPAREITEM)
			ccode = UIW_WINDOW::Event(event);
		else if (message == WM_MEASUREITEM)
		{
			MEASUREITEMSTRUCT *measure = (MEASUREITEMSTRUCT *)event.message.lParam;
			measure->itemWidth = cellWidth * display->cellWidth;
			measure->itemHeight = cellHeight * (display->cellHeight - display->preSpace - display->postSpace - 2);
		}
		else if (message == WM_LBUTTONDBLCLK && current && Current()->userFunction)
		{
			ccode = UI_WINDOW_OBJECT::Event(event);
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*Current()->userFunction)(Current(), uEvent, L_DOUBLE_CLICK);
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		}
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_HZ_LIST::Event(const UI_EVENT &event)
{
	static initializedTime = FALSE;
	static UI_TIME lastTime;
	if (!initializedTime)
	{
		lastTime.Import();
		initializedTime = TRUE;
	}

	UI_WINDOW_OBJECT *object;
	UI_WINDOW_OBJECT *oldCurrent = Current();

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_HZ_LIST);
	switch (ccode)
	{
	case S_SIZE:
	case S_CREATE:
		// Compute the object region.
		clipList.Destroy();
		UI_WINDOW_OBJECT::Event(event);

		// Compute the support object regions.
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION))
			{
				UI_REGION region = object->true;
				clipList.Destroy();
				clipList.Add(new UI_REGION_ELEMENT(screenID, --region));
			}
			else if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, FALSE);
		}
		// Continue to S_REDISPLAY.
	case S_REDISPLAY:
		// Compute the list object regions.
		{
		int top = 0, left = 0;
		int itemWidth = cellWidth * display->cellWidth;
		int itemHeight = cellHeight * (display->cellHeight - display->preSpace - display->postSpace);
		int totalHeight = clipList.First() ?
			clipList.First()->region.bottom - clipList.First()->region.top + 1 :
			true.bottom - true.top + 1;
		for (object = First(); object; object = object->Next())
		{
			if (top + itemHeight > totalHeight)
			{
				top = 0;
				left += itemWidth;
			}
			object->relative.left = left;
			object->relative.top = top;
			object->relative.right = left + itemWidth - 1;
			top += itemHeight;
			object->relative.bottom = top - 1;
			object->woFlags &= ~(WOF_NON_SELECTABLE | WOF_BORDER);
			object->woFlags |= WOF_VIEW_ONLY;
			if (ccode == S_REDISPLAY)
				object->Event(UI_EVENT(S_CREATE));
			else
				object->Event(event);
		}

		if (ccode == S_REDISPLAY)
			UI_WINDOW_OBJECT::Event(event);

		oldCurrent = NULL;	// Ensure scroll region is updated.
		}
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	case L_LEFT:
	case L_RIGHT:
		{
		int currentTop = Current()->true.top;
		object = (ccode == L_LEFT) ? Current()->Previous() : Current()->Next();
		while (object && object != current)
			if (object->true.top == currentTop)
				UIW_WINDOW::Add(object);
			else
				object = (ccode == L_LEFT) ? object->Previous() : object->Next();
		}
		break;

	case L_UP:
	case L_DOWN:
		{
		int emptyQueue;
		UI_EVENT tEvent;
		tEvent = event;
		ccode = (ccode == L_UP) ? L_PREVIOUS : L_NEXT;
		do
		{
			UIW_WINDOW::Event(UI_EVENT(ccode));
			emptyQueue = eventManager->Get(tEvent, Q_NO_BLOCK | Q_NO_DESTROY);
			if (!emptyQueue && tEvent.type == event.type && tEvent.rawCode == event.rawCode)
				eventManager->Get(tEvent);
			else
				emptyQueue = TRUE;
		} while (!emptyQueue);
		}
		break;

	case L_PGUP:
	case L_PGDN:
		{
		// Go to bottom of current page.
		UI_WINDOW_OBJECT *nObject = Current();
		UI_REGION region;
		if (clipList.First())		// Un-optimized for Zortech bug.
			region = clipList.First()->region;
		else
			region = true;
		for (object = (ccode == L_PGUP) ? Current()->Previous() : Current()->Next(); object;
			 object = (ccode == L_PGUP) ? object->Previous() : object->Next())
			if ((ccode == L_PGUP && object->true.left >= region.left) ||
				(ccode == L_PGDN && object->true.right <= region.right))
				nObject = object;
			else
				break;
		// Go to bottom of next page if already at bottom of current page.
		if (nObject == Current())
		{
			object = (ccode == L_PGUP) ? Current()->Previous() : Current()->Next();
			while (object)
			{
				nObject = object;
				if (nObject->true.top == Current()->true.top)
					break;
				object = (ccode == L_PGUP) ? object->Previous() : object->Next();
			}
		}
		// Select a new current object.
		UIW_WINDOW::Add(nObject);
		}
		break;

	case L_FIRST:
	case L_LAST:
		UIW_WINDOW::Add((ccode == L_FIRST) ? First() : Last());
		break;

	case L_SELECT:
		ccode = UIW_WINDOW::Event(UI_EVENT(L_SELECT));
		if (FlagSet(woAdvancedFlags, WOAF_TEMPORARY))
			eventManager->Put(UI_EVENT(S_CLOSE_TEMPORARY));
		break;

	case L_END_SELECT:
		{
		ccode = UIW_WINDOW::Event(event);
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			break;
		UI_TIME currentTime;
		if (ccode == L_END_SELECT && current && Current()->userFunction && currentTime - lastTime < doubleClickRate)
		{
			UI_EVENT uEvent;
			uEvent = event;
			ccode = (*Current()->userFunction)(Current(), uEvent, L_DOUBLE_CLICK);
		}
		else if (ccode == L_END_SELECT)
			lastTime = currentTime;
		}
		break;

	case L_VIEW:
	case L_BEGIN_SELECT:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		// Continue to default.
	default:
		ccode = UIW_WINDOW::Event(event);
		if (ccode == S_CURRENT || ccode == L_BEGIN_SELECT)
		{
			eventManager->DeviceState(E_CURSOR, D_OFF);
			eventManager->DeviceState(E_MOUSE, DM_VIEW);
		}
		break;
	}

	// Check the horizontal scroll region.
	if (oldCurrent != current && ccode != S_INITIALIZE)
	{
		UI_EVENT hEvent(S_HSCROLL_SET);
		if (hScroll)
		{
			hEvent.scroll.minimum = First()->relative.left;
			hEvent.scroll.maximum = Last()->relative.left;
			hEvent.scroll.current = Current()->relative.left;
			hEvent.scroll.delta = Current()->relative.right - Current()->relative.left + 1;
			hEvent.scroll.showing = hEvent.scroll.delta;
			hScroll->Event(hEvent);
		}
		hEvent.type = S_HSCROLL_CHECK;
		if (ccode != S_CREATE && event.type != E_MOUSE)
			UIW_WINDOW::Event(hEvent);
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_HZ_LIST::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	UI_WINDOW_OBJECT *object;

	// Switch on the request.
	if (!objectID) objectID = ID_HZ_LIST;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_HZ_LIST;
		windowID[1] = ID_LIST;
		windowID[2] = ID_WINDOW;
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_WINDOW::Information(CHANGED_FLAGS, data, ID_HZ_LIST);
		dwStyle |= LBS_NOINTEGRALHEIGHT | LBS_MULTICOLUMN;
		if (FlagSet(wnFlags, WNF_BITMAP_CHILDREN))
		{
			dwStyle |= LBS_OWNERDRAWFIXED;
			woStatus |= WOS_OWNERDRAW;
		}
		if (FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
			dwStyle |= LBS_MULTIPLESEL;
		if (compareFunction)
			dwStyle |= LBS_SORT;
		// We must do worst case fixup for bitmap children.
		if (screenID && request == CHANGED_FLAGS && objectID == ID_HZ_LIST)
		{
			DestroyWindow(screenID);
			if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
			{
				RegisterObject("UIW_HZ_LIST", "LISTBOX", &_hlistOffset,
					&_hlistJumpInstance, &_hlistCallback, NULL);
				SendMessage(screenID, LB_SETCOLUMNWIDTH, cellWidth * display->cellWidth, 0);
				SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
				UI_EVENT event(S_CREATE);
				for (object = First(); object; object = object->Next())
					object->Event(event);
			}
			else
				Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_HZ_LIST)
		{
			if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
				return (parent->Information(request, data));
			UI_EVENT event(S_INITIALIZE);
			Event(event);
			event.type = S_CREATE;
			Event(event);
		}
#endif
		wnFlags |= WNF_CONTINUE_SELECT;
		break;

	case GET_TEXT:
	case GET_BITMAP_ARRAY:
	case SET_BITMAP_ARRAY:
		if (current)
			return (current->Information(request, data, objectID));
		else
			return(NULL);

	default:
		data = UIW_WINDOW::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_HZ_LIST::UIW_HZ_LIST(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_WINDOW(0, 0, 20, 6, WOF_NO_FLAGS)
{
	// Initialize the text information.
	UIW_HZ_LIST::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	UIW_HZ_LIST::Information(INITIALIZE_CLASS, NULL);
}

void UIW_HZ_LIST::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the horizontal list information.
	UIW_WINDOW::Load(name, directory, file);
	short _value; file->Load(&_value); cellWidth = _value;
	file->Load(&_value); cellHeight = _value;
}

void UIW_HZ_LIST::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the horizontal list information.
	UIW_WINDOW::Store(name, directory, file);
	short _value = cellWidth; file->Store(_value);
	_value = cellHeight; file->Store(_value);
}
#endif

