//	Zinc Interface Library - MOUSE.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#include <mem.h>
#endif
#include "ui_evt.hpp"
#pragma hdrstop

// ----- UID_MOUSE ----------------------------------------------------------

#ifdef _WINDOWS
UID_MOUSE::UID_MOUSE(DEVICE_STATE _state, DEVICE_IMAGE _image) :
	UI_DEVICE(E_MOUSE, _state), image(_image)
{
	// Initialize the new mouse information.
	position.column = position.line = 0;
	installed = TRUE;
}

UID_MOUSE::~UID_MOUSE(void)
{
}

EVENT_TYPE UID_MOUSE::Event(const UI_EVENT &event)
{
	// Initialize the standard windows cursors.
	struct MOUSE_ITEM
	{
		EVENT_TYPE image;
		LPSTR identification;
	} _mouseTable[] = {
		{ DM_VIEW, 			(LPSTR)IDC_ARROW },
		{ DM_EDIT, 			(LPSTR)IDC_IBEAM },
		{ DM_WAIT, 			(LPSTR)IDC_WAIT },
		{ DM_MOVE, 			(LPSTR)IDC_SIZE },
		{ DM_HORIZONTAL, 	(LPSTR)IDC_SIZEWE },
		{ DM_VERTICAL, 		(LPSTR)IDC_SIZENS },
		{ DM_DIAGONAL_ULLR, (LPSTR)IDC_SIZENWSE },
		{ DM_DIAGONAL_LLUR, (LPSTR)IDC_SIZENESW },
		{ DM_POSITION, 		(LPSTR)IDC_CROSS },
		{ 0, 0 } };

	// Switch on the raw code.
	EVENT_TYPE ccode = event.type;
	switch (ccode)
	{
	case S_POSITION:
		position = event.position;
		SetCursorPos(position.column, position.line);
		break;

	case S_INITIALIZE:
		ccode = image;
		image = 0;
		// Continue to default.
	default:
		for (int i = 0; _mouseTable[i].image && image != ccode; i++)
			if (_mouseTable[i].image == ccode)
			{
				HCURSOR hCursor = LoadCursor(0, _mouseTable[i].identification);
				SetCursor(hCursor);
			}
		break;
	}

	// Return the control code.
	return (ccode);
}

void UID_MOUSE::Poll(void)
{
}
#else
#include <dos.h>
#include "ui_evt.hpp"

#define gnd	BACKGROUND
#define BLK	BLACK
#define wte	WHITE
#define gRy	LIGHTGRAY

static UCHAR _view[164] =
{
	0, 0, 10, 16,
	BLK,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,wte,BLK,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,wte,wte,BLK,gnd,
	BLK,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,
	BLK,wte,wte,wte,wte,wte,BLK,BLK,BLK,gnd,
	BLK,wte,wte,BLK,wte,wte,BLK,gnd,gnd,gnd,
	BLK,BLK,BLK,BLK,BLK,wte,wte,BLK,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd
};

static UCHAR _edit[79] =
{
	2, 6, 5, 15,
	BLK,BLK,gnd,BLK,BLK,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	gnd,gnd,BLK,gnd,gnd,
	BLK,BLK,gnd,BLK,BLK
};

static UCHAR _wait[409] =
{
	7, 13, 15, 27,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,
	BLK,BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,
	gnd,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,wte,wte,wte,wte,gRy,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,gRy,wte,gRy,wte,gRy,wte,gRy,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,gRy,wte,gRy,wte,gRy,wte,wte,BLK,BLK,gnd,
	gnd,gnd,BLK,BLK,wte,wte,gRy,wte,gRy,wte,wte,BLK,BLK,gnd,gnd,
	gnd,gnd,gnd,BLK,BLK,wte,wte,gRy,wte,wte,BLK,BLK,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,BLK,wte,wte,wte,BLK,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,wte,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,wte,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,wte,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,BLK,wte,wte,wte,BLK,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,BLK,BLK,wte,wte,wte,wte,wte,BLK,BLK,gnd,gnd,gnd,
	gnd,gnd,BLK,BLK,wte,wte,wte,gRy,wte,wte,wte,BLK,BLK,gnd,gnd,
	gnd,BLK,BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,wte,wte,gRy,wte,wte,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,wte,gRy,wte,gRy,wte,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,wte,gRy,wte,gRy,wte,gRy,wte,wte,BLK,BLK,gnd,
	gnd,BLK,BLK,wte,gRy,wte,gRy,wte,gRy,wte,gRy,wte,BLK,BLK,gnd,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,
	BLK,BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,BLK,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK
};

static UCHAR _move[365] =
{
	9, 9, 19, 19,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,
	gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,
	gnd,BLK,wte,wte,BLK,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,BLK,wte,wte,BLK,gnd,
	BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,gnd,wte,wte,wte,BLK,
	gnd,BLK,wte,wte,BLK,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,BLK,wte,wte,BLK,gnd,
	gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,
	gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd,gnd
};

static UCHAR _horizontal[196] =
{
	7, 5, 16, 12,
	gnd,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,
	gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,
	gnd,BLK,wte,wte,wte,BLK,BLK,BLK,BLK,BLK,BLK,wte,wte,wte,BLK,gnd,
	BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,
	BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,
	gnd,BLK,wte,wte,wte,BLK,BLK,BLK,BLK,BLK,BLK,wte,wte,wte,BLK,gnd,
	gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,
	gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,BLK,gnd,gnd,gnd,gnd,gnd,
};

static UCHAR _vertical[196] =
{
	5, 7, 12, 16,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,
	gnd,gnd,BLK,wte,wte,wte,wte,wte,wte,BLK,gnd,gnd,
	gnd,BLK,wte,wte,wte,wte,wte,wte,wte,wte,BLK,gnd,
	BLK,BLK,BLK,BLK,BLK,wte,wte,BLK,BLK,BLK,BLK,BLK,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	BLK,BLK,BLK,BLK,BLK,wte,wte,BLK,BLK,BLK,BLK,BLK,
	gnd,BLK,wte,wte,wte,wte,wte,wte,wte,wte,BLK,gnd,
	gnd,gnd,BLK,wte,wte,wte,wte,wte,wte,BLK,gnd,gnd,
	gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,gnd,BLK,BLK,gnd,gnd,gnd,gnd,gnd
};

static UCHAR _left[173] =
{
	5, 5, 12, 12,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,BLK,
	BLK,wte,BLK,BLK,wte,wte,wte,BLK,gnd,gnd,BLK,BLK,
	BLK,BLK,gnd,gnd,BLK,wte,wte,wte,BLK,BLK,wte,BLK,
	BLK,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK
};

static UCHAR _right[173] =
{
	5, 5, 12, 12,
	gnd,gnd,gnd,gnd,BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,
	gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,
	gnd,gnd,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,BLK,
	BLK,gnd,gnd,gnd,gnd,BLK,wte,wte,wte,wte,wte,BLK,
	BLK,BLK,gnd,gnd,BLK,wte,wte,wte,BLK,BLK,wte,BLK,
	BLK,wte,BLK,BLK,wte,wte,wte,BLK,gnd,gnd,BLK,BLK,
	BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,BLK,
	BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,gnd,
	BLK,wte,wte,wte,wte,wte,BLK,gnd,gnd,gnd,gnd,gnd,
	BLK,BLK,BLK,BLK,BLK,BLK,BLK,BLK,gnd,gnd,gnd,gnd
};

static UCHAR _position[125] =
{
	5, 5, 11, 11,
	gnd,gnd,gnd,gnd,BLK,BLK,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	BLK,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,BLK,
	BLK,wte,wte,wte,wte,wte,wte,wte,wte,wte,BLK,
	BLK,BLK,BLK,BLK,BLK,wte,BLK,BLK,BLK,BLK,BLK,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,wte,BLK,gnd,gnd,gnd,gnd,
	gnd,gnd,gnd,gnd,BLK,BLK,BLK,gnd,gnd,gnd,gnd
};

static unsigned _oldMouseMask;
static unsigned _oldMouseSegment;
static unsigned _oldMouseOffset;

static unsigned char _oldMouseState = 0;
static unsigned _mouseState = 0;
static int _horizontalMickeys = 0;
static int _verticalMickeys = 0;
static int _horizontalRatio = 1;
static int _verticalRatio = 1;
static UID_MOUSE *_mouse = NULL;

#define MAX_EVENTS	20
static UI_EVENT _mouseEvent[MAX_EVENTS];
static int _endEvent = -1;
static char _stopInterrupt = FALSE;

#ifdef __ZTC__
#include <msmouse.h>

static char _mouseStack[1024];

void _cdecl MouseISR(unsigned mask, unsigned state, unsigned column, unsigned line)
{
	// Get the mouse information.
	if (!_stopInterrupt && _mouse)
	{
		_stopInterrupt = TRUE;	// Prevent the interrupt from multiple calls.

		// Set up new mouse state. (Current state is in state)
		_mouseState = ((((_oldMouseState ^ state) << 4) | state) << 8) | (*((UCHAR far *) 0x417L) & 0xF);
		_oldMouseState = state;

		// Compute the mouse movement.
		int xCounter, yCounter;
		msm_readcounters(&xCounter, &yCounter);
		_horizontalMickeys += xCounter;
		_verticalMickeys += yCounter;
		int deltaX = _horizontalMickeys / _horizontalRatio;
		int deltaY = _verticalMickeys / _verticalRatio;

		// Make sure the mouse is still on the screen.
		if (_mouse->position.column + deltaX >= _mouse->display->columns ||
			_mouse->position.column + deltaX < 0)
			deltaX = 0;
		if (_mouse->position.line + deltaY >= _mouse->display->lines ||
			_mouse->position.line + deltaY < 0)
			deltaY = 0;
		_mouse->position.column += deltaX;
		_mouse->position.line += deltaY;

		// Update the mickey count.
		_horizontalMickeys %= _horizontalRatio;
		_verticalMickeys %= _verticalRatio;

		// Set the internal mouse information for use in the Poll routine.
		if (_endEvent == -1 || _mouseEvent[_endEvent].rawCode != _mouseState)
			_endEvent = ++_endEvent % MAX_EVENTS;
		_mouseEvent[_endEvent].type = E_MOUSE;
		_mouseEvent[_endEvent].rawCode = _mouseState;
		_mouseEvent[_endEvent].position = _mouse->position;

		// Update the mouse screen position.
//		if (_mouse->display && (deltaX || deltaY))
//			_mouse->display->DeviceMove(MOUSE_IMAGE,
//				_mouse->position.column - _mouse->offset.column,
//				_mouse->position.line - _mouse->offset.line);
		_stopInterrupt = FALSE;
	}
}

UID_MOUSE::UID_MOUSE(DEVICE_STATE _state, DEVICE_IMAGE _image) :
	UI_DEVICE(E_MOUSE, _state), image(_image)
{
	if (msm_init() != -1)	// Try to initialize the mouse driver.
		return;

	// Initialize the new mouse information.
	msm_signal(0xFF, MouseISR, &_mouseStack[1024]);
	position.column = position.line = 0;
	offset.column = offset.line = 0;
	_mouse = this;
	installed = TRUE;
}

UID_MOUSE::~UID_MOUSE(void)
{
	// Restore the old mouse information.
	if (installed)
		msm_term();
}


#endif
#ifdef _MSC_VER

static char _mouseStack[2048];
static unsigned _oldSP;
static unsigned _oldSS;

#pragma check_stack (off)
extern "C" void __interrupt MouseISR(void)
{
	// The following should be accounted for in the stack frame below.
	unsigned state;
	union REGS regs;
	int deltaX;
	int deltaY;
	
	// Lockout interupts until popf.
	__asm	pushf
	__asm	cli

	// Get the mouse information.
	if (!_stopInterrupt && _mouse)
	{
		_stopInterrupt = TRUE;	// Prevent the interrupt from multiple calls.
		__asm	popf

		// Setup stack.
		__asm	mov		_oldSS, ss
		__asm	mov		_oldSP, sp
		__asm	mov		ax, ds
		__asm	mov		ss, ax
		__asm	mov		sp, OFFSET _mouseStack[2048];

		// Setup stack frame.
		__asm	push	bp
		__asm	mov		bp, sp

		// WARNING!!!!  make sure the size of the locals are really big enough
		__asm	sub		sp, 30
		__asm	mov		state, bx

		// Set up new mouse state. (Current state is in state)
		_mouseState = ((((_oldMouseState ^ state) << 4) | state) << 8) | (*((UCHAR far *) 0x417L) & 0xF);
		_oldMouseState = state;

		// Compute the mouse movement.
		regs.x.ax = 11;
		int86(0x33, &regs, &regs);
		_horizontalMickeys += regs.x.cx;
		_verticalMickeys += regs.x.dx;
		deltaX = _horizontalMickeys / _horizontalRatio;
		deltaY = _verticalMickeys / _verticalRatio;

		// Make sure the mouse is still on the screen.
		if (_mouse->position.column + deltaX >= _mouse->display->columns ||
			_mouse->position.column + deltaX < 0)
			deltaX = 0;
		if (_mouse->position.line + deltaY >= _mouse->display->lines ||
			_mouse->position.line + deltaY < 0)
			deltaY = 0;
		_mouse->position.column += deltaX;
		_mouse->position.line += deltaY;

		// Update the mickey count.
		_horizontalMickeys %= _horizontalRatio;
		_verticalMickeys %= _verticalRatio;

		// Set the internal mouse information for use in the Poll routine.
		if (_endEvent == -1 || _mouseEvent[_endEvent].rawCode != _mouseState)
			_endEvent = ++_endEvent % MAX_EVENTS;
		_mouseEvent[_endEvent].type = E_MOUSE;
		_mouseEvent[_endEvent].rawCode = _mouseState;
		_mouseEvent[_endEvent].position = _mouse->position;

		// Update the mouse screen position.
//		if (_mouse->display && (deltaX || deltaY))
//			_mouse->display->DeviceMove(MOUSE_IMAGE,
//				_mouse->position.column - _mouse->offset.column,
//				_mouse->position.line - _mouse->offset.line);

		// Clean up stack and stack frame.
		__asm	mov	sp, bp
		__asm	pop	bp
		__asm	mov ss, _oldSS
		__asm	mov sp, _oldSP
		_stopInterrupt = FALSE;
	}
	else
		__asm	popf

	// Restore the world.
	__asm	mov sp,bp
	__asm	pop es
	__asm	pop ds
	__asm	pop di
	__asm	pop si
	__asm	pop bp
	__asm	pop bx
	__asm	pop bx
	__asm	pop dx
	__asm	pop cx
	__asm	pop ax
	__asm	retf
}

UID_MOUSE::UID_MOUSE(DEVICE_STATE _state, DEVICE_IMAGE _image) :
	UI_DEVICE(E_MOUSE, _state), image(_image)
{
	union _REGS inregs, outregs;
	struct _SREGS segregs;

/* START BLOCK COMMENT
**		unsigned char far *far *mouseVector = (unsigned char far *far *) (4 * 0x33);
**		if (*mouseVector && **mouseVector != 0xCF)
**		{
**			inregs.x.ax = 20;
**			inregs.x.bx = 0;
**			inregs.x.cx = 0;
**			inregs.x.dx = 0;
**			_int86x(0x33, &inregs, &outregs, &segregs);
**			_oldMouseMask = outregs.x.cx;
**			_oldMouseSegment = segregs.es;
**			_oldMouseOffset = outregs.x.dx;
**		}
END BLOCK COMMENT */

	// Set the new interrupt subroutine.
	inregs.x.ax = 0;
	_int86(0x33, &inregs, &outregs);
	if (outregs.x.ax != (USHORT)-1)
		return;

	void *pfunc = (void *)MouseISR;
	inregs.x.cx = 0xFF;
	inregs.x.dx = FP_OFF(pfunc);
	segregs.es = FP_SEG(pfunc);
	inregs.x.ax = 12;
	_int86x(0x33, &inregs, &outregs, &segregs);

	// Initialize the new mouse information.
	position.column = position.line = 0;
	offset.column = offset.line = 0;
	_mouse = this;
	installed = TRUE;
}

UID_MOUSE::~UID_MOUSE(void)
{
	union _REGS inregs, outregs;
	struct _SREGS segregs;

	// Restore the old mouse information.
	_mouse = NULL;
}

#endif
#if defined(__BCPLUSPLUS__) | defined(__TCPLUSPLUS__)
#pragma inline

static void MouseISR(void)
{
	// Get the proper value to ds.
	asm push ds
	asm mov ax, DGROUP
	asm mov ds, ax

	union REGS regs;
	
	// Get the mouse information.
	if (!_stopInterrupt && _mouse)
	{
		_stopInterrupt = TRUE;	// Prevent the interrupt from multiple calls.

		// Set up new mouse state. (Current state is in _BL)
		_mouseState = ((((_oldMouseState ^ _BL) << 4) | _BL) << 8) | (*((UCHAR far *) 0x417L) & 0xF);
		_oldMouseState = _BL;

		// Compute the mouse movement.
		regs.x.ax = 11;
		int86(0x33, &regs, &regs);
		_horizontalMickeys += regs.x.cx;
		_verticalMickeys += regs.x.dx;
		int deltaX = _horizontalMickeys / _horizontalRatio;
		int deltaY = _verticalMickeys / _verticalRatio;


		// Make sure the mouse is still on the screen.
		if (_mouse->position.column + deltaX >= _mouse->display->columns ||
			_mouse->position.column + deltaX < 0)
			deltaX = 0;
		if (_mouse->position.line + deltaY >= _mouse->display->lines ||
			_mouse->position.line + deltaY < 0)
			deltaY = 0;
		_mouse->position.column += deltaX;
		_mouse->position.line += deltaY;

		// Update the mickey count.
		_horizontalMickeys %= _horizontalRatio;
		_verticalMickeys %= _verticalRatio;

		// Set the internal mouse information for use in the Poll routine.
		if (_endEvent == -1 || _mouseEvent[_endEvent].rawCode != _mouseState)
			_endEvent = ++_endEvent % MAX_EVENTS;
		_mouseEvent[_endEvent].type = E_MOUSE;
		_mouseEvent[_endEvent].rawCode = _mouseState;
		_mouseEvent[_endEvent].position = _mouse->position;

		// Update the mouse screen position.
//		if (_mouse->display && (deltaX || deltaY))
//			_mouse->display->DeviceMove(MOUSE_IMAGE,
//				_mouse->position.column - _mouse->offset.column,
//				_mouse->position.line - _mouse->offset.line);
		_stopInterrupt = FALSE;
	}

	// Restore the ds register.
	asm pop ds
}

UID_MOUSE::UID_MOUSE(DEVICE_STATE _state, DEVICE_IMAGE _image) :
	UI_DEVICE(E_MOUSE, _state), image(_image)
{
	// Save the old mouse information.
	unsigned char far *far *mouseVector = (unsigned char far *far *) (4 * 0x33);
	if (*mouseVector && **mouseVector != 0xCF)
	{
		_AX = 20;
		_BX = 0;
		_DX = 0;
		_CX = 0;
		geninterrupt(0x33);
		_oldMouseMask = _CX;
		_BX = _ES;
		_oldMouseSegment = _BX;
		_oldMouseOffset = _DX;
	}

	// Set the new interrupt subroutine.
	_AX = 0;
	geninterrupt(0x33);
	if (_AX != (USHORT)-1)
		return;
	_CX = 0xFF;
	_DX = FP_OFF(MouseISR);
	_BX = FP_SEG(MouseISR);
	_ES = _BX;
	_AX = 12;
	geninterrupt(0x33);

	// Initialize the new mouse information.
	position.column = position.line = 0;
	offset.column = offset.line = 0;
	_mouse = this;
	installed = TRUE;
}

UID_MOUSE::~UID_MOUSE(void)
{
	// Restore the old mouse information.
	_mouse = NULL;
	if (installed)
	{
		_CX = _oldMouseMask;
		_DX = _oldMouseOffset;
		_BX = _oldMouseSegment;
		_ES = _BX;
		_AX = 12;
		geninterrupt(0x33);
	}
}

#endif

// Initialize the standard zinc cursors.
struct MOUSE_ITEM
{
	EVENT_TYPE image;
	UCHAR *bitmapImage;
} _mouseTable[] = {
	{ DM_VIEW, 			_view },
	{ DM_EDIT, 			_edit },
	{ DM_WAIT, 			_wait },
	{ DM_MOVE, 			_move },
	{ DM_HORIZONTAL, 	_horizontal },
	{ DM_VERTICAL, 		_vertical },
	{ DM_DIAGONAL_ULLR, _left },
	{ DM_DIAGONAL_LLUR, _right },
	{ DM_POSITION, 		_position },
	{ 0, 0 } };

EVENT_TYPE UID_MOUSE::Event(const UI_EVENT &event)
{
	// Make sure the mouse is installed.
	if (!installed)
		return (D_OFF);

	// Switch on the raw code.
	DEVICE_STATE oldState = state;
	DEVICE_IMAGE oldImage = image;
	UI_POSITION oldPosition = position;
	EVENT_TYPE ccode = event.type;
	switch (ccode)
	{
	case S_INITIALIZE:
		_horizontalRatio = display->isText ? (display->columns == 40 ? 16 : 8) : 1;
		_verticalRatio = display->isText ? 8 : 1;
		position.column = display->columns / 2;
		position.line = display->lines / 2;
		oldImage = 0;
		oldState = D_OFF;
		break;

	case S_POSITION:
		position = event.position;
		break;

	case DM_VIEW:
	case DM_EDIT:
	case DM_WAIT:
	case DM_MOVE:
	case DM_HORIZONTAL:
	case DM_VERTICAL:
	case DM_DIAGONAL_ULLR:
	case DM_DIAGONAL_LLUR:
	case DM_POSITION:
		image = ccode;
		state = D_ON;
		break;

	case D_OFF:
	case D_ON:
		state = event.type;
		break;
	}

	// Reset the mouse information.
	if (oldImage == image && oldState == state && oldPosition == position)
		return (ccode);
	if (oldPosition != position)
		display->DeviceMove(MOUSE_IMAGE, position.column - offset.column,
			position.line - offset.line);
	if (oldImage != image)
	{
		for (int i = 0; _mouseTable[i].image; i++)
			if (_mouseTable[i].image == image && display->isText)
			{
				display->DeviceSet(MOUSE_IMAGE, position.column, position.line, 1, 1, NULL);
				break;
			}
			else if (_mouseTable[i].image == image)
			{
				UCHAR *bitmapImage = _mouseTable[i].bitmapImage;
				image = _mouseTable[i].image;
				offset.column = bitmapImage[0];
				offset.line = bitmapImage[1];
				display->DeviceSet(MOUSE_IMAGE, position.column - offset.column,
					position.line - offset.line, bitmapImage[2], bitmapImage[3],
					&bitmapImage[4]);
				break;
			}
	}

	// Return the control code.
	return (ccode);
}

void UID_MOUSE::Poll(void)
{
	// Make sure the mouse is valid.
	_stopInterrupt = TRUE;
	if (installed && state != D_OFF && _endEvent >= 0)
	{
		_mouse->display->DeviceMove(MOUSE_IMAGE,
			_mouse->position.column - _mouse->offset.column,
			_mouse->position.line - _mouse->offset.line);
		for (int i = 0; i <= _endEvent; i++)
			eventManager->Put(_mouseEvent[i]);
	}
	_endEvent = -1;
	_stopInterrupt = FALSE;
}
#endif
