//	Zinc Interface Library - SCROLL.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include "ui_win.hpp"
#pragma hdrstop

// Internal class definition for the scroll buttons.
typedef UIF_FLAGS BUTTON_TYPE;
const BUTTON_TYPE VTOP_BUTTON			= 0x0001;
const BUTTON_TYPE VMIDDLE_BUTTON		= 0x0002;
const BUTTON_TYPE VBOTTOM_BUTTON		= 0x0004;
const BUTTON_TYPE HLEFT_BUTTON			= 0x0010;
const BUTTON_TYPE HMIDDLE_BUTTON		= 0x0020;
const BUTTON_TYPE HRIGHT_BUTTON			= 0x0040;

const BUTTON_TYPE VBUTTON				= 0x000F;
const BUTTON_TYPE HBUTTON				= 0x00F0;
const BUTTON_TYPE LTBUTTON				= 0x0011;
const BUTTON_TYPE MBUTTON				= 0x0022;
const BUTTON_TYPE RBBUTTON				= 0x0044;

class UIW_SCROLL_BUTTON : public UIW_BUTTON
{
	friend class EXPORT UIW_SCROLL_BAR;
public:
	BUTTON_TYPE btType;

	UIW_SCROLL_BUTTON(BUTTON_TYPE btType);
	~UIW_SCROLL_BUTTON(void);
	virtual EVENT_TYPE Event(const UI_EVENT &event);
};

// ----- UIW_WINDOW ---------------------------------------------------------

UIW_SCROLL_BAR::UIW_SCROLL_BAR(int left, int top, int width, int height,
	SBF_FLAGS _sbFlags, WOF_FLAGS _woFlags) :
	UIW_WINDOW(left, top, width, height, _woFlags), sbFlags(_sbFlags)
{
	// Initialize the slide information.
	UIW_SCROLL_BAR::Information(INITIALIZE_CLASS, NULL);
#ifdef _WINDOWS
	if (FlagSet(sbFlags, SBF_VERTICAL) && !width)
		dwStyle |= SBS_LEFTALIGN;
	else if (FlagSet(sbFlags, SBF_HORIZONTAL) && !height)
		dwStyle |= SBS_TOPALIGN;
	else if (FlagSet(sbFlags, SBF_CORNER) && !width && !height)
		dwStyle |= SBS_SIZEBOXTOPLEFTALIGN;
#endif
}

UIW_SCROLL_BAR::~UIW_SCROLL_BAR(void)
{
}

#ifdef _WINDOWS
static int _scrollOffset = -1;
static FARPROC _scrollCallback = (FARPROC)DefWindowProc;
static FARPROC _scrollJumpInstance = NULL;

long FAR PASCAL _export ScrollJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _scrollOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_SCROLL_BAR::Event(const UI_EVENT &event)
{
	// Ignore the corner scroll bar if windows is running.
	if (FlagSet(sbFlags, SBF_CORNER) && FlagsSet(woFlags, WOF_SUPPORT_OBJECT))
		return (S_ERROR);

	// Switch on the event type.
	EVENT_TYPE ccode = event.type;
	WORD message = event.message.message;
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_scrollJumpInstance)
			_scrollJumpInstance = (FARPROC)ScrollJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
		if (FlagsSet(woFlags, WOF_SUPPORT_OBJECT))
		{
			if (FlagSet(sbFlags, SBF_VERTICAL))
				dwStyle = WS_VSCROLL;
			else if (FlagSet(sbFlags, SBF_HORIZONTAL))
				dwStyle = WS_HSCROLL;
			parent->Information(SET_DWSTYLE, &dwStyle);
		}
		else
		{
			if (FlagSet(sbFlags, SBF_VERTICAL))
				dwStyle |= SBS_VERT;
			else if (FlagSet(sbFlags, SBF_HORIZONTAL))
				dwStyle |= SBS_HORZ;
			else
				dwStyle |= SBS_SIZEBOX;
		}
		break;

	case S_CREATE:
		if (!FlagSet(woFlags, WOF_SUPPORT_OBJECT))
		{
			if (FlagSet(sbFlags, SBF_CORNER | SBF_VERTICAL))
				relative.left = relative.right - display->cellHeight * 3 / 4;
			if (FlagSet(sbFlags, SBF_CORNER | SBF_HORIZONTAL))
				relative.top = relative.bottom - display->cellHeight * 3 / 4;
			UI_WINDOW_OBJECT::Event(event);
			RegisterObject("UIW_SCROLL_BAR", "SCROLLBAR", &_scrollOffset,
				&_scrollJumpInstance, &_scrollCallback, NULL);
		}
		break;

	case S_VSCROLL_SET:
	case S_HSCROLL_SET:
		{
		if ((ccode == S_HSCROLL_SET && !FlagSet(sbFlags, SBF_HORIZONTAL)) ||
			(ccode == S_VSCROLL_SET && !FlagSet(sbFlags, SBF_VERTICAL)))
			return (S_ERROR);
		scroll = event.scroll;
		int nbar = FlagSet(sbFlags, SBF_VERTICAL) ? SB_VERT : SB_HORZ;
		if (scroll.maximum > 0)
		{
			SetScrollRange(parent->screenID, nbar, scroll.minimum, scroll.maximum, FALSE);
			SetScrollPos(parent->screenID, nbar, scroll.current, FALSE);
		}
		else
			SetScrollRange(parent->screenID, nbar, 0, 0, FALSE);
		}
		break;

	default:
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message != WM_HSCROLL && message != WM_VSCROLL)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else
			ccode = S_ERROR;
		break;
	}

	if (ccode != S_ERROR || event.type != E_MSWINDOWS ||
		(FlagSet(sbFlags, SBF_VERTICAL) && message != WM_VSCROLL) ||
		(FlagSet(sbFlags, SBF_HORIZONTAL) && message != WM_HSCROLL))
		return (ccode);

	int tCurrent = scroll.current;
	int nbar = FlagSet(sbFlags, SBF_VERTICAL) ? SB_VERT : SB_HORZ;
	ccode = event.message.message;
	switch (event.message.wParam)
	{
	case SB_BOTTOM:
		scroll.current = scroll.maximum;
		break;

	case SB_TOP:
		scroll.current = scroll.minimum;
		break;

	case SB_LINEDOWN:
		scroll.current++;
		break;

	case SB_LINEUP:
		scroll.current--;
		break;

	case SB_THUMBPOSITION:
	case SB_THUMBTRACK:
		scroll.current = LOWORD(event.message.lParam);
		break;
	}
	if (scroll.current < scroll.minimum)
		scroll.current = scroll.minimum;
	else if (scroll.current > scroll.maximum)
		scroll.current = scroll.maximum;

	if (tCurrent != scroll.current)
	{
		SetScrollPos(parent->screenID, nbar, scroll.current, TRUE);
		if (!FlagSet(woFlags, WOF_NON_FIELD_REGION))
			;
		else if (FlagSet(sbFlags, SBF_VERTICAL))
			ScrollWindow(parent->screenID, 0, tCurrent - scroll.current, NULL, NULL);
		else
			ScrollWindow(parent->screenID, tCurrent - scroll.current, 0, NULL, NULL);
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_SCROLL_BAR::Event(const UI_EVENT &event)
{
	UI_EVENT sEvent;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_SCROLL_BAR);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!FlagSet(woFlags, WOF_NON_FIELD_REGION))
			;
		else if (FlagSet(sbFlags, SBF_VERTICAL) && parent)
			parent->Information(SET_VSCROLL, this);
		else if (FlagSet(sbFlags, SBF_HORIZONTAL) && parent)
			parent->Information(SET_HSCROLL, this);
		ccode = UIW_WINDOW::Event(event);
		break;

	case S_SIZE:
	case S_CREATE:
		{
		// Compute the object region.
		clipList.Destroy();
		UI_WINDOW_OBJECT::Event(event);
		if (display->isText)
		{
			if (FlagSet(sbFlags, SBF_CORNER | SBF_VERTICAL))
				true.left = true.right;
			if (FlagSet(sbFlags, SBF_CORNER | SBF_HORIZONTAL))
				true.top = true.bottom;
		}
		else
		{
			if (FlagSet(woFlags, WOF_BORDER))
			{
				true.right++;
				true.bottom++;
			}
			if (FlagSet(sbFlags, SBF_CORNER | SBF_VERTICAL))
				true.left = true.right - display->cellHeight * 3 / 4;
			else if (FlagSet(woFlags, WOF_BORDER))
				true.left--;
			if (FlagSet(sbFlags, SBF_CORNER | SBF_HORIZONTAL))
				true.top = true.bottom - display->cellHeight * 3 / 4;
			else if (FlagSet(woFlags, WOF_BORDER))
				true.top--;
		}
		// Compute the scroll button regions.
		clipList.Destroy();
		UI_REGION region = true;
		if (FlagSet(woFlags, WOF_BORDER) && !display->isText)
			--region;
		clipList.Add(new UI_REGION_ELEMENT(screenID, region));
		for (UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, FALSE);
		}
		for (object = First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, FALSE);
		}
		}
		break;

	case S_VSCROLL_SET:
	case S_HSCROLL_SET:
		scroll = event.scroll;
		// Continue to S_VSCROLL.
	case S_VSCROLL:
	case S_HSCROLL:
		{
		int tScroll = (ccode == S_VSCROLL || ccode == S_HSCROLL) ?
			scroll.current + event.scroll.delta : scroll.current;
		if (tScroll < scroll.minimum)
			tScroll = scroll.minimum;
		else if (tScroll > scroll.maximum)
			tScroll = scroll.maximum;
		if (scroll.maximum - scroll.minimum == 0)
			break;
		else if (ccode == S_VSCROLL_SET || ccode == S_HSCROLL_SET ||
			(event.scroll.delta != 0 && tScroll != scroll.current))
		{
			scroll.current = tScroll;
			int size = display->cellHeight * 3 / 4;
			UI_REGION region = clipList.First()->region;
			UI_REGION update = mButton->true;
			if (FlagSet(sbFlags, SBF_VERTICAL) && display->isText)
			{
				long value = region.bottom - region.top;	// Use long for precision.
				value *= scroll.current;
				value /= (scroll.maximum - scroll.minimum);
				mButton->true.top = mButton->true.bottom = region.top + (int)value;
			}
			else if (FlagSet(sbFlags, SBF_HORIZONTAL) && display->isText)
			{
				long value = region.right - region.left;	// Use long for precision.
				value *= scroll.current;
				value /= (scroll.maximum - scroll.minimum);
				mButton->true.left = mButton->true.right = region.left + (int)value;
			}
			else if (FlagSet(sbFlags, SBF_VERTICAL))
			{
				long value = region.bottom - region.top - size + 2;	// Use long for precision.
				value *= scroll.current;
				value /= (scroll.maximum - scroll.minimum);
				mButton->true.top = region.top - 1 + (int)value;
				mButton->true.bottom = mButton->true.top + size;
			}
			else
			{
				long value = region.right - region.left - size + 2;	// Use long for precision.
				value *= scroll.current;
				value /= (scroll.maximum - scroll.minimum);
				mButton->true.left = region.left - 1 + (int)(value);
				mButton->true.right = mButton->true.left + size;
			}
			if (screenID && (mButton->true.top != update.top ||
				mButton->true.left != update.left))
			{
				update.left = Min(update.left, mButton->true.left);
				update.top = Min(update.top, mButton->true.top);
				update.right = Max(update.right, mButton->true.right);
				update.bottom = Max(update.bottom, mButton->true.bottom);
				Event(UI_EVENT(S_DISPLAY_ACTIVE, 0, update));
			}
		}
		}
		break;

	case L_BEGIN_SELECT:
		// Check for the edit mode.
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));

		// Selection on the middle button.
		else if (mButton->true.Overlap(event.position))
		{
			UI_REGION newRegion = mButton->true;
			UI_POSITION position = event.position;
			UI_REGION updateRegion;
			if (clipList.First())	// Un-optimized for Zortech bug.
				updateRegion = clipList.First()->region;
			else
				updateRegion = true;
			if (!display->isText)
				++updateRegion;
			display->Rectangle(ID_DIRECT, newRegion, display->xorPalette, 1, FALSE, TRUE);
			while (ccode != L_END_SELECT)
			{
				eventManager->Get(sEvent);
				ccode = LogicalEvent(sEvent, ID_SCROLL_BAR);
				int deltaX = sEvent.position.column - position.column;
				int deltaY = sEvent.position.line - position.line;
				if (FlagSet(sbFlags, SBF_VERTICAL))
					deltaX = 0;
				else if (newRegion.left + deltaX < updateRegion.left)
					deltaX = updateRegion.left - newRegion.left;
				else if (newRegion.right + deltaX > updateRegion.right)
					deltaX = updateRegion.right - newRegion.right;
				if (FlagSet(sbFlags, SBF_HORIZONTAL))
					deltaY = 0;
				else if (newRegion.top + deltaY < updateRegion.top)
					deltaY = updateRegion.top - newRegion.top;
				else if (newRegion.bottom + deltaY > updateRegion.bottom)
					deltaY = updateRegion.bottom - newRegion.bottom;
				if (deltaX || deltaY)
				{
					UI_REGION oldRegion = newRegion;
					newRegion.left += deltaX;
					newRegion.top += deltaY;
					newRegion.right += deltaX;
					newRegion.bottom += deltaY;
					position.column += deltaX;
					position.line += deltaY;
					display->RectangleXORDiff(oldRegion, newRegion);
				}
			}
			display->Rectangle(ID_DIRECT, newRegion, display->xorPalette, 1, FALSE, TRUE);
			if (!true.Overlap(sEvent.position))
				break;
			else if (FlagSet(sbFlags, SBF_VERTICAL))
			{
				long val1 = newRegion.top - updateRegion.top;
				long val2 = scroll.maximum - scroll.minimum;
				long val3 = updateRegion.bottom - updateRegion.top - display->cellHeight * 3 / 4;
				sEvent.type = S_VSCROLL;
				sEvent.scroll.delta = val3 ? (int)(val1 * val2 / val3) - scroll.current : 0;
			}
			else if (FlagSet(sbFlags, SBF_HORIZONTAL))
			{
				long val1 = newRegion.left - updateRegion.left;
				long val2 = scroll.maximum - scroll.minimum;
				long val3 = updateRegion.right - updateRegion.left - display->cellHeight * 3 / 4;
				sEvent.type = S_HSCROLL;
				sEvent.scroll.delta = val3 ? (int)(val1 * val2 / val3) - scroll.current : 0;
			}
			// Reset the scroll region.
			if (sEvent.scroll.delta && FlagSet(woFlags, WOF_NON_FIELD_REGION))
				parent->Event(sEvent);
			else if (sEvent.scroll.delta)
				Event(sEvent);
			break;	// Do not continue to default operation of other buttons.
		}

		// Selection on the left/top button.
		else if (lButton->true.Overlap(event.position))
		{
			current = lButton;
			sEvent.type = FlagSet(sbFlags, SBF_VERTICAL) ? S_VSCROLL : S_HSCROLL;
			sEvent.scroll.delta = -scroll.delta;
			lButton->Event(event);
			sEvent.scroll.delta = -scroll.delta;
		}

		// Selection on the left/top button.
		else if (rButton->true.Overlap(event.position))
		{
			current = rButton;
			sEvent.type = FlagSet(sbFlags, SBF_VERTICAL) ? S_VSCROLL : S_HSCROLL;
			sEvent.scroll.delta = scroll.delta;
			rButton->Event(event);
		}

		// Selection of vertical page (previous/next).
		else if (FlagSet(sbFlags, SBF_VERTICAL))
		{
			current = NULL;
			sEvent.type = S_VSCROLL;
			sEvent.scroll.delta = (event.position.line < mButton->true.top) ?
				-scroll.showing : scroll.showing;
		}

		// Selection of horizontal page (previous/next).
		else if (FlagSet(sbFlags, SBF_HORIZONTAL))
		{
			current = NULL;
			sEvent.type = S_HSCROLL;
			sEvent.scroll.delta = (event.position.column < mButton->true.left) ?
				-scroll.showing : scroll.showing;
		}

		// Unknown option.
		else
			break;

		// Continue the process until L_END_SELECT is detected.
		{
		if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
			parent->Event(sEvent);
		else
			Event(sEvent);
		UI_EVENT tEvent;
		tEvent = event;
		UI_TIME currentTime, lastTime;
		while (ccode != L_END_SELECT)
		{
			currentTime.Import();
			if (eventManager->Get(tEvent, Q_NO_BLOCK) != -2)
			{
				ccode = LogicalEvent(tEvent, ID_SCROLL_BAR);
				if (current)
					Current()->Event(tEvent);
			}
			else if (current && Current()->true.Overlap(tEvent.position) &&
				currentTime - lastTime > repeatRate)
			{
				if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
					parent->Event(sEvent);
				else
					Event(sEvent);
				lastTime = currentTime;
			}
		}
		current = mButton;
		}
		break;

	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		break;

	default:
		ccode = UIW_WINDOW::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_SCROLL_BAR::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_SCROLL_BAR;
	switch (request)
	{
	case INITIALIZE_CLASS:
		if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
			woFlags |= WOF_SUPPORT_OBJECT;
		if (FlagSet(woFlags, WOF_SUPPORT_OBJECT))
			woAdvancedFlags |= WOAF_NON_CURRENT;
		if (FlagSet(sbFlags, SBF_CORNER | SBF_HORIZONTAL))
			woFlags |= WOF_JUSTIFY_RIGHT;
		searchID = windowID[0] = ID_SCROLL_BAR;
		windowID[1] = ID_WINDOW;
		scroll.minimum = scroll.current = 0;
		scroll.maximum = 1;
#ifndef _WINDOWS
		// Add the scroll buttons.
		if (FlagSet(sbFlags, SBF_VERTICAL))
		{
			Add((lButton = new UIW_SCROLL_BUTTON(VTOP_BUTTON)));
			Add((mButton = new UIW_SCROLL_BUTTON(VMIDDLE_BUTTON)));
			Add((rButton = new UIW_SCROLL_BUTTON(VBOTTOM_BUTTON)));
		}
		else if (FlagSet(sbFlags, SBF_HORIZONTAL))
		{
			Add((lButton = new UIW_SCROLL_BUTTON(HLEFT_BUTTON)));
			Add((mButton = new UIW_SCROLL_BUTTON(HMIDDLE_BUTTON)));
			Add((rButton = new UIW_SCROLL_BUTTON(HRIGHT_BUTTON)));
		}
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (!parent)
			break;
		else if (FlagsSet(woFlags, WOF_SUPPORT_OBJECT))
			parent->Information(SET_DWSTYLE, &dwStyle);
		else if (request == CHANGED_FLAGS)
		{
			if (request == CHANGED_FLAGS)
				UIW_WINDOW::Information(CHANGED_FLAGS, data, ID_SCROLL_BAR);
			if (screenID && request == CHANGED_FLAGS && objectID == ID_SCROLL_BAR)
			{
				DestroyWindow(screenID);
				if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
					RegisterObject("UIW_SCROLL_BAR", "SCROLLBAR", &_scrollOffset,
						&_scrollJumpInstance, &_scrollCallback, NULL);
				else
					Event(UI_EVENT(S_CREATE));
				SetWindowPos(screenID, previous ? Previous()->screenID : 0,
					0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
			}
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_SCROLL_BAR)
		{
			if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
				return (parent->Information(request, data));
			UI_EVENT event(S_INITIALIZE);
			Event(event);
			event.type = S_CREATE;
			Event(event);
		}
#endif
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_SCROLL_BAR)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &sbFlags;
		else if (request == GET_FLAGS)
			*(SBF_FLAGS *)data = sbFlags;
		else if (request == SET_FLAGS)
			sbFlags |= *(SBF_FLAGS *)data;
		else
			sbFlags &= ~(*(SBF_FLAGS *)data);
		break;

	case PRINT_INFORMATION:
	case PRINT_USER_FUNCTION:
		UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;

	default:
		data = UIW_WINDOW::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

#ifndef _WINDOWS
UIW_SCROLL_BUTTON::UIW_SCROLL_BUTTON(BUTTON_TYPE _btType) :
	UIW_BUTTON(0, 0, 0, NULL, BTF_NO_FLAGS, WOF_BORDER | WOF_JUSTIFY_CENTER | WOF_NON_FIELD_REGION),
	btType(_btType)
{
	depth = 1;
	if (FlagSet(btType, LTBUTTON | RBBUTTON))
		woFlags |= WOF_SUPPORT_OBJECT;

	static char *textUpArrow = "";
	static char *textDownArrow = "";
	static char *textLeftArrow = "";
	static char *textRightArrow = "";
	static char *textMiddleArrow = "";
	if (display->isText)
	{
		if (btType == VTOP_BUTTON)
			text = textUpArrow;
		else if (btType == VBOTTOM_BUTTON)
			text = textDownArrow;
		else if (btType == HLEFT_BUTTON)
			text = textLeftArrow;
		else if (btType == HRIGHT_BUTTON)
			text = textRightArrow;
		else
		{
			text = textMiddleArrow;
			woFlags &= ~WOF_NON_FIELD_REGION;
		}
		return;
	}

	static UCHAR upArrow[49] =
	{
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BACKGROUND,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND
	};
	static UCHAR downArrow[49] =
	{
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BACKGROUND,	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND
	};
	static UCHAR leftArrow[49] =
	{
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
		BACKGROUND,	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BACKGROUND,	BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND
	};
	static UCHAR rightArrow[49] =
	{
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BACKGROUND,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,
		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BLACK,		BACKGROUND,
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BLACK,		BACKGROUND,	BACKGROUND,
		BACKGROUND,	BACKGROUND,	BACKGROUND,	BLACK,		BACKGROUND,	BACKGROUND,	BACKGROUND
	};
	if (btType == VTOP_BUTTON)
		bitmapArray = upArrow;
	else if (btType == VBOTTOM_BUTTON)
		bitmapArray = downArrow;
	else if (btType == HLEFT_BUTTON)
		bitmapArray = leftArrow;
	else if (btType == HRIGHT_BUTTON)
		bitmapArray = rightArrow;
	else
		woFlags &= ~WOF_NON_FIELD_REGION;
	if (bitmapArray)
		bitmapWidth = bitmapHeight = 7;
}

UIW_SCROLL_BUTTON::~UIW_SCROLL_BUTTON(void)
{
	text = NULL;
	bitmapArray = NULL;
}

EVENT_TYPE UIW_SCROLL_BUTTON::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_SCROLL_BAR);
	switch (ccode)
	{
	case S_SIZE:
	case S_CREATE:
		{
		UI_WINDOW_OBJECT::Event(event);
		if (!display->isText && FlagSet(parent->woFlags, WOF_BORDER))
			++true;
		if (btType == VTOP_BUTTON)
			true.bottom = true.top + (true.right - true.left);
		else if (btType == VBOTTOM_BUTTON)
			true.top = true.bottom - (true.right - true.left);
		else if (btType == HLEFT_BUTTON)
			true.right = true.left + (true.bottom - true.top);
		else if (btType == HRIGHT_BUTTON)
			true.left = true.right - (true.bottom - true.top);
		else if (FlagSet(btType, MBUTTON))
		{
			if (display->isText)
			{
				true.left = true.right = clip.left;
				true.top = true.bottom = clip.top;
			}
			else
			{
				true.left = clip.left;
				true.top = clip.top;
				if (FlagSet(parent->woFlags, WOF_BORDER))
				{
					true.left--;
					true.top--;
				}
				true.right = true.left + display->cellHeight * 3 / 4;
				true.bottom = true.top + display->cellHeight * 3 / 4;
			}
		}
		}
		break;

	default:
		ccode = UIW_BUTTON::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_SCROLL_BAR::UIW_SCROLL_BAR(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UIW_WINDOW(0, 0, 2, 6, WOF_NO_FLAGS)
{
	// Initialize the slide information.
	Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	UIW_SCROLL_BAR::Information(INITIALIZE_CLASS, NULL);
}

void UIW_SCROLL_BAR::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the scroll-bar information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	file->Load(&sbFlags);
	short _value; file->Load(&_value); scroll.minimum = _value;
	file->Load(&_value); scroll.maximum = _value;
	file->Load(&_value); scroll.current = _value;
}

void UIW_SCROLL_BAR::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the scroll-bar information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	file->Store(sbFlags);
	short _value = scroll.minimum; file->Store(_value);
	_value = scroll.maximum; file->Store(_value);
	_value = scroll.current; file->Store(_value);
}
#endif
