//	Zinc Interface Library - TEXT.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <string.h>
#include "ui_win.hpp"
#pragma hdrstop
const int STRING_OFFSET = 5;

// ----- UIW_TEXT -----------------------------------------------------------

UIW_TEXT::UIW_TEXT(int left, int top, int width, int height, char *_text,
	int _maxLength, WNF_FLAGS _wnFlags, WOF_FLAGS _woFlags,
	USER_FUNCTION _userFunction) :
#ifdef _WINDOWS
	UIW_WINDOW(left, top, width, height, _woFlags, WOAF_NO_FLAGS),
	maxLength(_maxLength)
#else
	UIW_WINDOW(left, top, width, height, _woFlags, WOAF_NO_FLAGS),
	iString(0, 0, 0, NULL, 0),
	maxLength(_maxLength)
#endif
{
	// Initialize the text information.
	wnFlags |= _wnFlags;
	if (maxLength == -1)
		maxLength = _text ? ui_strlen(_text) + 1 : width * height;
	if (!FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = new char[maxLength+1];
	userFunction = _userFunction;
	UIW_TEXT::Information(INITIALIZE_CLASS, NULL);
	UIW_TEXT::DataSet(_text);
}

UIW_TEXT::~UIW_TEXT(void)
{
#ifndef _WINDOWS
	iString.text = NULL;
#endif
	if (text && !FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		delete text;
}

char *UIW_TEXT::DataGet(void)
{
#ifdef _WINDOWS
	if (screenID && !FlagSet(woFlags, WOF_VIEW_ONLY))
		SendMessage(screenID, WM_GETTEXT, maxLength, (LONG)text);
#endif
	return (text);
}

void UIW_TEXT::DataSet(char *_text, int _maxLength)
{
	// Reset the length.
	if (FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		maxLength = (_maxLength != -1) ? _maxLength : maxLength;
	else if (_maxLength != -1)
	{
		if (_maxLength > maxLength)
		{
			char *tText = new char[_maxLength+1];
			strncpy(tText, text, maxLength);
			tText[maxLength] = '\0';
			if (_text == text)
				_text = tText;
			if (text)
				delete text;
			text = tText;
		}
		maxLength = _maxLength;
	}

	// Reset the text.
	if (text == _text || FlagSet(woFlags, WOF_NO_ALLOCATE_DATA))
		text = _text;
	else
	{
		strncpy(text, _text ? _text : "", maxLength);
		text[maxLength] = '\0';
	}
#ifndef _WINDOWS
	iString.maxLength = maxLength;
	topLine = iString.text = text;
	iString.length = ui_strlen(text);
	iString.cursor = 0;
	iString.beginMark = iString.endMark = -1;
#endif

	// Redisplay the text field.
	if (screenID)
	{
#ifdef _WINDOWS
		InvalidateRect(screenID, NULL, TRUE);
		if (_maxLength != -1)
			SendMessage(screenID, EM_LIMITTEXT, maxLength - 1, FALSE);
		SendMessage(screenID, WM_SETTEXT, 0, (LONG)text);
#else
		RecomputeFields(FALSE, TRUE);
		Event(UI_EVENT(S_REDISPLAY));
#endif
	}
}

#ifdef _WINDOWS
static int _textOffset = -1;
static FARPROC _textCallback = (FARPROC)DefWindowProc;
static FARPROC _textJumpInstance = NULL;

long FAR PASCAL _export TextJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _textOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_TEXT::DrawItem(const UI_EVENT &event, EVENT_TYPE ccode)
{
	// Initialize the variables.
	DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
	UI_REGION region = { draw->rcItem.left, draw->rcItem.top,
		draw->rcItem.right, draw->rcItem.bottom };
	region.left += relative.left;
	UI_MSWINDOWS_DISPLAY::hDC = draw->hDC;

	// Check the string status.
	if (FlagSet(draw->itemState, ODS_SELECTED))
		woStatus |= WOS_SELECTED;
	else
		woStatus &= ~WOS_SELECTED;
	if (FlagSet(draw->itemAction, ODA_FOCUS) && FlagSet(draw->itemState, ODS_FOCUS))
		woStatus |= WOS_CURRENT;
	else if (FlagSet(draw->itemAction, ODA_FOCUS))
		woStatus &= ~WOS_CURRENT;

	// Draw the text information.
	if (FlagSet(draw->itemAction, ODA_DRAWENTIRE | ODA_SELECT))
	{
		lastPalette = UI_PALETTE_MAP::MapPalette(paletteMapTable, FlagSet(woStatus, WOS_SELECTED) ? PM_SELECTED : PM_ACTIVE, ID_TEXT);
		HBRUSH fillBrush = CreateSolidBrush(display->MapColor(lastPalette, FALSE));
		FillRect(draw->hDC, &draw->rcItem, fillBrush);
		DeleteObject(fillBrush);
		DrawText(ID_DIRECT, region, text, lastPalette, FALSE, ccode);
	}
	if (FlagSet(draw->itemAction, ODA_FOCUS))
		DrawFocusRect(draw->hDC, &draw->rcItem);

	return (TRUE);
}

EVENT_TYPE UIW_TEXT::Event(const UI_EVENT &event)
{
	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_TEXT);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_textJumpInstance)
			_textJumpInstance = (FARPROC)TextJumpProcedure;
		UIW_WINDOW::Event(event);
		break;

	case S_CREATE:
		UI_WINDOW_OBJECT::Event(event);
		RegisterObject("UIW_TEXT", "EDIT", &_textOffset, &_textJumpInstance,
			&_textCallback, text);
		SendMessage(screenID, EM_LIMITTEXT, maxLength - 1, FALSE);
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	case L_SELECT:
		ccode = CallWindowProc((FARPROC)defaultCallback, screenID,
			event.message.message, event.message.wParam, event.message.lParam);
		break;

	case L_BACKSPACE:
	case L_DELETE:
	case L_DELETE_WORD:
	case L_DELETE_EOL:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		// Continue to default.
	default:
		WORD message = event.message.message;
		if (event.type != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else if (message == WM_DRAWITEM)
			ccode = DrawItem(event, ccode);
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) || message != WM_CHAR)
			ccode = UI_WINDOW_OBJECT::Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
extern char _dialogSize[];

static int NextLine(char **tText, int totalWidth)
{
	// Compute the line width.
	int width = 1;
	int length = 0;
	char *text = *tText;
	while (*text && *text != '\r' && *text != '\n' && width < totalWidth)
	{
		width += _dialogSize[*text];
		length++;
		text++;
	}
	if (width >= totalWidth)
	{
		int tLength = length - 1;
		char *text1 = *tText;
		char *text2 = text;
		for (text2--; text2 > text1 && *text2 != ' '; text2--)
			tLength--;
		if (*text2 == ' ')
		{
			length = tLength + 1;
			text = text2 + 1;
		}
	}

	// Move past the newline characters.
	int rCount = 0, nCount = 0;
	while (*text)
		if ((*text == '\r' && ++rCount == 1) || (*text == '\n' && ++nCount == 1))
			text++;
		else
			break;
	*tText = text;
	return (length);
}

static int PreviousLine(char *startText, char **text, int totalWidth)
{
	// Make sure we can find a previous line.
	if (*text <= startText)
		return (0);

	// Find the hard-return previous line.
	char *currentLine = *text - 1;
	int rCount = 0, nCount = 0;
	while (currentLine > startText)
		if ((*currentLine == '\r' && ++rCount == 1) ||
			(*currentLine == '\n' && ++nCount == 1))
			currentLine--;
		else
			break;
	char *previousLine = currentLine;
	while (currentLine > startText && *previousLine != '\r' &&
		*previousLine != '\n')
	{
		currentLine = previousLine;
		previousLine--;
	}

	// Walk through the string to find the last line.
	int length = totalWidth;
	char *nextLine = currentLine;
	while (nextLine < *text)
	{
		currentLine = nextLine;
		length = NextLine(&nextLine, totalWidth);
	}

	*text = currentLine;
	return (length);
}

void UIW_TEXT::RecomputeFields(int updateFields, int updateStrings)
{
	UI_EVENT event;

	// Optimize the screen output.
	if (updateFields && display->isText)
		display->VirtualGet(screenID, true);

	// Check the cursor position within the buffer.
	if (iString.cursor < (int)(topLine - text))
	{
		topLine = text + iString.cursor;
		PreviousLine(text, &topLine, iString.lineWidth);
	}
	else if (iString.cursor > (int)(Last()->text + Last()->length))
	{
		topLine = text + iString.cursor;
		int count = UI_LIST::Count();
		while (--count > 0 && topLine > text)
			PreviousLine(text, &topLine, iString.lineWidth);
	}

	// Recompute the string field text information.
	current = NULL;
	int nonSelectable = FALSE;
	char *bottomLine = topLine;
	for (UIW_STRING *string = First(); string; string = string->NextString())
	{
		// Compute the new string information.
		char *_text = string->text;
		int _length = string->length;
		int _leftClip = string->leftClip;
		int _beginMark = string->beginMark;
		int _endMark = string->endMark;
		WOF_FLAGS _woStatus = string->woStatus;

		if (updateStrings)
		{
			string->text = bottomLine;
			if (nonSelectable)
				string->woFlags |= WOF_NON_SELECTABLE;
			else
				string->woFlags &= ~WOF_NON_SELECTABLE;
			string->length = string->maxLength = NextLine(&bottomLine, iString.lineWidth);
			nonSelectable = string->text[string->length] ? FALSE : TRUE;
		}

		// Check the string status and cursor position.
		int offset = (int)(string->text - text);
		if (!FlagSet(string->woFlags, WOF_NON_SELECTABLE) &&
			((iString.cursor > offset && iString.cursor < offset + string->length) ||
			 (iString.cursor == offset && !current) ||
			 (iString.cursor == offset + string->length && string->cursor == string->length) ||
			 (!current && (current == last || !string->text[string->length]))))
		{
			current = string;
			string->woStatus |= WOS_CURRENT;
			string->woFlags &= ~WOF_NON_SELECTABLE;
			string->cursor = iString.cursor - offset;
		}
		else
		{
			string->woStatus &= ~WOS_CURRENT;
			string->cursor = iString.leftClip;
		}
		string->leftClip = iString.leftClip;

		// Check the mark region.
		if (iString.beginMark != -1)
		{
			string->beginMark = Max(0, iString.beginMark - (int)(string->text - text));
			string->endMark = Min(string->length, iString.endMark - (int)(string->text - text));
			if (string->beginMark > string->endMark)
				string->beginMark = string->endMark = -1;
		}
		else
			string->beginMark = string->endMark = -1;

		// Check the update status for the string and the cursor.
		if (updateFields && (_text != string->text ||
			_length != string->length || _leftClip != string->leftClip ||
			_beginMark != string->beginMark || _endMark != string->endMark ||
			_woStatus != string->woStatus))
		{
			event.type = S_REDISPLAY;
			string->Event(event);
			// Give the event manager time to poll its devices.
			eventManager->Get(event, Q_NO_BLOCK | Q_NO_DESTROY);
		}
	}

	// Optimize the screen output.
	if (updateFields && display->isText)
		display->VirtualPut(screenID);
}

int UIW_TEXT::ResetMark(void)
{
	if (!FlagSet(iString.woStatus, WOS_AUTO_MARK))
		return (FALSE);
	iString.woStatus &= ~WOS_AUTO_MARK;
	for (UIW_STRING *string = First(); string; string = string->NextString())
	{
		string->beginMark = -1;	// This causes to automatically be reset.
		string->woStatus &= ~WOS_AUTO_MARK;
	}
	iString.beginMark = iString.endMark = -1;
	return (TRUE);
}

EVENT_TYPE UIW_TEXT::Event(const UI_EVENT &event)
{
	// Initialize the default function variables.
	int updateFields = FALSE, updateStrings = FALSE, updateCurrent = FALSE;
	int insertCount = 0, deleteCount = 0, scrollWidth = 0, oldCursor = iString.cursor;
	UIW_STRING *string = Current(), *oldCurrent = Current(), *newCurrent = NULL;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_TEXT);
	switch (ccode)
	{
	case S_CREATE:
	case S_SIZE:
		{
		// Compute the text region.
		UI_LIST::Destroy();
		clipList.Destroy();
		UIW_WINDOW::Event(event);
		if (clipList.First())		// Un-optimized for Zortech bug.
			iString.active = clipList.First()->region;
		else
		{
			iString.active = true;
			if (!display->isText && FlagSet(woFlags, WOF_BORDER))
				--iString.active;
		}
		iString.lineWidth = FlagSet(wnFlags, WNF_NO_WRAP) ?
			0x0FFF : iString.active.right - iString.active.left + 1;
		if (!display->isText)
			iString.lineWidth -= (STRING_OFFSET + STRING_OFFSET);
		int top = 0;
		int bottom = iString.active.bottom - iString.active.top + 1;
		int width = iString.active.right - iString.active.left + 1;
		int lineHeight = display->TextHeight(NULL, screenID, font) + display->preSpace + display->postSpace;
		// Create the text fields.
		while (top + lineHeight <= bottom)
		{
			WOF_FLAGS flags = WOF_NO_ALLOCATE_DATA;
			flags |= (woFlags & (WOF_VIEW_ONLY | WOF_NON_SELECTABLE));
			string = new UIW_STRING(0, top, width, text, 1024, STF_SUBSTRING, flags);
			top += lineHeight;
			UI_LIST::Add(string);
			string->parent = this;
			string->paletteMapTable = paletteMapTable;
			string->woStatus |= (woStatus & WOS_GRAPHICS);
			string->relative.bottom = top - 1;
			string->Event(event);
		}

		// Set up the text fields.
		updateStrings = TRUE;
		}
		break;

	case S_CURRENT:
		iString.insertMode = insertMode;
		for (string = First(); string; string = string->NextString())
			string->insertMode = insertMode;
		if (FlagSet(woFlags, WOF_AUTO_CLEAR) && !FlagSet(woFlags, WOF_VIEW_ONLY | WOF_NON_SELECTABLE))
		{
			iString.woStatus |= WOS_AUTO_MARK;
			for (string = First(); string; string = string->NextString())
				string->woStatus |= WOS_AUTO_MARK;
			iString.beginMark = 0;
			iString.endMark = iString.cursor = iString.length;
			RecomputeFields(TRUE, TRUE);
		}
		return (UIW_WINDOW::Event(event));

	case S_NON_CURRENT:
		updateFields = UIW_TEXT::ResetMark();
		if (updateFields)
			RecomputeFields(TRUE, FALSE);
		return (UIW_WINDOW::Event(event));

	case S_HSCROLL_CHECK:
	case S_VSCROLL_CHECK:
		break;

	case S_HSCROLL:
		if (string->leftClip + event.scroll.delta < 0 ||
			string->leftClip + event.scroll.delta > string->length)
			break;
		string->leftClip += event.scroll.delta;
		string->cursor += event.scroll.delta;
		if (string->cursor > string->length)
			string->cursor = string->length;
		else if (string->cursor < 0)
			string->cursor = 0;
		updateCurrent = TRUE;
		break;

	case S_VSCROLL:
		// Make sure we can scroll on the text field.
		if (!event.scroll.delta ||
			event.scroll.delta < 0 && oldCurrent->text == text ||
			event.scroll.delta > 0 && !oldCurrent->text[oldCurrent->length])
			break;

		// Scroll up.
		{
		int cursor = oldCurrent->cursor;
		if (event.scroll.delta < 0)
		{
			int vDelta = -event.scroll.delta - (oldCurrent->length + 1);
			if (vDelta < 0)
				vDelta = 0;
			while (vDelta >= 0 && topLine > text)
			{
				char *line = topLine;
				PreviousLine(text, &topLine, iString.lineWidth);
				vDelta -= (int)(line - topLine);
				iString.cursor -= (int)(line - topLine);
			}
			if (iString.cursor < 0)
				iString.cursor = 0;
			for (newCurrent = Current(); vDelta >= 0 && newCurrent != first; newCurrent = newCurrent->PreviousString())
				vDelta -= newCurrent->length + 1;
		}

		// Scroll down.
		else
		{
			int vDelta = event.scroll.delta - (oldCurrent->length + 1);
			if (vDelta < 0)
				vDelta = 0;
			char *lastLine = Last()->text;
			NextLine(&lastLine, iString.lineWidth);
			while (vDelta >= 0 && *lastLine)
			{
				char *line = topLine;
				NextLine(&topLine, iString.lineWidth);
				vDelta -= (int)(topLine - line);
				iString.cursor += (int)(topLine - line);
				NextLine(&lastLine, iString.lineWidth);
			}
			if (iString.cursor > iString.length)
				iString.cursor = iString.length;
			for (newCurrent = Current(); vDelta >= 0 && newCurrent != last; newCurrent = newCurrent->NextString())
				if (!FlagSet(newCurrent->Next()->woFlags, WOF_NON_SELECTABLE))
					vDelta -= newCurrent->length + 1;
		}

		// Update the text field.
		if (newCurrent == current)
		{
			RecomputeFields(TRUE, TRUE);
			current = NULL;
		}
		newCurrent->cursor = Min(cursor, newCurrent->length);
		}
		break;

	case L_INSERT_TOGGLE:
		if (!FlagSet(woFlags, WOF_VIEW_ONLY))
		{
			insertMode = !insertMode;
			iString.insertMode = insertMode;
			string->Event(event);
			for (string = First(); string; string = string->NextString())
				string->insertMode = insertMode;
		}
		break;

	case L_PREVIOUS:
	case L_NEXT:
		ccode = S_UNKNOWN;
		break;

	case L_TOP:
		updateFields = UIW_TEXT::ResetMark();
		if (iString.beginMark != -1)
		{
			if (iString.cursor == iString.endMark)
				iString.endMark = iString.beginMark;
			iString.beginMark = 0;
			updateFields = TRUE;
		}
		if (oldCurrent->cursor != 0 || oldCurrent->text != text)
		{
			iString.cursor = 0;
			First()->text = Current()->text = NULL;
			RecomputeFields(TRUE, TRUE);
			updateFields = FALSE;
		}
		break;

	case L_BOL:
	case L_LEFT:
	case L_WORD_LEFT:
		updateFields = UIW_TEXT::ResetMark();
		if (oldCurrent->Event(event) != S_UNKNOWN || oldCurrent->text == text)
			break;
		else if (oldCurrent->PreviousString())
			string = newCurrent = oldCurrent->PreviousString();
		else
		{
			oldCurrent->length = PreviousLine(text, &topLine, iString.lineWidth);
			oldCurrent->text = topLine;
			updateFields = updateStrings = updateCurrent = TRUE;
		}
		string->cursor = string->length;
		if (ccode == L_WORD_LEFT && string->length != 0)
			eventManager->Put(event, Q_BEGIN);
		if (iString.beginMark != -1)
		{
			string->beginMark = string->endMark = string->cursor;
			updateFields = TRUE;
		}
		break;

	case L_BOTTOM:
		updateFields = UIW_TEXT::ResetMark();
		if (iString.endMark != -1)
		{
			if (iString.cursor == iString.beginMark)
				iString.beginMark = iString.endMark;
			iString.endMark = iString.length;
			updateFields = TRUE;
		}
		if (oldCurrent->cursor != oldCurrent->length || oldCurrent->text[oldCurrent->length])
		{
			iString.cursor = iString.length;
			Last()->text = Current()->text = NULL;
			RecomputeFields(TRUE, TRUE);
			updateFields = FALSE;
		}
		break;

	case L_EOL:
	case L_RIGHT:
	case L_WORD_RIGHT:
		updateFields = UIW_TEXT::ResetMark();
		if (oldCurrent->Event(event) != S_UNKNOWN || !oldCurrent->text[oldCurrent->length])
			break;
		else if (oldCurrent->NextString())
			string = newCurrent = oldCurrent->NextString();
		else
		{
			char *lastLine = Last()->text;
			oldCurrent->length = NextLine(&lastLine, iString.lineWidth);
			if (!lastLine[0])
				break;
			NextLine(&topLine, iString.lineWidth);
			oldCurrent->text = lastLine;
			updateFields = updateStrings = updateCurrent = TRUE;
		}
		string->cursor = 0;
		if (iString.beginMark != -1)
		{
			string->beginMark = string->endMark = string->cursor;
			updateFields = TRUE;
		}
		break;

	case L_UP:
	case L_PGUP:
		updateFields = UIW_TEXT::ResetMark();
		if (oldCurrent->text == text)
			break;
		else if (oldCurrent->PreviousString())
			string = newCurrent = (ccode == L_UP) ? oldCurrent->PreviousString() : First();
		else
		{
			int count = (ccode == L_UP) ? 1 : UI_LIST::Count() - 1;
			while (--count >= 0 && topLine > text)
				oldCurrent->length = PreviousLine(text, &topLine, iString.lineWidth);
			oldCurrent->text = topLine;
			updateFields = updateStrings = updateCurrent = TRUE;
		}
		string->cursor = Min(oldCurrent->cursor, string->length);
		if (iString.beginMark != -1)
		{
			string->beginMark = string->endMark = string->cursor;
			updateFields = TRUE;
		}
		break;

	case L_DOWN:
	case L_PGDN:
		updateFields = UIW_TEXT::ResetMark();
		if (!oldCurrent->text[oldCurrent->length])
			break;
		else if (oldCurrent->NextString())
		{
			for (string = oldCurrent->NextString(); string; string = string->NextString())
			{
				if (!FlagSet(string->woFlags, WOF_NON_SELECTABLE))
					newCurrent = string;
				if (ccode == L_DOWN)
					break;
			}
			string = newCurrent;
		}
		else
		{
			int count = (ccode == L_DOWN) ? 1 : UI_LIST::Count() - 1;
			char *lastLine = Last()->text;
			int lastLength = Last()->length;
			while (--count >= 0 && lastLine[lastLength])
			{
				NextLine(&topLine, iString.lineWidth);
				lastLength = NextLine(&lastLine, iString.lineWidth);
			}
			oldCurrent->text = lastLine;
			oldCurrent->length = NextLine(&lastLine, iString.lineWidth);
			updateFields = updateStrings = updateCurrent = TRUE;
		}
		string->cursor = Min(oldCurrent->cursor, string->length);
		if (iString.beginMark != -1)
		{
			string->beginMark = string->endMark = string->cursor;
			updateFields = TRUE;
		}
		break;

	case L_BACKSPACE:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		updateFields = UIW_TEXT::ResetMark();
		iString.Event(event);
		RecomputeFields(TRUE, TRUE);
		break;

	case L_DELETE:
	case L_DELETE_WORD:
	case L_DELETE_EOL:
	case L_CUT:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		// Check for a mark above the top line of the text field.
		if (iString.beginMark != -1 && text + iString.beginMark < topLine)
		{
			topLine = text + iString.beginMark;
			PreviousLine(text, &topLine, iString.lineWidth);
		}
		// Process the delete messages.
		deleteCount = iString.length;
		iString.Event(event);
		deleteCount -= iString.length;
		updateFields = UIW_TEXT::ResetMark();
		if (deleteCount)
		{
			string->length -= deleteCount;
			string->maxLength -= deleteCount;
			string->StringUpdate(insertCount, deleteCount, scrollWidth);
			for (string = string->NextString(); string; string = string->NextString())
				string->text -= deleteCount;
			RecomputeFields(TRUE, TRUE);
		}
		break;

	case E_KEY:
	case L_PASTE:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		{
		int cursorCount = iString.cursor;
		insertCount = iString.length;
		iString.Event(event);
		cursorCount = iString.cursor - cursorCount;
		insertCount = iString.length - insertCount;
		updateFields = UIW_TEXT::ResetMark();
		if (insertCount || !insertMode)
		{
			oldCurrent->length += insertCount;
			oldCurrent->maxLength += insertCount;
			for (string = oldCurrent->NextString(); string; string = string->NextString())
				string->text += insertCount;
			char *line = Last()->text;
			if (current == last && Current()->cursor + cursorCount > NextLine(&line, iString.lineWidth))
				NextLine(&topLine, iString.lineWidth);
			else
			{
				scrollWidth = oldCurrent->CursorUpdate(oldCurrent->cursor + cursorCount, 0, TRUE);
				if (oldCurrent->leftClip == 0 || FlagSet(wnFlags, WNF_NO_WRAP))
					oldCurrent->StringUpdate(insertCount, deleteCount, scrollWidth);
			}
			RecomputeFields(TRUE, TRUE);
		}
		}
		break;

	case L_SELECT:
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) && iString.length + 2 <= iString.maxLength)
		{
			if (iString.beginMark != -1)
				iString.Event(UI_EVENT(L_CUT));
			updateFields = UIW_TEXT::ResetMark();
			if (iString.cursor != iString.length)
			{
				memmove(&text[iString.cursor+2], &text[iString.cursor], iString.length - iString.cursor + 1);
				text[iString.cursor] = '\r';
				text[iString.cursor+1] = '\n';
			}
			else
			{
				text[iString.cursor] = '\r';
				text[iString.cursor+1] = '\n';
				text[iString.cursor+2] = '\0';
			}
			iString.cursor += 2;
			iString.length += 2;
			oldCurrent->text = NULL;
			if (oldCurrent->NextString())
				current = string = oldCurrent->NextString();
			else
				NextLine(&topLine, iString.lineWidth);
			iString.leftClip = 0;
			RecomputeFields(TRUE, TRUE);
		}
		break;

	case L_MARK:
		updateFields = UIW_TEXT::ResetMark();
		ccode = UIW_WINDOW::Event(event);
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
			break;
		else if (iString.beginMark == -1)
			iString.beginMark = iString.endMark = (int)(oldCurrent->text + oldCurrent->beginMark - text);
		else
		{
			iString.beginMark = iString.endMark = -1;
			updateFields = TRUE;
		}
		break;

	case L_BEGIN_MARK:
	case L_CONTINUE_MARK:
	case L_END_MARK:
		if (FlagSet(woStatus, WOS_EDIT_MODE) || !true.Overlap(event.position))
			return (UI_WINDOW_OBJECT::Event(event));
		else if (ccode == L_BEGIN_MARK && 
			((vScroll && vScroll->true.Overlap(event.position)) ||
			 (hScroll && hScroll->true.Overlap(event.position))))
			return (UIW_WINDOW::Event(event));
		updateFields = UIW_TEXT::ResetMark();
		if (FlagSet(woFlags, WOF_VIEW_ONLY))
		{
			ccode = UIW_WINDOW::Event(event);
			break;
		}
		else if (ccode == L_BEGIN_MARK)
		{
			if (iString.beginMark != iString.endMark)
				updateFields = TRUE;
			ccode = UIW_WINDOW::Event(event);
			iString.beginMark = iString.endMark = (int)(Current()->text + Current()->beginMark - text);
			break;
		}
		else if (ccode == L_END_MARK)
		{
			ccode = UIW_WINDOW::Event(event);
			if (iString.beginMark == iString.endMark)
				iString.beginMark = iString.endMark = -1;
			iString.woStatus |= WOS_AUTO_MARK;
			break;
		}
		// L_CONTINUE_MARK needs to check for movement to another field.
		else if (string->true.Overlap(event.position))
		{
			ccode = UIW_WINDOW::Event(event);
			break;
		}
		for (string = First(); string; string = string->NextString())
			if (string->true.Overlap(event.position) && !FlagSet(string->woFlags, WOF_NON_SELECTABLE))
			{
				string->beginMark = string->endMark = -1;
				ccode = UIW_WINDOW::Event(event);
				updateFields = TRUE;
				break;
			}
		break;

	case L_VIEW:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		else if (!FlagSet(woFlags, WOF_VIEW_ONLY) &&
			clipList.First() && clipList.First()->region.Overlap(event.position))
			eventManager->DeviceState(E_MOUSE, DM_EDIT);
		else
			ccode = UIW_WINDOW::Event(event);
		break;

	default:
		ccode = UIW_WINDOW::Event(event);
		break;
	}

	// Check for a current update.
	if (updateCurrent)
	{
		string = Current();
		scrollWidth = string->CursorUpdate(string->cursor, 0, TRUE);
		string->StringUpdate(0, 0, scrollWidth);
	}

	// Check the current and mark information (see UIW_STRING code).
	if (newCurrent)
		UIW_WINDOW::Add(newCurrent);
	if (current)
	{
		string = Current();
		// Compute the new mark regions.
		int tCursor = (int)(string->text + string->cursor - text);
		if (string->beginMark == -1)
			;
		else if (iString.cursor == iString.beginMark &&
			(iString.cursor != iString.endMark || tCursor < iString.cursor))
			iString.beginMark = tCursor;
		else
			iString.endMark = tCursor;
		// Check for reversed marks (case when moving onto beginMark or endMark lines).
		iString.cursor = tCursor;
		if (iString.beginMark > iString.endMark)
		{
			tCursor = iString.beginMark;
			iString.beginMark = iString.endMark;
			iString.endMark = tCursor;
		}
		// Check the left-clip region of a no-wrap field.
		if (iString.leftClip != string->leftClip)
		{
			if (FlagSet(wnFlags, WNF_NO_WRAP))
				iString.leftClip = string->leftClip;
			updateFields = updateStrings = TRUE;
		}
	}

	// Update the field information.
	if (updateFields || updateStrings)
		RecomputeFields(updateFields, updateStrings);

	// Recompute the horizontal and vertical scroll regions.
	if (!current)
		return (ccode);
	string = Current();
	if (hScroll && (oldCursor != iString.cursor))
	{
		UI_EVENT event(S_HSCROLL_SET);
		event.scroll.minimum = 0;
		event.scroll.delta = 1;
		event.scroll.showing = iString.lineWidth / display->cellWidth;
		if (event.scroll.showing > string->length)
			event.scroll.showing = string->length;
		event.scroll.maximum = string->length;
		event.scroll.current = string->cursor;
		hScroll->Event(event);
	}
	if (vScroll && (newCurrent || updateFields || updateStrings))
	{
		UI_EVENT event(S_VSCROLL_SET);
		event.scroll.minimum = 0;
		event.scroll.maximum = iString.length - string->length;
		event.scroll.current = (int)(string->text - text);
		event.scroll.delta = string->length + 1;
		event.scroll.showing = 0;
		for (string = First(); string; string = string->NextString())
			event.scroll.showing += string->length;
		vScroll->Event(event);
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_TEXT::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	if (!objectID) objectID = ID_TEXT;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_TEXT;
		windowID[1] = ID_STRING;
		windowID[2] = ID_WINDOW;
#ifndef _WINDOWS
		iString.maxLength = maxLength;
		topLine = iString.text = text;
		iString.length = ui_strlen(text);
		iString.cursor = 0;
		iString.beginMark = iString.endMark = -1;
		iString.insertMode = insertMode = TRUE;
		iString.woAdvancedFlags = WOAF_NO_DESTROY;
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UIW_WINDOW::Information(CHANGED_FLAGS, data, ID_TEXT);
		dwStyle |= ES_AUTOVSCROLL | ES_MULTILINE | ES_OEMCONVERT;
		if (FlagSet(wnFlags, WNF_NO_WRAP))
			dwStyle |= ES_AUTOHSCROLL;
		if (FlagSet(woFlags, WOF_JUSTIFY_CENTER))
			dwStyle |= ES_CENTER;
		else if (FlagSet(woFlags, WOF_JUSTIFY_RIGHT))
			dwStyle |= ES_RIGHT;
		if (screenID && request == CHANGED_FLAGS && objectID == ID_TEXT)
		{
			DestroyWindow(screenID);
			if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
			{
				RegisterObject("UIW_TEXT", "EDIT", &_textOffset,
					&_textJumpInstance, &_textCallback, text);
				SendMessage(screenID, EM_LIMITTEXT, maxLength - 1, FALSE);
			}
			else
				Event(UI_EVENT(S_CREATE));
			SetWindowPos(screenID, previous ? Previous()->screenID : 0,
				0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_TEXT)
		{
			if (FlagSet(woFlags, WOF_NON_FIELD_REGION))
				return (parent->Information(request, data));
			UI_EVENT event(S_INITIALIZE);
			Event(event);
			event.type = S_CREATE;
			Event(event);
		}
#endif
		wnFlags |= WNF_CONTINUE_SELECT;
		break;

	case GET_TEXT:
		if (!data)
			return (DataGet());
		*(char **)data = DataGet();
		break;

	case SET_TEXT:
		DataSet((char *)data);
		break;

	case GET_TEXT_LENGTH:
		if (!data)
			return (&maxLength);
		*(int *)data = maxLength;
		break;

	case SET_TEXT_LENGTH:
		DataSet(text, *(int *)data);
		break;

	default:
		data = UIW_WINDOW::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_TEXT::UIW_TEXT(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
#ifdef _WINDOWS
	UIW_WINDOW(0, 0, 20, 6, WOF_NO_FLAGS)
#else
	UIW_WINDOW(0, 0, 20, 6, WOF_NO_FLAGS),
	iString(0, 0, 0, NULL, 0)
#endif
{
	// Initialize the text information.
	UIW_TEXT::Load(name, directory, file);
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	UIW_TEXT::Information(INITIALIZE_CLASS, NULL);
	UIW_TEXT::DataSet(text);
}

void UIW_TEXT::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Load the text information.
	UI_WINDOW_OBJECT::Load(name, directory, file);
	short _value; file->Load(&_value); maxLength = _value;
	text = new char[maxLength+1]; file->Load(text, maxLength);
	short noOfObjects;
	file->Load(&noOfObjects);
	for (int i = 0; i < noOfObjects; i++)
		Add(UI_WINDOW_OBJECT::New(NULL, directory, file));
	file->Load(&wnFlags);
}

void UIW_TEXT::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Store the text information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	short _value = maxLength; file->Store(_value);
	file->Store(text);
	short noOfObjects = support.Count();
	file->Store(noOfObjects);
	for (UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
	{
		_value = object->SearchID();
		file->Store(_value);
		object->Store(NULL, directory, file);
	}
	file->Store(wnFlags);
}
#endif

