//	Zinc Interface Library - WIN.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#define USE_RAW_KEYS
#include "ui_win.hpp"
#pragma hdrstop

SCREENID UI_WINDOW_MANAGER::currentScreenID = ID_SCREEN;

// ----- UI_WINDOW_MANAGER --------------------------------------------------

UI_WINDOW_MANAGER::UI_WINDOW_MANAGER(UI_DISPLAY *_display, UI_EVENT_MANAGER *_eventManager,
	EVENT_TYPE (*_exitFunction)(UI_DISPLAY *display, UI_EVENT_MANAGER *eventManager, UI_WINDOW_MANAGER *windowManager)) :
	UIW_WINDOW(0, 0, _display->columns, _display->lines),
	exitFunction(_exitFunction)
{
	extern void z_win2_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_win2_dummy();
	extern void z_event_dummy(void);	// Bug fix for Zortech & Microsoft linkers.
	z_event_dummy();
	extern void z_pnorm_dummy(void);	// Bug fix for Zortech & Microsoft linkers.
	z_pnorm_dummy();
	extern void z_hotkey_dummy(void);	// Bug fix for Zortech & Microsoft linkers.
	z_hotkey_dummy();
	extern void z_jump_dummy(void);		// Bug fix for Zortech & Microsoft linkers.
	z_jump_dummy();

	// Initialize the window manager information.
	windowID[0] = ID_WINDOW_MANAGER;
	windowID[1] = ID_WINDOW;

	if (!_display->isText)
		woStatus |= WOS_GRAPHICS;
	currentScreenID = screenID = ID_SCREEN;
	display = _display;
	eventManager = _eventManager;
	windowManager = this;

#ifdef _WINDOWS
	// Initialize the class registration module.
	extern int ZincClassInitialize(HANDLE hInstance);
	ZincClassInitialize(display->hInstance);
#endif
}

UI_WINDOW_MANAGER::~UI_WINDOW_MANAGER(void)
{
	UI_WINDOW_OBJECT *object;

	// Remove all objects from the window manager.
	for (object = First(); object; object = First())
	{
		UI_LIST::Subtract(object);
#ifdef _WINDOWS
		if (object->screenID)
		{
			object->woStatus |= WOS_WINDOWS_ACTION;
			DestroyWindow(object->screenID);
		}
#endif
		if (!FlagSet(object->woAdvancedFlags, WOAF_NO_DESTROY))
			delete object;
	}

#ifdef _WINDOWS
	extern int ZincClassRestore(HANDLE hInstance);
	ZincClassRestore(display->hInstance);
	PostQuitMessage(0);
#endif

	windowManager = NULL;
	eventManager = NULL;
	display = NULL;
}

UI_WINDOW_OBJECT *UI_WINDOW_MANAGER::Add(UI_WINDOW_OBJECT *object)
{
	// Check for a null object.
	if (!object)
		return (NULL);

	// Remove any temporary objects if we have a non-temporary window.
	UI_WINDOW_OBJECT *firstObject = First();
	if (!FlagSet(object->woAdvancedFlags, WOAF_TEMPORARY))
	{
		while (firstObject && FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
			firstObject = First();
		}
	}
	// Remove any temporary objects if the window is already in the list.
	else if (UI_LIST::Index(object) != -1)
	{
		while (firstObject && firstObject != object)
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
			firstObject = First();
		}
	}
	// See if the object is already the first on the list.
	if (object == First())
		return (object);

	// Find the maximum update region.
	UI_REGION updateRegion = { 0x0FFF, 0x0FFF, 0x0000, 0x0000 };
	for (UI_WINDOW_OBJECT *tObject = First(); tObject && tObject != object;
		tObject = tObject->Next())
		if (object->true.Overlap(tObject->true))
		{
			updateRegion.left = Min(updateRegion.left, tObject->true.left);
			updateRegion.top = Min(updateRegion.top, tObject->true.top);
			updateRegion.right = Max(updateRegion.right, tObject->true.right);
			updateRegion.bottom = Max(updateRegion.bottom, tObject->true.bottom);
		}

	// Initialize the object region.
	if (!object->screenID || FlagSet(object->woAdvancedFlags, WOAF_TEMPORARY))
	{
#ifdef _WINDOWS
		object->Event(UI_EVENT(S_INITIALIZE));
		object->Event(UI_EVENT(S_CREATE));
#else
		object->Event(UI_EVENT(S_INITIALIZE));
		if (!object->screenID)
		{
			object->screenID = ++currentScreenID;
//			if (screenID == ID_SCREEN)			// Windows default operation.
//				screenID = object->screenID;
		}
		object->Event(UI_EVENT(S_CREATE));

		// Make sure the temporary window is visible on the screen.
		if (FlagSet(object->woAdvancedFlags, WOAF_TEMPORARY))
		{
			if (object->relative.left < 0)
			{
				object->relative.right += object->relative.left;
				object->relative.left = 0;
			}
			else if (object->relative.right >= display->columns)
			{
				object->relative.left += (display->columns - 1 - object->relative.right);
				object->relative.right = display->columns - 1;
			}
			if (object->relative.bottom >= display->lines && object->parent)
			{
				int height = object->relative.bottom - object->relative.top;
				object->relative.bottom = object->parent->true.top - 1;
				object->relative.top = object->relative.bottom - height;
			}
			object->Event(UI_EVENT(S_CREATE));
		}
		updateRegion = object->true;

//		// Check the horizontal and vertical scroll regions.
//		object->woStatus |= WOS_INTERNAL_ACTION;
//		object->Event(UI_EVENT(S_VSCROLL_CHECK));
//		object->Event(UI_EVENT(S_HSCROLL_CHECK));
//		object->woStatus &= ~WOS_INTERNAL_ACTION;
#endif
	}

	// Bring the object to the front of the object queue.
	if (object != firstObject)
	{
		if (UI_LIST::Index(object) != -1)
			UI_LIST::Subtract(object);
		else
			updateRegion = object->true;
		UI_EVENT event(S_REGION_DEFINE);
		object->Event(event);
		event.type = S_NON_CURRENT;
		event.region.left = event.region.top = event.region.right = event.region.bottom = -1;
		if (firstObject && !FlagSet(object->woAdvancedFlags, WOAF_TEMPORARY))
			firstObject->Event(event);
		UI_LIST::Add(firstObject, object);
	}
		
	// See if the object needs to be re-displayed.
#ifdef _WINDOWS
	if (!FlagSet(object->woStatus, WOS_WINDOWS_ACTION))
	{
		object->woStatus |= WOS_WINDOWS_ACTION;
		if (currentScreenID == ID_SCREEN)
		{
//			screenID = currentScreenID = object->screenID;	// Default Windows operation.
			currentScreenID = object->screenID;
			ShowWindow(object->screenID, display->nCmdShow);
		}
		else
		{
			int showState = SW_SHOWNORMAL;
			if (FlagSet(object->woStatus, WOS_MAXIMIZED))
				showState = SW_SHOWMAXIMIZED;
			else if (FlagSet(object->woStatus, WOS_MINIMIZED))
				showState = SW_SHOWMINIMIZED;
			ShowWindow(object->screenID, showState);
		}
		UpdateWindow(object->screenID);
		SetFocus(object->screenID);
		object->woStatus &= ~WOS_WINDOWS_ACTION;
	}
#else
	eventManager->DeviceState(E_CURSOR, D_OFF);
	eventManager->DeviceState(E_MOUSE, DM_VIEW);
#endif
	object->Event(UI_EVENT(S_CURRENT, 0, updateRegion));

	// Return a pointer to the object.
	return (object);
}

#ifdef _WINDOWS
EVENT_TYPE UI_WINDOW_MANAGER::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *firstObject = First();

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_MANAGER);
	switch (ccode)
	{
	case S_CLOSE:
		if (!firstObject)
			ccode = S_NO_OBJECT;
		else if (FlagSet(firstObject->woAdvancedFlags, WOAF_LOCKED) &&
			screenID != firstObject->screenID)
			break;
		else if (!FlagSet(firstObject->woAdvancedFlags, WOAF_LOCKED))
		{
			Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
		}
		break;

	case S_ADD_OBJECT:
		if (FlagSet(woStatus, WOS_WINDOWS_ACTION) &&
			Index((UI_WINDOW_OBJECT *)event.data) != -1)
		{
			firstObject->woStatus &= ~WOS_CURRENT;
			firstObject = Add((UI_WINDOW_OBJECT *)event.data);
			firstObject->woStatus |= WOS_CURRENT;
		}
		break;

	case L_EXIT_FUNCTION:
		if (exitFunction && (*exitFunction)(display, eventManager, this) != L_EXIT)
			break;
		// Continue to L_EXIT.
	case L_EXIT:
		break;

	case L_PREVIOUS:
	case L_NEXT:
	case L_UP:
	case L_DOWN:
	case L_LEFT:
	case L_RIGHT:
		if (firstObject)
			ccode = firstObject->Event(event);
		break;

	default:
		if (event.type == E_MSWINDOWS)
		{
			MSG message = event.message;
			if (message.message == WM_KEYDOWN || message.message == WM_KEYUP ||
				message.message == WM_SYSKEYDOWN || message.message == WM_SYSKEYUP)
				TranslateMessage(&message);
			if (message.message == WM_QUIT)
				ccode = L_EXIT;
			else
				ccode = DispatchMessage(&message);
		}
		else if (firstObject)
			ccode = firstObject->Event(event);
		else
			ccode = S_NO_OBJECT;
		break;
	}

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UI_WINDOW_MANAGER::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;

	// Check for an empty list.
	UI_WINDOW_OBJECT *firstObject = First();
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW_MANAGER);
	if (!firstObject && ccode == L_EXIT)
		return (L_EXIT);
	else if (!firstObject && ccode != L_EXIT_FUNCTION)
		return (S_NO_OBJECT);

	// Switch on the event type.
	switch (ccode)
	{
	case S_CASCADE:
		{
		UI_EVENT event;
		int column = 0, line = 0;
		display->RegionDefine(ID_SCREEN, 0, 0, display->columns, display->lines);
		// Recompute the object regions.
		for (object = Last(); object; object = object->Previous())
		{
			int width = object->relative.right - object->relative.left + 1;
			int height = object->relative.bottom - object->relative.top + 1;
			if (line + height >= display->lines)
				line = 0;
			object->relative.left = column;
			object->relative.top = line;
			object->relative.right = column + width - 1;
			object->relative.bottom = line + height - 1;
			event.type = S_CREATE;
			object->Event(event);
			event.type = S_REGION_DEFINE;
			object->Event(event);
			column += display->cellWidth * 3;
			line += display->cellHeight * 3 / 2;
		}
		}
		// Continue to S_REDISPLAY.
	case S_REDISPLAY:
		// Redisplay the objects on the screen.
		if (display->isText)
			display->VirtualGet(ID_SCREEN, 0, 0, display->columns, display->lines);
		display->Rectangle(ID_SCREEN, true, display->backgroundPalette, 0, TRUE);
		for (object = Last(); object; object = object->Previous())
			object->Event(UI_EVENT((object == First()) ? S_CURRENT : S_DISPLAY_INACTIVE, 0, object->true));
		if (display->isText)
			display->VirtualPut(ID_SCREEN);
		break;

	case S_RESET_DISPLAY:
		{
		while (firstObject && FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
			firstObject = First();
		}
		if (!event.data)
		{
			woStatus &= ~WOS_GRAPHICS;

			// Convert each object's coordinates from graphics.
			for (UI_WINDOW_OBJECT *object = First(); object; object = object->Next())
				object->Event(event);
			break;
		}
		display = (UI_DISPLAY *)event.data;
		if (!display->isText)
			woStatus |= WOS_GRAPHICS;
		UI_WINDOW_OBJECT *object1 = Last();
		UI_WINDOW_OBJECT *object2;
		first = current = last = NULL;
		currentScreenID = screenID = ID_SCREEN;
		while (object1)
		{
			object2 = object1;
			object1 = object2->Previous();
			UI_WINDOW_MANAGER::Add(object2);
		}
		eventManager->Put(UI_EVENT(S_REDISPLAY));
		}
		break;

	case S_CLOSE_TEMPORARY:
		if (firstObject->parent && FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
		}
		else if (event.type != S_CLOSE_TEMPORARY)
			ccode = firstObject->Event(event);
		break;

	case S_CLOSE:
	case L_MOVE:
	case L_SIZE:
	case L_MINIMIZE:
	case L_MAXIMIZE:
	case L_RESTORE:
		while (firstObject && firstObject->parent &&
			FlagSet(firstObject->woAdvancedFlags, WOAF_TEMPORARY))
		{
			UI_WINDOW_MANAGER::Subtract(firstObject);
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
				delete firstObject;
			firstObject = First();
		}
		if (!firstObject)
			return (S_NO_OBJECT);
		else if (ccode != S_CLOSE)
			return (firstObject->Event(event));
		else if (screenID != firstObject->screenID)
		{
			// Make sure we can remove the window from the display.
			if (!FlagSet(firstObject->woAdvancedFlags, WOAF_LOCKED))
			{
				UI_WINDOW_MANAGER::Subtract(firstObject);
				if (!FlagSet(firstObject->woAdvancedFlags, WOAF_NO_DESTROY))
					delete firstObject;
				if (!First())
					eventManager->Put(UI_EVENT(S_NO_OBJECT));
			}
			break;
		}
		// Continue to L_EXIT_FUNCTION.
	case L_EXIT_FUNCTION:
		if (FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL))
			break;
		else if (exitFunction && (*exitFunction)(display, eventManager, this) != L_EXIT)
			break;
		// Continue to L_EXIT.
	case L_EXIT:
		ccode = L_EXIT;
		break;

	case L_NEXT_WINDOW:
		if (!FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL) &&
			firstObject != Last())
			UI_WINDOW_MANAGER::Add(Last());
		break;

	case L_BEGIN_SELECT:
	case L_VIEW:
		for ( ; firstObject; firstObject = firstObject->Next())
			if (!FlagSet(firstObject->woFlags, WOF_NON_SELECTABLE) &&
				firstObject->true.Overlap(event.position))
			{
				// Bring the new object to the front.
				if (firstObject != first && ccode == L_BEGIN_SELECT)
				{
					// Check for an overlapping parent.
					if (!FlagSet(firstObject->woStatus, WOS_MINIMIZED) &&
						FlagSet(First()->woAdvancedFlags, WOAF_TEMPORARY) &&
						First()->Next() == firstObject)
					{
						for (UI_WINDOW_OBJECT *object = First()->parent; object;
							object = object->parent)
							if (object == firstObject || !object->true.Overlap(event.position))
								break;
						// Temporary object is direct child of firstObject.
						if (object == firstObject)
							return (firstObject->Event(event));
					}
					UI_WINDOW_MANAGER::Add(firstObject);
				}
				EVENT_TYPE tCode = firstObject->Event(event);
				if (tCode != S_UNKNOWN)
					return (tCode);
			}
			else if (FlagSet(firstObject->woAdvancedFlags, WOAF_MODAL))
				break;
		eventManager->DeviceState(E_MOUSE, DM_VIEW);
		break;

	default:
		ccode = firstObject->Event(event);
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UI_WINDOW_MANAGER::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	// Switch on the request.
	switch (request)
	{
	case GET_NUMBERID_OBJECT:
	case GET_STRINGID_OBJECT:
		{
		void *match = NULL;
		for (UI_WINDOW_OBJECT *object = First(); object && !match; object = object->Next())
			match = object->UI_WINDOW_OBJECT::Information(request, data, objectID);
		data = match;
		}
		break;

	default:
		return (NULL);
	}

	// Return the information.
	return (data);
}

UI_WINDOW_OBJECT *UI_WINDOW_MANAGER::Subtract(UI_WINDOW_OBJECT *object)
{
	// Make sure there is an object to subtract.
	if (!object)
		return (NULL);
	object->woStatus &= ~WOS_CURRENT;

	// Update the windows according to their new list positions.
	UI_EVENT event;
	UI_WINDOW_OBJECT *nextObject = object->Next();
	if (UI_LIST::Index(object) == -1)
		return (object);
#ifdef _WINDOWS
	else if (FlagSet(object->woStatus, WOS_WINDOWS_ACTION))
	{
		UI_LIST::Subtract(object);
		if (object->screenID)
		{
			if (object->screenID == screenID)
				eventManager->Put(UI_EVENT(L_EXIT));
			else if (!First())
				eventManager->Put(UI_EVENT(S_NO_OBJECT));
			object->screenID = 0;
			event.type = S_DEINITIALIZE;
			object->Event(event);
		}
	}
#endif
	else
	{
		event.type = S_NON_CURRENT;
		event.region.left = event.region.top = event.region.right = event.region.bottom = -1;
		object->Event(event);		// Call any user-functions.
#ifdef _WINDOWS
		if (object->screenID)
			DestroyWindow(object->screenID);	// This re-calls Subtract Internally (see previous else).
#else
		event.type = S_REGION_DEFINE;
		event.region = object->true;
		display->RegionDefine(ID_SCREEN, 0, 0, display->columns, display->lines);
		for (UI_WINDOW_OBJECT *tObject = Last(); tObject; tObject = tObject->Previous())
			if (tObject != object)
				tObject->Event(event);
		if (display->isText)
			display->VirtualGet(ID_SCREEN, 0, 0, display->columns, display->lines);
		display->Rectangle(ID_SCREEN, event.region, display->backgroundPalette, 0, TRUE);
		for (tObject = Last(); tObject; tObject = tObject->Previous())
		{
			int updateRegion = FALSE;
			if (tObject == object)
				;
			else if (tObject == nextObject)
			{
				event.type = S_CURRENT;
				eventManager->DeviceState(E_CURSOR, D_OFF);
				eventManager->DeviceState(E_MOUSE, DM_VIEW);
				tObject->Event(event);
				updateRegion = TRUE;
				break;
			}
			else if (tObject->true.Overlap(object->true))
			{
				event.type = (tObject == nextObject || FlagSet(tObject->woAdvancedFlags, WOAF_TEMPORARY)) ?
					S_CURRENT : S_DISPLAY_INACTIVE;
				tObject->Event(event);
				updateRegion = TRUE;
			}
			if (updateRegion && display->isText)	// Fix for shadow border problem.
			{
				event.region.left = Min(event.region.left, tObject->true.left);
				event.region.top = Min(event.region.top, tObject->true.top);
				event.region.right = Max(event.region.right, tObject->true.right);
				event.region.bottom = Max(event.region.bottom, tObject->true.bottom);
				updateRegion = FALSE;
			}
		}
		if (display->isText)
			display->VirtualPut(ID_SCREEN);
		UI_LIST::Subtract(object);
		event.type = S_DEINITIALIZE;
		object->Event(event);
#endif
	}

	// Return a pointer to the next object.
	return (nextObject);
}
