//	Zinc Interface Library - WIN1.CPP
//	COPYRIGHT (C) 1990-1992.  All Rights Reserved.
//	Zinc Software Incorporated.  Pleasant Grove, Utah  USA
/* This file is part of OpenZinc

OpenZinc is free software: You can redistribute it and/or modify it
 under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the license or
 (at your option) any later version.

OpenZinc is distributed in the hope that it will be useful,
but without ANY WARRANTY; without even the implied warranty of
MARCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU Lessor Public license for more details

You should have received a copy of the GNU Lessor Public License
along with OpenZinc. If not, see <http://www.gnu.org/licenses/>.
*/


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include "ui_win.hpp"
#pragma hdrstop

// ----- UIW_WINDOW ---------------------------------------------------------

UIW_WINDOW::UIW_WINDOW(int left, int top, int width, int height,
	WOF_FLAGS _woFlags, WOAF_FLAGS _woAdvancedFlags,
	UI_HELP_CONTEXT _helpContext, UI_WINDOW_OBJECT *minObject) :
	UI_WINDOW_OBJECT(left, top, width, height, _woFlags, _woAdvancedFlags),
	UI_LIST(), support(), clipList(), wnFlags(WNF_NO_FLAGS)
{
	// Initialize the window information.
	compareFunctionName = NULL;
	helpContext = _helpContext;
	UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
	UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	if (minObject)
	{
		minObject->woFlags |= WOF_SUPPORT_OBJECT;
		UIW_WINDOW::Add(minObject);
	}
}

UIW_WINDOW::~UIW_WINDOW(void)
{
	if (compareFunctionName)
		delete compareFunctionName;
}

UI_WINDOW_OBJECT *UIW_WINDOW::Add(UI_WINDOW_OBJECT *object)
{
	// See if the new object exists or is already current.
	if (!object)
		return (NULL);
	else if (object == current)
	{
		object->woStatus |= WOS_CURRENT;
#ifdef _WINDOWS
		if (object->screenID && GetFocus() != object->screenID)
			SetFocus(object->screenID);
#endif
		return (object);
	}
	// Add the object to the list.
	else if (FlagSet(object->woFlags, WOF_SUPPORT_OBJECT) &&
		support.Index(object) == -1)
	{
		support.Add(object);
		object->parent = this;
	}
	else if (!FlagSet(object->woFlags, WOF_SUPPORT_OBJECT) &&
		((!object->Previous() && !object->Next() && object != First()) ||
		 UI_LIST::Index(object) == -1))
	{
		UI_LIST::Add(object);
		object->parent = this;
		if (FlagSet(object->woAdvancedFlags, WOAF_MDI_OBJECT))
		{
			if (current)
				Current()->woStatus &= ~WOS_CURRENT;
			object->woStatus |= WOS_CURRENT;
			current = object;
		}
	}
	else
	{
		UI_WINDOW_OBJECT *tObject = Current();
		UI_REGION updateRegion;
		if (tObject)
			updateRegion = tObject->true;
		else
			updateRegion = object->true;
		// Place MDI objects at the end of the list.
		if (FlagSet(object->woAdvancedFlags, WOAF_MDI_OBJECT))
		{
			UI_LIST::Subtract(object);
			UI_LIST::Add(object);
			object->Event(UI_EVENT(S_REGION_DEFINE));
			updateRegion = object->true;
		}
		// Make the old current field non-current.
		current = object;
		if (tObject && tObject != object)
		{
#ifdef _WINDOWS
			if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE) &&
				!Inherited(ID_GROUP))
				tObject->woStatus &= ~WOS_SELECTED;
#else
			if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
				tObject->woStatus &= ~WOS_SELECTED;
#endif
			if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
				tObject->woStatus &= ~WOS_CURRENT;
			else
				tObject->Event(UI_EVENT(S_NON_CURRENT, 0, object->true));
		}
		// Update the new current field.
		if (tObject != object)
		{
#ifdef _WINDOWS
			if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE) &&
				!Inherited(ID_GROUP))
				object->woStatus |= WOS_SELECTED;
#else
			if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
				object->woStatus |= WOS_SELECTED;
#endif
			if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
				object->woStatus |= WOS_CURRENT;
			else
			{
				eventManager->DeviceState(E_CURSOR, D_OFF);
				object->Event(UI_EVENT(S_CURRENT, 0, updateRegion));
			}
		}
	}

	// Display the object if its parent is already shown on the screen.
	if (screenID && !object->screenID)
	{
#ifdef _WINDOWS
		if (!FlagSet(woStatus, WOS_WINDOWS_ACTION))
			SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
#endif
		object->Event(UI_EVENT(S_INITIALIZE));
		object->Event(UI_EVENT(S_CREATE));
		if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
			clipList.Split(screenID, object->true, parent ? FALSE : TRUE);
		if (!current && !FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT) &&
			!FlagSet(object->woFlags, WOF_NON_SELECTABLE | WOF_SUPPORT_OBJECT))
			current = object;
	}

	// Return a pointer to the object.
	return (object);
}

void UIW_WINDOW::CheckSelection(void)
{
	// Check for proper single-select status.
	if (!current || !FlagSet(Current()->woStatus, WOS_SELECTED) ||
		FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
		return;

	// Make sure non-current items are not selected.
	for (UI_WINDOW_OBJECT *object = First(); object; object = object->Next())
		if (object != current && FlagSet(object->woStatus, WOS_SELECTED))
		{
			object->woStatus &= ~WOS_SELECTED;
#ifdef _WINDOWS
			if (object->screenID && object->Inherited(ID_BUTTON))
				SendMessage(object->screenID, BM_SETCHECK, 0, 0);
#else
			object->Event(UI_EVENT(S_REDISPLAY));
#endif
		}
}

void UIW_WINDOW::Destroy(void)
{
	// Destroy the list elements.
#ifdef _WINDOWS
	if (screenID)
	{
		SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
		UI_EVENT event(S_DEINITIALIZE);
		for (UI_WINDOW_OBJECT *object = First(); object; object = object->Next())
		{
			if (object->screenID)
				DestroyWindow(object->screenID);
			object->screenID = 0;
			object->Event(event);
		}
	}
#endif
	UI_LIST::Destroy();
}

#ifdef _WINDOWS
static int _windowOffset = -1;
static FARPROC _windowCallback = (FARPROC)DefWindowProc;
static FARPROC _windowJumpInstance = NULL;

long FAR PASCAL _export WindowJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	// Call the user class object event function.
	UIW_WINDOW *object = (UIW_WINDOW *)GetWindowLong(hWnd, _windowOffset);
	long ccode = object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam));

	// Check for a destroyed window.
	if (wMsg == WM_SYSCOMMAND && (wParam & 0xFFF0) == SC_CLOSE &&
		!object->screenID && !object->parent &&
		!FlagSet(object->woAdvancedFlags, WOAF_NO_DESTROY))
		delete object;
	return (ccode);
}

static int _mdiFrameOffset = -1;
static FARPROC _mdiFrameCallback = NULL;
static FARPROC _mdiFrameJumpInstance = NULL;

long FAR PASCAL _export MDIFrameJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	// Call the user class object event function.
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _mdiFrameOffset);
	long ccode = object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam));

	// Check for a destroyed window.
	if (wMsg == WM_SYSCOMMAND && (wParam & 0xFFF0) == SC_CLOSE &&
		!object->screenID && !object->parent &&
		!FlagSet(object->woAdvancedFlags, WOAF_NO_DESTROY))
		delete object;
	return (ccode);
}

static int _mdiChildOffset = -1;
static FARPROC _mdiChildCallback = (FARPROC)DefMDIChildProc;
static FARPROC _mdiChildJumpInstance = NULL;

long FAR PASCAL _export MDIChildJumpProcedure(HWND hWnd, WORD wMsg, WORD wParam, LONG lParam)
{
	UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)GetWindowLong(hWnd, _mdiChildOffset);
	return (object->Event(UI_EVENT(E_MSWINDOWS, hWnd, wMsg, wParam, lParam)));
}

EVENT_TYPE UIW_WINDOW::Event(const UI_EVENT &event)
{
	WOS_STATUS oldStatus = current ? Current()->woStatus : WOS_NO_STATUS;
	UI_WINDOW_OBJECT *object;

	// Switch on the event type.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);
	switch (ccode)
	{
	case S_INITIALIZE:
		if (!_windowJumpInstance)
			_windowJumpInstance = (FARPROC)WindowJumpProcedure;
		if (!_mdiFrameJumpInstance)
			_mdiFrameJumpInstance= (FARPROC)MDIFrameJumpProcedure;
		if (!_mdiChildJumpInstance)
			_mdiChildJumpInstance= (FARPROC)MDIChildJumpProcedure;
		UI_WINDOW_OBJECT::Event(event);
//		if (!parent && !screenID)
//		{
//			relative.right -= 2 * display->cellWidth;
//			relative.bottom -= 2 * display->cellHeight;
//			relative.right += 2 * GetSystemMetrics(SM_CXFRAME);
//			relative.bottom += 2 * GetSystemMetrics(SM_CYFRAME) + GetSystemMetrics(SM_CYCAPTION) + 2;
//		}
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Event(event);
		for (object = First(); object; object = object->Next())
		{
			if (FlagSet(object->woStatus, WOS_CURRENT) ||
				(!current && !FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
				 !FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT)))
			{
				if (current)
					Current()->woStatus &= ~WOS_CURRENT;
				if (current && FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					Current()->woStatus &= ~WOS_SELECTED;
				current = object;
				object->woStatus |= WOS_CURRENT;
				if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					object->woStatus |= WOS_SELECTED;
			}
			else
				object->woStatus &= ~WOS_CURRENT;
			object->Event(event);
		}
		break;

	case S_DEINITIALIZE:
		for (object = First(); object; object = object->Next())
		{
			object->screenID = 0;
			object->Event(event);
		}
		break;

	case S_CREATE:
		{
		clipList.Destroy();
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_MAXIMIZED))
			dwStyle |= WS_MAXIMIZE;
		else if (FlagSet(woStatus, WOS_MINIMIZED))
			dwStyle |= WS_MINIMIZE;
		if (parent && parent->Inherited(ID_LIST))
			SendMessage(parent->screenID, LB_ADDSTRING, 0, (LONG)this);
		else if (parent && parent->Inherited(ID_COMBO_BOX))
			SendMessage(parent->screenID, CB_ADDSTRING, 0, (LONG)this);
		else if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && !parent)
		{
			RegisterObject("UIW_MDIFRAME", NULL, &_mdiFrameOffset,
				&_mdiFrameJumpInstance, &_mdiFrameCallback,
				title ? (char *)title->Information(GET_TEXT, NULL) : NULL, menu);
			CLIENTCREATESTRUCT clientInfo;
			clientInfo.hWindowMenu = 0;
			clientInfo.idFirstChild = 0x1000;
			DWORD mdiStyle = WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE;
			mdiClient = CreateWindow("MDICLIENT", NULL, mdiStyle, 0, 0, 0, 0,
				screenID, 0,  display->hInstance, (LPSTR)&clientInfo);
		}
		else if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
			RegisterObject("UIW_MDICHILD", NULL, &_mdiChildOffset,
				&_mdiChildJumpInstance, &_mdiChildCallback,
				title ? (char *)title->Information(GET_TEXT, NULL) : NULL, menu);
		else
			RegisterObject("UIW_WINDOW", NULL, &_windowOffset,
				&_windowJumpInstance, &_windowCallback,
				title ? (char *)title->Information(GET_TEXT, NULL) : NULL, menu);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Event(event);
		for (object = First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, parent ? FALSE : TRUE);
		}
		}
		break;

	case S_SIZE:
		if (!screenID || FlagSet(woStatus, WOS_INTERNAL_ACTION))
			break;
		{
		if (parent)
			return (UI_WINDOW_OBJECT::Event(event));
		ccode = S_UNKNOWN;
		RECT region;
		GetWindowRect(screenID, &region); region.right--; region.bottom--;	// Fix Windows right/bottom "feature".
		true.Assign(region);
		if (!parent)
			relative = true;
		GetClientRect(screenID, &region); region.right--; region.bottom--;	// Fix Windows right/bottom "feature".
		clipList.Destroy();
		for (object = First(); object; object = object->Next())
		{
			UI_REGION region = object->true;
			object->Event(UI_EVENT(S_SIZE));
			if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, parent ? FALSE : TRUE);
			if (object->true != region)
			{
				WORD sizeFlags = FlagSet(woStatus, WOS_MINIMIZED) ? SWP_NOSIZE : 0;
				SetWindowPos(object->screenID, object->screenID,
					object->true.left, object->true.top,
					object->true.right - object->true.left + 1,
					object->true.bottom - object->true.top + 1,
					SWP_NOZORDER | SWP_NOREDRAW | sizeFlags);
				ccode = S_SIZE;
			}
		}
		if (mdiClient && clipList.First())
		{
			UI_REGION region = clipList.First()->region;
			SetWindowPos(mdiClient, mdiClient, region.left, region.top,
				region.right - region.left + 1, region.bottom - region.top + 1,
				SWP_NOZORDER | SWP_NOREDRAW);
		}
		if (ccode == S_SIZE)
		{
			SendMessage(screenID, WM_SETREDRAW, TRUE, 0);
			InvalidateRect(screenID, NULL, TRUE);
			UpdateWindow(screenID);
		}
		}
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
		woStatus |= WOS_WINDOWS_ACTION;
		UI_WINDOW_OBJECT::Event(event);
		if (current)
			ccode = Current()->Event(UI_EVENT(ccode));
		woStatus &= ~WOS_WINDOWS_ACTION;
		break;

	case S_REDISPLAY:
		ccode = UI_WINDOW_OBJECT::Event(event);
		break;

	case S_ADD_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (FlagSet(woStatus, WOS_WINDOWS_ACTION))
		{
			if (!FlagSet(object->woStatus, WOS_CURRENT))
				UIW_WINDOW::Add(object);
			if (!FlagSet(woStatus, WOS_CURRENT))
				UI_WINDOW_OBJECT::Event(event);
		}
		else if (UI_LIST::Index(object) == -1 && FlagSet(object->woStatus, WOS_EDIT_MODE))
		{
			UI_REGION region;
			if (clipList.First())
				region = clipList.First()->region;
			else
			{
				region.left = region.top = 0;
				region.right = relative.right - relative.left;
				region.bottom = relative.bottom - relative.top;
			}
			if (!FlagSet(object->woStatus, WOS_GRAPHICS))
			{
				region.left /= display->cellWidth;
				region.top /= display->cellHeight;
			}
			object->relative.left -= region.left;
			object->relative.top -= region.top;
			object->relative.right -= region.left;
			object->relative.bottom -= region.top;
			UIW_WINDOW::Add(object);
		}
		else
			UIW_WINDOW::Add(object);
		break;

	case S_SUBTRACT_OBJECT:
		Subtract((UI_WINDOW_OBJECT *)event.data);
		break;

	case L_HELP:
		// See if the low level object processes the help.
		if (event.type == E_MSWINDOWS)
			eventManager->Put(UI_EVENT(ccode));
		else if (current && Current()->Event(event) != S_UNKNOWN)
			break;
		else if (parent && helpContext == NO_HELP_CONTEXT)
			ccode = S_UNKNOWN;
		else if (helpSystem)
			helpSystem->DisplayHelp(windowManager, helpContext);
		break;

	case L_PREVIOUS:
		{
		EVENT_TYPE tcode = ccode;
		ccode = S_UNKNOWN;
		// Make sure there is a current object.
		if (!current)
			UIW_WINDOW::Add(Last());
		// See if the current object processes the message.
		else if (Current()->Event(event) != S_UNKNOWN)
			ccode = tcode;
		// Don't move if on a support object.
		else if (FlagSet(Current()->woFlags, WOF_SUPPORT_OBJECT))
			UI_ERROR_SYSTEM::Beep();
		// Don't move if on the first field of a non-wrapping window.
		else if (!Current()->Previous() && FlagSet(wnFlags, WNF_NO_WRAP))
			break;
		// Go to the previous field in the window.
		else
		{
			object = Current()->Previous() ? Current()->Previous() : Last();
			while (object && object != current && ccode == S_UNKNOWN)
				if (!FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
					!FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
				{
					UIW_WINDOW::Add(object);
					ccode = tcode;
				}
				else if (object->Previous())
					object = object->Previous();
				else if (FlagSet(wnFlags, WNF_NO_WRAP))
					break;
				else
					object = Last();
		}
		}
		break;

	case L_NEXT:
		{
		EVENT_TYPE tcode = ccode;
		ccode = S_UNKNOWN;
		// Make sure there is a current object.
		if (!current)
			UIW_WINDOW::Add(First());
		// See if the current object processes the message.
		else if (Current()->Event(event) != S_UNKNOWN)
			ccode = tcode;
		// Don't move if on a support object.
		else if (FlagSet(Current()->woFlags, WOF_SUPPORT_OBJECT))
			UI_ERROR_SYSTEM::Beep();
		// Don't move if on the last field of a non-wrapping window.
		else if (!Current()->Next() && FlagSet(wnFlags, WNF_NO_WRAP))
			break;
		// Go to the next field in the window.
		else
		{
			object = Current()->Next() ? Current()->Next() : First();
			while (object && object != current && ccode == S_UNKNOWN)
				if (!FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
					!FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
				{
					UIW_WINDOW::Add(object);
					ccode = tcode;
				}
				else if (object->Next())
					object = object->Next();
				else if (FlagSet(wnFlags, WNF_NO_WRAP))
					break;
				else
					object = First();
		}
		}
		break;

	// Default operations.
	default:
		if (ccode != E_MSWINDOWS && current)
			ccode = Current()->Event(event);
		else if (ccode != E_MSWINDOWS)
			ccode = UI_WINDOW_OBJECT::Event(event);
		else
			ccode = S_UNKNOWN;
		break;
	}
	if (ccode != S_UNKNOWN || event.type != E_MSWINDOWS)
		return (ccode);

	// Switch on the windows message.
	int callFunction = FALSE;
	WORD message = event.message.message;
	switch (message)
	{
	case WM_COMPAREITEM:
		{
		COMPAREITEMSTRUCT *compare = (COMPAREITEMSTRUCT *)event.message.lParam;
		UI_WINDOW_OBJECT *item1 = (UI_WINDOW_OBJECT *)compare->itemData1;
		UI_WINDOW_OBJECT *item2 = (UI_WINDOW_OBJECT *)compare->itemData2;
		if (item1->parent && item1->parent != this)
			return (item1->parent->Event(event));
		else if (compareFunction)
			ccode = (*compareFunction)((void *)item1, (void *)item2);
		else
			return (-1);
		}
		break;

	case WM_MEASUREITEM:
		{
		MEASUREITEMSTRUCT *measure = (MEASUREITEMSTRUCT *)event.message.lParam;
		for (object = First(); object; object = object->Next())
			if (FlagSet(object->woStatus, WOS_OWNERDRAW) &&
				GetWindowWord(object->screenID, GWW_ID) == measure->CtlID)
				break;
		if (object)
			ccode = object->Event(event);
		else
			callFunction = TRUE;
		}
		break;

 	case WM_DRAWITEM:
		{
		DRAWITEMSTRUCT *draw = (DRAWITEMSTRUCT *)event.message.lParam;
		if (draw->CtlType == ODT_BUTTON)
			ccode = SendMessage(draw->hwndItem, WM_DRAWITEM, 0, (LONG)draw);
		else if (draw->itemID != 0xFFFF)
		{
			object = (UI_WINDOW_OBJECT *)draw->itemData;
			ccode = object->Event(event);
		}
		}
		break;

	case WM_GETMINMAXINFO:
		if (parent && !FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
		{
			POINT *rgpt = (POINT *)event.message.lParam;
			rgpt[1].x = parent->true.right - parent->true.left + 1;
			rgpt[1].y = parent->true.bottom - parent->true.top + 1;
			rgpt[2].x = rgpt[2].y = 0;
		}
		else
			callFunction = TRUE;
		break;

	case WM_INITMENUPOPUP:
		// Check the system-button for invalid options.
		if (event.message.wParam != menu)
		{
			EnableMenuItem(event.message.wParam, SC_MOVE, MF_BYCOMMAND |
				(FlagSet(woAdvancedFlags, WOAF_NO_MOVE) ? MF_GRAYED : MF_ENABLED));
			EnableMenuItem(event.message.wParam, SC_SIZE, MF_BYCOMMAND |
				(FlagSet(woAdvancedFlags, WOAF_NO_SIZE) ? MF_GRAYED : MF_ENABLED));
			EnableMenuItem(event.message.wParam, SC_CLOSE, MF_BYCOMMAND |
				((parent || FlagSet(woAdvancedFlags, WOAF_LOCKED)) ? MF_GRAYED : MF_ENABLED));
		}
		else
			callFunction = TRUE;
		break;

	case WM_ERASEBKGND:
		if (FlagSet(woStatus, WOS_MINIMIZED) && icon)
		{
			HDC screenHDC = GetDC(ID_SCREEN);
			HDC objectHDC = event.message.wParam;
			SetBkColor(objectHDC, GetBkColor(screenHDC));
			SetBkMode(objectHDC, GetBkMode(screenHDC));
			UI_EVENT tEvent;
			tEvent = event;
			tEvent.message.message = WM_ICONERASEBKGND;
			if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && !parent)
				DefFrameProc(screenID, mdiClient, tEvent.message.message,
					tEvent.message.wParam, tEvent.message.lParam);
			else
				UI_WINDOW_OBJECT::Event(tEvent);
			ReleaseDC(ID_SCREEN, screenHDC);
			ccode = TRUE;
		}
		else
			callFunction = TRUE;
		break;

	case WM_PAINT:
	case WM_NCPAINT:
		if (FlagSet(woStatus, WOS_MINIMIZED) && icon)
		{
			PAINTSTRUCT ps;
			HDC objectHDC = BeginPaint(screenID, &ps);
			RECT region;
			GetClientRect(screenID, &region);
			region.left = (region.right - GetSystemMetrics(SM_CXICON)) / 2;
			region.top = (region.bottom - GetSystemMetrics(SM_CYICON)) / 2;
			DrawIcon(objectHDC, region.left, region.top, icon);
			EndPaint(screenID, &ps);
		}
		else
			callFunction = TRUE;
		break;

	case WM_QUERYDRAGICON:
		if (FlagSet(woStatus, WOS_MINIMIZED) && icon)
			ccode = icon;
		else
			callFunction = TRUE;
		break;

	case WM_SETFOCUS:
		if (current && Current()->screenID)
			SetFocus(Current()->screenID);
		else
			callFunction = TRUE;
		break;

	case WM_COMMAND:
		if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && !parent)
			callFunction = TRUE;
		else
		{
			object = (LOWORD(event.message.lParam) != 0) ? NULL :
				(UI_WINDOW_OBJECT *)Information(GET_NUMBERID_OBJECT, (void *)&event.message.wParam);
			if (object)
				ccode = object->Event(UI_EVENT(L_SELECT));
			else
				callFunction = TRUE;
		}
		break;

	case WM_SYSCOMMAND:
		{
		callFunction = TRUE;
		WORD command = event.message.wParam & 0xFFF0;
		if (command == SC_RESTORE)
		{
			woStatus &= ~(WOS_MINIMIZED | WOS_MAXIMIZED);
			woStatus |= WOS_REDISPLAY;
		}
		else if (command == SC_MAXIMIZE)
		{
			woStatus &= ~(WOS_MINIMIZED | WOS_MAXIMIZED);
			woStatus |= WOS_MAXIMIZED;
			woStatus |= WOS_REDISPLAY;
		}
		else if (command == SC_MINIMIZE)
		{
			woStatus &= ~(WOS_MINIMIZED | WOS_MAXIMIZED);
			woStatus |= WOS_MINIMIZED;
		}
		else if (command == SC_MOVE && FlagSet(woAdvancedFlags, WOAF_NO_MOVE))
			callFunction = FALSE;
		else if (command == SC_SIZE && FlagSet(woAdvancedFlags, WOAF_NO_SIZE))
			callFunction = FALSE;
		else if (command == SC_SIZE)
			woStatus |= WOS_REDISPLAY;
		else if (command == SC_CLOSE && screenID == windowManager->screenID)
		{
 			eventManager->Put(UI_EVENT(L_EXIT_FUNCTION));
			callFunction = FALSE;
		}
		else if (command == SC_CLOSE && FlagSet(woAdvancedFlags, WOAF_LOCKED))
			callFunction = FALSE;
		}
		break;

	case WM_MENUCHAR:
		callFunction = TRUE;
		for (object = First(); object && callFunction; object = object->Next())
			if (object->hotKey == HOT_KEY_SUB_WINDOW && object->Event(event) != S_UNKNOWN)
			{
				UIW_WINDOW::Add(object);
				return (ccode);
			}
			else if (object->hotKey == event.message.wParam && !FlagSet(object->woFlags, WOF_NON_SELECTABLE))
			{
				UIW_WINDOW::Add(object);
				object->Event(event);
				callFunction = FALSE;
				ccode = 0x00010000L;
				break;
			}
		break;

	case WM_MOVE:
		if (!parent && screenID)
		{
			RECT region;
			GetWindowRect(screenID, &region); region.right--; region.bottom--;	// Fix Windows right/bottom "feature".
			true.Assign(region);
		}
		break;

	case WM_SIZE:
		if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && !parent)
		{
			ccode = DefFrameProc(screenID, mdiClient, event.message.message,
				event.message.wParam, event.message.lParam);
			woStatus |= WOS_REDISPLAY;
		}
		else
			ccode = UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_REDISPLAY))
		{
			woStatus &= ~WOS_REDISPLAY;
			Event(UI_EVENT(S_SIZE));
		}
		break;

	case WM_LBUTTONDBLCLK:
	case WM_LBUTTONDOWN:
	case WM_RBUTTONDBLCLK:
	case WM_RBUTTONDOWN:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
		{
			for (object = First(); object; object = object->Next())
			{
				UI_REGION region = object->true;
				if (object->Inherited(ID_COMBO_BOX))
					region.bottom = region.top + display->cellHeight -
						display->preSpace - display->postSpace;
				if (region.Overlap(event.position))
				{
					UI_EVENT tEvent;
					tEvent = event;
					tEvent.position.column -= object->true.left;
					tEvent.position.line -= object->true.top;
					return (object->Event(tEvent));
				}
			}
		}
		callFunction = TRUE;
		break;

	default:
		callFunction = TRUE;
		break;
	}
	if (callFunction && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && !parent)
	{
		ccode = DefFrameProc(screenID, mdiClient, event.message.message,
			event.message.wParam, event.message.lParam);
		if (event.message.message == WM_NCDESTROY)
		{
			woStatus |= WOS_WINDOWS_ACTION;
			*windowManager - this;
			woStatus &= ~WOS_WINDOWS_ACTION;
		}
	}
	else if (callFunction)
		ccode = UI_WINDOW_OBJECT::Event(event);

	// Check the current status.
	if (current && !FlagSet(oldStatus, WOS_SELECTED) && FlagSet(Current()->woStatus, WOS_SELECTED))
		CheckSelection();

	// Return the control code.
	return (ccode);
}
#else
EVENT_TYPE UIW_WINDOW::Event(const UI_EVENT &event)
{
	UI_WINDOW_OBJECT *object;

	// Check for a minimized object.
	EVENT_TYPE ccode = LogicalEvent(event, ID_WINDOW);
	if (FlagSet(woStatus, WOS_MINIMIZED) && icon &&
		ccode != S_CREATE && ccode != L_MOVE && ccode != L_RESTORE && 
		ccode != L_MAXIMIZE && ccode != L_MINIMIZE && 
		(ccode != E_KEY || !FlagSet(event.key.shiftState, S_ALT)))
		return (icon->Event(event));

	// Switch on the event type.
	switch (ccode)
	{
	case S_INITIALIZE:
		UI_WINDOW_OBJECT::Event(event);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Event(event);
		for (object = First(); object; object = object->Next())
		{
			if (FlagSet(object->woStatus, WOS_CURRENT) ||
				(!current && !FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
				 !FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT)))
			{
				if (current)
					Current()->woStatus &= ~WOS_CURRENT;
				if (current && FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					Current()->woStatus &= ~WOS_SELECTED;
				current = object;
				object->woStatus |= WOS_CURRENT;
				if (FlagSet(wnFlags, WNF_AUTO_SELECT) && !FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					object->woStatus |= WOS_SELECTED;
			}
			else
				object->woStatus &= ~WOS_CURRENT;
			object->Event(event);
		}
		break;

	case S_SIZE:
	case S_CREATE:
		if (FlagSet(woStatus, WOS_MINIMIZED) && parent)
		{
			UI_WINDOW_OBJECT::Event(event);
			icon->relative = relative;
			icon->clip = clip;
			icon->true = true;
			icon->Event(event);
			break;
		}
		else if (FlagSet(woStatus, WOS_MINIMIZED))
		{
			true = clip = relative;
			icon->relative = icon->true = icon->clip = relative;
			icon->Event(event);
			break;
		}

		// Compute the object region.
		{
		clipList.Destroy();
		UI_WINDOW_OBJECT::Event(event);
		if (FlagSet(woStatus, WOS_MAXIMIZED) && parent && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
			true = clip;
		UI_REGION region = true;
		if (FlagSet(woFlags, WOF_BORDER) && (!parent || !display->isText ||
			FlagSet(woAdvancedFlags, WOAF_TEMPORARY)))
			--region;
		clipList.Add(new UI_REGION_ELEMENT(screenID, region));

		// Compute the support object regions.
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION))
			{
				UI_REGION region = object->true;
				clipList.Destroy();
				if (display->isText)
					clipList.Add(new UI_REGION_ELEMENT(screenID, region));
				else
					clipList.Add(new UI_REGION_ELEMENT(screenID, --region));
			}
			else if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, parent ? FALSE : TRUE);
		}
		// Compute the remaining object regions.
		for (object = First(); object; object = object->Next())
		{
			object->Event(event);
			if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
				clipList.Split(screenID, object->true, parent ? FALSE : TRUE);
		}
		}
		break;

	case S_RESET_DISPLAY:
		{
		UI_WINDOW_OBJECT::Event(event);

		// Convert each object's coordinates from graphics.
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Event(event);
		for (object = First(); object; object = object->Next())
			object->Event(event);
		}
		break;

	case S_REGION_DEFINE:
		{
		UI_REGION region = true;
		if (parent)
		{
			region.left = Max(region.left, clip.left);
			region.top = Max(region.top, clip.top);
			region.right = Min(region.right, clip.right);
			region.bottom = Min(region.bottom, clip.bottom);
		}
		if (display->isText)
		{
			for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
				if (FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION) &&
					!FlagSet(woStatus, WOS_MAXIMIZED))
				{
					if (region.right == true.right)
						region.right--;
					if (region.bottom == true.bottom)
						region.bottom--;
					break;
				}
		}
		display->RegionDefine(screenID, region);
		if (!parent && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
		{
			for (object = First(); object; object = object->Next())
				if (FlagSet(object->woAdvancedFlags, WOAF_MDI_OBJECT))
					object->Event(event);
		}
		}
		break;

	case L_MOVE:
	case L_SIZE:
		if (event.type != E_KEY && !event.rawCode && event.data != this)
			Current()->Event(event);
		else if ((ccode == L_MOVE && !FlagSet(woStatus, WOS_MAXIMIZED) &&
			(!FlagSet(woAdvancedFlags, WOAF_NO_MOVE) ||
			  FlagSet(woStatus, WOS_MINIMIZED))) ||
			(ccode == L_SIZE && !FlagSet(woStatus, WOS_MINIMIZED) &&
			!FlagSet(woAdvancedFlags, WOAF_NO_SIZE)))
		{
			UI_EVENT sEvent;
			sEvent = event;
			if (ccode != sEvent.type)
			{
				sEvent.type = ccode;
				sEvent.rawCode = 0;
			}
			Modify(sEvent);
		}
		break;

	case L_RESTORE:
	case L_MINIMIZE:
	case L_MAXIMIZE:
		if (event.type != E_KEY && !event.rawCode && event.data != this)
		{
			Current()->Event(event);
			break;
		}
		{
		UI_REGION region = true;
		if ((ccode == L_RESTORE && FlagSet(woStatus, WOS_MAXIMIZED | WOS_MINIMIZED)) ||
			(ccode == L_MINIMIZE && FlagSet(woStatus, WOS_MINIMIZED)) ||
			(ccode == L_MAXIMIZE && FlagSet(woStatus, WOS_MAXIMIZED)))
		{
			relative = restore;
			woStatus &= ~(WOS_MAXIMIZED | WOS_MINIMIZED);
		}
		else if (ccode == L_MINIMIZE && !FlagSet(woStatus, WOS_MAXIMIZED) && icon)
		{
			restore = relative;
			if (parent)
				relative = icon->relative;
			else
				true = relative = icon->true;
			woStatus |= WOS_MINIMIZED;
		}
		else if (ccode == L_MAXIMIZE && !FlagSet(woStatus, WOS_MINIMIZED))
		{
			restore = relative;
			relative.left = relative.top = 0;
			if (parent)
			{
				relative.right = parent->relative.right - parent->relative.left;
				relative.bottom = parent->relative.bottom - parent->relative.top;
			}
			else
			{
				relative.right = display->columns - 1;
				relative.bottom = display->lines - 1;
			}
			woStatus |= WOS_MAXIMIZED;
		}
		else
			return (S_UNKNOWN);
		Event(UI_EVENT(S_CREATE));
		eventManager->DeviceState(E_CURSOR, D_OFF);
		eventManager->DeviceState(E_MOUSE, DM_VIEW);
		if (!parent)
		{
			display->RegionDefine(ID_SCREEN, 0, 0, display->columns, display->lines);
			for (UI_WINDOW_OBJECT *object = windowManager->Last(); object; object = object->Previous())
				object->Event(UI_EVENT(S_REGION_DEFINE));
			display->Rectangle(ID_SCREEN, region, display->backgroundPalette, 0, TRUE);
			for (object = windowManager->Last(); object; object = object->Previous())
				if (object == this)
					object->Event(UI_EVENT(S_CURRENT, 0, true));
				else if (object->true.Overlap(region))
					object->Event(UI_EVENT(S_DISPLAY_INACTIVE, 0, region));
		}
		else
		{
			region.left = Min(region.left, true.left);
			region.top = Min(region.top, true.top);
			region.right = Max(region.right, true.right);
			region.bottom = Max(region.bottom, true.bottom);
			if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
				parent->Event(UI_EVENT(S_REGION_DEFINE));
			parent->Event(UI_EVENT(S_CURRENT, 0, region));
		}
		}
		break;

	case S_HSCROLL_CHECK:
	case S_VSCROLL_CHECK:
	case S_HSCROLL:
	case S_VSCROLL:
		{
		object = Current();
		if (!object)
			return (S_UNKNOWN);
		int hDelta = 0, vDelta = 0;
		UI_REGION region;
		if (clipList.First())
			region = clipList.First()->region;
		else
			region = true;
		if (ccode == S_VSCROLL_CHECK && object->true.bottom > region.bottom)
		{
			for (; object && object->true.bottom > region.bottom; object = object->Previous())
				vDelta += (object->true.top - object->true.bottom - 1);
		}
		else if (ccode == S_HSCROLL_CHECK && object->true.right > region.right)
		{
			int position = -1;
			for (; object && object->true.right > region.right; object = object->Previous())
				if (object->true.right != position)
				{
					hDelta += (object->true.left - object->true.right - 1);
					position = object->true.right;
				}
		}
		else if (ccode == S_VSCROLL_CHECK && object->true.top < region.top)
		{
			for (; object && object->true.top < region.top; object = object->Next())
				vDelta += (object->true.bottom - object->true.top + 1);
		}
		else if (ccode == S_HSCROLL_CHECK && object->true.left < region.left)
		{
			int position = -1;
			for (; object && object->true.left < region.left; object = object->Next())
				if (object->true.left != position)
				{
					hDelta += (object->true.right - object->true.left + 1);
					position = object->true.left;
				}
		}
		else if (ccode == S_HSCROLL && hScroll)
		{
			hScroll->Event(event);
			hDelta = -event.scroll.delta;
		}
		else if (ccode == S_VSCROLL && vScroll)
		{
			vScroll->Event(event);
			vDelta = -event.scroll.delta;
		}

		// Make sure we need to scroll the window objects.
		if (hDelta && First()->true.left + hDelta > region.left)
			hDelta = region.left - First()->true.left;
		else if (hDelta && Last()->true.left + hDelta < region.left)
			hDelta = region.left - Last()->true.left;
		if (vDelta && First()->true.top + vDelta > region.top)
			vDelta = region.top - First()->true.top;
		else if (vDelta && Last()->true.top + vDelta < region.top)
			vDelta = region.top - Last()->true.top;
		if (vDelta == 0 && hDelta == 0)
			return (S_UNKNOWN);

		// Update the window object positions.
		for (object = First(); object; object = object->Next())
			if (!FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
			{
				object->true.left += hDelta;
				object->true.top += vDelta;
				object->true.right += hDelta;
				object->true.bottom += vDelta;
			}
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
			break;

		// Update the screen information.
		if (true.top < 0 || true.left < 0 ||
			true.right >= display->columns || true.bottom >= display->lines)
			region = true;
		else if ((vDelta < 0 && Abs(vDelta) < region.bottom - region.top + 1) ||
			(hDelta < 0 && Abs(hDelta) < region.right - region.left + 1))
		{
			region.left -= hDelta;
			region.top -= vDelta;
			display->RegionMove(region, region.left + hDelta, region.top + vDelta);
			if (hDelta < 0)
				region.left = region.right + hDelta;
			else if (vDelta < 0)
				region.top = region.bottom + vDelta;
		}
		else if ((vDelta > 0 && vDelta < region.bottom - region.top + 1) ||
			(hDelta > 0 && hDelta < region.right - region.left + 1))
		{
			region.right -= hDelta;
			region.bottom -= vDelta;
			display->RegionMove(region, region.left + hDelta, region.top + vDelta);
			if (hDelta > 0)
				region.right = region.left + hDelta;
			else if (vDelta > 0)
				region.bottom = region.top + vDelta;
		}

		// Update the window according to the scrolled region.
		UIW_WINDOW::Event(UI_EVENT(S_CURRENT, 0, region));
		}
		break;

	case S_CURRENT:
	case S_NON_CURRENT:
	case S_DISPLAY_ACTIVE:
	case S_DISPLAY_INACTIVE:
		if (!screenID)
			break;
		{
		UI_WINDOW_OBJECT::Event(event);
		if (ccode == S_CURRENT)
			eventManager->DeviceState(E_CURSOR, D_OFF);

		int virtualGet = FALSE;
		UI_REGION region = true;
		if (display->isText || (parent && true.Overlap(event.region)))
		{
			display->VirtualGet(screenID, true);
			virtualGet = TRUE;
		}

		// Draw the window's border and fill the background.
		if (FlagSet(woFlags, WOF_BORDER) && true.Overlap(event.region))
			DrawBorder(screenID, region, FALSE, ccode);
		if (!support.First() && !First())
		{
			region = true;
			if (FlagSet(woFlags, WOF_BORDER) && (!display->isText || FlagSet(woAdvancedFlags, WOAF_TEMPORARY)))
				--region;
			clipList.Add(new UI_REGION_ELEMENT(screenID, region));
		}
		UI_REGION tRegion;
		for (UI_REGION_ELEMENT *element = clipList.First(); element; element = element->Next())
			if (element->region.Overlap(event.region, tRegion))
				display->Rectangle(screenID, tRegion, lastPalette, 0, TRUE, FALSE, &clip);

		// Update the children.
		UI_EVENT cEvent, aEvent;
		cEvent = event;
		aEvent = event;
		if (ccode == S_CURRENT)
			aEvent.type = S_DISPLAY_ACTIVE;
		else if (ccode == S_NON_CURRENT)
			aEvent.type = S_DISPLAY_INACTIVE;
		else if (ccode == S_DISPLAY_ACTIVE && FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT) && parent)
			cEvent.type = aEvent.type = S_DISPLAY_INACTIVE;
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			if (object == current || FlagSet(object->woStatus, WOS_CURRENT))
				object->Event(cEvent);
			else
				object->Event(aEvent);
		for (object = First(); object; object = object->Next())
		{
			if (object == current)
			{
				if (!FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					object->woStatus |= WOS_CURRENT;
				object->Event(cEvent);
			}
			else if (object->true.Overlap(event.region))
			{
				if (!FlagSet(wnFlags, WNF_SELECT_MULTIPLE))
					object->woStatus &= ~WOS_CURRENT;
				object->Event(aEvent);
			}
		}
		if (virtualGet)
			display->VirtualPut(screenID);
		woStatus &= ~WOS_REDISPLAY;
		}
		break;

	case S_REDISPLAY:
		UI_WINDOW_OBJECT::Event(event);
		break;

	case S_ADD_OBJECT:
		object = (UI_WINDOW_OBJECT *)event.data;
		if (UI_LIST::Index(object) == -1 && FlagSet(object->woStatus, WOS_EDIT_MODE))
		{
			UI_REGION region;
			if (clipList.First())
				region = clipList.First()->region;
			else
				region = true;
			if (!FlagSet(object->woStatus, WOS_GRAPHICS))
			{
				region.left /= display->cellWidth;
				region.top /= display->cellHeight;
			}
			object->relative.left -= region.left;
			object->relative.top -= region.top;
			object->relative.right -= region.left;
			object->relative.bottom -= region.top;
		}
		UIW_WINDOW::Add(object);
		CheckSelection();
		break;

	case S_SUBTRACT_OBJECT:
		// Subtract the window object from the window.
		Subtract((UI_WINDOW_OBJECT *)event.data);
		break;

	case L_HELP:
		// See if the low level object processes the help.
		if (current && Current()->Event(event) != S_UNKNOWN)
			break;
		else if (parent && helpContext == NO_HELP_CONTEXT)
			ccode = S_UNKNOWN;
		else if (helpSystem)
			helpSystem->DisplayHelp(windowManager, helpContext);
		break;

	case L_PREVIOUS:
		{
		EVENT_TYPE tcode = ccode;
		ccode = S_UNKNOWN;
		// Make sure there is a current object.
		if (!current)
			UIW_WINDOW::Add(Last());
		// See if the current object processes the message.
		else if (Current()->Event(event) != S_UNKNOWN)
			ccode = tcode;
		// Don't move if on a support object.
		else if (FlagSet(Current()->woFlags, WOF_SUPPORT_OBJECT))
			UI_ERROR_SYSTEM::Beep();
		// Don't move if on the first field of a non-wrapping window.
		else if (!Current()->Previous() && FlagSet(wnFlags, WNF_NO_WRAP))
			break;
		// Go to the previous field in the window.
		else
		{
			object = Current()->Previous() ? Current()->Previous() : Last();
			while (object && object != current && ccode == S_UNKNOWN)
				if (!FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
					!FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
				{
					UIW_WINDOW::Add(object);
					ccode = tcode;
				}
				else if (object->Previous())
					object = object->Previous();
				else if (FlagSet(wnFlags, WNF_NO_WRAP))
					break;
				else
					object = Last();
		}
		}
		break;

	case L_NEXT:
		{
		EVENT_TYPE tcode = ccode;
		ccode = S_UNKNOWN;
		// Make sure there is a current object.
		if (!current)
			UIW_WINDOW::Add(First());
		// See if the current object processes the message.
		else if (Current()->Event(event) != S_UNKNOWN)
			ccode = tcode;
		// Don't move if on a support object.
		else if (FlagSet(Current()->woFlags, WOF_SUPPORT_OBJECT))
			UI_ERROR_SYSTEM::Beep();
		// Don't move if on the last field of a non-wrapping window.
		else if (!Current()->Next() && FlagSet(wnFlags, WNF_NO_WRAP))
			break;
		// Go to the next field in the window.
		else
		{
			object = Current()->Next() ? Current()->Next() : First();
			while (object && object != current && ccode == S_UNKNOWN)
				if (!FlagSet(object->woFlags, WOF_NON_SELECTABLE) &&
					!FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
				{
					UIW_WINDOW::Add(object);
					ccode = tcode;
				}
				else if (object->Next())
					object = object->Next();
				else if (FlagSet(wnFlags, WNF_NO_WRAP))
					break;
				else
					object = First();
		}
		}
		break;

	case L_CANCEL:
	case S_ALT_KEY:
		// Bring up or take down the pull-down menu (if any).
		for (object = (UI_WINDOW_OBJECT *)support.Last(); object; object = object->Previous())
			if (object->Inherited(ID_PULL_DOWN_MENU))
			{
				// Restore the old selected window object.
				if (FlagSet(object->woStatus, WOS_CURRENT))
					UIW_WINDOW::Add((UI_WINDOW_OBJECT *)object->userObject);
				// Activate the pull-down menu.
				else if (ccode == S_ALT_KEY)
				{
					object->userObject = current;
					UIW_WINDOW::Add(object);
				}
				return (ccode);
			}
			else if (ccode == S_ALT_KEY && object->Inherited(ID_SYSTEM_BUTTON))
			{
				object->Event(UI_EVENT(L_SELECT));
				return (ccode);
			}
		ccode = S_UNKNOWN;
		break;

	case L_BEGIN_ESCAPE:
	case L_CONTINUE_ESCAPE:
	case L_END_ESCAPE:
		if (FlagSet(woStatus, WOS_EDIT_MODE))
			return (UI_WINDOW_OBJECT::Event(event));
		// Continue to E_MOUSE.
	case E_MOUSE:
	case L_VIEW:
	case L_BEGIN_SELECT:
	case L_CONTINUE_SELECT:
	case L_END_SELECT:
		// Check for mouse out-of-range.
		if (!true.Overlap(event.position))
		{
			if (current)
				return (Current()->Event(event));
			eventManager->DeviceState(E_MOUSE, DM_VIEW);
			return (S_UNKNOWN);
		}

		// Check for an active support object.
		object = (UI_WINDOW_OBJECT *)support.Current();
		if (object && !object->true.Overlap(event.position))
		{
			if (!FlagSet(woStatus, WOS_EDIT_MODE))
				object->Event(event);
			support.SetCurrent(NULL);
		}

		// Let object unactivate its region.
		if ((ccode == L_END_SELECT || ccode == L_END_ESCAPE) &&
			current && !Current()->true.Overlap(event.position))
		{
			Current()->Event(event);
			if (FlagSet(Current()->woAdvancedFlags, WOAF_NON_CURRENT))
			{
				for (object = First(); object; object = object->Next())
					if (FlagSet(object->woStatus, WOS_CURRENT))
					{
						current = object;
						break;
					}
			}
		}

		// See if the message should go to the current object.
		if (ccode == L_CONTINUE_SELECT && FlagSet(wnFlags, WNF_CONTINUE_SELECT))
			;
		else if (current && event.rawCode != 0 &&
			!FlagSet(event.rawCode, M_LEFT_CHANGE | M_MIDDLE_CHANGE | M_RIGHT_CHANGE))
			return (Current()->Event(event));

		object = support.Last() ? (UI_WINDOW_OBJECT *)support.Last() : Last();
		while (object)
		{
			if ((!FlagSet(object->woFlags, WOF_NON_SELECTABLE) || FlagSet(woStatus, WOS_EDIT_MODE)) && 
				((FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION) &&
				  true.Overlap(event.position) && !object->true.Overlap(event.position)) ||
				 (!FlagSet(object->woAdvancedFlags, WOAF_OUTSIDE_REGION) &&
				  object->true.Overlap(event.position))))
			{
				if (FlagSet(object->woAdvancedFlags, WOAF_NON_CURRENT))
				{
					if (FlagSet(object->woFlags, WOF_SUPPORT_OBJECT))
						;
					else if (ccode == L_BEGIN_SELECT)
						current = object;
					else if (ccode == L_END_SELECT)
					{
						for (UI_WINDOW_OBJECT *tObject = First(); tObject; tObject = tObject->Next())
							if (FlagSet(tObject->woStatus, WOS_CURRENT))
						{
							current = tObject;
							break;
						}
					}
				}
				else if (ccode == L_BEGIN_SELECT || (ccode == L_CONTINUE_SELECT &&
					FlagSet(wnFlags, WNF_CONTINUE_SELECT)))
				{
					if (object != current && FlagSet(object->woFlags, WOF_SUPPORT_OBJECT))
						object->userObject = current;
					UIW_WINDOW::Add(object);
				}
				if (event.rawCode && FlagSet(object->woFlags, WOF_SUPPORT_OBJECT))
				{
					support.SetCurrent(object);
					object->Event(event);
					if (FlagSet(woStatus, WOS_EDIT_MODE))
						UI_WINDOW_OBJECT::Event(event);
				}
				else
					object->Event(event);
				if (ccode == L_END_SELECT)
					CheckSelection();
				return (ccode);
			}
			// Get the next object.
			object = (!object->Previous() && FlagSet(object->woFlags, WOF_SUPPORT_OBJECT)) ?
				Last() : object->Previous();
		}

		// Object does not overlap any regions.
		woStatus &= ~WOS_AUTO_MARK;
		eventManager->DeviceState(E_MOUSE, DM_VIEW);
		UI_WINDOW_OBJECT::Event(event);
		break;

	default:
		if (ccode == E_KEY && (FlagSet(event.key.shiftState, S_ALT) ||
			FlagSet(woAdvancedFlags, WOAF_NORMAL_HOT_KEYS)))
		{
			UCHAR tHotKey = FlagSet(woAdvancedFlags, WOAF_NORMAL_HOT_KEYS) ?
				event.key.value : UI_EVENT_MAP::MapEvent(hotKeyMapTable, event);
			tHotKey = tolower(tHotKey);
			for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
				if (object->hotKey == HOT_KEY_SUB_WINDOW && object->Event(event) != S_UNKNOWN)
				{
					object->userObject = current;
					UIW_WINDOW::Add(object);
					return (ccode);
				}
				else if (object->hotKey == tHotKey && !FlagSet(object->woFlags, WOF_NON_SELECTABLE))
				{
					ccode = object->Event(UI_EVENT(L_SELECT));
					CheckSelection();
					return (ccode);
				}
			for (object = First(); object; object = object->Next())
				if (object->hotKey == HOT_KEY_SUB_WINDOW && object->Event(event) != S_UNKNOWN)
				{
					UIW_WINDOW::Add(object);
					return (ccode);
				}
				else if (object->hotKey == tHotKey && !FlagSet(object->woFlags, WOF_NON_SELECTABLE))
				{
					UIW_WINDOW::Add(object);
					ccode = object->Event(UI_EVENT(L_SELECT));
					CheckSelection();
					return (ccode);
				}
			ccode = S_UNKNOWN;
		}
		else
		{
			EVENT_TYPE code = ccode;
			ccode = current ? Current()->Event(event) : S_UNKNOWN;
			if (code == L_SELECT)
				CheckSelection();
		}
		break;
	}

	// Return the control code.
	return (ccode);
}
#endif

void *UIW_WINDOW::Information(INFO_REQUEST request, void *data, OBJECTID objectID)
{
	UI_WINDOW_OBJECT *object;

	// Switch on the request.
	if (!objectID) objectID = ID_WINDOW;
	switch (request)
	{
	case INITIALIZE_CLASS:
		searchID = windowID[0] = ID_WINDOW;
#ifdef _WINDOWS
		hScroll = vScroll = title = NULL;
		title = NULL;
		mdiClient = menu = icon = 0;
		mdiCallback = NULL;
#else
		vScroll = hScroll = icon = NULL;
#endif
		// Continue to CHANGED_FLAGS.
	case CHANGED_FLAGS:
		if (FlagSet(wnFlags, WNF_AUTO_SORT))
			compareFunction = &UIW_WINDOW::StringCompare;
#ifdef _WINDOWS
		if (request == CHANGED_FLAGS)
			UI_WINDOW_OBJECT::Information(CHANGED_FLAGS, data, ID_WINDOW);
		if (FlagSet(woAdvancedFlags, WOAF_MDI_OBJECT))
			dwStyle |= WS_CLIPCHILDREN;
		if (screenID)
		{
			if (!parent)
				dwStyle |= WS_CLIPSIBLINGS;
			menu = 0, icon = 0, title = NULL;
			for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
				object->Information(request, data, objectID);
			if (request != CHANGED_FLAGS || objectID != ID_WINDOW)
				break;
			else if (parent)
			{
				DestroyWindow(screenID);
				Event(UI_EVENT(S_CREATE));
				SetWindowPos(screenID, previous ? Previous()->screenID : 0,
					0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_NOREDRAW);
			}
			else
			{
				SetWindowLong(screenID, GWL_STYLE, dwStyle);
				if (menu)
					SetMenu(screenID, menu);
				Event(UI_EVENT(S_SIZE));
			}
		}
#else
		if (screenID && request == CHANGED_FLAGS && objectID == ID_WINDOW)
		{
			UI_EVENT event(S_INITIALIZE);
			Event(event);
			event.type = S_SIZE;
			Event(event);
		}
#endif
		break;

	case CHECK_SELECTION:
		CheckSelection();
		break;

	case GET_NUMBERID_OBJECT:
	case GET_STRINGID_OBJECT:
		{
		void *match = UI_WINDOW_OBJECT::Information(request, data, objectID);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object && !match; object = object->Next())
			match = object->Information(request, data, objectID);
		for (object = First(); object && !match; object = object->Next())
			match = object->Information(request, data, objectID);
		data = match;
		}
		break;

	case GET_TEXT:
	case SET_TEXT:
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			if (object->Inherited(ID_TITLE))
				return (object->Information(request, data, objectID));
		break;

	case GET_FLAGS:
	case SET_FLAGS:
	case CLEAR_FLAGS:
		if (objectID && objectID != ID_WINDOW)
			data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		else if (request == GET_FLAGS && !data)
			data = &wnFlags;
		else if (request == GET_FLAGS)
			*(WNF_FLAGS *)data = wnFlags;
		else if (request == SET_FLAGS)
			wnFlags |= *(WNF_FLAGS *)data;
		else
			wnFlags &= ~(*(WNF_FLAGS *)data);
		break;

	case SET_VSCROLL:	vScroll = (UI_WINDOW_OBJECT *)data;		break;
	case SET_HSCROLL:	hScroll = (UI_WINDOW_OBJECT *)data;		break;
#ifdef _WINDOWS
	case GET_MDIHANDLE:	*(SCREENID *)data = mdiClient;			break;
	case SET_TITLE:		title = (UI_WINDOW_OBJECT *)data;		break;
	case SET_MENU:		menu = *(HMENU *)data;					break;
	case SET_ICON:		icon = *(HICON *)data;					break;
#else
	case SET_ICON:		icon = (UI_WINDOW_OBJECT *)data;		break;
#endif

	case PRINT_INFORMATION:
	case PRINT_USER_FUNCTION:
	case PRINT_COMPARE_FUNCTION:
		if (request == PRINT_COMPARE_FUNCTION && compareFunctionName)
		{
			UI_STORAGE_OBJECT *cppEntry = (UI_STORAGE_OBJECT *)data;
			cppEntry->Store(compareFunctionName);
		}
		else if (request != PRINT_COMPARE_FUNCTION)
			UI_WINDOW_OBJECT::Information(request, data, objectID);
		for (object = First(); object; object = object->Next())
			object->Information(request, data, objectID);
		break;

	default:
		data = UI_WINDOW_OBJECT::Information(request, data, objectID);
		break;
	}

	// Return the information.
	return (data);
}

void UIW_WINDOW::RegionMax(UI_WINDOW_OBJECT *object)
{
	// Check to see if it is the top level object.
	if (object == this && (!parent || FlagSet(woAdvancedFlags, WOAF_TEMPORARY)))
	{
#ifdef _WINDOWS
		true = relative;
#else
		clip = true = relative;
#endif
		clipList.Destroy();
		return;
	}

	// Find the appropriate region element.
	UI_REGION_ELEMENT *element = clipList.First();
	UI_REGION region;
	if (element)						// Get first available region.
		region = element->region;
	else								// Determine the update region.
	{
#ifdef _WINDOWS
		RECT tRegion;
		GetClientRect(screenID, &tRegion); tRegion.right--; tRegion.bottom--;	// Fix Windows right/bottom "feature".
		region.Assign(tRegion);
#else
		region = true;
		if (FlagSet(woFlags, WOF_BORDER) && (!display->isText || FlagSet(woAdvancedFlags, WOAF_TEMPORARY)))
			--region;
#endif
		clipList.Add(new UI_REGION_ELEMENT(screenID, region));
	}
	int bottomRegion = FlagsSet(object->woFlags, WOF_NON_FIELD_REGION | WOF_JUSTIFY_RIGHT);
	for ( ; element; element = element->Next())
		if (!bottomRegion && region.top > element->region.top)
			region = element->region;
		else if (bottomRegion && region.bottom < element->region.bottom)
			region = element->region;

	// Compute the actual field region.
#ifdef _WINDOWS
	object->true = region;
#else
	object->clip = object->true = region;
	if (object->clip.left < object->parent->clip.left)
		object->clip.left = object->parent->clip.left;
	if (object->clip.top < object->parent->clip.top)
		object->clip.top = object->parent->clip.top;
	if (object->clip.right > object->parent->clip.right)
		object->clip.right = object->parent->clip.right;
	if (object->clip.bottom > object->parent->clip.bottom)
		object->clip.bottom = object->parent->clip.bottom;
#endif
	if (FlagSet(object->woFlags, WOF_NON_FIELD_REGION))
	{
		object->relative.right = object->relative.right - object->relative.left;
		object->relative.bottom = object->relative.bottom - object->relative.top;
		object->relative.left = object->relative.top = 0;
	}
	else
	{
		object->true.left = region.left + object->relative.left;
		object->true.top = region.top + object->relative.top;
		object->true.right = region.left + object->relative.right;
		object->true.bottom = region.top + object->relative.bottom;
	}
}

#ifdef _WINDOWS
UI_WINDOW_OBJECT *UIW_WINDOW::Subtract(UI_WINDOW_OBJECT *object)
{
	UI_WINDOW_OBJECT *nextObject = object ? object->Next() : NULL;

	// Remove the object from the list.
	if (!object)
		return (nextObject);
	else if (screenID)
		SendMessage(screenID, WM_SETREDRAW, FALSE, 0);
	if (UI_LIST::Index(object) != -1)
	{
		UI_EVENT event;
		if (FlagSet(woStatus, WOS_INTERNAL_ACTION))
			;
		else if (object->screenID)
		{
			woStatus |= WOS_INTERNAL_ACTION;
			DestroyWindow(object->screenID);
			event.type = S_DEINITIALIZE;
			object->screenID = 0;
			object->Event(event);
			woStatus &= ~WOS_INTERNAL_ACTION;
		}
		else
		{
			woStatus |= WOS_INTERNAL_ACTION;
			event.type = S_SUBTRACT_OBJECT;
			event.data = object;
			Event(event);
			woStatus &= ~WOS_INTERNAL_ACTION;
		}
		if (UI_LIST::Index(object) != -1)	// Make sure the object is still in the list.
			UI_LIST::Subtract(object);
	}
	else if (support.Index(object) != -1)
	{
		support.Subtract(object);
		if (object == current)	// Check for a current support object.
			current = First();
		if (object == vScroll)
			vScroll = NULL;
		else if (object == hScroll)
			hScroll = NULL;
		else if (object == title)
			title = NULL;
	}

	// Return a pointer to the next object.
	return (nextObject);
}
#else
UI_WINDOW_OBJECT *UIW_WINDOW::Subtract(UI_WINDOW_OBJECT *object)
{
	UI_WINDOW_OBJECT *nextObject = object ? object->Next() : NULL;

	// Remove the object from the list.
	if (!object)
		return (nextObject);
	else if (UI_LIST::Index(object) != -1)
	{
		UI_LIST::Subtract(object);
		object->Event(UI_EVENT(S_DEINITIALIZE));
	}
	else if (support.Index(object) != -1)
	{
		support.Subtract(object);
		if (object == current)	// Check for a current support object.
			current = First();
		if (object == vScroll)
			vScroll = NULL;
		else if (object == hScroll)
			hScroll = NULL;
		else if (object == icon)
			icon = NULL;
	}
	object->parent = NULL;

	// Return a pointer to the next object.
	return (nextObject);
}
#endif

int UIW_WINDOW::StringCompare(void *object1, void *object2)
{
	char *string1, *string2;
	((UI_WINDOW_OBJECT *)object1)->Information(GET_TEXT, &string1);
	if (!string1)
		return (-1);
	((UI_WINDOW_OBJECT *)object2)->Information(GET_TEXT, &string2);
	if (!string2)
		return (1);
	return (strcmp(string1, string2));
}

// ----- ZIL_PERSISTENCE ----------------------------------------------------

#ifdef ZIL_PERSISTENCE
UIW_WINDOW::UIW_WINDOW(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file) :
	UI_WINDOW_OBJECT(0, 0, 40, 10, WOF_NO_FLAGS, WOAF_NO_FLAGS),
	UI_LIST(), support(), clipList()
{
	// Initialize the window information.
	if (name && !file)
	{
		long miniNumeratorX = display->miniNumeratorX;
		long miniDenominatorX = display->miniDenominatorX;
		long miniNumeratorY = display->miniNumeratorY;
		long miniDenominatorY = display->miniDenominatorY;
		Load(name, directory, file);
		UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
		if (FlagSet(woFlags, WOF_MINICELL))
			Event(UI_EVENT(S_INITIALIZE));
		display->miniNumeratorX = miniNumeratorX;
		display->miniDenominatorX = miniDenominatorX;
		display->miniNumeratorY = miniNumeratorY;
		display->miniDenominatorY = miniDenominatorY;
	}
	else
	{
		Load(name, directory, file);
		UI_WINDOW_OBJECT::Information(INITIALIZE_CLASS, NULL);
		UIW_WINDOW::Information(INITIALIZE_CLASS, NULL);
	}
}

void UIW_WINDOW::Load(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Check for a valid directory and file.
	int tempDirectory = FALSE, tempFile = FALSE;
	if (name && !file)
	{
		char pathName[128], fileName[32], objectName[32], objectPathName[128];
		UI_STORAGE::StripFullPath(name, pathName, fileName, objectName, objectPathName);
		if (!directory)
		{
			tempDirectory = TRUE;
			UI_STORAGE::AppendFullPath(pathName, pathName, fileName);
			UI_STORAGE::ChangeExtension(pathName, ".DAT");
			directory = new UI_STORAGE(pathName, UIS_READ);
			if (directory->storageError)
			{
				UIW_WINDOW::Error(WOS_READ_ERROR, "File %s was not found.", fileName);
				goto LOAD_ERROR;
			}
		}
		if (!file)
		{
			tempFile = TRUE;
			if (objectPathName[0] != '\0' && !directory->ChDir(objectPathName))
				;
			else
				directory->ChDir("~UIW_WINDOW");
			if (objectName[0] == '\0')
				strcpy(objectName, fileName);
			file = new UI_STORAGE_OBJECT(*directory, objectName, ID_WINDOW, UIS_READ);
			if (file->objectError)
			{
				UIW_WINDOW::Error(WOS_READ_ERROR, "Object %s was not found.", objectName);
				goto LOAD_ERROR;
			}
			file->Load(&display->miniNumeratorX);	// miniNumeratorX
			file->Load(&display->miniDenominatorX);	// miniDenominatorX
			file->Load(&display->miniNumeratorY);	// miniNumeratorY
			file->Load(&display->miniDenominatorY);	// miniDenominatorY
		}
	}

	// Load the window information.
	UI_WINDOW_OBJECT::Load(name, directory, file);

	// Load the object information.
	{
 	short noOfObjects;
	file->Load(&noOfObjects);
	for (int i = 0; i < noOfObjects; i++)
	{
		UI_WINDOW_OBJECT *object = UI_WINDOW_OBJECT::New(NULL, directory, file);
		if (!object)
		{
			UIW_WINDOW::Error(WOS_READ_ERROR, "Unknown window object.");
			goto LOAD_ERROR;
		}
		Add(object);
	}
	file->Load(&wnFlags);
	file->Load(&compareFunctionName);
	}

LOAD_ERROR:
	// Clean up the file and storage.
	if (tempFile)
		delete file;
	if (tempDirectory)
		delete directory;
}

void UIW_WINDOW::Store(const char *name, UI_STORAGE *directory, UI_STORAGE_OBJECT *file)
{
	// Check for a valid directory and file.
	int tempDirectory = FALSE, tempFile = FALSE;
	if (name && !file)
	{
		char pathName[128], fileName[32], objectName[32], objectPathName[128];
		UI_STORAGE::StripFullPath(name, pathName, fileName, objectName, objectPathName);
		if (!directory)
		{
			UI_STORAGE::AppendFullPath(pathName, pathName, fileName);
			UI_STORAGE::ChangeExtension(pathName, ".DAT");
			directory = new UI_STORAGE(pathName, UIS_OPENCREATE | UIS_READWRITE);
			if (directory->storageError)
			{
				delete directory;
				return;
			}
			tempDirectory = TRUE;
		}
		if (!file)
		{
			if (objectPathName[0] != '\0' && !directory->ChDir(objectPathName))
				;
			else if (directory->ChDir("~UIW_WINDOW"))
			{
				directory->ChDir("~");
				directory->MkDir("UIW_WINDOW");
				directory->ChDir("~UIW_WINDOW");
			}
			if (objectName[0] == '\0')
				strcpy(objectName, fileName);
			StringID(objectName);
			file = new UI_STORAGE_OBJECT(*directory, objectName, ID_WINDOW, UIS_CREATE | UIS_READWRITE);
			file->Store(display->miniNumeratorX);	// miniNumeratorX
			file->Store(display->miniDenominatorX);	// miniDenominatorX
			file->Store(display->miniNumeratorY);	// miniNumeratorY
			file->Store(display->miniDenominatorY);	// miniDenominatorY
			tempFile = TRUE;

			// Reset the object table status.
			if (objectTable)
			{
				for (int i = 0; objectTable[i].data; i++)
					objectTable[i].flags = FALSE;
			}
		}
	}

	// Store the object information.
	UI_WINDOW_OBJECT::Store(name, directory, file);
	short noOfObjects = support.Count() + UI_LIST::Count();
	file->Store(noOfObjects);
	for (UI_WINDOW_OBJECT *object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
	{
		file->Store(object->searchID);
		object->Store(NULL, directory, file);
	}
	for (object = First(); object; object = object->Next())
	{
		file->Store(object->searchID);
		object->Store(NULL, directory, file);
	}
	file->Store(wnFlags);
	file->Store(compareFunctionName);

	// Write out the header information.
	if (!parent && stringID[0] != '\0')
	{
		if (directory->ChDir("~UI_HPP"))
		{
			directory->ChDir("~");
			directory->MkDir("UI_HPP");
			directory->ChDir("~UI_HPP");
		}
		UI_STORAGE_OBJECT hppEntry(*directory, stringID, ID_WINDOW, UIS_CREATE | UIS_READWRITE);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Information(PRINT_INFORMATION, &hppEntry);
		for (object = First(); object; object = object->Next())
			object->Information(PRINT_INFORMATION, &hppEntry);
		hppEntry.Store("");	// End of HPP string entries.
	}

	// Clean up the file and storage.
	if (tempFile)
	{
		delete file;

		// Create the CPP entries.
		if (directory->ChDir("~UI_CPP"))
		{
			directory->ChDir("~");
			directory->MkDir("UI_CPP");
			directory->ChDir("~UI_CPP");
		}
		directory->MkDir(stringID);
		directory->ChDir(stringID);

		if (objectTable)
		{
			OBJECTID objectID;
			UI_STORAGE_OBJECT cppEntry(*directory, "OBJECTID", 0, UIS_CREATE | UIS_READWRITE);
			for (int i = 0; objectTable[i].data; i++)
				if (objectTable[i].flags)
				{
					objectID = objectTable[i].value;
					cppEntry.Store(objectID);
				}
				objectID = 0; cppEntry.Store(objectID);	// End of CPP object entries.
		}

		UI_STORAGE_OBJECT userEntry(*directory, "USER_FUNCTION", 0, UIS_CREATE | UIS_READWRITE);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Information(PRINT_USER_FUNCTION, &userEntry);
		for (object = First(); object; object = object->Next())
			object->Information(PRINT_USER_FUNCTION, &userEntry);
		userEntry.Store("");	// End of CPP user-function entries.

		UI_STORAGE_OBJECT compareEntry(*directory, "COMPARE_FUNCTION", 0, UIS_CREATE | UIS_READWRITE);
		for (object = (UI_WINDOW_OBJECT *)support.First(); object; object = object->Next())
			object->Information(PRINT_COMPARE_FUNCTION, &compareEntry);
		for (object = First(); object; object = object->Next())
			object->Information(PRINT_COMPARE_FUNCTION, &compareEntry);
		compareEntry.Store("");	// End of CPP compare-function entries.
	}
	if (tempDirectory)
	{
		directory->Save();
		delete directory;
	}
}
#endif

// ----- ERROR_HANDLING -----------------------------------------------------

void UIW_WINDOW::Error(UIS_STATUS status, const char *format, ...)
{
#ifdef ZINC_DEBUG
	char *message = new char[256];
	va_list arguments;
   	va_start(arguments, format);
	vsprintf(message, format, arguments);
	va_end(arguments);
#ifdef _WINDOWS
	MessageBox(0, message, NULL, MB_OK);
#else
	if (display)
		delete display;
	printf(message);
#endif
	abort();
#endif

	// Flag parents with the error.
	for (UI_WINDOW_OBJECT *object = this; object; object = object->parent)
		object->woStatus |= status;
}

